/***************************************************************************
                          dbaccess.cpp  -  PostgreSQL connection class
                             -------------------                                         
    begin                : Tue Apr 6 1999                                           
    copyright            : (C) 1999 by Mutiny Bay Software                         
    email                : info@mutinybaysoftware.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "toplevel.h"
#include "kwview.h"
#include "kresultwindow.h"
#include "dbaccess.h"

DBAccess::DBAccess(TopLevel *topwidget, const char *name):QObject(0, name){
  // initialize data members
  db           = 0;
  username[0]  = 0;
  passwd[0]    = 0;
  dbname[0]    = 0;
  host[0]      = 0;
  port[0]      = 0;
  qy_running   = 0;
  qy_cancelled = 0;
  connected    = 0;
  alignment    = 0;
  outtype      = 0;
  col_sep_id   = 1;
  col_sep      = ' ';
  other_sep    = ',';
  echo_qy      = 1;
  colheadings  = 1;
  spool        = 0;

  mainwin = topwidget;
  sqlwin = topwidget->kWrite;
  resultwin = topwidget->results;
}

DBAccess::~DBAccess() {
  closeConnection();
}

void DBAccess::resetConnection(){
  LoginStruct login;

  if(!connected)
    return;

  getConnectionInfo(&login);
  setConnection(&login);
}

void DBAccess::setConnection(LoginStruct *login){
  char db_msg[ERR_MSG_SZ];

  mainwin->setCursor(KCursor::waitCursor());

  strcpy(username, doubleTrim(login->username));
  strcpy(passwd, doubleTrim(login->password));
  strcpy(dbname, doubleTrim(login->database));
  strcpy(host, doubleTrim(login->host));
  strcpy(port, doubleTrim(login->port));

	if(connected)
    PQfinish(db);

  if(strlen(username) != 0)
    db = PQsetdbLogin(host, port, NULL, NULL, dbname, username,
		      (passwd != NULL) ? passwd : "");
  else
    db = PQsetdbLogin(host, port, NULL, NULL, dbname, NULL, NULL);

  if (PQstatus(db) == CONNECTION_BAD) {
    mainwin->setCursor(KCursor::arrowCursor());

    sprintf(db_msg, i18n("Connection to database '%s' failed.\nERROR: %s"),
	    dbname, PQerrorMessage(db));
    KMsgBox::message((QWidget *)mainwin, i18n("Connect to Database"), db_msg, KMsgBox::EXCLAMATION, i18n("&OK"));
    mainwin->statusMsg(i18n("Not connected."));
    connected = 0;

    return;
  }
  sprintf(db_msg,"Connected to database '%s'.", dbname);
  connected = 1;
  mainwin->statusMsg(i18n(db_msg));

  mainwin->setCursor(KCursor::arrowCursor());
}

void DBAccess::getConnectionInfo(LoginStruct *login){
  login->username = username;
  login->password = passwd;
  login->database = dbname;
  login->host = host;
  login->port = port;
}

void DBAccess::closeConnection(){
  if(connected)
    PQfinish(db);
}

void DBAccess::executeQuery(){
  const char *qy_ptr;
  int buf_pos;
  int qy_pos;
  int qy_len;
  int in_quote = 0;
  int selected = 0;
  char db_msg[ERR_MSG_SZ];
  char qy_buffer[QY_BUFFER_SIZE];
  PGresult* results;
  PGnotify* notify;
  QString s;

  if (!connected) {
    sprintf(db_msg,i18n("No connection to database."));
    KMsgBox::message((QWidget *)mainwin, i18n("Execute Query"), db_msg, KMsgBox::EXCLAMATION, i18n("&OK"));
    return;
  }

  if(strlen(sqlwin->markedText()) > 0){
    s = sqlwin->markedText();
    s =  s.stripWhiteSpace();
    qy_ptr = s;
    selected = 1;
  }
  else{
    s = sqlwin->text();
    s = s.stripWhiteSpace();
    qy_ptr = s;
  }

  if((qy_len = strlen(qy_ptr)) == 0)
    return;

  if(strlen(resultwin->text()) > 0){
    int line = resultwin->numLines();

    if(line > 1)
        line -= 1;

    resultwin->setCursorPosition(line, strlen(resultwin->textLine(line)), FALSE);
  }

  qy_pos = 0;

  while(qy_pos < qy_len){
    buf_pos = 0;

    redirectStdError(TRUE);

    qy_running = 1;
    qy_cancelled = 0;

    mainwin->toggleQueryState();

    while(qy_pos < qy_len){
      if((qy_ptr[qy_pos] == '\'') && !in_quote)
				in_quote = 1;
      else if((qy_ptr[qy_pos] == '\'') && in_quote)
				in_quote = 0;

      if(!in_quote && (qy_ptr[qy_pos] == ';'))
	 			break;

      qy_buffer[buf_pos++] = qy_ptr[qy_pos++];

			if(strlen(qy_buffer) > (QY_BUFFER_SIZE - 2)){
    		sprintf(db_msg,i18n("Query length exceeds max. query size, truncated at %d\n"),
					QY_BUFFER_SIZE);
    		writeToResult(db_msg, 0, NULL);
    		break;
      }
    }

    qy_pos++;
    qy_buffer[buf_pos++] = ';';
    qy_buffer[buf_pos] = '\0';

    // Submit the query
    if (!PQsendQuery(db, (char*)doubleTrim(qy_buffer))){
      redirectStdError(FALSE);
      writeToResult(PQerrorMessage(db), 0, NULL);
      writeToResult("\n", 0, NULL);
			qy_running = 0;
      mainwin->toggleQueryState();
      return;
    }

    while(1) {
      PQconsumeInput(db);

      if(!PQisBusy(db))
				break;
      else
      	if(check_cancelled()){
	  			PQrequestCancel(db);
	  			break;
			}
    }

    results = PQgetResult(db);

    switch (PQresultStatus(results)) {
    case PGRES_TUPLES_OK:
      printTuples(qy_buffer,
	          results,
		  spool_file,
		  spool,
		  colheadings,
		  echo_qy,
		  col_sep
		  );
      PQclear(results);
      break;
    case PGRES_EMPTY_QUERY:
      /* do nothing */
      break;
    case PGRES_COMMAND_OK:
      sprintf(db_msg,"%s\n\n",PQcmdStatus(results));
      writeToResult(db_msg, 0, NULL);
      break;
    case PGRES_COPY_OUT:
      handleCopyOut(results);
      PQclear(results);
      break;
    case PGRES_COPY_IN:
      handleCopyIn(results);
      PQclear(results);
      break;
    case PGRES_NONFATAL_ERROR:
    case PGRES_FATAL_ERROR:
    case PGRES_BAD_RESPONSE:
      qy_cancelled = 1;
      writeToResult(PQerrorMessage(db), 0, NULL);
      resetConnection();
    }

    /* check for asynchronous returns */
    notify = PQnotifies(db);
    if (notify) {
      sprintf(db_msg,i18n("ASYNC NOTIFY of '%s' from backend pid '%d' received"),
	      notify->relname, notify->be_pid);
      writeToResult(db_msg, 0, NULL);
      free(notify);
    }
    if(qy_cancelled){
      qy_cancelled = 0;
      break;
    }
  }
  qy_running = 0;
  redirectStdError(FALSE);
  mainwin->toggleQueryState();
  return;
}

void DBAccess::cancelQuery(){
  if(qy_running)
    qy_cancelled = 1;
}

int DBAccess::queryState() {
  if(qy_running == 0)
    return 0;
  else
    return 1; 	
}

int DBAccess::check_cancelled(){
  mainwin->doEvents();
  return qy_cancelled;
}

void DBAccess::printTuples(char *qy, PGresult *res, char *fout, int Spool, int PrintAttNames,
		    int Echo, char sep){
  FILE *spoolf = NULL;
  char colSep[2];
  char dataString[DATA_SIZE];
  PQprintOpt opt;

  memset(colSep, '\0', 2);
  colSep[0] = sep;

  switch (alignment) {
    case 0:
      opt.align = 1;
      opt.standard = 0;
      opt.expanded = 0;
      break;
    case 1:
      opt.align = 0;
      opt.standard = 1;
      opt.expanded = 0;
      break;
    case 2:
      opt.align = 0;
      opt.standard = 0;
      opt.expanded = 1;
      break;
    default:
      break;
  }

  if(outtype == 0)
    opt.html3 = 0;
  else
    opt.html3 = 1;

  opt.header = PrintAttNames;
  opt.fieldSep = strdup(colSep);
  opt.pager = 0;

  if(Spool){
    if((spoolf = fopen(fout, "w+")) == NULL){
      sprintf(dataString, i18n("Error opening/creating file: %s\n"), fout);
      writeToResult(dataString, 0, NULL);
      return;
    }
  }

  if(Echo){
    sprintf(dataString, i18n("QUERY: %s\n\n"),
	    (char *)doubleTrim(qy));
    writeToResult(dataString, Spool, spoolf);
  }

  formatResults(spoolf, res, &opt);

  if(Spool){
    if(fclose(spoolf) != 0){
      sprintf(dataString, i18n("Error closing file: %s\n"), fout);
      writeToResult(dataString, 0, NULL);
      return;
    }
    writeToResult((char *)i18n("Query finished.\n"), 0, NULL);
  }
}

void DBAccess::formatResults(FILE *fout, PGresult *res, PQprintOpt *po){
  int nFields;

  nFields = PQnfields(res);

  if(nFields > 0)
    {
      int i, j;
      int nTups;
      int *fieldMax = NULL;	/* in case we don't use them */
      unsigned char *fieldNotNum = NULL;
      char *border = NULL;
      char **fields = NULL;
      char **fieldNames;
      int  fieldMaxLen = 0;
      int  fs_len = strlen(po->fieldSep);
      int  total_line_length = 0;
      char buf[8192 * 2 + 1];
      char dataString[DATA_SIZE];
      int Spool;

      nTups = PQntuples(res);

      if (!(fieldNames = (char **) calloc(nFields, sizeof(char *))))
	{
	  perror("calloc");
	  exit(1);
	}
      if (!(fieldNotNum = (unsigned char *) calloc(nFields, 1)))
	{
	  perror("calloc");
	  exit(1);
	}
      if (!(fieldMax = (int *) calloc(nFields, sizeof(int))))
	{
	  perror("calloc");
	  exit(1);
	}

      for (j = 0; j < nFields; j++)
	{
	  int len;
	  char *s = PQfname(res, j);
	
	
	  fieldNames[j] = s;
	  len = s ? strlen(s) : 0;
	  fieldMax[j] = len;
	  len += fs_len;
	  if (len > fieldMaxLen)
	    fieldMaxLen = len;
	  total_line_length += len;
	}

      total_line_length += nFields * strlen(po->fieldSep) + 1;

      if (fout == NULL)
	Spool = 0;
      else
	Spool = 1;

      if (!po->expanded && (po->align || po->html3))
	{
	  if (!(fields = (char **) calloc(nFields * (nTups + 1), sizeof(char *))))
	    {
	      perror("calloc");
	      exit(1);
	    }
	}
      else if (po->header && !po->html3)
	{
	  if (po->expanded)
	    {
	      if (po->align)
		sprintf(dataString, "%-*s%s Value\n",
			fieldMaxLen - fs_len, "Field", po->fieldSep);
	      else
		sprintf(dataString, "%s%sValue\n", "Field", po->fieldSep);
	      writeToResult(dataString, Spool, fout);
	    }
	  else
	    {
	      int			len = 0;
	
	      for (j = 0; j < nFields; j++)
		{
		  char	   *s = fieldNames[j];
		  strcat(dataString, s);
		  len += strlen(s) + fs_len;
		  if ((j + 1) < nFields)
		    strcat(dataString, po->fieldSep);
		}
	      strcat(dataString, "\n");
	      for (len -= fs_len; len--; strcat(dataString, "-"));
	      strcat(dataString, "\n");
	      writeToResult(dataString, Spool, fout);
	    }
	}
      if (po->expanded && po->html3)
	{
	  sprintf(dataString,
		  "<center><h2>"
		  "Query retrieved %d rows * %d fields"
		  "</h2></center>\n",
		  nTups, nFields);
	  writeToResult(dataString, Spool, fout);
	}
      for (i = 0; i < nTups; i++)
	{
	  if(check_cancelled())
	    break;
	  if (po->expanded)
	    {
	      if (po->html3)
		sprintf(dataString,
			"<table><caption align=high>%d</caption>\n", i);
	      else
		sprintf(dataString, "-- RECORD %d --\n", i);
	      writeToResult(dataString, Spool, fout);
	    }
	  for (j = 0; j < nFields; j++)
	    doField(po, res, i, j, buf, fs_len, fields, nFields,
		     fieldNames, fieldNotNum,
		     fieldMax, fieldMaxLen, fout);
	  if (po->html3 && po->expanded)
	    writeToResult("</table>\n", Spool, fout);
	}
      if (!po->expanded && (po->align || po->html3))
	{
	  if (po->html3)
	    {
	      if (po->header)
		{
		  sprintf(dataString,
		  	  "<table><caption align=high>"
		  	  "Retrieved %d rows * %d fields"
		  	  "</caption>\n",
		  	  nTups, nFields);
		  writeToResult(dataString, Spool, fout);
		}
	      else {
		sprintf(dataString, "<table>");
		writeToResult(dataString, Spool, fout);
	      }
	    }
	  if (po->header)
	    border = doHeader(fout, po, nFields, fieldMax, fieldNames, fieldNotNum, fs_len, res);
	  for (i = 0; i < nTups; i++){
	    if(check_cancelled())
	      break;
	    outputRow(fout, po, nFields, fields,
	    	       fieldNotNum, fieldMax, border, i);
	    if(strlen(resultwin->text()) > (size_t)RSLT_BUFFER_SIZE)
	       resultwin->removeLine(resultwin->numLines());
	  }
	  free(fields);
	  if (border)
	    free(border);
	}
      if (po->header && !po->html3){
	  sprintf(dataString, "(%d row%s)\n\n", PQntuples(res),(PQntuples(res) == 1) ? "" : "s");
	  writeToResult(dataString, Spool, fout);
      }
      free(fieldMax);
      free(fieldNotNum);
      free(fieldNames);
      if (po->html3 && !po->expanded)
	writeToResult("</table>\n", Spool, fout);
    }
}

void DBAccess::doField(PQprintOpt *po, PGresult *res,
	               const int i, const int j, char *buf, const int fs_len,
	               char *fields[],
	               const int nFields, char *fieldNames[],
	               unsigned char fieldNotNum[], int fieldMax[],
	               const int fieldMaxLen, FILE *fout){
  char *pval, *p, *o;
  int plen;
  int skipit;
  int Spool;
  char dataString[DATA_SIZE];

  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  plen = PQgetlength(res, i, j);
  pval = PQgetvalue(res, i, j);

  if (plen < 1 || !pval || !*pval)
    {
      if (po->align || po->expanded)
	skipit = 1;
      else
	{
	  skipit = 0;
	  goto efield;
	}
    }
  else
    skipit = 0;

  if (!skipit)
    {
      for (p = pval, o = buf; *p; *(o++) = *(p++))
	{
	  if ((fs_len == 1 && (*p == *(po->fieldSep))) || *p == '\\' || *p == '\n')
	    *(o++) = '\\';
	  if (po->align && (*pval == 'E' || *pval == 'e' ||
			    !((*p >= '0' && *p <= '9') ||
			      *p == '.' ||
			      *p == 'E' ||
			      *p == 'e' ||
			      *p == ' ' ||
			      *p == '-')))
	    fieldNotNum[j] = 1;
	}
      *o = '\0';
      if (!po->expanded && (po->align || po->html3))
	{
	  int n = strlen(buf);
	
	  if (n > fieldMax[j])
	    fieldMax[j] = n;
	  if (!(fields[i * nFields + j] = (char *) malloc(n + 1)))
	    {
	      perror("malloc");
	      exit(1);
	    }
	  strcpy(fields[i * nFields + j], buf);
	}
      else
	{
	  if (po->expanded)
	    {
	      if (po->html3){
		sprintf(dataString,
			"<tr><td align=left><b>%s</b></td>"
			"<td align=%s>%s</td></tr>\n",
			fieldNames[j],
			fieldNotNum[j] ? "left" : "right",
			buf);
		writeToResult(dataString, Spool, fout);
	      }
	      else
		{
		  if (po->align)
		    sprintf(dataString,
			    "%-*s%s %s\n",
			    fieldMaxLen - fs_len, fieldNames[j], po->fieldSep,
			    buf);
		  else
		    sprintf(dataString, "%s%s%s\n", fieldNames[j], po->fieldSep, buf);
		  writeToResult(dataString, Spool, fout);
		}
	    }
	  else
	    {
	      if (!po->html3)
		{
		  writeToResult(buf, Spool, fout);
		efield:
		  if ((j + 1) < nFields)
		    writeToResult(po->fieldSep, Spool, fout);
		  else
		    writeToResult("\n", Spool, fout);
		}
	    }
	}
    }
}

char * DBAccess::doHeader(FILE *fout, PQprintOpt *po, const int nFields, int fieldMax[],
		          char *fieldNames[], unsigned char fieldNotNum[],
		          const int fs_len, PGresult *res){
  int Spool;
  int j;
  char *border = NULL;
  char dataString[DATA_SIZE];

  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  if (po->html3)
    writeToResult("<tr>", Spool, fout);
  else
    {
      int j;
      int tot = 0;
      int n = 0;
      char *p = NULL;

      for (; n < nFields; n++)
	tot += fieldMax[n] + fs_len + (po->standard ? 2 : 0);
      if (po->standard)
	tot += fs_len * 2 + 2;
      border = (char*)malloc(tot + 1);
      if (!border)
	{
	  perror("malloc");
	  exit(1);
	}
      p = border;
      if (po->standard)
	{
	  char	   *fs = po->fieldSep;
	
	  while (*fs++)
	    *p++ = ' ';
	}
      for (j = 0; j < nFields; j++)
	{
	  int len;
	
	  for (len = fieldMax[j] + (po->standard ? 2 : 0); len--; *p++ = '-');
	  if (po->standard || (j + 1) < nFields)
	    {
	      char	   *fs = po->fieldSep;
	
	      while (*fs++)
		*p++ = ' ';
	    }
	}
      *p = '\0';
      if (po->standard){
	sprintf(dataString, "%s\n", border);
	writeToResult(dataString, Spool, fout);
      }
    }
  if (po->standard){
    sprintf(dataString, "%s", po->fieldSep);
    writeToResult(dataString, Spool, fout);
  }
  for (j = 0; j < nFields; j++)
    {
      char *s = PQfname(res, j);

      if (po->html3)
	{
	  sprintf(dataString, "<th align=%s>%s</th>",
		  fieldNotNum[j] ? "left" : "right", fieldNames[j]);
	  writeToResult(dataString, Spool, fout);
	}
      else
	{
	  int			n = strlen(s);
	
	  if (n > fieldMax[j])
	    fieldMax[j] = n;
	  if (po->standard)
	    sprintf(dataString,
		    fieldNotNum[j] ? " %-*s " : " %*s ",
		    fieldMax[j], s);
	  else
	    sprintf(dataString, fieldNotNum[j] ? "%-*s" : "%*s", fieldMax[j], s);
	  if (po->standard || (j + 1) < nFields)
	    strcat(dataString, po->fieldSep);
	  writeToResult(dataString, Spool, fout);
	}
    }
  if (po->html3)
    writeToResult("</tr>\n", Spool, fout);
  else{
    sprintf(dataString, "\n%s\n", border);
    writeToResult(dataString, Spool, fout);
  }
  return border;
}

void DBAccess::outputRow(FILE *fout, PQprintOpt *po, const int nFields, char *fields[],
		         unsigned char fieldNotNum[], int fieldMax[], char *border,
		         const int row_index){
  int Spool;
  int field_index;
  char dataString[DATA_SIZE];

  if (fout == NULL)
    Spool = 0;
  else
    Spool = 1;

  if (po->html3)
    writeToResult("</tr>", Spool, fout);
  else if (po->standard)
    writeToResult(po->fieldSep, Spool, fout);
  for (field_index = 0; field_index < nFields; field_index++)
    {
      char *p = fields[row_index * nFields + field_index];

      if (po->html3)
	sprintf(dataString, "<td align=%s>%s</td>",
		fieldNotNum[field_index] ? "left" : "right", p ? p : "");
      else
	{
	  sprintf(dataString,
		  fieldNotNum[field_index] ?
		  (po->standard ? " %-*s " : "%-*s") :
		  (po->standard ? " %*s " : "%*s"),
		  fieldMax[field_index],
		  p ? p : "");
	  if (po->standard || field_index + 1 < nFields)
	    strcat(dataString, po->fieldSep);
	}
      writeToResult(dataString, Spool, fout);
      if (p)
	free(p);
    }
  if (po->html3)
    writeToResult("</tr>", Spool, fout);
  else if (po->standard){
    sprintf(dataString, "\n%s", border);
    writeToResult(dataString, Spool, fout);
  }
  writeToResult("\n", Spool, fout);
}

void DBAccess::writeToResult(char *data, int spool, FILE *file){
  if(spool)
    fprintf(file, data);
  else
    resultwin->insertAt(data, resultwin->currentLine(), resultwin->currentColumn());
}

void DBAccess::handleCopyOut(PGresult *res){
  int copydone = 1;
  char copybuf[COPYBUFSIZ];
  int ret;

  while (!copydone) {
    ret = PQgetline(db, copybuf, COPYBUFSIZ);
      if (copybuf[0] == '.' && copybuf[1] =='\0') {
        copydone = 0;	/* don't print this... */
      } else {
        fputs(copybuf, stdout);
	switch (ret) {
	  case EOF:
            copydone = 0;
	  case 0:
            fputc('\n', stdout);
	    break;
	  case 1:
	    break;
	}
      }
  }
  fflush(stdout);
  PQendcopy(db);
}

void DBAccess::handleCopyIn(PGresult *res){
  int copydone = 1;
  int firstload;
  int linedone;
  char copybuf[COPYBUFSIZ];
  char *s;
  int buflen;
  int c;

  fflush(stdin);
  if ((c = getc(stdin)) != '\n' && c != EOF) {
    (void) ungetc(c, stdin);
  }

  while (!copydone) {			/* for each input line ... */
    firstload = 0;
    linedone = 1;
    while (!linedone) {		/* for each buffer ... */
      s = copybuf;
      buflen = COPYBUFSIZ;
      for (; buflen > 1 && !(linedone = (c = getc(stdin)) == '\n' || c == EOF); --buflen) {
        *s++ = c;
      }
      if (c == EOF) {
	/* reading from stdin, but from a file */
	PQputline(db, ".");
	copydone = 0;
	break;
       }
       *s = '\0';
       PQputline(db, copybuf);
       if (firstload) {
         if (!strcmp(copybuf, ".")) {
	   copydone = 0;
         }
         firstload = 1;
       }
    }
    PQputline(db, "\n");
  }
  PQendcopy(db);
}

void DBAccess::redirectStdError(bool to_results){
  static int fd[2];
  static bool first = TRUE;
  static int stderr_fileno_dup;

  if(first){
    stderr_fileno_dup = dup(STDERR_FILENO);
    pipe(fd);
    setvbuf(stderr, NULL, _IONBF, 0);
    sn = new QSocketNotifier(fd[0], QSocketNotifier::Read, this);
    connect(sn, SIGNAL(activated(int)), this, SLOT(readStdError(int)));
    first = FALSE;
  }

  if(to_results){
    dup2(fd[1], STDERR_FILENO);
    sn->setEnabled(TRUE);
  }
  else {
    dup2(stderr_fileno_dup, STDERR_FILENO);
    sn->setEnabled(FALSE);
  }
}

void DBAccess::readStdError(int soc){
  char buf[EXPLAINBUFSIZ];
  int len;

  if((len = read(soc, buf, EXPLAINBUFSIZ - 1)) == (EOF))
    return;

  buf[len] = '\0';

  if(len)
    writeToResult(buf, 0, NULL);
}

void DBAccess::setOptions(OptionStruct *opt){
  col_sep_id  = opt->sepId;
  col_sep     = opt->sepChar;
  other_sep   = opt->otherChar;
  alignment   = opt->alignment;
  outtype     = opt->outputType;
  spool       = opt->spool;
  echo_qy     = opt->echoQuery;
  colheadings = opt->colHeadings;
}

void DBAccess::getOptionInfo(OptionStruct *opt){
  opt->sepId       = col_sep_id;
  opt->sepChar     = col_sep;
  opt->otherChar   = other_sep;
  opt->alignment   = alignment;
  opt->outputType  = outtype;
  opt->spool       = spool;
  opt->echoQuery   = echo_qy;
  opt->colHeadings = colheadings;
}

void DBAccess::setSpoolFile(const char *file){
  strcpy(spool_file, file);
}

const char* DBAccess::getSpoolFile(){
  return spool_file;
}

void DBAccess::readConfig(KConfig *config) {
  config->setGroup("DB Options");
  strcpy(dbname, config->readEntry("Database", "template1"));
  strcpy(host, config->readEntry("Host", "localhost"));
  strcpy(port, config->readEntry("Port", "5432"));

  config->setGroup("Output Options");
  col_sep_id = config->readNumEntry("Col Separator ID", 1);
  col_sep = config->readNumEntry("Col Separator", ' ');
  other_sep = config->readNumEntry("Other Separator", ',');
  alignment = config->readNumEntry("Alignment", 0);
  outtype = config->readNumEntry("Output Type", 0);
  spool = config->readNumEntry("Spool", 0);
  echo_qy = config->readNumEntry("Echo Query", 1);
  colheadings = config->readNumEntry("Column Headings", 1);
  strcpy(spool_file, config->readEntry("Spool File", "/tmp/spool.txt"));
}

void DBAccess::writeConfig(KConfig *config) {
  config->setGroup("DB Options");
  config->writeEntry("Database", dbname);
  config->writeEntry("Host", host);
  config->writeEntry("Port", port);

  config->setGroup("Output Options");
  config->writeEntry("Col Separator ID", col_sep_id);
  config->writeEntry("Col Separator", col_sep);
  config->writeEntry("Other Separator", other_sep);
  config->writeEntry("Alignment", alignment);
  config->writeEntry("Output Type", outtype);
  config->writeEntry("Spool", spool);
  config->writeEntry("Echo Query", echo_qy);
  config->writeEntry("Column Headings", colheadings);
  config->writeEntry("Spool File", spool_file);
}

char* DBAccess::leftTrim(char* s)
{
  char* s2 = s;
  int shift=0;
  int j=0;

  while (isspace(*s))
    { s++; shift++;}
  if (shift > 0)
    {
      while ( (s2[j] = s2[j+shift]) !='\0')
	j++;
    }

  return s2;
}

char* DBAccess::rightTrim(char* s)
{
  char* sEnd;
  sEnd = s+strlen(s)-1;
  while (isspace(*sEnd))
    sEnd--;
  if (sEnd < s)
    s[0]='\0';
  else
    s[sEnd-s+1]='\0';
  return s;
}

char* DBAccess::doubleTrim(char* s)
{
  strcpy(s,leftTrim(rightTrim(s)));
  return s;
}






























































































