//-< STORAGE.H >-----------------------------------------------------*--------*
// POST++                     Version 1.0        (c) 1998  GARRET    *     ?  *
// (Persistent Object Storage)                                       *   /\|  *
//                                                                   *  /  \  *
//                          Created:      2-Feb-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 21-Jan-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Storage interface
//-------------------------------------------------------------------*--------*

#ifndef __STORAGE_H__
#define __STORAGE_H__

#define POST_VERSION 119 // 1.19

#include "file.h"

const size_t defaultMaxDatabaseSize = 8*1024*1024;

#if defined(_WIN32)
    class critical_section { 
	CRITICAL_SECTION* cs;
      public:
	critical_section() { EnterCriticalSection(cs); }
	~critical_section() { LeaveCriticalSection(cs); }
	critical_section(CRITICAL_SECTION* cs) { this->cs = cs; }
    };
    #define CS(storage) critical_section __cs(&(storage)->cs)
#elif defined(_REENTRANT)
    #include <pthread.h>
    class critical_section { 
	pthread_mutex_t* cs;
      public:
	critical_section() { pthread_mutex_lock(cs); }
	~critical_section() { pthread_mutex_unlock(cs); }
	critical_section(pthread_mutex_t* cs) { this->cs = cs; }
    };
    #define CS(storage) critical_section __cs(&(storage)->cs)
#else
    #define CS(storage) 
#endif

class object;

class class_descriptor;

struct object_header { 
    enum { 
	free_object   = 0,
	metaclass_cid = 1,
	page_map_cid  = 2,
	first_cid     = 3,
	gc_marked     = 0x80000000 
    };
    int      cid;
    unsigned size;
};  

struct storage_free_block : public object_header { 
    storage_free_block* next;
};       

class storage_class_descriptor;

class storage_page_map : public object_header { 
  public: 
    //
    // All pages except first occupied by large object are marked with non-zero
    // bit in this bitmap. Bits correspond to all other pages are zero.
    //
    char bits[1];
};

struct storage_header : public file_header { 
    object* root_object;

    size_t  n_classes;
    storage_class_descriptor* class_dictionary;

    int     version;

    storage_page_map* page_map;

    storage_free_block* free_page_chain;
    storage_free_block* free_block_chain[13];
    char    timestamp[32];

    void    adjust_references(long shift);
};

struct gc_segment { 
    object** ptr;
    size_t   len;
};

class storage { 
    friend class object;
  public:
    static long base_address_shift;
    static gc_segment* gc_stack;

    enum open_flags { 
	use_transaction_log       = 0x01, // apply all changes as transactions
	no_file_mapping           = 0x02, // do not use file mapping mechanism
	fault_tolerant            = 0x04, // preserve storage consistency
	read_only                 = 0x08, // read only file view
	support_virtual_functions = 0x10, // storage objects contain virt.func.
	do_garbage_collection     = 0x20  // perform GC on loading
    };
    virtual bool open(int flags = 0);
    //
    // Commit current transaction: save all modified pages in data file and
    // truncate transaction log. This function works only in 
    // in transaction mode (when flag use_transaction_log is set).
    // Commit is implicitly called by close().
    //
    virtual void commit(); 
    //
    // Rollback current transaction. This function works only in 
    // in transaction mode (when flag use_transaction_log is set).
    //
    virtual void rollback(); 
    //
    // Flush all changes on disk. 
    // Behavior of this function depends on storage mode:
    // 1. In transaction mode this function is equivalent to commit(). 
    // 2. In other modes this functions performs atomic update of storage file 
    //    by flushing data to temporary file and then renaming it to original 
    //    one. All modified objects will not be written in data file until 
    //    flush().
    //
    virtual void flush(); 

    //
    // Close storage. To store modified objects you should first call flush(),
    // otherwise all changes will be lost.
    //
    virtual void close();
    //
    // Free all objects unreachable from the storage root object.
    // This function will be called from open() if 'do_garbage_collection'
    // attribute is specified. It is also possible to call this function 
    // explicitly, but you should be sure that no program variable points 
    // to object unreachable from the storage root. 
    // Method returns number of deallocated objects. Even if you are using
    // explicit memory deallocation you can call this method to search for
    // memory leak and check reference consisteny.
    //
    virtual int  garbage_collection();

    //
    // Report error in storage method "func". Applciation can override this
    // method to perform application dependent error handling.
    // Default implementation of this method out put message and abort 
    // application
    // 
    virtual void handle_error(const char* func);

    void set_root_object(object* obj) { 
	hdr->root_object = obj; 
    }

    object* get_root_object() const { 
	return hdr->root_object; 
    }

    void set_version(int version) { 
	hdr->version = version; 
    }

    int get_version() const { 
	return hdr->version; 
    }

    static storage* find_storage(object const* obj) { 
	for (storage* sp = chain; sp != NULL; sp = sp->next) { 
	    if (size_t((char*)obj - (char*)sp->hdr) < sp->hdr->file_size) { 
		return sp;
	    }
	} 
	return NULL;
    }

    class_descriptor* get_object_class(int cid) const { 
	return dbs_class_mapping[cid];
    }

    bool has_object(object const* obj) const { 
	return size_t((char*)obj - (char*)hdr) < hdr->file_size;
    }

    char* get_error_text(char* buf, size_t buf_size);

    object_header* get_header(object const* obj);

    //
    // Allocate object in storage.
    //
    void* allocate(int app_class_id, size_t size) { 
	CS(this);
	int cid = app_class_mapping[app_class_id];
	assert(cid != 0);
	if (size + sizeof(object_header) <= page_size/2) { 
	    int n = block_chain[((size + sizeof(object_header) + 7) >> 3) - 1];
	    storage_free_block* bp = hdr->free_block_chain[n];
	    if (bp != NULL) { 
		hdr->free_block_chain[n] = bp->next;
		bp->size = size;
		bp->cid = cid;
		return (object_header*)bp+1;
	    }
	}
	return alloc_page(cid, size);
    }

    //
    // Explicit deallocation of object in storage.
    //
    void  free(object* obj) { 
	CS(this);
	object_header* hp = get_header(obj);
	if (hp->size + sizeof(object_header) <= page_size/2) {
	    int n = block_chain[((hp->size+sizeof(object_header)+7) >> 3) - 1];
	    storage_free_block* bp = (storage_free_block*)hp;
	    bp->cid = object_header::free_object;
	    bp->next = hdr->free_block_chain[n];
	    hdr->free_block_chain[n] = bp;
	} else { 
	    free_page(hp);
	}
    }

    //
    // Parameter max_file_size specifies maximal data file extension for 
    // this storage.
    // Parameter max_locked_pages is used only in transaction mode and 
    // specifies number of pages which can be locked in memory to provide
    // buffering sadow pages writes to the log file. For Windows-NT number
    // of locked pages by default should not be greater then 30. 
    // If larger number of pages is specified, POST++ will try to extend 
    // process working set.
    //
    storage(const char* name, 
	    size_t max_file_size = defaultMaxDatabaseSize,
	    size_t max_locked_pages = 30)
    : data_file(name, max_file_size, max_locked_pages) 
    {
#if defined(_WIN32)
	InitializeCriticalSection(&cs);
#elif defined(_REENTRANT)
	pthread_mutex_init(&cs, NULL);
#endif
    }

    virtual~storage();

  protected:
    file data_file;
    storage_header* hdr;
    storage_free_block* curr_free_page;

    storage* next;
    static storage* chain; // chain of opened storages

    class_descriptor** dbs_class_mapping;
    int*               app_class_mapping;
    
#if defined(_WIN32)
    CRITICAL_SECTION cs;
#elif defined(_REENTRANT)
    pthread_mutex_t cs;
#endif

    enum { 
	page_size = 512,        // bytes
	init_page_map_size = 8  // pages
    };

    static const int block_size[page_size/16];
    static const int block_chain[page_size/16];

    void* alloc_page(int cid, size_t size);
    void  free_page(object_header* hp);

    virtual void adjust_references();
    virtual void load_class_dictionary();
    virtual bool update_class_dictionary();

    void  call_constructor(object_header* hp); 
};

#endif
