/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include <ctype.h>
#include <signal.h>
#include <getopt.h>

#include "kdbs.h"
#include "vacuum.h"
#include "sm.h"
#include "net.h"
#include "lock.h"

Database db;
unsigned short port = 0;	/* the port */
char *host = NULL;		/* the host */
char *usrname = "root";

int connectid = 0;		/* the connection id */
char *connkey = NULL;		/* the connection key */
char *client_usrname = NULL;	/* the client user name */
Uid clientuid = 0;		/* die User ID des clients */
Gid clientgid = 0;		/* die (default) Group ID des clients */
int timeoutval = 30;		/* 30 seconds timeout */

/*-----------------------------------
  Das eigentlich Hauptprogramm
------------------------------------*/
void showhelp (void);
void showversion (void);
void showerror (int errnr);
void Welcome (void);

#define HELP_OPT       0
#define USER_OPT       1
#define BASE_OPT       2
#define SAVEDEL_OPT    3
#define FID_OPT        4
#define HOST_OPT       5
#define PORT_OPT       6
#define VERSION_OPT    7
#define CONF_OPT       8

void
main(int argc, char **argv)
{
  int c, i;
  char 
    *dbname = NULL,
    *usrname = NULL,
    *filename = NULL,
    *portstring = NULL;
  char *configfilename = NULL;
  int deldeleted = 1;
  int sock, retval;
  int tmpsize = 1024;
  char *tmp = (char*) malloc (tmpsize);
  char resstr[MAXMSG];
  int fno;
  VOretval *voretval;

  Welcome ();
  
  opterr = 0;
  
  while (1) {
    static struct option long_options[] = {
      {"help", 0, 0, HELP_OPT},
      {"user", 1, 0, USER_OPT},
      {"base", 1, 0, BASE_OPT},
      {"savedel", 0, 0, SAVEDEL_OPT},
      {"file", 1, 0, FID_OPT},
      {"port", 1, 0, PORT_OPT},
      {"host", 1, 0, HOST_OPT},
      {"version", 0, 0, VERSION_OPT},
      {"conffile", 1, 0, CONF_OPT},
      {0, 0, 0, 0}
    };
    int option_index = 0;
    
    c = getopt_long (argc, argv, "hu:b:sf:p:a:d:", 
		     long_options, &option_index);
    
    if (c == -1)
      break;
    
    switch (c) {
    case HELP_OPT:
    case 'h':
      showhelp();
      exit (0);
    case PORT_OPT:
    case 'p':			/* port to address the daemon */
      portstring = optarg;
      port = atoi (portstring);
      break;
    case HOST_OPT:
    case 'a':			/* host to connect to */
      host = optarg;
      break;
    case USER_OPT:
    case 'u':
      usrname = optarg;
      break;
    case BASE_OPT:
    case 'b':
      dbname = optarg;
      break;
    case SAVEDEL_OPT:
    case 's':
      deldeleted = 0;
      break;
    case FID_OPT:
    case 'd':
      filename = optarg;
      break;
    case CONF_OPT:
    case 'f':
      configfilename = optarg;
      break;
    case VERSION_OPT:
      showversion ();
      exit (EXIT_SUCCESS);
    case '?':
      exit (EXIT_FAILURE);
    default:
      exit (EXIT_FAILURE);
    }
  }

  if (!dbname) {
    perror ("No database specified");
    exit (EXIT_FAILURE);
  }

  /* read global configuration file */
  read_configuration (configfilename);

  sock = make_connection (&port, &host);
  if (sock < 0)
    VC_GOERR ("Connect error (vacuum)");
  printf ("Connected to daemon (%s:%d)\n", host, port);
  
  if (make_login (sock, KCP_SUPERVISOR, usrname, NULL, &tmp, &tmpsize) < 0)
    goto errhd;
  if (parse_login_data (tmp, &connectid, &connkey, &client_usrname, 
			&clientuid, &clientgid) < 0)
    goto errhd;
  
  printf ("Username %s UID: %d GID: %d\n",
	  client_usrname, clientuid, clientgid);
  
  
  /* ----------------------------------------------------------------------
     at first lock the database 
     ---------------------------------------------------------------------- */
  retval = verify_database (sock, connectid, connkey, clientuid, 
			    dbname, &voretval);
  switch (retval) {
  case -1:
    VC_GOERR ("no access to database.  Some error occured");
  case 0:
    VC_GOERR ("permission denied");
  }

  resstrcpy (resstr, lock_LOCKED, lock_DATABASE, voretval->dbid, 0);
  retval = getresources (sock, connectid, connkey, 
			 KCP_BEGINTRANS, 0, resstr, timeoutval);
  switch (retval) {
  case -1:
    VC_GOERR ("no access to database.  Some error occured");
  case -2:
    VC_GOERR ("timeout");
  case -3:
    VC_GOERR ("explizit halt request by daemon");
  case 0:
    VC_GOERR ("permission denied");
  }

  /* ----------------------------------------------------------------------
     now open the database 
     ---------------------------------------------------------------------- */
  printf ("Vacuum the database: %s\n", voretval->path); 
  if (mountdb (&db, voretval->path) < 0) {
    PRINTERR ("Can't mount database");
    exit (EXIT_FAILURE);
  }
  db.dbid = voretval->dbid;
  db.asowner = voretval->dbowner;
  
  printf ("Database: %s\n", db.basename);
  printf ("Path:     %s\n", db.path);
  printf ("DBID:     %d\n", db.dbid);
  if (db.asowner)
    printf ("You are the owner of this database!\n");
  fno = get_datafileno (&db);
  printf ("No. of datafiles: %i\n", fno);
  for (i = 0; i < fno ; i++) {
    printf ("%i. Datafile name: %s\n", i, get_datafile_name (&db, i));
  }
  fno = get_indexfileno (&db);
  printf ("No. of indexfiles: %i\n", fno);
  for (i = 0; i < fno ; i++) {
    printf ("%i. Index name: %s\n", i, get_index_name (&db, i));
  }
  
  if (opendb (&db) < 0) {
    PRINTERR ("Can't open the database.");
    exit (EXIT_FAILURE);
  }
  
  /* ----------------------------------------------------------------------
     ok.  Now we may compact it. 
     ---------------------------------------------------------------------- */
  printf (">VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! \n");

  vacuum (&db, deldeleted, filename);

  printf ("<VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! VACUUM! \n");

  /* ----------------------------------------------------------------------
     and at least: UNLOCK the database 
     ---------------------------------------------------------------------- */
  closedb (&db);
  retval = freeresources (sock, connectid, connkey, 0, timeoutval);
  switch (retval) {
  case -1:
    VC_GOERR ("no access to database.  Some error occured");
  case -2:
    VC_GOERR ("timeout");
  case -3:
    VC_GOERR ("explizit halt request by daemon");
  case 0:
    VC_GOERR ("permission denied");
  }
  
  exit (0);
  
errhd:
  PRINTERR(errstr);
  exit (1);
}

/* -----------------------------------------------------
   Kurzhilfe & Fehlerausgabe
--------------------------------------------------------*/

void
Welcome (void)
{
  printf("
vacuum - tool to cleanup databases
This file is part of the kaenguru database system
Copyright (c) %s by Gregor Klinke

", copyrightyear);
}

void
showversion (void)
{
  fprintf (stdout, "

This is Kaenguru %s - The Kaenguru Database System (vacuum)
Copyright (c) %s by Gregor Klinke

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.
  
This program ist distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public Lincense for more details.

", version, copyrightyear);

}

void
showhelp (void)
{
  printf("\
vacuum is a tool to clean up database in the kaenguru database system.  It 
takes one specific (or all) datafiles from a databases and compacts the 
saved item.  After a vacuum run there are no superflous space left in the
datafiles, and the Openlistfiles are zerolength again.

NOTE: Don't trust vacuum's security!  Make backups before you run it!
Since Vacuums requests a total database lock before running it may take
some time before vacuums starts to act.

General usage:
  creatdb [-h] [-u<user>] [-b<basename>] [-f<file>] [-s]

Options:

  -h, --help       This help

  -u, --user=NAME   (Not implemented yet)

  -b, --base=NAME  The database to be cleaned up.  If no name is given
                   vacuum tries the environment variable KAENBASE.  Is
                   this is also undefined it takes the user login name.

  -f, --file=FILE  Specifies the datafile to be compacted. NIY  By default 
                   the complete database is compacted.

  -s, --savedel    By default vacuum kills all saved items which are marked
                   deleted.  The option -s surpress this feature (i.e. 
                   deleted objects may be restore with undelete).

      --version    print out version and copyright.

"
	 );    
}





