/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <getopt.h>
#include <locale.h>

#include "kdbs.h"
#include "net.h"
#include "utils.h"

int exit_status = 0;
unsigned short port;		/* the port */
char *host = NULL;		/* the host */
char *usrname = "root";

int connectid = 0;		/* the connection id */
char *connkey = NULL;		/* the connection key */
char *client_usrname = NULL;	/* the client user name */
Uid clientuid = 0;		/* die User ID des clients */
Gid clientgid = 0;		/* die (default) Group ID des clients */
int timeoutval = 30;		/* 30 seconds timeout */

/* prototypes */
void showhelp (void);void showversion (void);
void Welcome();

/* ----------------------------------------------------------------------
   main and startup routines 
   ---------------------------------------------------------------------- */
#define NONE_FUNC     0
#define KILL_FUNC     1
#define RESTART_FUNC  2

#define HELP_OPT      0
#define PORT_OPT      1
#define HOST_OPT      2
#define USER_OPT      3
#define VERSION_OPT   4
#define KILL_OPT      5
#define RESTART_OPT   6
#define CONF_OPT      7

int
main(int argc, char **argv)
{
  char *portstring = NULL;		/* port argument */
  char c;
  int sock;
  int func = NONE_FUNC;
  int tmpsize = 1024;
  char *tmp = (char*) malloc (tmpsize);
  char *configfilename = NULL;

  setlocale (LC_MESSAGES, "");
  textdomain (PACKAGE);
  
  Welcome ();			/* a welcome at first */

  /* argumentpassing */
  
  while (1) {
    static struct option long_options[] = {
      {"help", 0, 0, HELP_OPT},
      {"port", 1, 0, PORT_OPT},
      {"host", 1, 0, HOST_OPT},
      {"user", 1, 0, USER_OPT},
      {"version", 0, 0, VERSION_OPT},
      {"kill", 0, 0, KILL_OPT},
      {"restart", 0, 0, RESTART_OPT},
      {"conffile", 1, 0, CONF_OPT},
      {0, 0, 0, 0}
    };
    int option_index;

    c = getopt_long (argc, argv, "hp:a:u:kr", long_options, &option_index);
    
    if (c == -1)
      break;

    switch (c) {
    case HELP_OPT:
    case 'h':			/* help */
      showhelp();
      exit (EXIT_SUCCESS);
    case PORT_OPT:
    case 'p':			/* port to address the daemon */
      portstring = optarg;
      port = atoi (portstring);
      break;
    case HOST_OPT:
    case 'a':			/* host to connect to */
      host = optarg;
      break;
    case USER_OPT:
    case 'u':
      usrname = optarg;
      break;
    case KILL_OPT:
    case 'k':
      func = KILL_FUNC;
      break;
    case RESTART_OPT:
    case 'r':
      func = RESTART_FUNC;
      break;
    case CONF_OPT:
    case 'f':
      configfilename = optarg;
      break;
    case VERSION_OPT:
      showversion ();
      exit (EXIT_SUCCESS);
    case '?':			/* wrong option */
      exit (EXIT_FAILURE);
    default:			/* other options */
      exit (EXIT_FAILURE);
    }
  }

  if (func == NONE_FUNC) {
    PRINTERR (_("No function given! (--kill, --restart)"));
    exit (EXIT_FAILURE);
  }

  /* read global configuration file */
  read_configuration (configfilename);

  sock = make_connection (&port, &host);
  if (sock < 0) {
    PRINTERR (_("connect error"));
    exit (EXIT_FAILURE);
  }
  printf(_("Connected to daemon (%s:%d)\n"), host, port);  

  if (make_login (sock, KCP_SUPERVISOR, usrname, NULL, &tmp, &tmpsize) < 0)
    goto errhd;
  if (parse_login_data (tmp, &connectid, &connkey, &client_usrname, 
			&clientuid, &clientgid) < 0)
    goto errhd;
  
  switch (func) {
  case KILL_FUNC:		/* kill daemon */
    if (writesockf (sock, "%s %d %s", KCP_KILL, connectid, connkey) <= 0)
      goto errhd;
    break;
  case RESTART_FUNC:
    if (writesockf (sock, "%s %d %s", KCP_REREAD, connectid, connkey) <= 0)
      goto errhd;
    if (writesockf (sock, "%s %d %s", KCP_CLOSE, connectid, connkey) <= 0)
      goto errhd;
    break;
  }
  
  close (sock);
  exit (EXIT_SUCCESS);

 errhd:
  close (sock);
  exit (EXIT_FAILURE);
}




/*------------------------------------------------------------
  General routines
  ------------------------------------------------------------*/	
void
Welcome()
{
  fprintf(stdout, _(
"kdutil - kaenguru database daemon utility\n"
"This is kd_util - a tool to manage the kaenguru multiuser networkdaemon.\n"
"Use kdutil --help for detailed help on syntax and use.\n"
"Copyright (c) %s by Gregor Klinke\n"), COPYRIGHTYEAR);
}

void
showversion (void)
{
  fprintf (stdout, _(
"This is Kaenguru %s - The Kaenguru Database System (kdutil)\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"
"This program is free software; you can redistribute it and/or modify it\n"
"under the terms of the GNU General Public License as published by the\n"
"Free Software Foundation; either version 2 of the License, or (at your\n"
"option) any later version.\n"
"\n"
"This program ist distributed in the hope that it will be useful, but\n"
"WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n"
"General Public Lincense for more details.\n"
"\n"), VERSION, COPYRIGHTYEAR);

}

void
showhelp (void)
{
  
  fprintf(stdout, _(
"kdutil manages a running multiuser daemon of the kaenguru database system.\n"
"(halt, restart)\n"
"\n"
"General usage:\n"
"  kdutil [-h] [-kr] [p<port>] [a<host>]\n"
"\n"
"Options:\n"
"  -k, --kill         Halt the daemon\n"
"  -r, --restart      Restart the daemon (reread the configuration data)\n"
"  -h, --help         This help\n"
"  -p, --port=PORT    The port, to which the daemon listens\n"
"  -a, --host=HOST    The host on which the daemon is running\n"
"      --version      print version and copyright\n"
"\n"
"You have to specify at least one function (--kill, --restart)\n"
"\n")
	  );    
}
