/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "sm.h"

#define VAC_ZEROLEN     0
#define VAC_OK          1
#define VAC_KILLED      2
#define VAC_DEL         3
#define VAC_FID         4
#define VAC_ERROR       5
#if DISTRIBUTED_DB
#define VAC_REMOTE      6
#endif

int
readVacuumItem (Database *db, Fid fid, Oid oid, char **buf, int *bufsize,
		Objstat *objstat, int deldeleted)
{
  Tableitem tableitem;
  Itemhead itemhead;
  int sizeread;
  int findex;

#if DISTRIBUTED_DB
  if (!db->remote) {
    return VAC_REMOTE;
  }
#endif
  
  objstat->fid = 0;
  objstat->uid = 0;
  objstat->gid = 0;
  objstat->accesscode = 0;
  objstat->status = status_UNDEF;
  objstat->oid = 0;
  objstat->length = 0;
  objstat->creat = 0;
  
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);			/* read tableitem */
  
  findex = get_fid_index (db, tableitem.fid);
  if (findex < 0)
    SM_GOERR (NWEEFID);      
  
  if (tableitem.fid != fid) { /* ein Datensatz im angegeben File? */
    *buf = NULL;
    return VAC_FID;
  }
  
  if (tableitem.ofs == status_KILLED) { /* item is killed! */
    *buf = NULL;
    return VAC_KILLED;
  }
  
  if (readItemhead (db->data[findex].file, tableitem.ofs, &itemhead) < 0)
    SM_GOERR (NWERITEMH);			/* read head of item */
  
  switch (itemhead.status) {
  case status_DEL:
    if (deldeleted) {		/* if deleted items should be removed */
      *buf = NULL;
      return VAC_DEL;
    }
    break;
  case status_ENTRY:		/* nothing happens */
    break;
  default:
    SM_GOERR (NWEDATACORRUPT);
  }
  
  if (itemhead.length > 0) {
    *buf = (char *) malloc (itemhead.length + 1);
    *bufsize = itemhead.length + 1;
    
    fseek (db->data[findex].file, tableitem.ofs + sizeof(Itemhead),
	   SEEK_SET);
    sizeread = fread (*buf, 1, *bufsize, db->data[findex].file);
        
    objstat->fid = tableitem.fid;
    objstat->uid = itemhead.uid;
    objstat->gid = tableitem.gid;
    objstat->accesscode = tableitem.accesscode;
    objstat->status = itemhead.status;
    objstat->oid = oid;
    objstat->length = itemhead.length;
    objstat->creat = itemhead.creat;
    
    return VAC_OK;
  }
  
  *buf = NULL;
  return VAC_ZEROLEN;
  
 errhd:
  return VAC_ERROR;
}
  
int
writeVacuumItem (Database *db, Oid oid, char *buf, int length, FILE *stream) 
{
  Tableitem tableitem, newtableitem;
  Itemhead itemhead, newitemhead;
  long sizewritten;
  long newpos;
  int findex;

#if DISTRIBUTED_DB
  if (!db->remote) {
    return -1;
  }
#endif

  /* first read the old headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);

  findex = get_fid_index (db, tableitem.fid);
  if (findex < 0)
    SM_GOERR (NWEEFID);
  if (readItemhead (db->data[findex].file, tableitem.ofs, &itemhead) < 0)
    SM_GOERR (NWERITEMH);		/* read head of item */
    
    /* get the last position of the file */
  fseek (stream, 0, SEEK_END);
  newpos = ftell (stream);
  if (newpos == -1) 
    SM_GOERR (NWEEPOS);
    
    /* nun setze die Header und Listeneintraege neu zusammen */
  newtableitem.ofs = newpos; 
    
  newtableitem.fid = tableitem.fid;
  newtableitem.accesscode = tableitem.accesscode;
  newtableitem.uid = tableitem.uid;
  newtableitem.gid = tableitem.gid;
    
  newitemhead.oid = oid;
  newitemhead.status = itemhead.status;
  newitemhead.uid = itemhead.uid;
  newitemhead.length = length;
  newitemhead.creat = itemhead.creat;
    
    /* nun speichere den neuen Datensatz ab */
  if (writeItemhead (stream, newpos, &newitemhead) < 0)
    SM_GOERR (NWEWITEMH);		/* save header */
    
  fseek (stream, newpos+sizeof(Itemhead), SEEK_SET);
  sizewritten = fwrite (buf, 1, length, stream);
  if (sizewritten != length)
    SM_GOERR (NWEWDATA);
    
  cleanoutput (stream);
    
  /* sofern das geklappt hat, schreibe den neuen Tabelleneintrag*/
  if (writeTableitem (db->tblfile, oid, &newtableitem) < 0)
    SM_GOERR (NWEWTABLE);
    
  return 1;
    
 errhd:
  return -1;
}

int
killVacuumItem (Database *db, Oid oid)
{
  Tableitem tableitem;

#if DISTRIBUTED_DB
  if (!db->remote) {
    return -1;
  }
#endif
  
  tableitem.ofs = status_KILLED;	/* kill tableentry -1 = no ref! */
  tableitem.fid = 0;
  tableitem.accesscode = 0;
  tableitem.uid = 0;
  tableitem.gid = 0;
  if (writeTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWEWTABLE);
  return 1;
  
 errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   This is the public vacuum function.  
   ---------------------------------------------------------------------- */
int
vacuumFile (Database *db, Fid fid, int deldeleted) 
{
  Oid current_oid, total_oid;
  Objstat objstat;
  char *buf = NULL;
  int bufsize, writesize;
  FILE *newfile = NULL;
  char *tmp_filename = NULL;
  int findex;
  char tmp[FILENAME_MAX];
  int retval;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Vaccum a datafile/database on a remote site.\n");
    return 1;
  }
#endif

  findex = get_fid_index (db, fid);
  if (findex < 0)
    SM_GOERR (NWEEFID);
  
  /* -------------------------------------------------------------------
     At first we have to create a new file with a temporary name.
     ------------------------------------------------------------------- */
  tmp_filename = get_filename (db);
  newfile = create_file (tmp_filename);
    
  if (!newfile)
    SM_GOERR (NWEVCTMP);
  
  /* -------------------------------------------------------------------
     now copy the data from the original file to the new file and update
     the pointers in the tables.  If deldeleted is true all items marked
     deleted are not copied and their pointers are set to killed.
     ------------------------------------------------------------------- */
  total_oid = gettotal_oid (db);
  current_oid = FIRSTOID;	/* start with the first oid */
  
  for ( ; current_oid < total_oid ; current_oid++) {
    buf = NULL;
    bufsize = 0;
    
    retval = readVacuumItem (db, fid, current_oid, &buf, &bufsize, &objstat,
			     deldeleted);
    
    switch (retval) {
    case VAC_ERROR:
      goto errhd;
#if DISTRIBUTED_DB
    case VAC_REMOTE:
      goto errhd;
#endif
    case VAC_KILLED:		/* let killed item untouched */
    case VAC_FID:
      break;
    case VAC_ZEROLEN:		/* this items have to be deleted */
    case VAC_DEL:
      if (killVacuumItem (db, current_oid) < 0)
	goto errhd;
      break;
    default:
      if ((bufsize > 0) &&
	  (buf)) {	
	/* ok, item must be copy to a new place */
	writesize = writeVacuumItem (db, current_oid, buf, 
				     bufsize, newfile);
	if (writesize < 0)
	  goto errhd;
      }
    }
    
    if (buf) {
      free (buf);
      buf = NULL;
    }
  }
  
  /* --------------------------------------------------------------------
     so if we come to this point, we are done.  Now close the original
     file and the temporaryfile; remove the original file and rename the
     tmpfile to the original name.
     ------------------------------------------------------------------- */
  if (closedatafile (db, findex) < 0)
    goto errhd;
  fclose (newfile);
  
  datafilename (tmp, db, findex); /* generate original filename */
  if (unlink (tmp) < 0)	/* delete the original file */
    SM_GOERR (NWEVDORG);
  
  if (rename (tmp_filename, tmp) < 0)	/* ren tmp file to original name */
    SM_GOERR (NWEVMTMP);
  if (opendatafile (db, findex) < 0) /* reopen the datafile */
    goto errhd;
  
  /* --------------------------------------------------------------------
     ok.  The same has to be done with the openlist file.  It has to be
     truncated to length zero (close, unlink, create)
     ------------------------------------------------------------------- */
  if (closeoplfile (db, findex) < 0) /* close the opl file */
    goto errhd;
  if (deloplfile (db, findex) < 0) /* delete the opl file */
    goto errhd;
  if (createoplfile (db, findex) < 0) /* create and open the opl file */
    goto errhd;
  
  return 0;

 errhd: 
  if (buf)
    free (buf);
  if (tmp_filename)
    free (tmp_filename);
  
  if (newfile)
    fclose (newfile);
  
  return -1;
} 

int
vacuum (Database *db, int deldeleted, Fid fid) 
{
  if (fid == UNKNOWNFID) {
    int i;
    
    for (i = 0; i < db->datafileno; i++) {
      if (vacuumFile (db, db->data[i].fid, deldeleted) < 0)
	goto errhd;
    }
  }
  else {
    Fid fid;
    int findex = get_fid_index (db, fid);
    if (findex < 0)
      SM_GOERR (NWEEFID);
    
    fid = get_index_id (db, findex);
    
    if (vacuumFile (db, fid, deldeleted) < 0)
      goto errhd;
  }
  return 0;
  
 errhd:
  return -1;
}
