/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "sm.h"
#include "net.h"
#include "btree.h"

/* ----------------------------------------------------------------------
   With a remote site you can't access the database-struct manual!  So we
   must handle the database structure as a opac type
   ---------------------------------------------------------------------- */
char *
get_index_name (Database *db, int iindex)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get index name from a remote site\n");
    return NULL;
  }
#endif

  if ((iindex < 0) || (iindex > db->indexno))
    SM_GOERR (NWEEFID);
  return db->index[iindex].name;

 errhd:
  return NULL;
}

char *
get_datafile_name (Database *db, int findex)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get datafile name from a remote site\n");
    return NULL;
  }
#endif

  if ((findex < 0) || (findex > db->datafileno))
    SM_GOERR (NWEEFID);
  return db->data[findex].name;

 errhd:
  return NULL;
}

Fid
get_index_id (Database *db, int iindex)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get index id from a remote site\n");
    return 0;
  }
#endif

  if ((iindex < 0) || (iindex > db->indexno))
    SM_GOERR (NWEEFID);
  return db->index[iindex].iid;

errhd:
  return 0;
}

Fid
get_datafile_id (Database *db, int findex)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get datafile id from a remote site\n");
    return 0;
  }
#endif

  if ((findex < 0) || (findex > db->datafileno))
    SM_GOERR (NWEEFID);
  return db->data[findex].fid;

errhd:
  return 0;
}

int
get_datafileno (Database *db)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get number of datafiles from a remote site\n");
    return 0;
  }
#endif

  return db->datafileno;
}

int
get_indexfileno (Database *db)
{
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Get number of indexfiles from a remote site\n");
    return 0;
  }
#endif

  return db->indexno;
}


/* ----------------------------------------------------------------------
   Interface routinen fr den Zugriff auf Indizes.  Man sollte
   grundstzlich nur ber diese Routinen auf den Index zugreifen, da diese
   Routinen zwischen verschiedenen Indextypen unterscheiden und vor allem
   das Remote Server Problem lsen.
   ---------------------------------------------------------------------- */
int
update_index (Database *db, Fid iid, char *token, Oid oid)
{
  if (token) {
    int iindex;

#if DISTRIBUTED_DB
    if (db->remote) {
      printf ("Update index key on remote site.\n");
      return 0;
    }
#endif
    
    iindex = get_iid_index (db, iid);
    if (iindex < 0)
      SM_GOERR (NWEEFID);
    
    if (updatebtree (&db->index[iindex], token, oid) < 0) {
      SM_GOERR (_("updating index"));
    }
    
    return 1;
  }
  return 0;
  
 errhd:
  return -1;
}

int
delete_index (Database *db, Fid iid, char *token, Oid oid)
{
  if (token) {
    int iindex;

#if DISTRIBUTED_DB
    if (db->remote) {
      printf ("Delete index key from remote site.\n");
      return 0;
    }
#endif

    iindex = get_iid_index (db, iid);
    if (iindex < 0)
      SM_GOERR (NWEEFID);
    
    if (deletebtree (&db->index[iindex], token, oid) < 0) {
      SM_GOERR (_("deleting key from index"));
    }
    return 1;
  }
  return 0;
  
errhd:
  return -1;
}


int
retrieve_index (Database *db, Fid iid, char *token, Keyinfo *keyinfo)
{
  if (token) {
    Index *index;
    int nr, found, iindex;
    Node lastwp;
    
#if DISTRIBUTED_DB
    if (db->remote) {
      printf ("Retrieve index key from remote site.\n");
      return 0;
    }
#endif
    
    iindex = get_iid_index (db, iid);
    if (iindex < 0) {
      SM_GOERR (NWEEFID);
    }
    
    index = &db->index[iindex];
    
    found = retrievebtree (token, index, &lastwp, &nr, keyinfo);
    
    if (found == 0)
      return 0;
    else if (found < 0)
      return -1;
    return 1;
  }
  return 0;

errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   ldt eine Seite NODE aus dem Index IID in der Datenbank DB.  Konnte
   nichts gelesen werden (o..) wird 0 zurckgegeben, trat ein Fehler auf
   -1.  Konnte die Seite geladen werden wird 1 geliefert und in WORKPAGE
   steht die entsprechende Seite.
   ---------------------------------------------------------------------- */
int
load_index_page (Database *db, Fid iid, Node node, Workpage *workpage)
{
  int iindex;
  Index *indx;
  
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Loading page from remote Server\n");
    return 0;
  }
#endif
  
  if (node == i_NOLINK)
    return 0;
  iindex = get_iid_index (db, iid);
  if (iindex < 0) {
    SM_GOERR (NWEEFID);
  }
  
  indx = &db->index[iindex];
  
  if (loadKnot (indx, node, workpage) < 0)
    goto errhd;
  
  return 1;
 errhd:
  return -1;
}

Node
get_index_root (Database *db, Fid iid)
{
  int iindex;
  Index *indx;
  Node wp;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Loading root page from remote Server\n");
    return 0;
  }
#endif
  
  iindex = get_iid_index (db, iid);
  if (iindex < 0) {
    SM_GOERR (NWEEFID);
  }
  
  indx = &db->index[iindex];

  cleaninput (indx->file);	/* reset input stream */

  if (loadBasispage (indx) < 0)
    goto errhd;
  wp = getRoot (indx);
  
  return wp;
  
 errhd:
  return i_NOLINK;
}
