/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include <sys/stat.h>
#include <errno.h>

#include "sm.h"
#include "btree.h"
#include "path.h"

char kaengurubase[FILENAME_MAX] = BASEPATH;
char tmppath[FILENAME_MAX] =  TMPPATH;

void
initdbvar (Database *db)
{
  int i;
  
  db->basename = NULL;
  db->path = NULL;
  for (i = 0; i < MAXDTAFILE; i++) {
    db->data[i].name = NULL;
    db->data[i].file = NULL;
    db->data[i].fid = 0;
  }
  for (i = 0; i < MAXINDEX; i++) {
    db->index[i].name = NULL;
    db->index[i].file = NULL;
    db->index[i].iid = 0;
  }
  db->datafileno = 0;
  db->status = 0;
  db->asowner = false;
  db->dbid = UNKNOWNDB;
  db->lockid = 0;

#if DISTRIBUTED_DB
  /* this are vars for the remote site access */
  db->server = NULL;
  db->port = 0;
  db->remote = 0;
  db->nettype = 0;
  db->connkey = NULL;
  db->connid = -1;
  db->sock = 0;
#endif
}

void
killdbvar (Database *db)
{
  int i;

  if (db->basename) {
    free (db->basename);
    db->basename = NULL;
  }
  if (db->path) {
    free (db->path);
    db->path = NULL;
  }

  for (i = 0; i < MAXDTAFILE; i++) {
    if (db->data[i].name) {
      free (db->data[i].name);
      db->data[i].name = NULL;
    }
    db->data[i].file = NULL;
  }
  for (i = 0; i < MAXINDEX; i++) {
    if (db->index[i].name) {
      free (db->index[i].name);
      db->index[i].name = NULL;
    }
    db->index[i].file = NULL;
  }
  
  db->datafileno = 0;
  db->status = 0;
  db->asowner = false;
  db->dbid = UNKNOWNDB;
  db->lockid = 0;

#if DISTRIBUTED_DB
  if (db->server) {
    free (db->server);
    db->server = NULL;
  }
  db->port = 0;
  db->remote = 0;
  db->nettype = 0;
  if (db->connkey) {
    free (db->connkey);
    db->connkey = NULL;
  }
  db->connid = -1;
  db->sock = 0;
#endif
}

/* this function reads the basic files file in the base-directory of a
   database wich tolds the system the fundamentals of a database (the
   different datafiles, their paths, etc.) */
int
readcfgfile (Database *db)
{
  char cfgname[FILENAME_MAX], iline[FILENAME_MAX];
  FILE *stream;
  const char spacedelim[] = " \t\r\n";
  const char keyworddelim[] = ":\n \t";
  char *token, *tail;
  int datafilecount, indexcount;
  
  sprintf (cfgname, "%s/%s", db->path, FILES_FILE_NAME);

  stream = fopen (cfgname, "r");
  if (stream == NULL)
    SM_GOERR (_("opening 'files' file"));
  
  datafilecount = 0;
  indexcount = 0;

  while ((!feof (stream)) && 
	 (datafilecount < MAXDTAFILE) &&
	 (indexcount < MAXINDEX)) {
    if (fgets (iline, FILENAME_MAX-1, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	SM_GOERR (_("reading 'files' file"));
    }
    token = strtok (iline, keyworddelim);
    if (token) {		/* datafile type: data OR index */
      if (strcmp (token, "data") == 0) {
	token = strtok (NULL, spacedelim);
	if (token) {		/* datafile name (phyiscal) */
	  db->data[datafilecount].name = strdup (token);
	  token = strtok (NULL, spacedelim);
	  if (token) {		/* the fid */
	    db->data[datafilecount].fid = strtol (token, &tail, 0);
	  }
	  else {
	    fprintf (stderr, _("no file id config in 'files' file\n"));
	    goto errhd;
	  }
	  datafilecount++;
	}
      }
      else if (strcmp (token, "index") == 0) {
	token = strtok (NULL, spacedelim);
	if (token != NULL) {	/* indexfile name (physical) */
	  db->index[indexcount].name = strdup (token);
	  token = strtok (NULL, spacedelim);
	  if (token) {		/* the iid */
	    db->index[indexcount].iid = strtol (token, &tail, 0);
	  }
	  else {
	    fprintf (stderr, _("no index id config in 'files' file\n"));
	    goto errhd;
	  }
	  indexcount++;
	}
      }
      else if (strcmp (token, "#") == 0) {
	/* comment */
      }
    }
  }
eofhd:
  db->datafileno = datafilecount;
  db->indexno = indexcount;
  
  if (fclose (stream) != 0)
    SM_GOERR (_("closing 'files' file"));
  return 1;
  
 errhd:
  return -1;
}


int
setdbfilenames (Database *db, char *basename, char *path)
{
  char *nstr, dbpath[FILENAME_MAX];
  char *dbbase;
  
  if (!basename) {
    SM_GOERR (_("undefined database"));
  }
  else 
    dbbase = basename;
  
  if (!path) {
    nstr = kaengurubase;
    if (!nstr) {
      sprintf (dbpath, "/var/lib/kaenguru/base/%s", dbbase);
    }
    else 
      sprintf (dbpath, "%s/%s", nstr, dbbase);
  }
  else { 
    if (strchr (path, '<')) {	/* this is a remote site! */
#if DISTRIBUTED_DB
      char *p, *p1, *tail;

      p = strchr (path, '<');	/* find the begin of the data */
      p++;
      p1 = strchr (p, '!');
      *p1 = '\0';		/* this is the transport protocol (tcp) */
      if (strcmp (p, "tcp") == 0) {
	db->remote = 1;		/* yes, a remote database via tcp */
	db->nettype = 0;	/* 0 = tcp */
	
	p = p1+1;
	p1 = strchr (p, '!');	/* this is the servername */
	*p1 = '\0';
	db->server = strdup (p);
	p = p1+1;			/* find the portnumber */
	p1 = strchr (p, '>');	
	*p1 = '\0';
	db->port = (unsigned short) strtol (p, &tail, 0);
	
	strcpy (dbpath, "");
      }
      else {
	db->remote = 0;		/* not supported type */
	db->nettype = 0;	/* 0 = tcp */
	goto errhd;
      }
#else /* not distributed_db */
      SM_GOERR (NODISTRDB); 
#endif
    }
    else
      strcpy (dbpath, path);
  }
  
  db->basename = strdup (dbbase);
  db->path = strdup (dbpath);
  
  return 1;

 errhd:
  return -1;
}


int
mountdb (Database *db, char *dbpath)
{
  char *basename = NULL, *path = NULL, *p;

  if ((dbpath) &&
      (strlen (dbpath) > 0)) {
    char tmp[FILENAME_MAX];
    strcpy (tmp, dbpath);
    if ((p = strchr (tmp, '@'))) { /* full explicit path */
      *p = '\0';		/* fix basename */
      basename = tmp;
      path = p+1;
    }
    else
      basename = dbpath;
  }
  
  if (setdbfilenames (db, basename, path) < 0)
    goto errhd;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("[Database %s at remote site %s port %d]\n", 
	    db->basename, 
	    db->server,
	    db->port);
    return 1;
  }
  else {
#endif

    if (readcfgfile (db) < 0)
      goto errhd;
    return 1;
#if DISTRIBUTED_DB
  }
#endif

 errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   Get ID numbers for data- and index-files .
   ---------------------------------------------------------------------- */
int
get_fid_index (Database *db, Fid fid)
{
  int i;
  for (i = 0; i < db->datafileno; i++) {
    if (db->data[i].fid == fid) {
      return i;
    }
  }
  return -1;
}

int
get_iid_index (Database *db, Fid iid)
{
  int i;
  for (i = 0; i < db->indexno; i++) {
    if (db->index[i].iid == iid) {
      return i;
    }
  }
  return -1;
}

int
get_iname_index (Database *db, char *indexname)
{
  int i;
  
  for (i = 0; i < db->indexno; i++) {
    if (strcmp (db->index[i].name, indexname) == 0) {
      return i;
    }
  }
  return -1;
}

int
get_dname_index (Database *db, char * datafilename)
{
  int i;
  for (i = 0; i < db->datafileno; i++) {
    if (strcmp (db->data[i].name, datafilename) == 0) {
      return i;
    }
  }
  return -1;
}

/* ----------------------------------------------------------------------
   Open a file (datafile, openlist, index)
   Create, delete ...
   ---------------------------------------------------------------------- */
int
opendatafile (Database *db, int datafileno)
{
  char buf[FILENAME_MAX];
  
  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  datafilename (buf, db, datafileno);
  
  db->data[datafileno].file = fopen (buf, "r+");
  if (db->data[datafileno].file == NULL)
    SM_GOERR (_("opening datafile"));
  return 1;
  
 errhd:
  return -1;
}

int
openoplfile (Database *db, int datafileno)
{
  char buf[FILENAME_MAX];
  
  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  oplfilename (buf, db, datafileno);
  
  db->data[datafileno].opfile = fopen (buf, "r+");
  if (db->data[datafileno].opfile == NULL)
    SM_GOERR (_("opening openlist"));
  return 1;
  
 errhd:
  return -1;
}

int
createoplfile (Database *db, int datafileno)
{
  char buf[FILENAME_MAX];
  
  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  oplfilename (buf, db, datafileno);
  
  db->data[datafileno].opfile = fopen (buf, "w+");
  if (db->data[datafileno].opfile == NULL)
    SM_GOERR (_("opening openlist"));
  return 1;
  
 errhd:
  return -1;
}

int
openIndex (Database *db, int indexfileno)
{
  char buf[FILENAME_MAX];

  if (indexfileno > db->indexno)
    SM_GOERR (NWEEFID);
  indxfilename (buf, db, indexfileno);

  db->index[indexfileno].file = fopen (buf, "r+");
  if (db->index[indexfileno].file == NULL)
    SM_GOERR (_("opening index"));

  db->index[indexfileno].basispage.root = 1;
  db->index[indexfileno].basispage.nextfree = i_NOLINK;
  db->index[indexfileno].bp_changed = false;
  return 1;

errhd:
  return -1;
}

int
closedatafile (Database *db, int datafileno)
{
  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  
  if (db->data[datafileno].file) 
    fclose (db->data[datafileno].file);
  return 1;

errhd:
  return -1;
}

int
deldatafile (Database *db, int datafileno)
{
  char buf[FILENAME_MAX];

  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  datafilename (buf, db, datafileno);
 
  if (unlink (buf) < 0)
    SM_GOERR (_("deleting datafile"));

  return 1;

errhd:
  return -1;
}

int
closeoplfile (Database *db, int datafileno)
{
  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  
  if (db->data[datafileno].opfile) 
    fclose (db->data[datafileno].opfile);
  return 1;
errhd:
  return -1;
}

int
deloplfile (Database *db, int datafileno)
{
  char buf[FILENAME_MAX];

  if (datafileno > db->datafileno)
    SM_GOERR (NWEEFID);
  oplfilename (buf, db, datafileno);
 
  if (unlink (buf) < 0)
    SM_GOERR (_("deleting openlist"));

  return 1;

errhd:
  return -1;
}

int
closeIndex (Database *db, int indexfileno)
{
  if (indexfileno > db->indexno)
    SM_GOERR (NWEEFID);
  
  if (fclose (db->index[indexfileno].file))
    SM_GOERR (_("closing index"));
  return 1;

errhd:
  return -1;
}


/* ---------------------------------------------------------------------- */

int
opendb (Database *db)
{
  char buf[FILENAME_MAX];
  int i;

#if DISTRIBUTED_DB
  if (db->remote) {
    /* open a remote site */
    printf ("Open database %s at remote site %s port %d\n", 
	    db->basename, 
	    db->server,
	    db->port);
    if (connect_to_remote_site (db) < 0)
      goto errhd;
    
    db->status = 1;
    return 1;
  }
#endif

  sprintf (buf, "%s/%s", db->path, TABLE_FILE_NAME);
  
  db->tblfile = fopen (buf, "r+"); /* ffne table file */
  if (db->tblfile == NULL)
    SM_GOERR (_("opening table"));
  
  for (i = 0; i < db->datafileno; i++) { /* ffne alle datenfiles */
    if (opendatafile (db, i) < 0)
      goto errhd;
    if (openoplfile (db, i) < 0)
      goto errhd;
  }
  for (i = 0; i < db->indexno; i++) { /* ffne alle Indexfiles */
    if (openIndex (db, i) < 0)
      goto errhd;
  }
  db->status = 1;
  return 1;
  
 errhd:
  return -1;
}

int
closedb (Database *db)
{
  char buf[FILENAME_MAX];
  int i;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Close database %s at remote site %s port %d\n", 
	    db->basename, 
	    db->server,
	    db->port);
    db->status = 0;
    if (disconnect_from_remote_site (db) < 0)
      goto errhd;
    
    return 1;
  }
#endif

  sprintf (buf, "%s/%s", db->path, TABLE_FILE_NAME);
    
  if (db->tblfile)
    fclose(db->tblfile);
    
  for (i = 0; i < db->datafileno; i++) {
    if (closedatafile (db, i) < 0)
      goto errhd;
    if (closeoplfile (db, i) < 0)
      goto errhd;
  }
  for (i = 0; i < db->indexno; i++) {
    if (closeIndex (db, i) < 0)
      goto errhd;
  }
  db->status = 0;
  return 1;

errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   Creation management
   ---------------------------------------------------------------------- */
int
createIndex (char *path, char *filename)
{
  Workpage workpage;
  Index *idx = NULL;
  char buf[FILENAME_MAX];
  
  idx = (Index *) alloca (sizeof (Index));
  idx->name = filename;
  sprintf (buf, "%s/%s.indx", path, filename);
  
  idx->file = fopen (buf, "w+");
  if (idx->file == NULL)
    SM_GOERR (_("creating index"));
  
  initBasispage (&idx->basispage, 1, i_NOLINK);
  
  if (writeBasispage (idx) < 0)
    goto errhd;
  
  initKnot (&workpage, i_KEYTREE);
  if (writeKnot (idx, 1, &workpage) < 0)
    goto errhd;
  
  if (fclose (idx->file) != 0)
    SM_GOERR (_("closing index"));
  
  return 1;
  
errhd:
  return -1;
}


/* createdb takes the path, basename, a database-var and two lists of all
   (inital) datafiles and indexes of a database, and creates this in the
   specified places.  The lists of datafiles and indexes are semicolon (or
   komma) separated names which are used as databasenames.  Each entry must
   be divided by a colon: the first half is the name of the file, the
   second half the id number (e.g. book:21 renders the datafile book.data
   and book.opl with the id 21).  An inital files-file is generated
   automatic.

   EX:  createdb (&db, "john", NULL, 
                       "Books:1 ; CDs:2 ; Cassettes:3 ; Maps:5",
                       "default:10 ; names:11 ; places:20 ; keys:0");

   will generate 14 files: Books.data as 1, Books.opl as 1, CDs.data as 2,
   CDs.opl as 2, Cassettes.data as 3, Cassettes.opl as 3, Maps.data as 5,
   Maps.opl as 5, default.indx as 10, names.indx as 11, places.indx as 20,
   keys.indx as 0, table, files

   NOTE: each file - except "files" and all indexes are created as zero
   length files, because the opendb-functions needs every possible file to be
   existent. 

   NOTE TOO: each database should be created first with this function,
   because the index-files must be initialize in a rather complicated way
   (different from mere creating) 

   WARNING: if there are any objects stored to the database yet you never
   should change the id values of the datafiles and indices.  On the other
   hand you may any datafiles */
int
createdb (Database *db, char *basename, char *path,
	  char *list_of_files, char *list_of_index)
{
  char buf[FILENAME_MAX];
  FILE *tfile, *filesfile;
  const char filelistdelim[] = " \t\r\n;,";
  int fleno;
  char *token;

#define touchfile(name, suffx)				\
{							\
  sprintf (buf, "%s/%s%s", db->path, name, suffx);	\
  tfile = fopen (buf, "w+");				\
  if (tfile == NULL)					\
    SM_GOERR (_("creating file"));			\
  fclose (tfile);					\
}							\
    
  /* set the variable */
  if (setdbfilenames (db, basename, path) < 0)
    goto errhd;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Create a remote database!\n");
    return 1;
  }
#endif

  /* create the basic path */
  
  if (mkdir (db->path, S_IRWXU) < 0) {
    switch (errno) {
    case EEXIST:
      SM_GOERR (_("database exists already"));
    case EACCES:
      SM_GOERR (_("no write permission (file system)"));
    case EROFS:
      SM_GOERR (_("read only file system"));
    default:
      SM_GOERR (_("creating database (unspecified)"));
    }
  }
  
  if (chdir (db->path) < 0)	/* change to the actual path */
    SM_GOERR (_("changing base path"));
  
  touchfile (TABLE_FILE_NAME, "");	/* create the table */
  
  /* create the files file */
  sprintf (buf, "%s/%s", db->path, FILES_FILE_NAME);
  if ((filesfile = fopen (buf, "w+")) == NULL)
    SM_GOERR (_("creating 'files' file"));
  
  fprintf (filesfile, 
	   "# This file was generated automatically by createdb\n"
	   "# Don't edit, if you don't know what you are doing\n"
	   "\n"
	   "# These are the datafiles\n");
  /* create the datafiles */
  fleno = 0;
  token = strtok (list_of_files, filelistdelim);
  while (token) {
    if (fleno < MAXDTAFILE) {
      char *p;
      
      p = strchr (token, ':');	
      if (p) {
	*p = '\0';
	p++;
	if (!*p)
	  SM_GOERR (_("missing file id"));
	
	touchfile (token, ".data");
	touchfile (token, ".opl");
	
	fprintf (filesfile, "data: %s %s\n", token, p);
      }
      else
	SM_GOERR (_("missing file id"));
      fleno++;
    }
    else
      SM_GOERR (_("too many datafiles"));
    token = strtok (NULL, filelistdelim);
  }
    
  fprintf (filesfile, 
	   "\n"
	   "# These are the indices\n");
  /* create the indexfiles */
  fleno = 0;
  token = strtok (list_of_index, filelistdelim);
  while (token) {
    if (fleno < MAXINDEX) {
      char *p;
      p = strchr (token, ':');
      if (p) {
	*p = '\0';
	p++;
	if (!*p)
	  SM_GOERR (_("missing index id"));
	
	if (createIndex(db->path, token) < 0)
	  goto errhd;
	fprintf (filesfile, "index: %s %s\n", token, p);
      }
      else
	SM_GOERR (_("missing index id"));

      fleno++;
    }
    else
      SM_GOERR (_("too many indexfiles"));
    token = strtok (NULL, filelistdelim);
  }
  
  if (fclose (filesfile) < 0)
    SM_GOERR (_("creating 'files' file"));

  /* at least create the directory for the blobs */
  if (mkdir (BLOB_DIR_NAME, S_IRWXU) < 0) {
    switch (errno) {
    case EEXIST:
      SM_GOERR (_("blobs directory exists already"));
    case EACCES:
      SM_GOERR (_("no write permission (file system)"));
    case EROFS:
      SM_GOERR (_("read only file system"));
    default:
      SM_GOERR (_("creating blob directory (unspecified)"));
    }
  }
  return 1;

 errhd:
  return -1;
}

/* Create a temporary filename.  Besteht aus 2 Komponenten: der Process ID
   und dem Erstellungsdatum */
static char tmpfilename[FILENAME_MAX];
static int tmpfilecount = 0;
char *
sm_tmpnam ()
{
  sprintf (tmpfilename, "#%d-%lx-%d#", (int)getpid(), 
	   get_system_time (), tmpfilecount);
  tmpfilecount++;
  
  return tmpfilename;
}

char *
get_tmp_filename ()
{
  char buf[FILENAME_MAX];
  
  sprintf (buf, "%s/%s", tmppath, sm_tmpnam ());
  return strdup (buf);
}

char *
get_filename (Database *db)
{
  char buf[FILENAME_MAX];
  
  sprintf (buf, "%s/%s", db->path, sm_tmpnam ());
  return strdup (buf);
}

FILE *
create_file (char *path)
{
  FILE *stream;

  if (path) {
    stream = fopen (path, "w+");
    if (stream == NULL)
      SM_GOERR (_("creating file"));
    return stream;
  }
  SM_GOERR (_("unspecifed path"));
  
errhd:
  return NULL;
}



