/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "sm.h"
#include "btree.h"

void
setistat (Indexstatus *istat, char pagetype)
{
  if (pagetype == i_KEYTREE) {
    istat->maxitems = i_MAXITEMS_KEY;
    istat->pagemidth = i_PAGEMIDTH_KEY;
    istat->keysize = i_KEYSIZE;
  }
  else if (pagetype == i_REFTREE) {
    istat->maxitems = i_MAXITEMS_REF;
    istat->pagemidth = i_PAGEMIDTH_REF;
    istat->keysize = 0;
  }
  istat->pagetype = pagetype;
  istat->emptypagegap = i_EMPTYPAGEGAP;
}

/* ----------------------------------------------------------------------
   Basics
   ---------------------------------------------------------------------- */
Node
nodeofs (Node node)
{
  return sizeof (Basispage) + (node - 1) * sizeof (Backpage);
}


/* Functions to access the basispage (holding tree-root and
   openlist-information */
int
writeBasispage (Index *index)
{
  if (index->bp_changed) {	/* only save if the page has changed */
    fseek (index->file, 0, SEEK_SET);
    if (fwrite (&index->basispage, sizeof(Basispage), 1, index->file) != 1) 
      SM_GOERR (_("writing index base page"));
  }
  index->bp_changed = false;
  return 1;
errhd:
  return -1;
}

int
loadBasispage (Index *index)
{
  fseek (index->file, 0, SEEK_SET);
  if (fread (&index->basispage, sizeof(Basispage), 1, index->file) != 1) 
    SM_GOERR (_("reading index base page"));
  index->bp_changed = false;
  return 1;
errhd:
  return -1;
}

void
changeNextfree (Index *index, Node nextfree)
{
  index->basispage.nextfree = nextfree;
  index->bp_changed = true;	/* basispage hat sich veraendert! */
}

void
changeRoot (Index *index, Node root)
{
  index->basispage.root = root;
  index->bp_changed = true;	/* basispage hat sich veraendert! */
}

Node
getNextfree (Index *index)
{
  return index->basispage.nextfree;
}

Node
getRoot (Index *index)
{
  return index->basispage.root;
}


/* load and write a sector in the main-tree and reference-trees
   sections. In future these routines should pack the data before storing
   and de-pack when loading. */
int
loadKnot (Index *index, Node node, Workpage *workpage)
{
  Node fofs = nodeofs(node);
  Backpage bp;
  int i;

  fseek (index->file, fofs, SEEK_SET);
  if (fread (&bp, sizeof(Backpage), 1, index->file) != 1)
    SM_GOERR(_("loading index knot"));
  
  if (bp.kt.pagetype == i_KEYTREE) {
    workpage->pagetype = i_KEYTREE;
    if (bp.kt.items > i_MAXITEMS_KEY - 1) 
      bp.kt.items = i_MAXITEMS_KEY - 1;
    workpage->items = bp.kt.items;
    workpage->itm[0].link = bp.kt.nulllink;    
    for (i = 0; i < bp.kt.items; i++) {
      workpage->itm[i+1].link = bp.kt.itm[i].link;
      workpage->itm[i+1].info = bp.kt.itm[i].info;
      workpage->itm[i+1].reftype = bp.kt.itm[i].reftype;
      strcpy (workpage->itm[i+1].key, bp.kt.itm[i].key);
    }
  }
  else if (bp.kt.pagetype == i_REFTREE) {
    workpage->pagetype = i_REFTREE;
    if (bp.ref.items > i_MAXITEMS_REF - 1) 
      bp.ref.items = i_MAXITEMS_REF - 1;
    workpage->items = bp.ref.items;
    workpage->itm[0].link = bp.ref.nulllink;
    for (i = 0; i < bp.ref.items; i++) {
      workpage->itm[i+1].link = bp.ref.itm[i].link;
      workpage->itm[i+1].info = bp.ref.itm[i].ref;
    }
  }
  else if (bp.kt.pagetype == i_EMPTYPAGE) {
    workpage->pagetype = i_EMPTYPAGE;
    workpage->nextfree = bp.ep.nextfree;
  }
  else
    SM_GOERR (_("index tree structure inconsistent"));
  
  return 1;
  
 errhd:
  return -1;
}
  
int
writeKnot (Index *index, Node node, Workpage *workpage)
{
  Node fofs = nodeofs(node);
  Backpage bp;
  int i;

  fseek (index->file, fofs, SEEK_SET);

  if (workpage->pagetype == i_KEYTREE) {
    for (i = 0; i < i_MAXITEMS_KEY - 1; i++) {  
      bp.kt.itm[i].link = 0;  
      bp.kt.itm[i].info = 0;  
      bp.kt.itm[i].reftype = '+';  
      memset (bp.kt.itm[i].key, '-', i_KEYSIZE);  
    }  
    
    bp.kt.items = workpage->items;
    bp.kt.pagetype = i_KEYTREE;
    bp.kt.nulllink = workpage->itm[0].link;
    for (i = 1; i <= workpage->items; i++) {
      bp.kt.itm[i-1].link = workpage->itm[i].link;
      bp.kt.itm[i-1].info = workpage->itm[i].info;
      bp.kt.itm[i-1].reftype = workpage->itm[i].reftype;
      strcpy (bp.kt.itm[i-1].key, workpage->itm[i].key);
    }
  }
  else if (workpage->pagetype == i_REFTREE) {
    for (i = 0; i < i_MAXITEMS; i++) {  
      bp.ref.itm[i].ref = 0; 
      bp.ref.itm[i].link = 0;  
    }  
    
    bp.ref.items = workpage->items;
    bp.ref.pagetype = i_REFTREE;
    bp.ref.nulllink = workpage->itm[0].link;
    for (i = 1; i <= workpage->items; i++) {
      bp.ref.itm[i-1].link = workpage->itm[i].link;
      bp.ref.itm[i-1].ref = workpage->itm[i].info;
    }
  }
  else if (workpage->pagetype == i_EMPTYPAGE) {
    bp.ep.nextfree = workpage->nextfree;
    bp.ep.pagetype = i_EMPTYPAGE;
    memset (bp.ep.space, '~', i_EMPTYPAGEGAP); /* muss eigentl. nicht sein */
  }
  else
    SM_GOERR (_("bad page type"));

  if (fwrite (&bp, sizeof (Backpage), 1, index->file) != 1)
    SM_GOERR (_("writing index knot"));

  fflush (index->file); 
  return 1;
errhd:
  return -1;
}

void
initItem (Item *item)
{
  item->link = i_NOLINK;
  item->info = i_NOREF;
  item->reftype = i_PAGELINK;	/* p:0 ist nirgendwo! */
  memset (item->key, '-', i_KEYSIZE); /* ist nicht noetig */
}

void
initKnot (Workpage *wp, char pagetype)
{
  int i;
  
  wp->pagetype = pagetype;
  wp->items = 0;
  wp->nextfree = 0;
  for (i = 0; i < i_MAXITEMS; i++)
    initItem (&wp->itm[i]);
}

void
initBasispage (Basispage *basispage, Node root, Node nextfree)
{
  basispage->root = root;
  basispage->nextfree = nextfree;
  memset (basispage->unused, ' ', i_BSPUNUSED);
}

Node
getnewKnot (Index *index)
{
  Node newpos;
  Node nextfree = getNextfree (index);
  
  if (nextfree == i_NOLINK) {
    /* keine freie Seite mehr vorhanden, neue Seite generieren */
    fseek (index->file, 0, SEEK_END);
    newpos = (ftell (index->file)-sizeof (Basispage)) / sizeof (Backpage) + 1;
  }
  else {
    Workpage wp;
    
    if (loadKnot (index, nextfree, &wp) < 0) 
      SM_GOERR (_("geting new knot"));
    newpos = nextfree;
    changeNextfree (index, wp.nextfree);
  }
  
  return newpos;
 errhd:
  return 0;			/* hier ist 0 error! nicht -1!  */
}

int
releaseKnot (Index *index, Workpage *wp, Node node)
{
  Node nextfree;
  
  if (node != i_NOLINK) {		/* aus vorsichtsmassnahme */
    nextfree = getNextfree (index);
    
    wp->pagetype = i_EMPTYPAGE;
    if (nextfree == i_NOLINK)	/* es gibt noch keine freien Seiten */
      wp->nextfree = 0;		/* NOLINK = ende der Kette */
    else 			/* put page to chain */
      wp->nextfree = nextfree;
    
    if (writeKnot (index, node, wp) < 0)
      goto errhd;
    
    changeNextfree (index, node);
  }
  
  return 1;
errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   Lokalisation und Suchen
   ---------------------------------------------------------------------- */
/* sucht aus einem Knoten (knot_key) den richtigen String raus.  Bedient
   sich des binaeren Suchens. */
void
keysearch (char *key,		/* key to search for */
	   Workpage *workpage, 
	   int *itemnr,		/* position of the item */
	   bool *itemfound)	/* item found? */
{
  int low, high, cmp, i;
  bool found;
  
  i = 0;
  found = false;
  low = 0;
  high = workpage->items + 1;
  
  while ((low+1 < high) && !found) {
    i = (low + high) / 2;
    cmp = strcasecmp (key, workpage->itm[i].key);
    if (cmp == 0)
      found = true;
    else if (cmp < 0)
      high = i;
    else
      low = i;
  }
  if (!found) 
    i = low;
  
  *itemnr = i;
  *itemfound = found;
}

/* sucht aus einem Knoten (knot_ref) die richte Reference heraus.  Bedient
   sich des binaeren Suchens. */
void
refsearch (Node ref,		/* ref to search for */
	   Workpage *workpage, 
	   int *itemnr,		/* position of the item */
	   bool *itemfound)	/* item found? */
{
  int low, high, cmp, i;
  bool found;
  
  i = 0;
  found = false;
  low = 0;
  high = workpage->items + 1;
  
  while ((low+1 < high) && !found) {
    i = (low + high) / 2;
    cmp = ref - workpage->itm[i].info;
    if (cmp == 0)
      found = true;
    else if (cmp < 0)
      high = i;
    else
      low = i;
  }
  if (!found) 
    i = low;
  
  *itemnr = i;
  *itemfound = found;
}

/* search for a key in an index. Im Ggs. zu den anderen Routinen nicht
   rekursiv! */
int
retrievebtree (char *key,	/* the key I search for */
	       Index *index,	/* the index */
	       long *lastwp,	/* last read workpage */
	       int *nr,		/* the number of the found item in wp*/
	       Keyinfo *keyinfo)
{
  Workpage wp;
  Node actwp;
  int itemnr;
  bool itemfound;
  char token[i_KEYSIZE + 1];

  cleaninput (index->file);	/* reset input stream */

  if (key) {
    if (loadBasispage (index) < 0)
      goto errhd;
    actwp = getRoot (index);
    
    if (actwp == i_NOLINK)
      goto notfoundhd;

    if (strlen (key) > i_KEYSIZE) {
      strncpy (token, key, i_KEYSIZE - 1);
      token[i_KEYSIZE] = '\0';
    }
    else
      strcpy (token, key);
    
    while (5) {
      if (loadKnot (index, actwp, &wp) < 0)
	goto errhd;
      keysearch (token, &wp, &itemnr, &itemfound);
      
      if (itemfound)
	goto foundhd;
      else {
	if (wp.itm[itemnr].link > i_NOLINK)
	  actwp = wp.itm[itemnr].link;
	else 
	  goto notfoundhd;
      }
    }
  }
  goto errhd;
  
 foundhd:
  *nr = itemnr;
  keyinfo->info = wp.itm[itemnr].info;
  keyinfo->reftype = wp.itm[itemnr].reftype;
  strcpy(keyinfo->key, wp.itm[itemnr].key);
  return 1;
  
 notfoundhd:
  *nr = 0;
  return 0;
  
 errhd:
  *nr = 0;
  return -1;
}

