/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */


/* WARNING: Der nachfolgende Code ist unsauber!  In tableitem gibt es ein
   Feld namens file.  Dieses enthlt nicht die file-Nummer sondern die fid!
   Alle ffentlichen Routinen erhalten -- sofern berhaupt -- die logische
   fid (d.h. die, die in der "files" Configuration steht.  Alle Routinen,
   die fr interne Zwecke benutzt werden, erhalten -- aus
   Geschwindigkeitsgrnden -- den bereits aufgelsten Index auf die
   data[]-Tabelle in der db-Structure.  Dummerweise heit alles noch
   irgendwie fid.  Irgendwann die ganze Struktur (data[] und index[]) auf
   Ptr-Listen umstellen, d.h. dynamisch machen.  Dann fllt diese etwas
   verzwickte Problematik weg.  Die ffentlichen Routinen arbeiten mit
   fids, die internen mit C-Pointer auf den entsprechenden data[]
   bzw. index[]-Eintrag.  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "sm.h"

int forcesync = 1;
int blob_limit = 8192;
#if defined HAVE_FDATASYNC
int pedanticforcesync = 0;
#endif

/* ----------------------------------------------------------------------
   allegemeine Routinen um Streams auf den aktuellsten Stand zu bringen 
   ---------------------------------------------------------------------- */
void
cleanoutput (FILE *stream)
{
  fflush (stream);		/* Stream aufrumen, alle Buffer ans System! */

#if defined HAVE_FDATASYNC
  if (pedanticforcesync) {
    fsync (fileno (stream));	/* synchronisiere diese Datei! */
  }
  else if (forcesync) {
    fdatasync (fileno (stream));
  }
#else
  if (forcesync) {
    fsync (fileno (stream));
  }
#endif
}

void
cleaninput (FILE *stream)
{
  fflush (stream);		/* setzte Stream zurck! */
}


/* ------------------------------------------------------------
   Itemhead-functions
   ------------------------------------------------------------ */
int
writeItemhead (FILE *iofile, long ofs, Itemhead *ihead) 
{
  int sizewritten;

  fseek (iofile, ofs, SEEK_SET);
  sizewritten = fwrite (ihead, sizeof(Itemhead), 1, iofile);

  if (sizewritten != 1) 
    return -1;

  cleanoutput (iofile);         /* flush stream */
  return 1;
}

int
readItemhead (FILE *iofile, long ofs, Itemhead *ihead)
{
  int sizeread;

  cleaninput (iofile);

  fseek (iofile, ofs, SEEK_SET);
  sizeread = fread (ihead, sizeof(Itemhead), 1, iofile);
  
  if (sizeread != 1) 
    return -1;
  
  return 1;
}


/* ------------------------------------------------------------
   Table-functions
   ------------------------------------------------------------ */
int
writeTableitem (FILE *tblfile, Oid oid, Tableitem *titem) 
{
  int sizewritten;

  if (oid == 0)
    return -1;
  fseek (tblfile, oid*sizeof(Tableitem), SEEK_SET);
  sizewritten = fwrite (titem, sizeof(Tableitem), 1, tblfile);
  
  if (sizewritten != 1) 
    return -1;

  cleanoutput (tblfile);
  return 1;
}

int
readTableitem (FILE *tblfile, Oid oid, Tableitem *titem)
{
  int sizeread;
  
  if (oid == 0)
    return -1;

  cleaninput (tblfile);

  fseek (tblfile, oid*sizeof(Tableitem), SEEK_SET);
  sizeread = fread (titem, sizeof(Tableitem), 1, tblfile);
  
  if (sizeread != 1) 
    return -1;
  
  return 1;
}

/* ------------------------------------------------------------ 
   openlist functions, and functions to get new oid and offsets
   ------------------------------------------------------------ */

long 
put2oplist (Database *db, int file, long ofs, long length)
{
  long fpos;
  int found = 0;
  Oplitem oplitem, newoplitem;
  
  if (file > db->datafileno)
    SM_GOERR (NWEEFID);

  cleaninput (db->data[file].opfile);

  fseek (db->data[file].opfile, 0, SEEK_SET);
  
  while ((!found) && (!feof (db->data[file].opfile))) {
    if (fread (&oplitem, sizeof (Oplitem), 1, db->data[file].opfile) != 1) {
      if (feof(db->data[file].opfile))
	goto eofhd;		/* ende der Datei */
      else
	SM_GOERR (NWEWOPL);
    }
    if (oplitem.length == 0)
      found = 1;
  }
  
eofhd:
  if (found) {
    fpos = ftell(db->data[file].opfile) - sizeof(Oplitem); /* fileposition */
    fseek (db->data[file].opfile, fpos, SEEK_SET);
  }
  else {
    fseek(db->data[file].opfile, 0, SEEK_END); 	/* ermittle endoffile! */
  }
  
  /* nun schreibe die Daten in die Tabelle */
  newoplitem.ofs = ofs;
  newoplitem.length = length;
  if (fwrite(&newoplitem, sizeof(Oplitem), 1, db->data[file].opfile) != 1)
    SM_GOERR (NWEWOPL);

  cleanoutput (db->data[file].opfile);
  return 1;
errhd:
  return -1;
}

/* Sucht aus der Leerliste einen Eintrag, der dem gesuchten Groesse am
   naechsten kommt; die Auswahl geht von vorne nach hinten; wird lediglich
   ein annaehernd grosser Platz gefunden, gibt die Routine den Restplatz
   fuer zukuenftige Suchen frei */
#define FOUND_NOTHING    0
#define FOUND_BAD        1
#define FOUND_VERY_GOOD  2
long
getnewpos(Database *db, int file, long minlength)
{
  Oplitem oplitem, newoplitem;
  long 
    smallestfoundOfs = 0,
    smallestfoundSize = 0xffff,	/* at first assume largest! */
    inOfs = 0;
  int 
    found = FOUND_NOTHING;
  long 
    ofs = 0;			/* the return offset */

  if (file > db->datafileno)
    SM_GOERR (NWEEFID);

  cleaninput (db->data[file].opfile);
  minlength += sizeof (Itemhead);
  
  while ((!found) && (!feof(db->data[file].opfile))) {
    if (fread(&oplitem, sizeof(Oplitem), 1, db->data[file].opfile) != 1) { 
      if (feof(db->data[file].opfile))
	goto eofhd; /* ende der Datei */
      else
	SM_GOERR (NWEGETNEWPOS);
    }
    
    if (oplitem.length == minlength) {	/* gefunden! */
      inOfs = ftell(db->data[file].opfile) - sizeof(Oplitem);
      smallestfoundOfs = oplitem.ofs;
      smallestfoundSize = oplitem.length;
      found = FOUND_VERY_GOOD;	/* very good! */
    }
    else {
      if (oplitem.length > minlength) {	/* groesseres gefunden! */
	if (oplitem.length < smallestfoundSize) {
	  inOfs = ftell(db->data[file].opfile) - sizeof(Oplitem);
	  smallestfoundOfs = oplitem.ofs;
	  smallestfoundSize = oplitem.length;
	  found = FOUND_BAD;	/* nicht so good */
	}
      }
    }
  }

 eofhd:
  
  switch (found) {
  case FOUND_BAD:		/* den rest wieder eintragen */
    ofs = smallestfoundOfs;
    fseek (db->data[file].opfile, inOfs, SEEK_SET);
    if (smallestfoundSize - minlength > sizeof (Itemhead) ) {
      newoplitem.ofs = smallestfoundOfs + minlength;
      newoplitem.length = smallestfoundSize - minlength;
    }
    else {
      newoplitem.ofs = 0;
      newoplitem.length = 0;
    }
    if (fwrite(&newoplitem, sizeof(Oplitem), 1, db->data[file].opfile) != 1)
      SM_GOERR (NWEUPDOPL);
    break;
    
  case FOUND_VERY_GOOD:		/* new item fills complete the gap */
    ofs = smallestfoundOfs;
    fseek (db->data[file].opfile, inOfs, SEEK_SET);
    newoplitem.ofs = 0;		/* kill old openlist item */
    newoplitem.length = 0;
    if (fwrite(&newoplitem, sizeof(Oplitem), 1, db->data[file].opfile) != 1)
      SM_GOERR (NWEUPDOPL);
    break;
    
  case FOUND_NOTHING:		/* no gap found, put to end of file */
  default:
    fseek (db->data[file].file, 0, SEEK_END); /* search in datafile! */
    ofs = ftell (db->data[file].file);
    if (ofs == -1) 
      SM_GOERR (NWEGETNEWPOS);
    break;
  }
  
  cleanoutput (db->data[file].opfile);
  return ofs;
  
 errhd:
  return -1;
}

/* ermittelt die naechste OID -- im Moment noch rein rechnerisch.
   Irgendwann sollte es eine Art Openlist fr die Tabledatei geben, in der
   wieder frei- gegeben OIDs abgelegt werden (Sure?). getnew_oid holt die
   sich dann daraus.  */
Oid
getnew_oid (Database *db)
{
  Oid fend;

  cleaninput (db->tblfile);
  
  fseek(db->tblfile, 0, SEEK_END);
  fend = ftell(db->tblfile) / sizeof(Tableitem); /* rein rechnerisch */
  if (fend < FIRSTOID)
    fend = FIRSTOID;		/* to start at specific point  */
  return fend;
}

Oid
gettotal_oid (Database *db)
{
  Oid fend;

  cleaninput (db->tblfile);
  
  fseek(db->tblfile, 0, SEEK_END);
  fend = ftell(db->tblfile) / sizeof(Tableitem); /* rein rechnerisch */
  if (fend < FIRSTOID)
    fend = FIRSTOID;		/* to start at specific point  */
  return fend;
}

/* ------------------------------------------------------------ 
   basic blob functions
   ------------------------------------------------------------ */
void
build_external_datafile_name (Database *db, char *fname, Oid oid)
{
  sprintf (fname, "%s/%s/%ld", db->path, BLOB_DIR_NAME, oid);
}

int
del_external_datafile (Database *db, Oid oid)
{
  char fname[FILENAME_MAX];

  build_external_datafile_name (db, fname, oid);
  if (unlink (fname) < 0)
    SM_GOERR (NWEDBLOB);
  
  return 0;

 errhd:
  return -1;
}

int
write_external_datafile (Database *db, Oid oid, char *buf, int buflen)
{
  FILE *stream;
  char fname[FILENAME_MAX];
  int sizewritten = 0;
  Blobhead blobhead;

  build_external_datafile_name (db, fname, oid);
  
  /* now write the buffer to this file */
  stream = fopen (fname, "w");
  if (!stream) 
    SM_GOERR (NWECBLOB);

  blobhead.oid = oid;
  blobhead.length = buflen;
  blobhead.r1 = blobhead.r2 = 0;

  fwrite (&blobhead, sizeof(Blobhead), 1, stream); /* write header */
  sizewritten = fwrite (buf, 1, buflen, stream); /* write data */

  if (sizewritten != buflen)
    SM_GOERR (NWEWBLOB);

  fclose (stream);
  
  return 0;

 errhd:
  return -1;
}

char *
read_external_datafile (Database *db, Oid oid, long *size)
{
  FILE *stream;
  char fname[FILENAME_MAX];
  int sizeread = 0;
  Blobhead blobhead;
  char *buf;

  *size = 0;

  build_external_datafile_name (db, fname, oid);
  
  /* now write the buffer to this file */
  stream = fopen (fname, "r");
  if (!stream) 
    SM_GOERR (NWERBLOB);
  
  sizeread = fread (&blobhead, sizeof(Blobhead), 1, stream);
  if (sizeread != 1)
    goto errhd;
  if (blobhead.length <= 0)
    goto errhd;

  buf = (char*) malloc (blobhead.length);
  sizeread = fread (buf, 1, blobhead.length, stream);
  *size = blobhead.length;

  fclose (stream);

  return buf;

 errhd:
  return NULL;
}

/* ======================================================================
   This are the `public' functions, which are used by other system
   functions.  All functions above must be considered as internal
   ====================================================================== */

/* ------------------------------------------------------------ 
   main function: read, new, rewrite 
   ------------------------------------------------------------ */
int
readItem (Database *db, Oid oid, char **buf, Objstat *objstat)
{
  Tableitem tableitem;
  Itemhead itemhead;
  long sizeread, bufsize;
  int findex;
  
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Read an item from a remote site.\n");
    return 0;
  }
#endif
  
  objstat->fid = 0;
  objstat->uid = 0;
  objstat->gid = 0;
  objstat->accesscode = 0;
  objstat->status = status_UNDEF;
  objstat->oid = 0;
  objstat->length = 0;
  objstat->creat = 0;

  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);			/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);
  
  findex = get_fid_index (db, tableitem.fid);
  
  if (findex < 0)
    SM_GOERR (NWEEFID);
  if (readItemhead (db->data[findex].file, tableitem.ofs, &itemhead) < 0)
    SM_GOERR (NWERITEMH);			/* read head of item */

  switch (itemhead.status) {
  case status_DEL:
    *buf = NULL;
    SM_GOERR (NWEIDEL);
    break;
  case status_ENTRY:		/* nothing happens */
    break;
  default:
    SM_GOERR (NWEDATACORRUPT);
  }
  if (itemhead.length > 0) {
    *buf = (char *) malloc (itemhead.length + 1);
    bufsize = itemhead.length + 1;
    
    fseek (db->data[findex].file, tableitem.ofs + sizeof(Itemhead),
	   SEEK_SET);
    sizeread = fread (*buf, 1, bufsize, db->data[findex].file);
    objstat->length = itemhead.length;
  }
  else {			/* otherwise it is a blob! */
    *buf = read_external_datafile (db, oid, &bufsize);
    objstat->length = bufsize;
  }
  
  objstat->fid = tableitem.fid;
  objstat->uid = itemhead.uid;
  objstat->gid = tableitem.gid;
  objstat->accesscode = tableitem.accesscode;
  objstat->status = status_UNDEF;
  objstat->oid = oid;
  objstat->creat = itemhead.creat;
  
  return bufsize;
  
 errhd:
  return -1;
}

int
rewriteItem (Database *db, Oid oid, char *buf, long length)
{
  Tableitem tableitem, newtableitem;
  Itemhead itemhead, newitemhead;
  long sizewritten;
  long newpos;
  int findex;
    
#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Rewrite an item on a remote site.\n");
    return 1;
  }
#endif

  /* first read the old headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);
  
  findex = get_fid_index (db, tableitem.fid);
  
  if (findex < 0)
    SM_GOERR (NWEEFID);
    
  if (readItemhead (db->data[findex].file, tableitem.ofs, &itemhead) < 0)
    SM_GOERR (NWERITEMH);		/* read head of item */
  
  /* nun setze die Header und Listeneintraege neu zusammen */
  newpos = newtableitem.ofs = getnewpos (db,
					 findex,
					 length);
  if (newpos < 0)
    SM_GOERR (NWEEPOS);
  
  newtableitem.ofs = newpos;
  newtableitem.fid = tableitem.fid;
  newtableitem.accesscode = tableitem.accesscode;
  newtableitem.uid = tableitem.uid;
  newtableitem.gid = tableitem.gid;
  
  newitemhead.oid = oid;
  newitemhead.status = (char) status_ENTRY;
  newitemhead.uid = itemhead.uid;
  newitemhead.length = length;
  newitemhead.creat = itemhead.creat;

  if (length >= blob_limit)
    newitemhead.length = 0;	/* this tells the system: it's a blob */

  /* nun speichere den neuen Datensatz ab */
  if (writeItemhead (db->data[findex].file, newpos, &newitemhead) < 0)
    SM_GOERR (NWEWITEMH);		/* save header */
  
  if (length < blob_limit) {
    fseek (db->data[findex].file, newpos + sizeof(Itemhead), SEEK_SET);
    sizewritten = fwrite (buf, 1, length, db->data[findex].file);
    if (sizewritten != length)
      SM_GOERR (NWEWDATA);
    
    cleanoutput (db->data[findex].file);
  }
  else {			/* save a blob */
    if (write_external_datafile (db, oid, buf, length) < 0)
      SM_GOERR (NWEWBLOB);
  }
  
  /* sofern das geklappt hat, schreibe den neuen Tabelleneintrag*/
  if (writeTableitem (db->tblfile, oid, &newtableitem) < 0)
    SM_GOERR (NWEWTABLE);
  
  /* ok, soweit, dann wird jetzt der alte Platz an die Openliste
     gemeldet, und zwar inclusive dem Itemhead, da dieser mit
     ueberschrieben werden kann! Danach ueberschreibe den alten
     Datensatz-Itemhead als frei! */
  if (put2oplist (db, findex, tableitem.ofs, 
		  itemhead.length + sizeof (Itemhead)) < 0)
    goto errhd;
  
  return 1;
  
 errhd:
  return -1;
}


Oid 
newItem (Database *db, 
	 Fid fid, Uid uid, Gid gid, ACode acode, 
	 char *buf, long length)
{
  long newpos;
  Oid newoid;
  Tableitem newtableitem;
  Itemhead newitemhead;
  long sizewritten;
  int findex;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Insert an item to a remote site.\n");
    return 0;
  }
#endif
    
  findex = get_fid_index (db, fid);
  if (findex < 0)
    SM_GOERR (NWEEFID);
  
  newpos = getnewpos (db, findex, length);
  if (newpos < 0)
    SM_GOERR (NWEEPOS);
  newoid = getnew_oid (db);
  
  newtableitem.ofs = newpos;
  newtableitem.fid = fid;
  newtableitem.accesscode = acode;
  newtableitem.uid = uid;
  newtableitem.gid = gid;
  
  newitemhead.oid = newoid;
  newitemhead.status = (char) status_ENTRY;
  newitemhead.uid = uid;
  newitemhead.length = length;
  newitemhead.creat = get_system_time ();

  if (length >= blob_limit)
    newitemhead.length = 0;	/* this tells the system: it's a blob */

  if (writeItemhead (db->data[findex].file, newpos, &newitemhead) < 0)
    SM_GOERR (NWEWITEMH);		/* save header */

  if (length < blob_limit) {
    fseek (db->data[findex].file, newpos+sizeof(Itemhead), SEEK_SET);
    sizewritten = fwrite (buf, 1, length, db->data[findex].file);
    if (sizewritten != length)
      SM_GOERR (NWEWDATA);
    cleanoutput (db->data[findex].file);
  }
  else {
    if (write_external_datafile (db, newoid, buf, length) < 0)
      SM_GOERR (NWEWBLOB);
  }
  
  /* sofern das geklappt hat, schreibe den neuen Tabelleneintrag*/
  if (writeTableitem (db->tblfile, newoid, &newtableitem) < 0)
    SM_GOERR (NWEWTABLE);
  
  return newoid;
  
 errhd:
  return 0;
}


/* ------------------------------------------------------------ 
   Kill and delete functions 
   ------------------------------------------------------------ */
/* markiert einen Datensatz als geloescht, indem es den Status im
   Datensatzkopf auf 'd' (= deleted) setzt.  Alle uebrigen Resourcen
   bleiben jedoch erhalten, sodass mit einem undelItem der Datensatz wieder
   hergestellt werden kann.  Der Datensatz sollte dann erst durch einen
   vacuum-befehl geloescht werden. */
int
delItem (Database *db, Oid oid)
{
  Tableitem tableitem;
  Itemhead itemhead;
  int findex;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Delete an item on a remote site.\n");
    return 0;
  }
#endif
    
  /* first read the old headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);
  
  findex = get_fid_index (db, tableitem.fid);
  
  if (findex < 0)
    SM_GOERR (NWEEFID);
  if (readItemhead (db->data[findex].file, tableitem.ofs, &itemhead) < 0)
    SM_GOERR (NWERITEMH);		/* read head of item */
  
  if (itemhead.status != (char) status_ENTRY) 
    SM_GOERR (NWEDNITEM);
  
  itemhead.status = (char) status_DEL;
  if (writeItemhead (db->data[findex].file, tableitem.ofs,
		     &itemhead) < 0)
    SM_GOERR (NWEWITEMH);		/* save header */
  
  return 1;
  
 errhd:
  return -1;
}

/* stellt einen Datensatz wieder her, indem der Itemhead-Status wieder auf
   `e' gesetzt wird. */
int
undelItem (Database *db, Oid oid)
{
  Tableitem tableitem;
  Itemhead itemhead;
  int findex;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Undelete an item on a remote site.\n");
    return 0;
  }
#endif
  
  /* first read the old headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);

  findex = get_fid_index (db, tableitem.fid);

  if (findex < 0)
    SM_GOERR (NWEEFID);
  if (readItemhead (db->data[findex].file, tableitem.ofs,
		    &itemhead) < 0)
    SM_GOERR (NWERITEMH);		/* read head of item */
    
  if (itemhead.status != (char) status_DEL) 
    SM_GOERR (NWEUDNDITEM);
    
  itemhead.status = (char) status_ENTRY;
  if (writeItemhead (db->data[findex].file, tableitem.ofs,
		     &itemhead) < 0)
    SM_GOERR (NWEWITEMH);		/* save header */
    
  return 1;
    
 errhd:
  return -1;
}


/* loescht ein Eintrag unwiderruflich: schreibt den belegten Speicherplatz
   in die openlist, und setzt den Tableeintrag auf -1 (= kein Item).  Ein
   Blob wird automatisch aus dem blob-verzeichnis gelscht */
int
killItem (Database *db, Oid oid)
{
  Tableitem tableitem;
  Itemhead itemhead;
  int findex;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Kill an item on a remote site.\n");
    return 0;
  }
#endif

  /* first read the old headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);
  
  findex = get_fid_index (db, tableitem.fid);
  
  if (findex < 0)
    SM_GOERR (NWEEFID);
  if (readItemhead (db->data[findex].file, tableitem.ofs,
		    &itemhead) < 0)
    SM_GOERR (NWERITEMH);		/* read head of item */
  
  /* now put the entire space to the open list */
  if (put2oplist (db, findex, tableitem.ofs, 
		  itemhead.length + sizeof (Itemhead)) < 0)
    goto errhd;

  tableitem.ofs = status_KILLED;	/* kill tableentry -1 = no ref! */
  tableitem.fid = 0;
  tableitem.accesscode = 0;
  tableitem.uid = 0;
  tableitem.gid = 0;
  if (writeTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWEWTABLE);

  /* at least kill a possible blob */
  if (itemhead.length == 0) {	/* a blob? */
    if (del_external_datafile (db, oid) < 0)
      goto errhd;		/* errmsg set by del_external_datafile */
  }
  
  return 1;
  
 errhd:
  return -1;
}

int
readItemCtrlData (Database *db, Oid oid, Objstat *objstat)
{
  Tableitem tableitem;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Read the item ctrl data from a remote site.\n");
    return 0;
  }
#endif
    
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);			/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);
  if (get_fid_index (db, tableitem.fid) < 0)
    SM_GOERR (NWEEFID);
    
  objstat->fid = tableitem.fid;
  objstat->uid = tableitem.uid;
  objstat->gid = tableitem.gid;
  objstat->accesscode = tableitem.accesscode;
  objstat->status = 0;
  objstat->oid = oid;
  objstat->length = 0;
  objstat->creat = 0;

  return 1;
    
 errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   NDERUNG UND SETZEN DER ZUGRIFFSRECHTE/FLAGS
   ---------------------------------------------------------------------- */
/* changeItem changes the flags and (accessable) data in an item
   table entry.  Depending on FUNC it sets: 

   0  UID_FLAG      the uid
   1  GID_FLAG      the gid
   2  ACODE_FLAG    the accesscode flag
*/
int
changeItem (Database *db, Oid oid, int func, int data)
{
  Tableitem tableitem;

#if DISTRIBUTED_DB
  if (db->remote) {
    printf ("Change item uid on a remote site.\n");
    return 0;
  }
#endif
    
  /* first read the headers */
  if (readTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWERTABLE);		/* read tableitem */
  if (tableitem.ofs == status_KILLED)
    SM_GOERR (NWEIDKILL);

  /* now change the tableitem and save it */
  switch (func) {
  case 0:			/* UID */
    tableitem.uid = (Uid) data;
    break;
  case 1:			/* GID */
    tableitem.gid = (Gid) data;
    break;
  case 2:			/* ACODE */
    tableitem.accesscode = (ACode) data;
    break;
  }
  if (writeTableitem (db->tblfile, oid, &tableitem) < 0)
    SM_GOERR (NWEWTABLE);
 
  return 1;
  
 errhd:
  return -1;
}

