/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#if defined HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#if defined TIME_WITH_SYS_TIME_H
# include <time.h>
#endif

#include "sm.h"
#include "kdbs.h"
#include "lock.h"
#include "net.h"

Reslistentry resourceslist[MAXRESOURCES]; /* die lockdaten */
Lockreqlist lockreqlist_data[MAXLOCKREQ]; /* die eigentlichen Daten */
Lockreqlist *lockreqlist[MAXLOCKREQ]; /* prt auf lockreqlist_data */
int lockreqlistcount = 0;
int lockchanged;


/* RESCMP vergleicht zwei Resourceneintrge; 
 * rckgabe: true equal, false unequal
 *
 * SETRESOURCE weit einer resourcevariablen die entsprechenden Werte zu
 *
 * RESCPY kopiert den Inhalt der Resource in die Andere
 */
bool
rescmp (Resource *fres, Resource *sres)
{
  if (fres->type != sres->type) 
    return false;
  if (fres->eid != sres->eid) 
    return false;
  if (fres->dbid != sres->dbid) 
    return false;
  
  return true;
}

void
setresource (Resource *res, int type, long eid, Dbid dbid)
{
  res->type = type;
  res->eid = eid;
  res->dbid = dbid;
}

void
rescpy (Resource *tres, Resource *sres)
{
  memcpy (tres, sres, sizeof (Resource));
}

void
clearresource (int listno)
{
  resourceslist[listno].lockid = 0;
  resourceslist[listno].status = lock_NULL;
  resourceslist[listno].rid.type = lock_NULL;
  resourceslist[listno].rid.eid = -1;
  resourceslist[listno].rid.dbid = UNKNOWNDB;
  lockchanged = true;
}

void
clearlock (int lockno)
{
  int i;

  for (i = 0; i < lockreqlist[lockno]->maxrid; i++) {
    if (lockreqlist[lockno]->reslist[i] != NULL) {
      free (lockreqlist[lockno]->reslist[i]);
      lockreqlist[lockno]->reslist[i] = NULL;
    }
  }
  lockreqlist[lockno]->maxrid = -1;
  lockreqlist[lockno]->announced = false;
  lockchanged = true;
}

void 
initreslist (void)
{
  int i, j;

  for (i = 0; i < MAXRESOURCES; i++) {
    clearresource (i);
  }
  
  for (i = 0; i < MAXLOCKREQ; i++) {
    lockreqlist[i] = NULL;
    for (j = 0; j < MAXSERVRES; j++) {
      lockreqlist_data[i].reslist[j] = NULL;
    }
    lockreqlist_data[i].maxrid = -1;
    lockreqlist_data[i].announced = false;
  }
  lockreqlistcount = 0;
  lockchanged = false;
}


Lockreqlist *
alloc_free_req_item ()
{
  int i;
  
  for (i = 0; i < MAXRESOURCES; i++) {
    if (lockreqlist_data[i].maxrid == -1) 
      return &lockreqlist_data[i];
  }
  return NULL;
}

int global_lockid_counter = 0;	/* starts with 0 */

int
get_next_lockid ()
{
  global_lockid_counter++;
  return global_lockid_counter;
}

int
get_server_no (int serverid)
{
  int i;
  
  for (i = 0; i < lockreqlistcount; i++) {
    if (lockreqlist[i]->serverid == serverid) {
      return i; 
    }
  }
  return -1;
}

int
get_lock_no (int lockid)
{
  int i;
  
  for (i = 0; i < lockreqlistcount; i++) {
    if (lockreqlist[i]->lockid == lockid) {
      return i; 
    }
  }
  return -1;
}

/* deletelockno ist nur fr interne Zwecke gedacht: es erhlt als
   bergabewert kein Handle auf ein Lock, sondern die direkte Arrayadresse
   auf einen Lock. */
void
deletelockno (int lockno)
{
  int i;
  
  if (lockno >= 0) {
    for (i = 0; i < lockreqlist[lockno]->maxrid; i++) {
      if (lockreqlist[lockno]->reslist[i]->status == lock_OK) {
	deleteresource (lockreqlist[lockno]->lockid,
			&lockreqlist[lockno]->reslist[i]->rid);
      }
    }
    clearlock (lockno);
    for (i = lockno; i < lockreqlistcount; i++) {
      lockreqlist[i] = lockreqlist[i+1];
    }
    lockreqlistcount--;
    lockreqlist[lockreqlistcount] = NULL;
  }
  lockchanged = true;
}

int
insertnewresource (char status, int lockid, Resource *rid)
{
  int i;
  
  for (i = 0; i < MAXRESOURCES; i++) {
    if (resourceslist[i].status == lock_NULL) {
      resourceslist[i].lockid = lockid;
      resourceslist[i].status = status;
      rescpy (&resourceslist[i].rid, rid);
      lockchanged = true;
      return i;
    }
  }
  return 0;
}

int
deleteresource (int lockid, Resource *rid)
{
  int i;
  
  for (i = 0; i < MAXRESOURCES; i++) {
    if (rescmp(&resourceslist[i].rid, rid) &&
	(resourceslist[i].lockid == lockid)) {
      clearresource (i);
      return 0;			/* kill only one resource! */
    }
  }
  lockchanged = true;
  return 0;
}

int
changeresource (char status, int lockid, Resource *rid)
{
  int i;
  
  for (i = 0; i < MAXRESOURCES; i++) {
    if (rescmp (&resourceslist[i].rid, rid) &&
	(resourceslist[i].lockid == lockid)) {
      resourceslist[i].status = status;
      return i;
    }
  }
  return 0;
}

/* islocked ist die zentrale Routine, die entscheidet, welchen Servern
   Resourcen zugeteilt werden sollen, und welchen nicht. */
int
islocked (char status, Resource *rid)
{
  int i;
  bool locked = false;
  
  for (i = 0; i < MAXRESOURCES; i++) {
    if (rescmp(&resourceslist[i].rid, rid)) {
      switch (status) {
      case lock_SHARED:
	switch (resourceslist[i].status) {
	case lock_SHARED:	/* shared:shared */
	case lock_RESTRICTED:	/* shared:restricted */
	  locked = false; 
	  break;
	case lock_LOCKED:	/* shared:locked */
	  return true; 
	  break;
	}
	break;
      case lock_LOCKED:		/* locked:shared/restricted/locked */
	return true;
	break;
      case lock_RESTRICTED:
	switch (resourceslist[i].status) {
	case lock_SHARED: 	/* restricted:shared */
	  locked = false; 
	  break;
	case lock_LOCKED:	/* restricted:locked */
	case lock_RESTRICTED:	/* restricted:restricted */
	  return true;
	  break;
	}
	break;
      }
    }
  }
  return locked;
}


/* ----------------------------------------------------------------------
   routinen die direkt von auen zu benutzen sind 
   ---------------------------------------------------------------------- */

/* beginlock: legt einen neuen Lockrequesteintrag an und gibt ein Handle
   zurck, mit dem dieser Lockeintrag anzusprechen ist.  */
int
beginlock (int serverid)
{
  int newlockno, lockid;
  
  newlockno = lockreqlistcount;
  lockreqlist[newlockno] = alloc_free_req_item ();
  lockreqlist[newlockno]->lockid = lockid = get_next_lockid ();
  lockreqlist[newlockno]->serverid = serverid;
  lockreqlist[newlockno]->maxrid = 0;
  lockreqlist[newlockno]->announced = false;
  lockreqlistcount++;
  lockchanged = true;
  return lockid;
}

/* fgt zu einem bereits bestehenden Lock eine weitere Resource hinzu.  Als
   erstes Parameter mu ein gltiges Handle stehen.  Ist das Handler
   ungltig gibt die Funktion -1 zurck, andernfalls 0. */
int
addtolock (int lockid, char reqstatus, Resource *rid) 
{
  int newrid = 0, lockno;
  
  lockno = get_lock_no (lockid);
  if (lockno >= 0) {
    newrid = lockreqlist[lockno]->maxrid;
    lockreqlist[lockno]->maxrid++;
    
    if (lockreqlist[lockno]->reslist[newrid] == NULL) {
      lockreqlist[lockno]->reslist[newrid] =
	(Reslistentry *) malloc (sizeof (Reslistentry));
    }
    
    memcpy (&lockreqlist[lockno]->reslist[newrid]->rid, rid,
	    sizeof (Resource));
    lockreqlist[lockno]->reslist[newrid]->reqstatus = reqstatus;
    lockreqlist[lockno]->reslist[newrid]->status = lock_PENDING;
    lockreqlist[lockno]->announced = false;
    
    lockchanged = true;
    return 0;
  }
  else {
    return -1;
  }
}

/* verndert eine Resource in einem bestehenden Lock.  Wurde die angegebene
   Resource rid nicht gefunden, so gibt die Funktion 0 zurck; ist das
   Handle lockid nicht gltig gibt's -1; ist alles ok dann 1 */
int
changelock (int lockid, char reqstatus, Resource *rid) 
{
  int i, lockno;
  
  lockno = get_lock_no (lockid);
  if (lockno >= 0) {
    for (i = 0; i < lockreqlist[lockno]->maxrid; i++) {
      if (lockreqlist[lockno]->reslist[i]) {
	if (rescmp (&lockreqlist[lockno]->reslist[i]->rid, rid)) {
	  lockreqlist[lockno]->reslist[i]->reqstatus = reqstatus;
	  lockreqlist[lockno]->reslist[i]->status = lock_PENDING;
	  lockreqlist[lockno]->announced = false;
	  goto okhd;
	}
      }
    }
    return 0;
  }
  else {
    return -1;
  }
  
okhd: 
  deleteresource (lockreqlist[lockno]->lockid,
		  &lockreqlist[lockno]->reslist[i]->rid);
  lockchanged = true;
  return 1;
}

/* lscht den Lock-Eintrag, der spezifisch unter dem Handle lockid gemeldet
   ist.  Ist die Operation ok abgelaufen ist der Returnvalue 0, bei einem
   Fehler -1 (ungltiges Handle) */
int
deletelock (int lockid)
{
  int lockno;
  
  lockno = get_lock_no (lockid);
  
  if (lockno >= 0) {
    deletelockno (lockno);
    lockchanged = true;
    return 0;
  }
  return -1;
}

/* lscht alle Locks, die vond einem spezifischen Server angemeldet sind.
   Nur fr generelles saubermachen sinnvoll */
void
deleteallserverlocks (int serverid)
{
  int i;
  
  for (i = 0; i < lockreqlistcount; ) {
    if (lockreqlist[i]->serverid == serverid) {
      deletelockno (i);
    }
    else {			/* i darf nur weiter gezhlt werden, */
      i++;			/* wenn nicht gelscht wurde, da das */
    }                           /* Lschen die Listeheranzieht! */
  }
  lockchanged = true;
}

/* updatelocklist sollte 'gelegentlich' aufgerufen werden, um die Lockliste
   zu berprfen: d.h., immer dann, wenn ein Server eine Anfrage startet,
   eine zurcknimmt, oder hnliches, mu diese Funktion ausgefhrt werden.
   Hat sich nichts am globalen Schwergewicht gendert gibt die Funktion
   false (0) zurck, ansonsten true (1). */
bool
updatelocklist ()
{
  int i, lockno;
  bool retval = false;		/* wurde was gendert? */

  for (lockno = 0; lockno < lockreqlistcount; lockno++) {
    
    for (i = 0; i < lockreqlist[lockno]->maxrid; i++) {
      
      if (lockreqlist[lockno]->reslist[i]->status == lock_PENDING) {
	if (!islocked (lockreqlist[lockno]->reslist[i]->reqstatus,
		       &lockreqlist[lockno]->reslist[i]->rid)) {
	  insertnewresource (lockreqlist[lockno]->reslist[i]->reqstatus,
			     lockreqlist[lockno]->lockid,
			     &lockreqlist[lockno]->reslist[i]->rid);
	  lockreqlist[lockno]->reslist[i]->status = lock_OK; 
	  lockchanged = true;
	  retval = true;
	}
      }
    }
  }
  return retval;
}

/* iscomplete geht die unter dem Handle lockid angemeldeten Locks durch und
   berprft sie auf ihre Wertigkeit.  Steht nur eine Resource auf PENDING,
   so gibt iscomplete false zurck, ansonsten true.  Mit true sind alle
   angeforderten Resourcen also vollstndig erhalten.  */
bool
iscomplete (int lockno)
{
  int i;
  
  for (i = 0; i < lockreqlist[lockno]->maxrid; i++) {
    if (lockreqlist[lockno]->reslist[i]->status == lock_PENDING)
      return false;		/* a pending resource -> not complete */
  }
  return true;
}



/* ------------------------------------------------------------
   Convienienceroutinen zu den Locks
   ------------------------------------------------------------ */

/* hngt eine resource an eine Kette von Resourcen (resourcestring) an. 
 * der Zielstring mu bereits ausreichend bemessen sein. */
void
resstrcat (char *resstr, int status, int type, Dbid db, long eid)
{
  char s[32];
  sprintf(s, "%c%c%02dX%08lX", status, type, db, eid);
  strcat (resstr, s);
}

/* kopiert eine resource in einen string (Pro resource 10 Bytes) */
void
resstrcpy (char *resstr, int status, int type, Dbid db, long eid)
{
  char s[32];
  sprintf(s, "%c%c%02dX%08lX", status, type, db, eid);
  strcpy (resstr, s);
}


/* ----------------------------------------------------------------------
   HIGHEND LOCK FUNCTIONS
   ---------------------------------------------------------------------- */
/* getresources takes 3 parameters: func is BEGINTRANS, ADDTRANS or
   CHANGETRANS and specifies the mode of resource locking; lockid is the
   ID-handle of the lock request (it is of course interesting only with
   ADDTRANS and CHANGETRANS---for BEGINTRANS it should be 0); resstr is a
   readymade string with the specific resource information.  getresources
   waits for the daemon reporting KCP_OK and returns the lockid. 
   -3 -> halt request from daemon
   -2 -> timeout 
   -1 -> error
   0 -> denied
   > 0 lockid
 */
int
getresources (int socket, int connectid, char *connkey, char *func,
	      int lockid, char *resstr, int timeoutval)
{
  int tmpsize = 1024;
  char *tmp = (char*) malloc (tmpsize), *token, *tail;
  int retlockid;
  fd_set active_fd_set, read_fd_set;	/* for select */
  int  selectval, sock;
  struct timeval timeout;			/* fr die timeoutroutine */
  
  if (writesockf (socket, "%s %d %s %d %s", 
		  func,		/* the func: BEGINTRANS ADDTRANS CHANGETRANS */
		  connectid, connkey, /* the passport */
		  lockid,	/* the lock ID */
		  resstr	/* the resource string */
		  ) <= 0)
    goto errhd;
  
  FD_ZERO (&active_fd_set);	/* init activ-sockets for select */
  FD_SET (socket, &active_fd_set);
  
  while (1) {
    
    read_fd_set = active_fd_set;

    /* this curious loop to catch signals */
  selectjump:
    if (timeoutval > 0) {
      timeout.tv_sec = timeoutval;
      timeout.tv_usec = 0;
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, &timeout);
    }
    else {
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, NULL);
    }
    if (selectval < 0) {	/* signal EINTR caught? */
      if (errno == EINTR) 
	goto selectjump;
      NET_GOERR (_("select error"));
    }
    if (selectval == 0) {	/* timeout! */
      NET_DESCERR (_("timeout. Resources locked"));
      goto timeout_hd;
    }
    
    for (sock = 0; sock < FD_SETSIZE; ++sock) {
      if (FD_ISSET (sock, &read_fd_set)) { /* activ socket? */

	if (read_from_sock (sock, &tmp, &tmpsize) < 0)
	  goto errhd;

	token = strtok (tmp, " \n\r");
	if (token) {
	  if (strcasecmp (token, KCP_TRANS) == 0) {
	    token = strtok (NULL, " \n\r");
	    if ((token) &&
		(strlen (token) > 0)) {
	      if (token[0] == KCP_OKC)
		goto okhd;
	      else if (token[0] == KCP_ERRC) 
		goto deniedhd;
	      else if (token[0] == KCP_WAITC) {
		token = strtok (NULL, " \n\r");
		retlockid = strtol (token, &tail, 0);
	      }
	    }
	  }
	  else if (strcasecmp (token, KCP_HALT) == 0)
	    goto halthd;
	}
      }
    }
  }
  
 okhd:
  free (tmp);
  token = strtok (NULL, " \n\r"); /* now take the lock ID number */
  retlockid = strtol (token, &tail, 0);
  return retlockid;
 deniedhd:
  free (tmp);
  return 0;
 errhd:
  free (tmp);
  return -1;
 timeout_hd:
  free (tmp);
  /* 1) if BEGINTRANS & ADDTRANS: free the resources, since WE have stopped
     the request!  2) if CHANGETRANS: just return -2: it is up to the
     higher level function to restore the original resource, since we don't
     know about the original state */
  if ((strcmp (func, KCP_BEGINTRANS) == 0)
      || (strcmp (func, KCP_ADDTRANS) == 0)) {
    freeresources (socket, connectid, connkey, retlockid, timeoutval);
  }
  return -2;
 halthd:
  free (tmp);
  return -3;
}


/* frees the locks requested for the handle lockid. If lockid is 0 EVERY
   lock is removed from the daemon (total cleanup) */
int
freeresources (int socket, int connectid, char *connkey, int lockid,
	       int timeoutval)
{
  int tmpsize = 1024;
  char *tmp = (char*) malloc (tmpsize), *token;
  fd_set active_fd_set, read_fd_set;	/* for select */
  int selectval, sock;
  struct timeval timeout;			/* fr die timeoutroutine */
  
  if (writesockf (socket, "%s %d %s %d\n",
		  KCP_ENDTRANS, connectid, connkey, lockid) <= 0)
    goto errhd;
  
  FD_ZERO (&active_fd_set);	/* init activ-sockets for select */
  FD_SET (socket, &active_fd_set);
  
  while (1) {
    
    read_fd_set = active_fd_set;
    
    /* this curious loop to catch signals */
  selectjump:
    if (timeoutval > 0) {
      timeout.tv_sec = timeoutval;
      timeout.tv_usec = 0;
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, &timeout);
    }
    else {
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, NULL);
    }
    if (selectval < 0) {	/* signal EINTR caught? */
      if (errno == EINTR) 
	goto selectjump;
      NET_GOERR (_("select error"));
    }
    if (selectval == 0) {	/* timeout! */
      NET_DESCERR (_("timeout"));
      goto timeout_hd;
    }
    
    for (sock = 0; sock < FD_SETSIZE; ++sock) {
      if (FD_ISSET (sock, &read_fd_set)) { /* activ socket? */
	
	if (read_from_sock (sock, &tmp, &tmpsize) < 0)
	  goto errhd;
	
	token = strtok (tmp, " \n\r");
	if (token) {
	  if (strcasecmp (token, KCP_ENDTRANS) == 0) {
	    token = strtok (NULL, " \n\r");
	    if ((token) &&
		(strlen (token) > 0)) {
	      if (token[0] == KCP_OKC)
		goto okhd;
	      else if (token[0] == KCP_ERRC) 
		goto deniedhd;
	    }
	  }
	  else if (strcasecmp (token, KCP_HALT) == 0)
	    goto halthd;
	}
      }
    }
  }
  
 okhd:
  free (tmp);
  return 1;
 deniedhd:
  free (tmp);
  return 0;
 errhd:
  free (tmp);
  return -1;
 timeout_hd:
  free (tmp);
  return -2;
 halthd:
  free (tmp);
  return -3;
}

/* ----------------------------------------------------------------------
   OPEN & CLOSE DATABASE
   ---------------------------------------------------------------------- */
int
verify_database (int socket, int connectid, char* connkey, Uid clientuid,
		 char *dbname, VOretval **voretval)
{
  int tmpsize = 1024;
  char *tmp = (char*) malloc (tmpsize), *token, *tail;
  
  *voretval = (VOretval *) malloc (sizeof (VOretval));

  if (writesockf (socket, "%s %d %s %d %s", 
		  KCP_VERIFYDB,
		  connectid, connkey,
		  clientuid, dbname) <= 0)
    goto errhd;
  
  if (read_from_sock (socket, &tmp, &tmpsize) < 0)
    goto errhd;
  
  token = strtok (tmp, " \r");	/* get +OK or -ERR */
  if ((token) &&
      (strlen (token) > 0) &&
      (token[0] == '+')) {
    token = strtok (NULL, " \r"); /* get the dbid */
    if (!token)
      goto errhd;
    (*voretval)->dbid = strtol (token, &tail, 0);
    
    token = strtok (NULL, " \r"); /* get the path */
    if (!token)
      goto errhd;
    (*voretval)->path = strdup (token);
    
    token = strtok (NULL, " \r"); /* get the 'are we the owner?' */
    if (!token)
      goto errhd;
    if (strcmp (token, "owner") == 0)
      (*voretval)->dbowner = true;
    else
      (*voretval)->dbowner = false;
  }
  else
    goto deniedhd;

  free (tmp);
  return 1;
deniedhd:
  free (tmp);
  return 0;
errhd:
  free (tmp);
  return -1;
}


