/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include <ctype.h>

#include "proto.h"
#include "hc.h"

static char s_vectorq[] = "vector?";
static char a_vectorq[] = "*";
Atom
f_vectorq (Atom list)
{
  return VECTORQ(CAR(list));
}

static char s_make_vector[] = "make-vector";
static char a_make_vector[] = "n.*";
Atom
f_make_vector (Atom list)
{
  int size = numv(CAR(list));
  Atom fill = ((CDR(list) != NIL) ? CADR(list) : NIL);
  Vector *v = make_vector (size, size, fill);
  return NEWVECTOR(v);
}

static char s_make_object[] = "make-object";
static char a_make_object[] = "i";
Atom
f_make_object (Atom list)
{
  Vector *v = empty_object (AHASH(h_CORE), CAR(list));
  return NEWVECTOR (v);
}

#define LIST2VECTOR(list)				\
({							\
  Atom _l = (list);					\
  int _size = LENGTH(_l);				\
  Vector *_v = make_vector (_size, _size, FALSE);	\
  Atom _s = _l;						\
  int _i = 0;						\
  while (TYP(_s) == CELL_P) {				\
    _v->array[_i] = CAR(_s);				\
    _s = CDR(_s);					\
    _i++;						\
  }							\
  NEWVECTOR(_v);					\
})

static char s_vector[] = "vector";
static char a_vector[] = ".>*";
Atom
f_vector (Atom list)
{
  return LIST2VECTOR(list);
}

static char s_list2vector[] = "list->vector";
static char a_list2vector[] = "l";
Atom
f_list2vector (Atom list)
{
  return LIST2VECTOR(CAR(list));
}

static char s_vector_length[] = "vector-length";
static char a_vector_length[] = "v";
Atom
f_vector_length (Atom list)
{
  return NEWINT (VECTOR_SIZE(CAR(list)));
}

static char s_vector_ref[] = "vector-ref";
static char a_vector_ref[] = "vn";
Atom
f_vector_ref (Atom list)
{
  return vector_ref(OBL_VECTOR(CAR(list)), numv(CADR(list)));
}

static char s_vector_set[] = "vector-set!";
static char a_vector_set[] = "vn*";
Atom
f_vector_set (Atom list)
{
  vector_set(OBL_VECTOR(CAR(list)), CADDR(list), numv(CADR(list)));
  return UNSPECIFIED;
}

static char s_vector2list[] = "vector->list";
static char a_vector2list[] = "v";
Atom
f_vector2list (Atom list)
{
  Atom rootx, rx, retval;
  int i, size = VECTOR_SIZE(CAR(list));
  Atom *a = VECTOR_ARRAY(CAR(list));
  
  PUSH_SCANL();
  rx = rootx = SAVE_ROOT();
  for (i = 0; i < size; i++)
    rx = CDR(rx) = CONS(a[i], NIL);
  retval = CDR(rootx);
  POP_SCANL();
  return retval;
}

static char s_vector_fill[] = "vector-fill!";
static char a_vector_fill[] = "v*";
Atom
f_vector_fill (Atom list)
{
  vector_fill (OBL_VECTOR(CAR(list)), CAR(CDR(list)));
  return UNSPECIFIED;
}

static char s_vector_index[] = "vector-index";
static char a_vector_index[] = "*v";
Atom
f_vector_index (Atom list)
{
  int f = vector_lookup (OBL_VECTOR(CADR(list)), CAR(list));
  if (f >= 0)
    return NEWINT (f);
  else
    return FALSE;
}

static char s_vector_setlen[] = "vector_set_length!";
static char a_vector_setlen[] = "vn";
Atom
f_vector_setlen (Atom list)
{
  int size = numv(CADR(list));
  if (size < 0)
    SETERR (list, _("negativ vector size"));
  vector_set_length (OBL_VECTOR(CAR(list)), size);
  return UNSPECIFIED;
}

static char s_vector_sort[] = "vector-sort!";
static char a_vector_sort[] = "v";
Atom
f_vector_sort (Atom list)
{
  Vector *v = OBL_VECTOR (CAR(list));
  vector_sort (v);
  return CAR(list);
}

static char s_vector_merge[] = "vector-merge";
static char a_vector_merge[] = "vvh";
Atom
f_vector_merge (Atom list)
{
  Vector *vr;

  switch (HASHV(CADDR(list))) {
  case h_OR:
    vr = vector_merge_or (OBL_VECTOR(CAR(list)), OBL_VECTOR(CADR(list)));
    break;
  case h_AND:
    vr = vector_merge_and (OBL_VECTOR(CAR(list)), OBL_VECTOR(CADR(list)));
    break;
  case h_XANY:
    vr = vector_merge_xany (OBL_VECTOR(CAR(list)), OBL_VECTOR(CADR(list)));
    break;
  case h_XOR:
    vr = vector_merge_xor (OBL_VECTOR(CAR(list)), OBL_VECTOR(CADR(list)));
    break;
  default:
    SETERR (list, _("unknown merge mode"));
  }
  return NEWVECTOR(vr);
}

/* ----------------------------------------------------------------------
   So, hier gibt's auch die Dfieldroutinen
   ---------------------------------------------------------------------- */
static char s_make_dfield[] = "make-dfield";
static char a_make_dfield[] = ".n";
Atom
f_make_dfield (Atom list)
{
  int size = 0;
  
  if (list != NIL) {
    size = CAR(list);
    if (size < 0) 
      SETERR (list, _("negativ field size"));
  }
  return NEWDFIELD(make_dfield (NULL, size));
}

static char s_free_dfield[] = "free-dfield";
static char a_free_dfield[] = "f";
Atom
f_free_dfield (Atom list)
{
  set_dfield (OBL_DFIELD(CAR(list)), NULL, 0);
  return UNSPECIFIED;
}

static char s_dfieldq[] = "dfield?";
static char a_dfieldq[] = "*";
Atom
f_dfieldq (Atom list)
{
  return DFIELDQ(CAR(list));
}

static char s_dfield_length[] = "dfield-length";
static char a_dfield_length[] = "f";
Atom
f_dfield_length (Atom list)
{
  return NEWINT(DFIELD_SIZE(CAR(list)));
}

static char s_dfield_id[] = "dfield-id";
static char a_dfield_id[] = "f";
Atom
f_dfield_id (Atom list)
{
  return NEWINT(DFIELD_BLOBID(CAR(list)));
}

static char s_dfield_binaryq[] = "dfield-binary?";
static char a_dfield_binaryq[] = "f.n";
Atom
f_dfield_binaryq (Atom list)
{
  int i = 0, bsize = DFIELD_SIZE(CAR(list));
  int testamount = ((bsize < 100) ? bsize : 100);
  char *buf = DFIELD_DATA(CAR(list));
  
  if (CDR(list) != NIL) {
    int n = numv(CADR(list));
    if (n < 0)
      testamount = bsize;
    else
      testamount = (n > bsize ? bsize : n);
  }
  
  for (i = 0; i < testamount; i++) {
    if (buf[i] == '\0')
      return TRUE;
  }
  return FALSE;
}

void
init_vectors ()
{
  Functiontable functbl[] = {
    {s_vectorq, h_VECTORQ, a_vectorq, f_vectorq},
    {s_make_vector, h_MAKE_VECTOR, a_make_vector, f_make_vector},
    {s_make_object, h_MAKE_OBJECT, a_make_object, f_make_object},
    {s_vector, h_VECTOR, a_vector, f_vector},
    {s_vector_length, h_VECTOR_LENGTH, a_vector_length, f_vector_length},
    {s_vector_ref, h_VECTOR_REF, a_vector_ref, f_vector_ref},
    {s_vector_set, h_VECTOR_SET, a_vector_set, f_vector_set},
    {s_vector2list, h_VECTOR2LIST, a_vector2list, f_vector2list},
    {s_list2vector, h_LIST2VECTOR, a_list2vector, f_list2vector},
    {s_vector_fill, h_VECTOR_FILL, a_vector_fill, f_vector_fill},
    {s_vector_index, h_VECTOR_INDEX, a_vector_index, f_vector_index},
    {s_vector_setlen, h_VECTOR_SETLEN, a_vector_setlen, f_vector_setlen},
    {s_vector_sort, h_VECTOR_SORT, a_vector_sort, f_vector_sort},
    {s_vector_merge, h_VECTOR_MERGE, a_vector_merge, f_vector_merge},
    
    {s_dfieldq, h_DFIELDQ, a_dfieldq, f_dfieldq},
    {s_dfield_length, h_DFIELD_LENGTH, a_dfield_length, f_dfield_length},
    {s_dfield_id, h_DFIELD_ID, a_dfield_id, f_dfield_id},
    {s_dfield_binaryq, h_DFIELD_BINARYQ, a_dfield_binaryq, f_dfield_binaryq},
    {s_make_dfield, h_MAKE_DFIELD, a_make_dfield, f_make_dfield},
    {s_free_dfield, h_FREE_DFIELD, a_free_dfield, f_free_dfield},
    {NULL, -1, NULL, NULL}
  };
  Functiontable *ft = functbl;
  int i;
  
  for (i = 0; (ft+i)->fname != NULL ; i++)
    make_fset ((ft+i)->fname, (ft+i)->hashcode, (ft+i)->args, (ft+i)->fnc);
}

