/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <obstack.h>

#include "proto.h"
#include "hc.h"
#include "net.h"

/* prototypes */
void o_show (Atom cell);

static struct obstack optr;

/* ----------------------------------------------------------------------
   Obstack pieces
   ---------------------------------------------------------------------- */
void *
obstack_malloc (size_t size)
{
  return smalloc (size);
}

#define obstack_chunk_alloc obstack_malloc
#define obstack_chunk_free free

#ifndef HAVE_OBSTACK_PRINTF
# define o_printf(TEMPLATE, VALUE)		\
({						\
  char *_tmp;					\
  asprintf (&_tmp, TEMPLATE, VALUE);		\
  obstack_grow (&optr, _tmp, strlen(_tmp));	\
  free (_tmp);					\
})
# define o_printstr(VALUE) 			\
({ 						\
  obstack_grow (&optr, VALUE, strlen (VALUE));	\
})
#else /* no obstack_printf */
# define o_printf(template, value)		\
({						\
  if (template)					\
    obstack_printf (&optr, template, value);	\
  else						\
    obstack_printf (&optr, value);		\
})
# define o_printstr(VALUE) 			\
({ 						\
  obstack_printf (&optr, VALUE);		\
})
#endif /* have_obstack_printf */

/* ----------------------------------------------------------------------
   Ausgabe routinen
   ---------------------------------------------------------------------- */
void
o_show_atom (Atom obj)
{
  int c;

  switch (TYP(obj)) {
  case BOOL_P:
    o_printf ("%s", 
	      (BOOLV(obj) ? 
	       (BOOLV(obj) == 2 ? "#<unspecified>" : "#t") : "#f"));
    break;
  case STR_P:
    o_printf ("\"%s\"", (char*) STR_DATA(obj));
    break;
  case HASH_P:
    o_printf ("%s", HASH_STR(obj));
    break;
  case INT_P:
    o_printf ("%d", INTV(obj));
    break;
  case CHAR_P:
    c = (int)CHARV(obj);
    if (c <= 32)
      o_printf ("#\\%s", charnametable[c]);
    else if (c == 0177)
      o_printstr ("#\\del");
    else
      o_printf ("#\\%c", CHARV(obj));
    break;
  case NIL_P:			/* the emptylist is a atom indeed! */
    o_printstr ("()");
    break;
  case OID_P:
    o_printf ("#o%ld", (Oid)OIDV(obj));
    break;
  case OBL_P:
    switch (OBL_TYP(obj)) {
    case DATE_T:
      o_printf ("#<%s>", 
		date2string ((Date) OBL_DATA(obj), h_DATEFORM_NORMAL));
      break;
    case OID_T:
      o_printf ("#o%ld", (Oid) OBL_OID(obj));
      break;
    case VECTOR_T:
      {
	int i;
	o_printstr ("#(");
	for (i = 0; i < VECTOR_SIZE(obj); i++) {
	  o_show (VECTOR_REF(obj,i));
	  o_printstr (" ");
	}
	o_printstr (")");
      }
      break;
    case DFIELD_T:
      o_printf ("#[Size: %d", DFIELD_SIZE(obj));
      o_printf (", ID: %d]", DFIELD_BLOBID(obj));
      break;
    case EXTINT_T:
      o_printf ("%d", *OBL_INT(obj));
      break;
    case SUBR_T:
      o_printstr ("#subr");
      break;
    default:
      o_printf ("#<unknown typ: %d>", OBL_TYP(obj));
      break;
    }
  }
}

void
o_show_list (Atom cell)
{
  if (TYP(CAR(cell)) == CELL_P)
    o_printstr ("("); 
  
  if (TYP(CAR(cell)) == CELL_P) 
    o_show_list (CAR(cell));
  else {
    if (TYP(CAR(cell)) == HASH_P) {
      if (HASHV(CAR(cell)) == h_CLOS) {
	o_printstr ("<closure ");
	o_show (CLOS_ARGS(cell));
	o_printstr (" ");
	o_show (CLOS_CODE(cell));
	o_printstr (">");
	return;
      }
      else if (HASHV(CAR(cell)) == h_FSET) {
	o_printstr ("<primitive instruction>");
	o_show (CDR(cell));
	return;
      }
    }
    o_show_atom (CAR(cell));
  }
  
  if (TYP(CAR(cell)) != CELL_P)
    if (TYP(CDR(cell)) != CELL_P)
      if (CDR(cell) != NIL)
	o_printstr (" ."); 
  o_printstr (" ");
  
  if (TYP(CDR(cell)) == CELL_P)
    o_show_list (CDR(cell));
  else
    if (CDR(cell) != NIL)
      o_show_atom (CDR(cell));
  
  if (TYP(CDR(cell)) != CELL_P)
    o_printstr (")"); 
}


void
o_show (Atom cell)
{
  if (TYP(cell) == CELL_P) {
    o_printstr ("(");
    o_show_list (cell);
  }
  else
    o_show_atom (cell);
}

/* ----------------------------------------------------------------------
   export the lisp code to a buffer.  Done with obstacks
   ---------------------------------------------------------------------- */
void
init_talklisp ()
{
  obstack_init (&optr);
}

void
show (Atom cell)
{
  void *s = NULL;
  
  o_show (cell);
  
#ifndef HAVE_OBSTACK_PRINTF
  obstack_grow (&optr, "\0", 1); /* string beenden! */
#endif
  
  s = obstack_finish (&optr);
  if (talklisp)
    printf ("%s\n", (char*) s);
  obstack_free (&optr, s);
}

int
show_to_sock (int socket, Atom cell)
{
  char *lispbuf;
  int sizeofbuffer;
  int retval;

  o_show (cell);

#ifndef HAVE_OBSTACK_PRINTF
  obstack_grow (&optr, "\0", 1); /* string beenden! */
#endif

  sizeofbuffer = obstack_object_size (&optr);
  lispbuf = obstack_finish (&optr);

  retval = writesockf (socket, "%c%s", KCP_OKC, KCP_OK);
  if (retval <= 0)
    return retval;
  retval = write_blob (socket, lispbuf, sizeofbuffer + 1);
  if (retval <= 0)
    return retval;
  obstack_free (&optr, lispbuf);
  return 1;
}


/* ----------------------------------------------------------------------
   Functions to send a k4 list to a frontend
   ---------------------------------------------------------------------- */
typedef struct _ScmObj {
  int typ;
  int length;
  union {
    int num;
    int blobid;
    long unum;
    char *str;
  } data;
} ScmObj;

#define SO_NONE     0 
#define SO_INT      1
#define SO_UINT     2
#define SO_STR      3
#define SO_DFIELD   4
#define SO_SYM      5
#define SO_CHAR     6
#define SO_BOOL     7

#define SETNONE_SCMOBJ(SO)			\
({						\
  SO->typ = SO_NONE;				\
  SO->length = 0;	         		\
})
#define SETINT_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_INT;				\
  SO->data.num = VAL;				\
  SO->length = sizeof (int);			\
})
#define SETUINT_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_UINT;				\
  SO->data.unum = VAL;				\
  SO->length = sizeof (long);			\
})
#define SETSTR_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_STR;				\
  SO->data.str = VAL;				\
  SO->length = strlen (VAL);			\
})
#define SETSYM_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_SYM;				\
  SO->data.str = VAL;				\
  SO->length = strlen (VAL);			\
})
#define SETCHAR_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_CHAR;				\
  SO->data.num = VAL;				\
  SO->length = sizeof (char);			\
})
#define SETBOOL_SCMOBJ(SO, VAL)			\
({						\
  SO->typ = SO_BOOL;				\
  SO->data.num = VAL;			\
  SO->length = sizeof (char);			\
})
#define SETDFIELD_SCMOBJ(SO, SIZE, BLOBID)	\
({						\
  SO->typ = SO_DFIELD;				\
  SO->data.blobid = BLOBID;			\
  SO->length = SIZE;            		\
})

int
write_scmobj (int sock, ScmObj *sobj)
{
  int retval = 1;

  if (sobj) {
    switch (sobj->typ) {
    case SO_INT:
      retval = writesockf (sock, "%c%s INT %d", KCP_OKC, KCP_OK,
			   sobj->data.num); 
      break;
    case SO_UINT:
      retval = writesockf (sock, "%c%s UINT %ld", KCP_OKC, KCP_OK,
			   sobj->data.unum); 
      break;
    case SO_STR:
      retval = writesockf (sock, "%c%s STR ", KCP_OKC, KCP_OK); 
      if (retval > 0)
	retval = write_blob (sock, sobj->data.str, sobj->length + 1);   
      break;
    case SO_DFIELD:
      retval = writesockf (sock, "%c%s DFIELD %d %d", KCP_OKC, KCP_OK,
			   sobj->length, sobj->data.blobid); 
      break;
    case SO_NONE:
      retval = writesockf (sock, "%c%s VOID", KCP_OKC, KCP_OK);
      break;
    case SO_CHAR:
      retval = writesockf (sock, "%c%s CHAR %d", KCP_OKC, KCP_OK,
			   sobj->data.num); 
      break;
    case SO_SYM:
      retval = writesockf (sock, "%c%s SYM ", KCP_OKC, KCP_OK); 
      if (retval > 0)
	retval = write_blob (sock, sobj->data.str, sobj->length + 1);   
      break;
    case SO_BOOL:
      retval = writesockf (sock, "%c%s BOOL %d", KCP_OKC, KCP_OK,
			   sobj->data.num); 
      break;
    }
  }
  return retval;
}

static ScmObj _scmobj;

ScmObj *
write_atom (Atom cell)
{
  ScmObj * sobj = &_scmobj;
  
  SETNONE_SCMOBJ (sobj);

  switch (TYP(cell)) {
  case BOOL_P:
    SETBOOL_SCMOBJ (sobj,  
		    (BOOLV(cell) ? 
		     (BOOLV(cell) == 2 ? 2 : 1) : 0));
    break;
  case STR_P:
    SETSTR_SCMOBJ (sobj, (char*) STR_DATA(cell));
    break;
  case HASH_P:
    SETSYM_SCMOBJ (sobj, (char*) HASH_STR(cell));
    break;
  case INT_P:
    SETINT_SCMOBJ (sobj, (int) INTV(cell));
    break;
  case OID_P:
    SETUINT_SCMOBJ (sobj, OIDV(cell));
    break;
  case CHAR_P:
    SETCHAR_SCMOBJ (sobj, (int) CHARV(cell));
    break;
  case OBL_P:
    switch (OBL_TYP(cell)) {
    case DATE_T:
      SETINT_SCMOBJ (sobj, (int) OBL_DATA(cell));
      break;
    case OID_T:
      SETUINT_SCMOBJ (sobj, (Oid) OBL_DATA(cell));
      break;
    case VECTOR_T:
      printf (_("No export of vectors\n"));
      SETNONE_SCMOBJ (sobj);
      break;
    case DFIELD_T:
      SETDFIELD_SCMOBJ (sobj, DFIELD_SIZE(cell), DFIELD_BLOBID(cell));
      break;
    case EXTINT_T:
      SETUINT_SCMOBJ (sobj, *OBL_INT(cell));
      break;
    default:
      SETNONE_SCMOBJ (sobj);
      break;
    }
  }  
  return sobj;
}

#define WRITE_SOBJ(SOCK, OBJ)			\
({						\
  int _retval = 1;                              \
  ScmObj *_sobj = write_atom (OBJ);		\
  if (_sobj)					\
    _retval = write_scmobj (SOCK, _sobj);	\
  _retval;                                      \
})

int
write_cell (int sock, Atom cell)
{
  Atom s = cell;
  int retval = 1;

  if (s == NIL)			/* sicherheitshalber */
    return 1;

  while (TYP(s) == CELL_P) {
    if ((TYP(CAR(s)) != CELL_P) 
	&& (CAR(s) != NIL)) {
      retval = WRITE_SOBJ (sock, CAR(s));
    }
    s = CDR(s);
  }
  if (s != NIL) {
    retval = WRITE_SOBJ (sock, s);
  }
  return retval;
}

int
k4data_to_sock (int sock, Atom cell)
{
  int retval = 1;

  /* single atoms are send 'as is', i.e. without header */
  if (TYP(cell) != CELL_P) {
    retval = WRITE_SOBJ(sock, cell);
  }
  
  else {
    /* list transmissions start with a BLIST command and terminates with a
       ELIST command */
    retval = writesockf (sock, "%c%s BLIST", KCP_OKC, KCP_OK);
    if (retval <= 0)
      return retval;
    
    retval = write_cell (sock, cell);
    if (retval <= 0)
      return retval;
    
    retval = writesockf (sock, "%c%s ELIST", KCP_OKC, KCP_OK);
  }
  
  return retval;
}

/* ----------------------------------------------------------------------
   blob functions
   ---------------------------------------------------------------------- */
int
blob_to_sock (int sock, Atom obj)
{
  int retval = 1;

  if (DFIELDQ(obj) != FALSE) {
    int size = DFIELD_SIZE (obj);
    char *buf = DFIELD_DATA (obj);

    if ((size > 0) 
	&& buf) {
      retval = writesockf (sock, "%c%s", KCP_OKC, KCP_OK);
      if (retval > 0)
	retval = write_blob (sock, buf, size);
    }
    else {
      retval = writesockf (sock, "%c%s empty or bad BLOB", KCP_ERRC, KCP_ERR);
    }
  }
  else {
    retval = writesockf (sock, "%c%s bad BLOB", KCP_ERRC, KCP_ERR);
  }
  
  return retval;
}

void
read_blob_from_sock (int sock, Atom obj)
{
  if (DFIELDQ(obj) != FALSE) {
    char *buf = NULL;
    int readsize;
    
    readsize = read_blob (sock, &buf);
    if (readsize < 0) {
      free (buf);
      SETERR (0, _("error reading blob")); 
    }
    set_dfield (OBL_DFIELD(obj), buf, readsize);
  }
}

