/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "proto.h"
#include "hc.h"

/* ----------------------------------------------------------------------
   REGULAR FUNCTIONS
   ---------------------------------------------------------------------- */
static char s_map[] = "map";
static char a_map[] = "xl.>l";
Atom
f_map (Atom list)
{
  Atom lphrase = NIL, argl, rx, r;
  
  PUSH_SCANL();
  rx = lphrase = SAVE_ROOT();
  /* at first we make bind the function item to fresh location */
  rx = CDR(rx) = CONS(CAR(list), NIL);
  rx = CAR(lphrase) = ALLOC();	/* root for the results */
  
  while (TYP(CADR(list)) == CELL_P) {
    r = CDR(lphrase);
    argl = CDR(list);
    while (TYP(argl) == CELL_P) {
      r = CDR(r) = CONS(CAAR(argl), NIL); /* link argl to the paramlist */
      CAR(argl) = CDAR(argl); /* get arg for next run in position */
      argl = CDR(argl);
    }
    rx = CDR(rx) = CONS(evaluate(CDR(lphrase)), NIL);
  }
  rx = CDAR(lphrase);
  POP_SCANL();
  return rx;
}


static char s_foreach[] = "for-each";
static char a_foreach[] = "xl.>l";
Atom
f_foreach (Atom list)
{
  Atom lphrase = NIL, argl, rx, r;
  
  PUSH_SCANL();
  rx = lphrase = SAVE_ROOT();
  /* at first we make bind the function item to fresh location */
  rx = CDR(rx) = CONS(CAR(list), NIL);
  
  /* second: we make a list of empty elements that will take the
     arguments from the sublists - we just have to link the elements
     to this list */
  r = CDR(list);
  while (TYP(r) == CELL_P) {
    rx = CDR(rx) = CONS(UNSPECIFIED, NIL);
    r = CDR(r);
  }
  
  while (TYP(CADR(list)) == CELL_P) {
    r = CDDR(lphrase);
    argl = CDR(list);
    while (TYP(argl) == CELL_P) {
      CAR(r) = CAAR(argl); /* link argl to the paramlist */
      CAR(argl) = CDAR(argl); /* get arg for next run in position */
      r = CDR(r);
      argl = CDR(argl);
    }
    evaluate(CDR(lphrase));
  }
  POP_SCANL();
  return UNSPECIFIED;
}

static char s_eval[] = "eval";
static char a_eval[] = "*";
Atom
f_eval (Atom list)
{
  return evaluate (CAR(list));
}

static char s_apply[] = "apply";
static char a_apply[] = "xl";
Atom
f_apply (Atom list)
{
  CDR(list) = CADR(list);	/* build a correct list */
  return evaluate (list);
}

static char s_load[] = "load";
static char a_load[] = "s.b";
Atom
f_load (Atom list)
{
  int mode = ((CDR(list) != NIL) ? (BOOLV(CADR(list))) : 0);
  return (run_file (STR_STR(CAR(list)), mode) ? TRUE : FALSE);
}

static char s_autoload[] = "autoload";
static char a_autoload[] = "hs";
Atom
f_autoload (Atom list)
{
  Atom p;
  p = lookup (CAR(list));
  if (PTR(p))
    SETERR (CAR(list), _("symbol defined"));
  PUTSYM (GENVL, CAR(list), h_AUTOLOAD, CADR(list));
  return UNSPECIFIED;
}

static char s_lookup[] = "db-lookup";
static char a_lookup[] = "*";
Atom
f_lookup (Atom list)
{
  return look_up_cache (CAR(list));
}

static char s_send[] = "send";
static char a_send[] = "oh.>*";
Atom
f_send (Atom list)
{
  Atom rx, rootx, ml, meth, retval, obj;
  
  obj = CAR(list);
 loop:
  ml = VECTOR_REF (obj, METHL_SLOT);
  meth = ASSOC(ml, CADR(list));
  
  if (meth != FALSE)
    goto method_found;
  
  obj = VECTOR_REF (obj, CLASS_SLOT);
  if ((obj == NIL)
      || (obj == FALSE))
    SETERR (CADR(list), _("unknown method"));
  
  /* look up the object in the cache */ 
  obj = look_up_cache (obj);
  if (!PTR(obj))
    SETERR (obj, _("unknow class"));
  
  if ((TYP(obj) == OBL_P) 
      && (OBL_TYP(obj) == VECTOR_T))
    goto loop;
  SETERR (obj, _("wrong class type"));
  
method_found:
  /* build a new list to be evaluated */
  PUSH_SCANL();
  rx = rootx = SAVE_ROOT();
  /* the first element is the found method */
  rx = CDR(rx) = CONS(CDR(meth), NIL);
  /* the second element is the object */
  rx = CDR(rx) = CONS(CAR(list), NIL);
  /* the third and so for element are the following elements */
  rx = CDR(rx) = CDDR(list);
  
  retval = evaluate(CDR(rootx));
  
  POP_SCANL();
  return retval;
}

static char errortext[255] = "";
static char s_error[] = "error";
static char a_error[] = "s";
Atom
f_error (Atom list)
{
  strcpy (errortext, STR_STR(CAR(list)));
  SETERR (0, errortext);
}

static char s_diagnostics[] = "diagnostics";
static char a_diagnostics[] = "n";
Atom
f_diagnostics (Atom list)
{
  int func = numv (CAR(list));

  if (func == 0) {
    printf (_("Choose one of the following diagnostic functions:\n"
	      " 0   this help\n"
	      " 1   show blobid list\n"
	      " 2   show global environment list\n")
	    );
  }
  else if (func == 1) {
    Blobrec *bls = bloblist.next;
    int i = 0;
    printf ("--- BLOBID LIST ---\n");
    while (bls) {
      printf (_("ID: %d SIZE: %d BOUND TO: %ld\n"), 
	      bls->id, DFIELD_SIZE (bls->ptr), bls->ptr);
      bls = bls->next;
      i++;
    }
    printf (_("TOTAL: %d blobs registered\n"), i);
  }
  else if (func == 2) {		/* show genvl */
    show (GENVL);
    printf ("\n");
  }
  return UNSPECIFIED;
}

void
init_sysfuncs ()
{
  Functiontable functbl[] = {
    {s_map, h_MAP, a_map, f_map},
    {s_foreach, h_FOREACH, a_foreach, f_foreach},
    {s_apply, h_APPLY, a_apply, f_apply},
    {s_eval, h_EVAL, a_eval, f_eval},
    {s_load, h_LOAD, a_load, f_load},
    {s_autoload, h_AUTOLOAD, a_autoload, f_autoload},
    {s_send, h_SEND, a_send, f_send},
    {s_lookup, h_LOOKUP, a_lookup, f_lookup},
    {s_error, h_ERROR, a_error, f_error},
    {s_diagnostics, h_DIAGNOSTICS, a_diagnostics, f_diagnostics},
    {NULL, -1, NULL, NULL}
  };
  Functiontable *ft = functbl;
  int i;
  
  for (i = 0; (ft+i)->fname != NULL ; i++)
    make_fset ((ft+i)->fname, (ft+i)->hashcode, (ft+i)->args, (ft+i)->fnc);
  
}
