/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <ctype.h>

#include "proto.h"
#include "hc.h"

/* this are the names of special chars (0 < x <= 32) in r4rs scheme */
char *charnametable[] = {
  "nul", "soh", "stx", "etx", "eot", "enq", "ack", "bel",
  "bs",  "ht",  "nl",  "vt",  "np",  "cr",  "so",  "si",
  "dle", "dc1", "dc2", "dc3", "dc4", "nak", "syn", "etb",
  "can", "em",  "sub", "esc", "fs",  "gs",  "rs",  "us",
  "space" };

struct {
  char *str;
  char chr;
} extcharnametable[] = {
  {"newline",   '\n'},
  {"return",    '\r'},
  {"escape",    '\e'},
  {"backspace", '\b'},
  {"null",        0 },
  {"bell",        7 },
  {"delete",    0177},
  {"tab",       '\t'},
  {"page",      '\f'},
  {NULL, 0}
};

char
decode_char (char *str)
{
  int k;
  
  if (!str[0])			/* empty string */
    return 0;
  else {
    if (!str[1]) {		/* one single char */
      return str[0];
    }
    else if (str[0] == '\\') {
      if (!str[2]) {		/* a \ seq, one char long */
	return str[1];
      }
      else {			/* a multibyte \ sequence */
	if (str[1] == '0') {
	  char *tail;
	  k = strtol ((str+1), &tail, 8); /* always octal! */
	  if (strlen(tail) == 0) {
	    return (char)k;
	  }
	}
	else {
	  int i;
	  for (i = 0; i < 33; i++) {
	    if (strcmp ((str+1), charnametable[i]) == 0)
	      return (char) i;
	  }
	  for (i = 0; extcharnametable[i].str != NULL; i++) {
	    if (strcmp ((str+1), extcharnametable[i].str) == 0)
	      return extcharnametable[i].chr;
	  }
	}
      }
    }
  }
  SETERR (0, LEESCSEQ);
}

char *toupcase (char *str)
{
  char *tmp = strdup (str), *p = tmp, *q = str;
  for ( ; *q ; q++, p++) {
    *p = toupper(*q);
  }
  *p = '\0';
  return tmp;
}

char *todowncase (char *str)
{
  char *tmp = strdup (str), *p = tmp, *q = str;
  for ( ; *q ; q++, p++) {
    *p = tolower(*q);
  }
  *p = '\0';
  return tmp;
}

char *tocapitalize (char *str)
{
  char *tmp = strdup (str), *p = tmp, *q = str;
  for ( ; *q ; q++, p++) {
    if ((q == str) ||
	((q > str) && 
	 (isspace(*(q - 1)) || 
	  ispunct(*(q - 1)))
	 ))
      *p = toupper(*q);
    else
      *p = tolower(*q);
  }
  *p = '\0';
  return tmp;
}

/* ----------------------------------------------------------------------
   String routines
   ---------------------------------------------------------------------- */
static char s_string_ref[] = "string-ref";
static char a_string_ref[] = "sn";
Atom
f_string_ref (Atom list)
{
  char *str = STR_STR (CAR(list));
  int k = numv(CAR(CDR(list)));
  
  if ((k < strlen (str)) &&
      (k >= 0)) {
    return ACHAR (str[k]);
  }
  SETERR (list, LERANGE);
}

static char s_string_append[] = "string-append";
static char a_string_append[] = ".>s";
Atom
f_string_append (Atom list)
{
  Atom s, retval;
  int tmpsize = 256;
  int tmplen = 0;
  char *tmp = (char *) smalloc (tmpsize);
  char *str;
  int len;

  tmp[0] = '\0';
  s = list;
  
  while (TYP(s) == CELL_P) {
    str = STR_STR(CAR(s));
    len = strlen(str);
    
    if (tmplen + len > tmpsize) {
      tmpsize += len + 256;
      tmp = (char *) realloc (tmp, tmpsize);
      if (!tmp) {
	SETERR (0, LEMEM);
      }
    }
    strcat (tmp, str);
    tmplen += len;
    s = CDR(s);
  }
  retval = NEWSTR(tmp);
  free (tmp);
  return retval;
}

static char s_string[] = "string";
static char a_string[] = ".>c";
Atom
f_string (Atom list)
{
  return LIST2STR(list);
}

static char s_list2string[] = "list->string";
static char a_list2string[] = "l";
Atom
f_list2string (Atom list)
{
  return LIST2STR(CAR(list));
}

static char s_string2list[] = "string->list";
static char a_string2list[] = "s";
Atom
f_string2list (Atom list)
{
  int i;
  Atom rootx, rx, retval = NIL;
  char *str = STR_STR (CAR(list));
  int len = strlen (str);
  
  PUSH_SCANL ();
  rx = rootx = SAVE_ROOT ();
  
  for (i = 0; i < len; i++) {
    rx = CDR(rx) = CONS (ACHAR(str[i]), NIL);
  }
  
  retval = CDR(rootx);
  POP_SCANL ();
  return retval;
}

static char s_string_split[] = "string-split";
static char a_string_split[] = "ss";
Atom
f_string_split (Atom list)
{
  char *p, *q;
  char *fstr = STR_STR (CAR(list));
  int flen = strlen (fstr);
  char *str = STR_STR (CADR(list));
  Atom rx, rootx, retval;
  
  p = q = str;
  PUSH_SCANL ();
  rx = rootx = SAVE_ROOT ();
  
  while (p) {
    p = strstr (q, fstr);
    
    if (p) {		/* needle in haystack? */
      char tmp[p - q + 1];
      strncpy (tmp, q, p - q);
      tmp[p-q] = '\0';
      rx = CDR(rx) = CONS (NEWSTR(tmp), NIL);
      q = p + flen;
    }
    else
      rx = CDR(rx) = CONS (NEWSTR(q), NIL);
  }
  retval = CDR(rootx);
  POP_SCANL();
  
  return retval;
}

static char s_string_splitv[] = "string-splitv";
static char a_string_splitv[] = "ss";
Atom
f_string_splitv (Atom list)
{
  char *p, *q;
  char *fstr = STR_STR (CAR(list));
  int flen = strlen (fstr);
  char *str = STR_STR (CADR(list));
  Atom rx, rootx, retval;

  p = q = str;
  PUSH_SCANL ();
  rx = rootx = SAVE_ROOT ();
  
  while (p) {
    p = strstr (q, fstr);
    
    if (p) {		/* needle in haystack? */
      rx = CDR(rx) = CONS (NEWINT(p-str), NIL);
      q = p + flen;
    }
    else
      rx = CDR(rx) = CONS (NEWINT(q-str), NIL);
  }
  retval = CDR(rootx);
  POP_SCANL();
  
  return retval;
}

static char s_string_copy[] = "string-copy";
static char a_string_copy[] = "s";
Atom
f_string_copy (Atom list)
{
  return NEWSTR(STR_STR(CAR(list)));
}

static char s_string_fill[] = "string-fill!";
static char a_string_fill[] = "sc";
Atom
f_string_fill (Atom list)
{
  char *str = STR_STR (CAR(list));
  char c = CHARV (CAR(CDR(list)));
  memset (str, c, strlen (str));
  return CAR(list);
}

static char s_limit_string[] = "limit-string";
static char a_limit_string[] = "sns.c";
Atom
f_limit_string (Atom list)
{
  char c = ' ';
  int wide = numv(CADR(list));
  int lng = strlen (STR_STR(CAR(list)));
  int elips = strlen (STR_STR(CADDR(list)));

  if (CDDDR(list) != NIL)
    c = CHARV (CAR(CDDDR(list)));

  if ((wide > 0)
      && (wide > elips)) {
    char tmp[wide + 1];

    if (lng < wide) {
      strcpy (tmp, STR_STR(CAR(list)));
      memset (&tmp[lng], c, wide - lng);
      tmp[wide] = '\0';
    }
    else if (lng == wide) {
      strcpy (tmp, STR_STR(CAR(list)));
    }
    else {			/* lng > wide */
      int blng = wide - elips;
      strncpy (tmp, STR_STR(CAR(list)), blng);
      strcpy (&tmp[blng], STR_STR(CADDR(list))); 
    }
    return NEWSTR (tmp);
  }
  SETERR (list, LERANGE);
}

static char s_make_string[] = "make-string";
static char a_make_string[] = "n.c";
Atom
f_make_string (Atom list)
{
  char c = ' ';
  int k = numv(CAR(list));
    
  if (CDR(list) != NIL)
    c = CHARV (CADR(list));
  
  if (k > 0) {
    char tmp[k+1];
    memset (tmp, c, k);
    tmp[k] = '\0';
    return NEWSTR(tmp);
  }
  SETERR (list, LERANGE);
}

static char s_string_set[] = "string-set!";
static char a_string_set[] = "snc";
Atom
f_string_set (Atom list)
{
  char *str = STR_STR (CAR(list));
  int k = numv(CAR(CDR(list)));
  char c = CHARV(CAR(CDR(CDR(list))));
  
  if ((k <strlen (str)) &&
      (k >= 0)) {
    str[k] = c;
    return CAR(list);
  }
  SETERR (list, LERANGE);
}

static char s_string_length[] = "string-length";
static char a_string_length[] = "s";
Atom
f_string_length (Atom list)
{
  return NEWINT (strlen(STR_STR(CAR(list))));
}

static char s_string2number[] = "string->number";
static char a_string2number[] = "s.n";
Atom
f_string2number (Atom list)
{
  char *str, *tail;
  int num = 0;
  str = STR_STR(CAR(list));
  
  if (CDR(list) == NIL) {
    num = strtol (str, &tail, 0);
    if (strlen(tail) > 0)
      return FALSE;
  }
  else {			/* a radix */
    int n = numv(CAR(CDR(list)));
    num = strtol (str, &tail, n);
    if (strlen(tail) > 0)
      return FALSE;
  }
  return NEWINT (num);
}

static char s_number2string[] = "number->string";
static char a_number2string[] = "n.n";
Atom
f_number2string (Atom list)
{
  char tmp[128];
  int radix = 10;
  int n = numv(CAR(list));
  
  if (CDR(list) != NIL)
    radix = numv (CAR(CDR(list)));
  
  switch (radix) {
  case 10:
    sprintf (tmp, "%d", n);
    break;
  case 16:
    sprintf (tmp, "%x", n);
    break;
  case 8:
    sprintf (tmp, "%o", n);
    break;
  default:
    SETERR (CAR(CDR(list)), LERADIX);
  }
  return NEWSTR(tmp);
}

static char s_substring[] = "substring";
static char a_substring[] = "snn";
Atom
f_substring (Atom list)
{
  char *str = STR_STR (CAR(list));
  int len = strlen (str);
  int ks = numv(CAR(CDR(list)));
  int ke = numv(CAR(CDR(CDR(list))));
  
  if ((ks <= len) &&
      (ks >= 0) &&
      (ke <= len) &&
      (ke >= 0)) {
    if (ks < ke) {
      char tmp[ke - ks + 1];
      strncpy (tmp, (str + ks), ke - ks);
      tmp[ke-ks] = '\0';
      return NEWSTR(tmp);
    }
    else if (ks == ke) {
      return NEWSTR("");
    }
  }
  SETERR (list, LERANGE);
}

static char s_stringq[] = "string?";
static char a_stringq[] = "*";
Atom
f_stringq (Atom list)
{
  return ((TYP(CAR(list)) == STR_P) ? TRUE : FALSE);
}



static char s_str_equal[] = "string=?";
static char a_str_equal[] = "ss";
Atom
f_str_equal (Atom list)
{
  return  ((strcmp(STR_STR(CAR(list)), 
		   STR_STR(CAR(CDR(list)))) == 0) ? TRUE : FALSE);
}

static char s_str_less[] = "string<?";
static char a_str_less[] = "ss";
Atom
f_str_less (Atom list)
{
  return ((strcmp(STR_STR(CAR(list)), 
		  STR_STR(CAR(CDR(list)))) < 0) ? TRUE : FALSE);
}

static char s_str_great[] = "string>?";
static char a_str_great[] = "ss";
Atom
f_str_great (Atom list)
{
  return ((strcmp(STR_STR(CAR(list)), 
		  STR_STR(CAR(CDR(list)))) > 0) ? TRUE : FALSE);
}

static char s_str_lesseq[] = "string<=?";
static char a_str_lesseq[] = "ss";
Atom
f_str_lesseq (Atom list)
{
  return ((strcmp(STR_STR(CAR(list)), 
		  STR_STR(CAR(CDR(list)))) <= 0) ? TRUE : FALSE);
}

static char s_str_greateq[] = "string>=?";
static char a_str_greateq[] = "ss";
Atom
f_str_greateq (Atom list)
{
  return ((strcmp(STR_STR(CAR(list)), 
		  STR_STR(CAR(CDR(list)))) >= 0) ? TRUE : FALSE);
}

static char s_str_upc[] = "string-upcase";
static char a_str_upc[] = "s";
Atom
f_str_upc (Atom list)
{
  /* look out! toupcase provides a strdup copy of str! */
  return PTRSTR(toupcase(STR_STR(CAR(list))));
}

static char s_str_downc[] = "string-downcase";
static char a_str_downc[] = "s";
Atom
f_str_downc (Atom list)
{
  /* look out! todowncase provides a strdup copy of str! */
  return PTRSTR(todowncase(STR_STR(CAR(list))));
}

static char s_str_capitalize[] = "string-capitalize";
static char a_str_capitalize[] = "s";
Atom
f_str_capitalize (Atom list)
{
  /* look out! tocapitalize provides a strdup copy of str! */
  return PTRSTR(tocapitalize(STR_STR(CAR(list))));
}

static char s_str_index[] = "string-index";
static char a_str_index[] = "ss";
Atom
f_str_index (Atom list)
{
  char *str = STR_STR(CAR(CDR(list)));
  char *p = strstr(str, STR_STR(CAR(list)));
  if (p)
    return NEWINT(p - str);
  else
    return FALSE;
}

void
init_strings ()
{
  Functiontable functbl[] = {
    {s_string_ref, h_STRREF, a_string_ref, f_string_ref},
    {s_string_append, h_STRAPPEND, a_string_append, f_string_append},
    {s_string, h_STRING, a_string, f_string},
    {s_list2string, h_LIST2STR, a_list2string, f_list2string},
    {s_string2list, h_STR2LIST, a_string2list, f_string2list},
    {s_string_copy, h_STRCPY, a_string_copy, f_string_copy},
    {s_string_fill, h_STRFILL, a_string_fill, f_string_fill},
    {s_make_string, h_MAKE_STR, a_make_string, f_make_string},
    {s_limit_string, h_LIMIT_STRING, a_limit_string, f_limit_string},
    {s_string_set, h_STRSET, a_string_set, f_string_set},
    {s_string_length, h_STRLEN, a_string_length, f_string_length},
    {s_string2number, h_STR2NUM, a_string2number, f_string2number},
    {s_number2string, h_NUM2STR, a_number2string, f_number2string},
    {s_substring, h_SUBSTRING, a_substring, f_substring},
    {s_stringq, h_STRINGQ, a_stringq, f_stringq},
    {s_str_equal, h_STR_EQUAL, a_str_equal, f_str_equal},
    {s_str_less, h_STR_LESS, a_str_less, f_str_less},
    {s_str_great, h_STR_GREAT, a_str_great, f_str_great},
    {s_str_lesseq, h_STR_LESSEQ, a_str_lesseq, f_str_lesseq},
    {s_str_greateq, h_STR_GREATEQ, a_str_greateq, f_str_greateq},
    {s_str_upc, h_STR_UPC, a_str_upc, f_str_upc},
    {s_str_downc, h_STR_DOWNC, a_str_downc, f_str_downc},
    {s_str_capitalize, h_STR_CAPITALIZE, a_str_capitalize, f_str_capitalize},
    {s_str_index, h_STR_INDEX, a_str_index, f_str_index},
    {s_string_split, h_STRING_SPLIT, a_string_split, f_string_split},
    {s_string_splitv, h_STRING_SPLITV, a_string_splitv, f_string_splitv},
    {NULL, -1, NULL, NULL}
  };
  Functiontable *ft = functbl;
  int i;
  
  for (i = 0; (ft+i)->fname != NULL ; i++)
    make_fset ((ft+i)->fname, (ft+i)->hashcode, (ft+i)->args, (ft+i)->fnc);
}
