/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#if defined TM_IN_SYS_TIME
# include <sys/time.h>
#else
# include <time.h>
#endif
#include <ctype.h>

#include "proto.h"
#include "hc.h"
#include "dbfunc.h"

/* ----------------------------------------------------------------------
   GLOBALE VARIABLEN
   ---------------------------------------------------------------------- */
Cell *heap;			/* der heap */
Oblist *oblist;			/* the oblist */
Strlist *strlist;		/* the Stringtable */
Hashlist *hashlist;		/* the hashlist */

int heapsize = 0;		/* current size of cell heap */
int oblistsize = 0;		/* current size of oblist */
int strlistsize = 0;		/* current size of string table */
int hashlistsize = 0;		/* current size of hash table */

int defaultheapsize = 512;
int defaultoblistsize = 128;
int defaultstrlistsize = 384;
int defaulthashlistsize = 512;
int defaultstacksize = 128;

int hashsize, systemhash;
int hashpagesize = 64;


/* vars that are accessable through slisp */
int verbose = 0;		/* testchating */
int talklisp = 0;		/* show retvalues ? */

Atom lastexp = 0;		/* recorded last errornous expression */

int classrequired = 1;

/* ----------------------------------------------------------------------
   INITIALISIERUNGSROUTINEN
   ---------------------------------------------------------------------- */
/* initialisiert den ILISP-Interpreter.  Mit startheapsize kann die
   anfngliche Gre des Heaps eingestellt werden, ist der Wert 0, so whlt
   initilisp den Wert der Seitengre DEFAULTHEAPSIZE.  Die Ziffer bezieht
   sich auf die Anzahl der freien sexp-Einheiten auf dem heap (zu je 8
   byte).  Der Lispinterpreter kann bei Mehranforderungen durch das
   Programm den Heap im Moment nur vergrern (sofern die globale Variable
   reallocheapavail gesetzt ist).  Die Routine fhrt zum Schlu einmal den
   Garbage collector aus, um den freien heap in die freelist zu hngen. */

void
init_memory (int startheapsize, 
	     int startoblistsize,
	     int startstrlistsize,
	     int starthashlistsize,
	     int startstacksize)
{
  int i, a, b, c;

  /* get memory for the heap */
  if (startheapsize == 0)
    heapsize = defaultheapsize;
  else 
    heapsize = startheapsize;
  heap = (Cell *) smalloc (heapsize * sizeof (Cell));
  
  /* get memory for the object list */
  if (startoblistsize == 0)
    oblistsize = defaultoblistsize;
  else 
    oblistsize = startoblistsize;
  oblist = (Oblist *) smalloc (oblistsize * sizeof (Oblist));
  
  /* get memory for the string table */
  if (startoblistsize == 0)
    strlistsize = defaultstrlistsize;
  else 
    strlistsize = startstrlistsize;
  strlist = (Strlist *) smalloc (strlistsize * sizeof (Strlist));
  /* We have to initialize the stringfield! */
  for (i = 0; i < strlistsize; i++) {
    STR_NEXT(i) = ASTR((i+1));
    STR_TYP(i) = STR_fl;
  }
  STR_NEXT((i-1)) = ASTR(0); /* free string list ptr */
  
  /* get memory for the hash table */
  if (starthashlistsize == 0)
    hashlistsize = defaulthashlistsize;
  else 
    hashlistsize = starthashlistsize;
  hashlist = (Hashlist *) smalloc (hashlistsize * sizeof (Hashlist));

  if (startstacksize == 0)
    initstack (defaultstacksize);
  else
    initstack (startstacksize);

  /* initialize default and system objects */
  
  /* initialize default and system lists */
  CAR(c_NIL) = NIL; CDR(c_NIL) = NIL;
  CAR(FL) = NIL;    CDR(FL) = NIL;
  CAR(ENVL) = NIL;  CDR(ENVL) = NIL;
  CAR(GENVL) = NIL; CDR(GENVL) = NIL;
  CAR(SCANL) = NIL; CDR(SCANL) = NIL;
  
  OBL_CDR(OFL) = o_NIL;
  
  gc (1, &a, &b, &c); /* put free heap to freelist */

  initactivl();
}

void
initscm (int initneeded)
{
  init_talklisp ();
  initparse ();
  init_memory (0, 0, 0, 0, 0);
  init_hash ();
  init_vars ();
  init_bloblist ();

  /* now define the functions */
  init_funcs ();
  init_sysfuncs ();
  init_numeric ();
  init_strings ();
  init_chars ();
  init_vectors ();
  init_db_funcs ();
  init_admin ();

  initpaths ();
  initdatabase();
  
  run_file ("init.k4", initneeded);
}


/* ----------------------------------------------------------------------
   Get and put a symbol to the ENVL
   ---------------------------------------------------------------------- */

/* finds a symbol in the Envl.  It returns a ptr to its value-structure or
   Nil if nothing is found */
Atom
lookup (int hash)
{
  int qhash = HASHV(hash);
  Atom s = CDR(ENVL);		/* at first in the local environment */
  while (TYP(s) == CELL_P) {
    if (HASHV(CAAR(s)) == qhash)
      return CAR(s);
    s = CDR(s);
  }
  s = CDR(GENVL);		/* not in the local environment */
  while (TYP(s) == CELL_P) {
    if (HASHV(CAAR(s)) == qhash)
      return CAR(s);
    s = CDR(s);
  }
  return NIL;
}

/* put a new symbol to the list list.  It returns a Ptr (Atom) to the new
   allocated Propertylist, that resides in the ENVL. 

   list: Die Liste, in die das Symbol eingefgt werden soll (ENVL)
   hash: Der hashcode des Symbolnamens
   modus: h_RO, h_RW
   cell: die Funktion (object als SUBR_T, oder als liste (cellptr))
   local: true = Lokale Variable
   */
int
addhash (char *name)
{
  int i;

  for (i = 0; i < hashsize; i++) {
    if (strcmp (hashlist[i].string, name) == 0)
      return AHASH(i);
  }
  if (hashsize > hashlistsize - 2) { /* warum - 2? aber es funktioniert */
    hashlistsize += hashpagesize;
    hashlist = (Hashlist*) realloc ((Hashlist*) hashlist,
				    hashlistsize * sizeof (Hashlist));
  }

  HASH_STR(hashsize) = strdup (name);
  hashsize++;
  return AHASH((hashsize - 1));
}

/* make_fset generates a complete primitive instructions representative in
   the global environment.  It serves only as startup installation
   system. */
void
make_fset (char *name, int hashcode, char *args, void *fnc)
{
  Atom rootx, rx;

  PUSH_SCANL();
  rootx = SAVE_ROOT();

  SETHASH (name, hashcode);
  
  /* Build a fset-list for external functions */
  rx = CDR(rootx) = CONS(AHASH (h_FSET), NIL);
  CDR(rx) = CONS (CONSSTR (args), NEWSUBR(fnc)); /* point arg string! */
  PUTSYM (GENVL, hashcode, h_RO, rx);
  
  POP_SCANL();
}

/* ----------------------------------------------------------------------
   Basic Lisp Functions
   ---------------------------------------------------------------------- */
Atom
eq (Atom first, Atom sec)
{
  if (TYP(first) == TYP(sec)) {
    if (TYP(first) == CELL_P) {
      if (PTR(first) == PTR(sec))
	return TRUE;
    }
    else {
      if (TYP(first) == OBL_P) {
	/* with external objects atomic types only: OID, Date, EXTINT_T.  No
           Vectors or Dfields */
	if (OBL_TYP(first) == OBL_TYP(sec)) {
	  switch (OBL_TYP(first)) {
	  case DATE_T:
	  case OID_T:
	  case EXTINT_T:
	    if (OBL_DATA(first) == OBL_DATA(sec))
	      return TRUE;
	  }
	}
      }
      else {
	/* dies gilt fr Int, Hash, Str, Bool.  Sind ihre ptr-Codes
           identisch, so sind auch die Ergebnisse gleich (Ausnahme fr
           Strings.  Aber das ist das typische Verhalten von scm, elisp,
           etc. */
	if (PTR(first) == PTR(sec))
	  return TRUE;
      }
    }
  }
  return FALSE;
}

Atom
equal (Atom first, Atom sec)
{
  if (TYP(first) == TYP(sec)) {
    if (TYP(first) == CELL_P) {
      if (PTR(first) == PTR(sec))
	return TRUE;
      if (equal(CAR(first), CAR(sec)) == FALSE) 
	return FALSE;
      if (equal(CDR(first), CDR(sec)) == FALSE)
	return FALSE;
      return TRUE;
    }
    else {
      if (TYP(first) == OBL_P) {
	/* with external objects atomic types only: OID, Date, EXTINT_T.  No
           Vectors or Dfields */
	if (OBL_TYP(first) == OBL_TYP(sec)) {
	  switch (OBL_TYP(first)) {
	  case DATE_T:
	  case OID_T:
	  case EXTINT_T:
	    if (OBL_DATA(first) == OBL_DATA(sec))
	      return TRUE;
	  }
	}
      }
      else if (TYP(first) == STR_P) {
	if (strcmp(STR_STR(first), (STR_STR(sec))) == 0)
	  return TRUE;
      }
      else {
	/* dies gilt fr Int, Hash.  Sind ihre ptr-Codes identisch, so
           sind auch die Ergebnisse gleich (Ausnahme fr Strings.  Aber das
           ist das typische Verhalten von scm, elisp, etc. */
	if (PTR(first) == PTR(sec))
	  return TRUE;
      }
    }
  }
  return FALSE;
}


int
numv(Atom x)
{
  switch (TYP(x)) {
  case INT_P: return INTV(x);
  case OID_P: return OIDV(x);
  case OBL_P:
    switch (OBL_TYP(x)) {
    case OID_T: return OBL_OID(x);
    case EXTINT_T: return *OBL_INT(x);
    }
  }
  return 0;
}

Atom
number(Atom x)
{
  if ((TYP(x) == INT_P)
      || (TYP(x) == OID_P)
      || ((TYP(x) == OBL_P) 
	  && ((OBL_TYP(x) == OID_T) 
	      || (OBL_TYP(x) == EXTINT_T))))
    return TRUE;
  return FALSE;
}
     
/* ----------------------------------------------------------------------
   Vector functions
   ---------------------------------------------------------------------- */
Vector *
make_vector (int alloc_size, int size, Atom fill)
{
  if ((alloc_size < 0) || (size < 0) || (size > alloc_size)) {
    return NULL;
  }
  else {
    Vector *vec = (Vector *) smalloc (sizeof (Vector));
    int i;
    if (alloc_size < 16)
      alloc_size = 16;
    vec->size = size;
    vec->alloc_size = alloc_size;
    vec->array = (Atom *) smalloc (alloc_size * sizeof (Atom));
    for (i = 0; i < size; i++)
      vec->array[i] = fill;
    return vec;
  }
}

void
vector_set (Vector *vec, Atom val, int ref)
{
  if (ref >= vec->alloc_size) {

    vec->alloc_size += (ref - vec->alloc_size) + 1;
    vec->array = (Atom *) realloc ((Atom*)vec->array,
				   vec->alloc_size*sizeof(Atom));
  }
  if (ref >= vec->size) {
    int i;
    for (i = vec->size; i < ref; i++)
      vec->array[i] = NIL;
    vec->size = ref + 1;
  }
  vec->array[ref] = val;
}

Atom
vector_ref (Vector *vec, int ref)
{
  if ((ref >= vec->size) ||
      (ref < 0))
    return FALSE;
  return vec->array[ref];
}

void
vector_fill (Vector *vec, Atom fill)
{
  if (vec->size > 0) {
    int size = vec->size, i;
    for (i = 0; i < size; i++)
      vec->array[i] = fill;
  }
}

int
vector_lookup (Vector *vec, Atom obj)
{
  if (vec->size > 0) {
    int size = vec->size, i;
    for (i = 0; i < size; i++)
      if (equal(vec->array[i], obj) != FALSE)
	return i;
  }
  return -1;
}

void
vector_set_length (Vector *vec, int size)
{
  if (size > 0) {
    if (size != vec->alloc_size) {
      vec->alloc_size = size;
      vec->array = (Atom *) realloc ((Atom*)vec->array,
				     vec->alloc_size*sizeof(Atom));
    }
    if (size > vec->size) {
      int i;
      for (i = vec->size; i < size; i++)
	vec->array[i] = FALSE;
    }
    vec->size = size;
  }
}

int
vector_equal (first, sec)
     Atom *first, *sec;
{
  if (TYP(*first) == TYP(*sec)) {
    if (TYP(*first) == CELL_P) {
      if (PTR(*first) == PTR(*sec))
	return 0;
      return vector_equal(&CAR(*first), &CAR(*sec));
    }
    else {
      if (TYP(*first) == OBL_P) {
	/* with external objects atomic types only: OID, Date, EXTINT_T.  No
           Vectors or Dfields */
	if (OBL_TYP(*first) == OBL_TYP(*sec)) {
	  switch (OBL_TYP(*first)) {
	  case DATE_T:
	  case OID_T:
	  case EXTINT_T:
	    return OBL_DATA(*first) - OBL_DATA(*sec);
	  }
	}
      }
      else if (TYP(*first) == STR_P) {
	return strcmp(STR_STR(*first), STR_STR(*sec));
      }
      else {
	/* dies gilt fr Int, Hash.  Sind ihre ptr-Codes identisch, so
           sind auch die Ergebnisse gleich (Ausnahme fr Strings.  Aber das
           ist das typische Verhalten von scm, elisp, etc. */
	return PTR(*first) - PTR(*sec);
      }
    }
  }
  return 0;
}

void
vector_sort (Vector *vec)
{
  if (vec->size > 0) {
    qsort (vec->array, vec->size, sizeof (Atom), vector_equal);
  }
}

Vector *
vector_merge_and (Vector *vec1, Vector *vec2)
{
  int vi, vj, vri, res;
  Vector *vr = make_vector (0, 0, NIL);
  
  vi = vj = vri = 0;
  while ((vi < vec1->size) 
	 && (vj < vec2->size)) {
    res = vector_equal (&vec1->array[vi], &vec2->array[vj]);
    if (res == 0) {		/* equal */
      vector_set (vr, vec1->array[vi], vri);
      vi++;
      vj++;
      vri++;
    }
    else if (res < 0) {		/* vec1 < vec2 */
      vi++;
    }
    else {			/* vec1 > vec2 */
      vj++;
    }
  }
  return vr;
}

Vector *
vector_merge_or (Vector *vec1, Vector *vec2)
{
  int vi, vj, vri, res;
  Vector *vr = make_vector (0, 0, NIL);

  vi = vj = vri = 0;
  while ((vi < vec1->size)
	 && (vj < vec2->size)) {
    res = vector_equal (&vec1->array[vi], &vec2->array[vj]);

    if (res == 0) {		/* equal */
      vector_set (vr, vec1->array[vi], vri);
      vi++;
      vj++;
      vri++;
    }
    else if (res < 0) {
      vector_set (vr, vec1->array[vi], vri);
      vi++;
      vri++;
    }
    else if (res > 0) {
      vector_set (vr, vec2->array[vj], vri);
      vj++;
      vri++;
    }
  }

  /* are the more data in vec1 ? */
  while (vi < vec1->size) {
    vector_set (vr, vec1->array[vi], vri);
    vi++;
    vri++;
  }
  
  /* are the more data in vec2 ? */
  while (vj < vec2->size) {
    vector_set (vr, vec2->array[vj], vri);
    vj++;
    vri++;
  }
  
  return vr;
}

/* xany means: x and not y */
Vector *
vector_merge_xany (Vector *vec1, Vector *vec2)
{
  int vi, vj, vri, res;
  Vector *vr = make_vector (0, 0, NIL);

  vi = vj = vri = 0;
  while ((vi < vec1->size) 
	 && (vj < vec2->size)) {
    res = vector_equal (&vec1->array[vi], &vec2->array[vj]);
    
    if (res == 0) {		/* equal: ignore both */
      vi++;
      vj++;
    }
    else if (res < 0) {
      vector_set (vr, vec1->array[vi], vri);
      vi++;
      vri++;
    }
    else if (res > 0) {
      vj++;
    }
  }

  /* are the more data in vec1 ? */
  while (vi < vec1->size) {
    vector_set (vr, vec1->array[vi], vri);
    vi++;
    vri++;
  }
  return vr;
}

Vector *
vector_merge_xor (Vector *vec1, Vector *vec2)
{
  int vi, vj, vri, res;
  Vector *vr = make_vector (0, 0, NIL);

  vi = vj = vri = 0;
  while ((vi < vec1->size)
	 && (vj < vec2->size)) {
    res = vector_equal (&vec1->array[vi], &vec2->array[vj]);

    if (res == 0) {		/* equal: ignoreboth */
      vi++;
      vj++;
    }
    else if (res < 0) {
      vector_set (vr, vec1->array[vi], vri);
      vi++;
      vri++;
    }
    else if (res > 0) {
      vector_set (vr, vec2->array[vj], vri);
      vj++;
      vri++;
    }
  }

  /* are the more data in vec1 ? */
  while (vi < vec1->size) {
    vector_set (vr, vec1->array[vi], vri);
    vi++;
    vri++;
  }
  
  /* are the more data in vec2 ? */
  while (vj < vec2->size) {
    vector_set (vr, vec2->array[vj], vri);
    vj++;
    vri++;
  }
  
  return vr;
}

Vector *
empty_object (Atom class, Atom oid)
{
  Vector *v = make_vector (SLOT_SIZE, SLOT_SIZE, NIL);
  vector_set (v, AHASH(h_OBJECT), OBJECT_ID_SLOT);
  vector_set (v, class, CLASS_SLOT);
  vector_set (v, oid, OID_SLOT);
  vector_set (v, NIL, METHL_SLOT);
  vector_set (v, NEWVECTOR(make_vector(1, 0, NIL)), DESC_SLOT); 
  vector_set (v, NEWVECTOR(make_vector(1, 0, NIL)), DATA_SLOT);
  return v;
}

/* ----------------------------------------------------------------------
   DFIELD FUNCTIONS
   ---------------------------------------------------------------------- */
Blobrec bloblist;

static int blobid_counter = 0;
#define next_blobid() ({ blobid_counter++; blobid_counter; })

Dfield *
make_dfield (char *buf, int size)
{
  Dfield *dfield = (Dfield*) smalloc (sizeof(Dfield));
  dfield->size = size;
  dfield->data = buf;
  dfield->blobid = next_blobid ();
  return dfield;
}

Dfield *
set_dfield (Dfield *dfield, char *buf, int size)
{
  if (dfield->size > 0) {
    free (dfield->data);
  }
  dfield->data = buf;
  dfield->size = size;
  
  return dfield;
}

#define ALLOC_BLOBREC(BLOBID, PTR, NEXT)			\
({								\
  Blobrec *_blob = (Blobrec*) smalloc (sizeof (Blobrec));	\
  _blob->id = BLOBID;						\
  _blob->ptr = PTR;						\
  _blob->next = NEXT;						\
  _blob;							\
})
     
void
init_bloblist ()
{
  bloblist.next = NULL;
  bloblist.id = 0;
  bloblist.ptr = NIL;
}

Blobrec *
record_blob (Atom obj, int blobid)
{
  Blobrec *blob = ALLOC_BLOBREC(blobid, obj, bloblist.next);
  bloblist.next = blob;
  return blob;
}

int
free_blob (int blobid)
{
  Blobrec *bls = &bloblist;
  
  if (bls) {
    while (bls->next) {
      if ((bls->next)->id == blobid) {
	Blobrec *nxt = (bls->next)->next;
	free (bls->next);
	bls->next = nxt;
	goto done;
      }
      bls = bls->next;
    }
  }
  return 0;			/* unknown blobid */
 done:
  return 1;			/* ok. */
}

Atom
blob_ref (int blobid)
{
  Blobrec *bls = bloblist.next;
  while (bls) {
    if (bls->id == blobid)
      return bls->ptr;
    bls = bls->next;
  }
  return FALSE;
}

/* ----------------------------------------------------------------------
   Date and Time-routines
   ---------------------------------------------------------------------- */
static char *months[] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
  "Nov", "Dec", NULL
};

static char *days[] = {
  "Sun", "Mon", "Tue", "Wed", "Thu", "Sat", "Sun", NULL
};

#define GETSTRNO(list,str)			\
({						\
  __label__ done;				\
  int _i = 0, _retval = -1;			\
  while (list[_i]) {				\
    if (strcasecmp(list[_i], str) == 0) {	\
      _retval = _i;				\
      goto done;				\
    }						\
    _i++;					\
  }						\
done:						\
  _retval;					\
})
     
char*
date2string (Date date, int format)
{
  struct tm *bdtime;
  char bstr[128], *str;
  
  bdtime = localtime (&date);
  switch (format) {
  case h_DATEFORM_NORMAL: 	/* Wed Jul 16, 1997 14:01:27 */
    sprintf (bstr, "%s %s %d, %d %d:%02d:%02d", 
	     days[bdtime->tm_wday], 
	     months[bdtime->tm_mon],
	     bdtime->tm_mday,
	     bdtime->tm_year+1900, 
	     bdtime->tm_hour, bdtime->tm_min, bdtime->tm_sec);
    break;
  case h_DATEFORM_AMERICAN:	/* 07/16/1997 14:01:27 */
    sprintf (bstr, "%02d/%02d/%4d %d:%02d:%02d",
	     bdtime->tm_mon+1,	/* in *tm beginnt es bei 0! */
	     bdtime->tm_mday,
	     bdtime->tm_year+1900,
	     bdtime->tm_hour, bdtime->tm_min, bdtime->tm_sec);
    break;
  case h_DATEFORM_GERMAN:	/* 16.7.1997 14:01:27 */
    sprintf (bstr, "%d.%d.%4d %d:%02d:%02d",
	     bdtime->tm_mday,
	     bdtime->tm_mon+1,	/* in *tm beginnt es bei 0! */
	     bdtime->tm_year+1900,
	     bdtime->tm_hour, bdtime->tm_min, bdtime->tm_sec);
    break;
  case h_DATEFORM_KOREAN:		/* 1997-7-16 14:01:27 */
    sprintf (bstr, "%d-%d-%d %d:%02d:%02d",
	     bdtime->tm_year+1900,
	     bdtime->tm_mon+1,	/* in *tm beginnt es bei 0! */
	     bdtime->tm_mday,
	     bdtime->tm_hour, bdtime->tm_min, bdtime->tm_sec);
    break;
  case h_DATEFORM_UNIX:		/* Wed Jul 16 14:03:46 MET DST 1997 */
    sprintf (bstr, "%s %s %d %d:%02d:%02d %s %4d",
	     days[bdtime->tm_wday], 
	     months[bdtime->tm_mon],
	     bdtime->tm_mday,
	     bdtime->tm_hour, bdtime->tm_min, bdtime->tm_sec,
	     (bdtime->tm_isdst ? tzname[1] : tzname[0]),
	     bdtime->tm_year+1900);
    break;
  }
  str = strdup (bstr);
  return str;
}

Date
string2date (char *strline)
{
  struct tm mkt;
  char *p, *r, *tail;
  char *str;

  if (!strline)
    return 0;
  
  str = strdup (strline);

  mkt.tm_mon = 0;
  mkt.tm_mday = 0;
  mkt.tm_year = 0;
  mkt.tm_hour = 0;
  mkt.tm_min = 0;
  mkt.tm_sec = 0;
  mkt.tm_wday = 0;


  /* ----------------------------------------------------------------------
     SCAN KAENGURU NORMAL FORMAT
     [wday] mon DD, YYYY HH:MM:SS
     ---------------------------------------------------------------------- */
  if (strchr (str, ',')) {	/* normal format */
    p = str;
    for ( ; isspace(*p) ; p++);	/* start of data */

    r = p;			/* week days name or month*/
    for ( ; *p && !isspace(*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_mon = GETSTRNO(months, r);
    if (mkt.tm_mon < 0) {
      r = p;			/* now definitly month */
      for ( ; *p && !isspace(*p) ; p++);
      *p = '\0';
      p++; for ( ; isspace(*p) ; p++);
      mkt.tm_mon = GETSTRNO(months, r);
      if (mkt.tm_mon < 0)
	mkt.tm_mon = 0;
    }
    
    r = p;			/* month day */
    for ( ; *p && (*p != ',') ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_mday = strtol (r, &tail, 0);

    r = p;			/* year */
    for ( ; *p && !isspace (*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_year = strtol (r, &tail, 0);
    if (mkt.tm_year > 1900)
      mkt.tm_year -= 1900;

    r = p;			/* hour */
    for ( ; *p && (*p != ':') ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_hour = strtol (r, &tail, 0);

    r = p;			/* min */
    for ( ; *p && (*p != ':') ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_min = strtol (r, &tail, 0);

    r = p;			/* seconds */
    for ( ; *p && !isspace (*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_sec = strtol (r, &tail, 0);

    free (str);
    return mktime (&mkt);
  }

  /* ----------------------------------------------------------------------
     SCAN STANDARD UNIX FORMAT
     [wday] mon DD HH:MM:SS timezone YYYY
     ---------------------------------------------------------------------- */
  else {			/* unix: Wed Jul 16 14:01:27 MST DST 1997*/
    char lp;

    p = str;
    for ( ; isspace(*p) ; p++);	/* start of data */

    r = p;			/* week days name or month*/
    for ( ; *p && !isspace(*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_mon = GETSTRNO(months, r);
    if (mkt.tm_mon < 0) {
      r = p;			/* now definitly month */
      for ( ; *p && !isspace(*p) ; p++);
      *p = '\0';
      p++; for ( ; isspace(*p) ; p++);
      mkt.tm_mon = GETSTRNO(months, r);
      if (mkt.tm_mon < 0)
	mkt.tm_mon = 0;
    }
    
    r = p;			/* month day */
    for ( ; *p && !isspace(*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_mday = strtol (r, &tail, 0);

    r = p;			/* hour */
    for ( ; *p && (*p != ':') && !isspace (*p) ; p++);
    lp = *p;
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_hour = strtol (r, &tail, 0);

    if (lp != ' ') {
      r = p;			/* min */
      for ( ; *p && (*p != ':') && !isspace (*p) ; p++);
      lp = *p;
      *p = '\0';
      p++; for ( ; isspace(*p) ; p++);
      mkt.tm_min = strtol (r, &tail, 0);
      
      if (lp != ' ') {
	r = p;			/* seconds */
	for ( ; *p && !isspace (*p) ; p++);
	*p = '\0';
	p++; for ( ; isspace(*p) ; p++);
	mkt.tm_sec = strtol (r, &tail, 0);
      }
    }

    r = p;			/* ignore Timezone */
    for ( ; *p && !(isdigit (*p)) ; p++);

    r = p;			/* year */
    for ( ; *p && !isspace (*p) ; p++);
    *p = '\0';
    p++; for ( ; isspace(*p) ; p++);
    mkt.tm_year = strtol (r, &tail, 0);
    if (mkt.tm_year > 1900)
      mkt.tm_year -= 1900;
    
    free (str);
    return mktime (&mkt);
  }

  free (str);
  return -1;
}




/* ----------------------------------------------------------------------
   Cache routines
   ---------------------------------------------------------------------- */
Atom 
look_up_cache (Atom obj)
{
  Atom p, retval, readobj, cl;
  Oid goid;

#define RECORD_OBJECT(clst, lobjname, lobj)		\
({							\
  Atom _c = (clst);					\
  Atom _t, _s;						\
  _t = _s = CONS(lobjname, NIL);			\
  _s = CDR(_s) = CONS(lobj, NIL);			\
  _s = CDR(_s) = CONS(NEWDATE(get_system_time()), NIL); \
  VAR_DATA(_c) = CONS(_t, VAR_DATA(_c));		\
})

  if (TYP(obj) != CELL_P) {
    cl = lookup (h_CACHE);
    if (!PTR(cl))
      SETERR (0, _("no cache found"));
    
    retval = ASSOC(VAR_DATA(cl), obj);
    
    if (retval != FALSE)
      return CADR(retval);	/* return the recorded object! */
    
    /* if come until here, the required object wasn't recorded yet! */
    switch (TYP(obj)) {
    case HASH_P:
      p = lookup (HASHV(obj));
      if (!PTR(p)) {
	if (classrequired)
	  SETERR (obj, _("required class not found"));
      }
      else {  
	RECORD_OBJECT (cl, obj, VAR_DATA(p));
	return VAR_DATA(p);
      }
      return NIL;
    case STR_P:
      goid = get_oid_by_index (STR_STR(obj), 
			       look_up_index (CLASS_INDEX_NAME));
      if (goid) {
	readobj = read_object (goid);
	RECORD_OBJECT (cl, obj, readobj);
	return readobj;
      }
      return NIL;
    case OID_P:
      goto load_oid_obj;
    case OBL_P:
      if (OBL_TYP(obj) == OID_T) {
load_oid_obj:
	readobj = read_object (oidv(obj)); /* here must be oidv! */
	if (readobj != FALSE) {
	  RECORD_OBJECT (cl, obj, readobj);
	  return readobj;
	}
      }
      SETERR (obj, _("object can't be inherited"));
    }
  }
  
  return NIL;
}


void
cleanup_cache (int timespan, Date acttime)
{
  Atom s, ls, cl;
  
  cl = lookup (h_CACHE);
  if (!PTR(cl))
    SETERR (0, _("no cache found"));
  
  s = ls = VAR_DATA(cl);
  while (TYP(s) == CELL_P) {
    if ((TYP(CAR(s)) == CELL_P) 
	&& (TYP(CAR(CDDAR(s))) == OBL_P) 
	&& (OBL_TYP(CAR(CDDAR(s))) == DATE_T)) {
      Date cachetime = OBL_DATE(CAR(CDDAR(s)));
      
      if (cachetime + timespan < acttime) {
	if (ls == VAR_DATA(cl)) {
	  ls = s = VAR_DATA(cl) = CDR (s);
	}
	else
	  s = CDR(ls) = CDR(s);
      }
      else {
	ls = s;
	s = CDR(s);
      }
    }
    else {
      ls = s;
      s = CDR(s);
    }
  }
}
