/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.


  */

%option noyywrap
%option never-interactive
%option nounput
%{
#if HAVE_CONFIG_H
# include "config.h"
#endif
#if defined HAVE_STRING_H || defined STDC_HEADERS
# include <string.h>
#else
# include <strings.h>
#endif

#include "proto.h"
#include "hc.h"

#define MAX_STR_CONST   1024

/* Category codes, for the p-code compilers */
#define cat_POP     1		/* parantheses open */
#define cat_PCL     2		/* parantheses close */
#define cat_QUO     3		/* quote */
#define cat_ATOM    4		/* atom */
#define cat_DOT     5		/* a dot */
#define cat_STOP    6		/* stop the interpreter */

char *yyfp;			/* ptr for compiling */
char *yybuffer;			/* buffer compiling */
char lastdata;
int yybufsize;
int paran;
char strc[1024];
char string_buf[MAX_STR_CONST];
char *string_buf_ptr;
Atom lasteval;

typedef struct {
  char nextaim;
  Atom act;
} Env;

/* ---- definitionen fr den parser ---- */
Env envstack[MAXENVSTACK];
int envsp;

char nextaim;
Atom actcell;
Atom yylval;

#define na_NULL          '0'
#define na_LEFT          'L'
#define na_RIGHT         'R'
#define na_APPEND        'A'
#define na_COMPL         'C'
#define na_QUOTE         'Q'
#define na_QUOTECL       'q'

/* ---- prototypen fr den parser ---- */
void initparse ();

%}

DIGIT    [0-9]
HEXDIGIT [0-9a-fA-F]
BDIGIT	 [0-1]
ODIGIT	 [0-7]
LABEL    [a-zA-Z\-+*%/!=<>\?\:\$_\&\~\^][a-zA-Z0-9\-+*%/!=<>\:\?\$_\&\~\^]*

%x comment
%x stringc
%x charac
%x eval
%x evalobj
%x oidr

%%


";"             BEGIN(comment);
<comment>[\n]   BEGIN(INITIAL);
<comment>[^\n]* /* eat the comment */

"#"            {
  BEGIN(eval);
}
<eval>"t" {
  BEGIN(INITIAL);
  yylval = TRUE;
  return cat_ATOM;
}
<eval>"f" {
  BEGIN(INITIAL);
  yylval = FALSE;
  return cat_ATOM;
}
<eval>"\:" {
  BEGIN(evalobj);
}
<evalobj>{DIGIT}+ {
  int i = atoi( yytext );
  Vector *v = empty_object (AHASH(h_CORE), NEWOID(i));
  BEGIN(INITIAL);
  yylval = NEWVECTOR (v);
  return cat_ATOM;
}
<evalobj><<EOF>> {
  BEGIN(INITIAL);
  SETERR (0, _("bad #: syntax"));
}
<eval>"o" {
  BEGIN(oidr);
}
<oidr>{DIGIT}+ {
  char *tail;
  int i = strtol( yytext, &tail, 0);
  BEGIN(INITIAL);
  yylval = NEWOID (i);
  return cat_ATOM;
}
<oidr><<EOF>> {
  BEGIN(INITIAL);
  SETERR (0, _("bad #o syntax"));
}
<eval>"()" {
  Vector *v = make_vector (0, 0, NIL);
  BEGIN(INITIAL);
  yylval = NEWVECTOR (v);
  return cat_ATOM;
}
<eval>\\       {
  BEGIN(charac);
}
<eval><<EOF>> {
  BEGIN(INITIAL);
  SETERR (0, _("bad # syntax"));
}
<charac>(\\|\"|'|\(|\)|@|\.|\[|\]) {
  BEGIN(INITIAL);
  yylval = NEWCHAR(yytext);
  return cat_ATOM;
}
<charac>[^\\ \n\"\'\()]+ {
  char tmp[strlen(yytext) + 2];
  BEGIN(INITIAL);
  sprintf (tmp, "\\%s", yytext);
  yylval = NEWCHAR(tmp);
  return cat_ATOM;
}
<charac><<EOF>> {
  BEGIN(INITIAL);
  SETERR (0, _("bad character code"));
}

\"             {
  string_buf_ptr = string_buf;
  BEGIN(stringc);
}

<stringc>"\""   {
  BEGIN(INITIAL);
  *string_buf_ptr = '\0';

  yylval = NEWSTR (string_buf);
  return cat_ATOM;
}

<stringc>\\[0-7]{1,3} {
  /* octal escape sequence */
  int result;
  
  (void) sscanf( yytext + 1, "%o", &result );
  
  if ( result > 0xff ) {
    BEGIN(INITIAL);
    SETERR(0, LEESCSEQ);
  }

  *string_buf_ptr++ = result;
}

<stringc>\\[0-9]+ {
  BEGIN(INITIAL);
  SETERR (0, LEESCSEQ);
}

<stringc>\\e      *string_buf_ptr++ = '\e';
<stringc>\\n      *string_buf_ptr++ = '\n';
<stringc>\\t      *string_buf_ptr++ = '\t';
<stringc>\\r      *string_buf_ptr++ = '\r';
<stringc>\\b      *string_buf_ptr++ = '\b';
<stringc>\\f      *string_buf_ptr++ = '\f';


<stringc>\\(.|\n)  *string_buf_ptr++ = yytext[1];

<stringc>[^\\\"]+ {
  char *yptr = yytext;
  while ( *yptr )
    *string_buf_ptr++ = *yptr++;
}


<stringc><<EOF>> {
  BEGIN(INITIAL);
  SETERR (0, LEUNTERMSTR);
}

{DIGIT}+         {
  int i;
  i = atoi( yytext );
  yylval = NEWINT (i);
  return cat_ATOM;
}


"0x"{HEXDIGIT}+  {
  char *tail;
  long i;
  i = strtol (yytext, &tail, 0);
  yylval = NEWINT (i);
  return cat_ATOM;
}


"0b"{BDIGIT}+    {
  char *tail;
  long i;
  i = strtol (&yytext[2], &tail, 2);
  yylval = NEWINT (i);
  return cat_ATOM;
}


"0c"{ODIGIT}+    {
  char *tail;
  Oid i;
  i = strtol (&yytext[2], &tail, 8);
  yylval = NEWINT (i);
  return cat_ATOM;
}

{LABEL}           {
  yylval = NEWHASH (yytext);
  return cat_ATOM;
}


"("               {
  paran++;
  return cat_POP;
}

")"               {
  if (paran > 0)
    paran--;
  return cat_PCL;
}

"'"               return cat_QUO;

"."               return cat_DOT;

"\n"              { }

[ \t\n\r]+          /* eat up whitespace */

.		  printf( _("unrecognized character: %d\n"), *yytext );

<<EOF>> {
  if (paran > 0)
    SETERR (0, LEPARAN);
  return cat_STOP;
}

%%


/* ----------------------------------------------------------------------
   Parsercode
   ---------------------------------------------------------------------- */
typedef struct {
  int nextaim;
  Atom actcell;
  YY_BUFFER_STATE lexbf;
} Parserstack;

#define MAXFILEDEPTH         64

Parserstack parsstack[MAXFILEDEPTH];
int parssp;
YY_BUFFER_STATE flxbf;


void
initparse ()
{
  envsp = 0;
  parssp = 0;
  flxbf = NULL;
  paran = 0;
}

void
saveparserenv (YY_BUFFER_STATE lexbf)
{
  if (parssp > MAXFILEDEPTH)
    SETERR (0, LEINCLDEPTH);
  parsstack[parssp].nextaim = nextaim;
  parsstack[parssp].actcell = actcell;
  parsstack[parssp].lexbf = lexbf;
  parssp++;
  PUSH_SCANL ();
}

YY_BUFFER_STATE
restoreparserenv ()
{
  YY_BUFFER_STATE lexbf;
  
  if (parssp <= 0)
    SETERR (0, LESTACKINCL);
  parssp--;
  nextaim = parsstack[parssp].nextaim;
  actcell = parsstack[parssp].actcell;
  lexbf = parsstack[parssp].lexbf;
  
  POP_SCANL ();
  
  return lexbf;
}

void
saveenv ()
{
  if (envsp > MAXENVSTACK)
    SETERR (0, LESTACKPARSE);
  envstack[envsp].nextaim = nextaim;
  envstack[envsp].act = actcell;
  envsp++;
}

void
restoreenv ()
{
  if (envsp <= 0)
    SETERR (0, LESTACKUFPARSE);
  envsp--;
  nextaim = envstack[envsp].nextaim;
  actcell = envstack[envsp].act;
}

#define PROGEND       1
#define PROGGOON      2

Atom
lispparse (char *source, int *mode)
{
  int token;
  
  nextaim = na_NULL;
  actcell = SAVE_ROOT();

  paran = 0;
  while (1) {
    token = yylex ();

    switch (token) {
      
    case cat_POP:
      switch (nextaim) {
      case na_LEFT:
	nextaim = na_APPEND;
	saveenv ();
	actcell = CAR(actcell) = ALLOC();
	nextaim = na_LEFT;
	break;

      case na_RIGHT:
	nextaim = na_COMPL;		/* den Parser nach ) beeinflussen! */
	saveenv ();
	nextaim = na_APPEND;
	break;

      case na_APPEND:
	actcell = CDR(actcell) = ALLOC();
	nextaim = na_APPEND;
	saveenv ();
	actcell = CAR(actcell) = ALLOC();
	nextaim = na_LEFT;
	break;

      case na_QUOTE:
	actcell = CDR(actcell) = ALLOC();
	nextaim = na_QUOTECL;
	saveenv ();
	actcell = CAR(actcell) = ALLOC();
	nextaim = na_LEFT;
	break;

      case na_COMPL:
	SETERR (0, LESYNTAX);
	
      case na_NULL:
	nextaim = na_NULL;
	saveenv ();
	actcell = CAR(actcell) = ALLOC();
	nextaim = na_LEFT;
	break;

      default:
	SETERR (0, LEPARSER);
      }
      break;

    case cat_PCL:
      switch (nextaim) {
      case na_LEFT:
	restoreenv ();
	CAR(actcell) = NIL;
	break;

      case na_RIGHT:
	SETERR (0, LESYNTAX);
	
      case na_APPEND:
	CDR(actcell) = NIL;
	restoreenv ();
	break;
	
      case na_COMPL:
	restoreenv ();
	break;
	
      case na_QUOTE:
      case na_NULL:
	SETERR (0, LESYNTAX);
	
      default:
	SETERR (0, LEPARSER);
      }
      break;

    case cat_QUO:
      switch (nextaim) {
      case na_LEFT:
	nextaim = na_APPEND;
	saveenv ();
	actcell = CAR(actcell) = CONS(AHASH(h_QUOTE), NIL);
	nextaim = na_QUOTE;
	break;

      case na_RIGHT:
	nextaim = na_COMPL;		/* den Parser nach ) beeinflussen! */
	saveenv ();
	actcell = CDR(actcell) = CONS(AHASH(h_QUOTE), NIL);
	nextaim = na_QUOTE;
	break;

      case na_APPEND:
	actcell = CDR(actcell) = ALLOC();
	nextaim = na_APPEND;
	saveenv ();
	actcell = CAR(actcell) = CONS(AHASH(h_QUOTE), NIL);
	nextaim = na_QUOTE;
	break;
	
      case na_QUOTE:
	actcell = CDR(actcell) = ALLOC();
	nextaim = na_QUOTECL;
	saveenv ();
	actcell = CAR(actcell) = CONS(AHASH(h_QUOTE), NIL);
	nextaim = na_QUOTE;
	break;
	
      case na_COMPL:
	SETERR (0, LESYNTAX);
	
      case na_NULL:
	nextaim = na_NULL;
	saveenv ();
	actcell = CAR(actcell) = CONS(AHASH(h_QUOTE), NIL);
	nextaim = na_QUOTE;
	break;
	
      default:
	SETERR (0, LEPARSER);
      }
      break;
      
    case cat_DOT:
      switch (nextaim) {
      case na_APPEND:
	nextaim = na_RIGHT;
	break;
	
      case na_QUOTE:
      case na_LEFT:
      case na_RIGHT:
      case na_COMPL:
      case na_NULL:
	SETERR (0, LESYNTAX);
	
      default:
	SETERR (0, LEPARSER);
      }
      break;

    case cat_ATOM:
      switch (nextaim) {
      case na_LEFT:
	CAR(actcell) = yylval;
	nextaim = na_APPEND;
	break;
	
      case na_RIGHT:
	CDR(actcell) = yylval;
	nextaim = na_COMPL;
	break;

      case na_APPEND:
	actcell = CDR(actcell) = CONS(yylval, NIL);
	nextaim = na_APPEND;
	break;

      case na_QUOTE:
	actcell = CDR(actcell) = CONS(yylval, NIL);
	nextaim = na_QUOTECL;
	break;
	
      case na_COMPL:
	SETERR (0, LESYNTAX);

      case na_NULL:
	actcell = CAR(actcell) = CONS(yylval, NIL);
	nextaim = na_NULL;
	break;

      default:
	SETERR (0, LEPARSER);
      }
      break;
    case cat_STOP:
      *mode = PROGEND;
      return NIL;
    }
    
    while (nextaim == na_QUOTECL) {
      restoreenv();
    }
    
    if (nextaim == na_NULL)
      break;
  }
  
  *mode = PROGGOON;
  return CAR(actcell);
}

int
runprog_plain (char *source)
{
  int mode;
  Atom parsedeval;

  if (source) {
    /* an dieser Stelle entscheiden, ob es sich um ein bereits kompiliertes
       Programm handelt, oder um einen Quellcode.  Davon abhngig den
       Runtimeparser oder einfach nur die Runmaschine aufrufen...! */

    saveparserenv (flxbf);	/* store parser environment */

    flxbf = yy_scan_string (source);

    do {
      parsedeval = lispparse (source, &mode);

      if (mode != PROGEND) {
	lasteval = evaluate (parsedeval);
/* 	show (lasteval); */
      }
    }
    while (mode != PROGEND);
    
    yy_delete_buffer (flxbf);
    
    flxbf = restoreparserenv ();	/* restore parser environment */
    yy_switch_to_buffer (flxbf);
    
    return 1;
  }
  
  return -1;
}
