/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include <ctype.h>
#include <time.h>

#include "proto.h"
#include "hc.h"
#include "path.h"
#include "sm.h"			/* for kaengurubase */

int initlispfilebufsize = 256;	/* default initbuffersize is 256 byte */

static char lisppath[FILENAME_MAX] = "";
char default_lisp_path[FILENAME_MAX] = DEFAULT_LISP_PATH;

void
initpaths ()
{
  char *pathname, *token;
  Atom var;
  
  var = lookup (h_LOAD_PATH); /* get the load-path */
  VAR_DATA (var) = NIL;
  
  pathname = (char*) alloca (strlen(default_lisp_path));
  strcpy (pathname, default_lisp_path);
  
  token = strtok (pathname, ":\n");
  while (token) {
    VAR_DATA(var) = CONS(NEWSTR(token), VAR_DATA(var));
    token = strtok (NULL, ":\n");
  }

  var = lookup (h_BASE_PATH);	/* set the base path */
  VAR_DATA (var) = NEWSTR(kaengurubase);
}


/* ffnet eine Lispdatei filename und liefert eine FILE * struktur zurck.
   Ist in filename bereits ein "/" enthalten, so sucht openlispfile direkt
   unter diesem Namen, ansonsten geht es die Suchpfade unter der internen
   Variable load-path durch.  Wurde die Datei in keinem dieser Pfade
   entdeckt liefert openlispfile NULL, ansonsten den gltigen Pfad.  Der
   Name der geffneten Datei findet sich in der globalen Variable lisppath
   (die von nachfolgenden Aufrufen berschrieben wird). */
FILE *
openlispfile (char *filename)
{
  FILE *iofile = NULL;

  lisppath[0] = '\0';
  
  if (index (filename, '/')) {	/* there is a path in the string! */
    strcpy (lisppath, filename);
    iofile = fopen (lisppath, "r");
    return iofile;
  }
  else {
    Atom s = VAR_DATA(lookup (h_LOAD_PATH)); /* get the load-path */
    
    while (TYP(s) == CELL_P) {
      if (TYP(CAR(s)) == STR_P) {
	char *str = STR_STR(CAR(s));
	if (str) {
	  sprintf (lisppath, "%s/%s", str, filename);
	  iofile = fopen (lisppath, "r");
	  if (iofile)
	    return iofile;
	}
      }
      s = CDR(s);
    }
  }
  return NULL;
}

/* ldt ein lisp-programm in einen buffer und gibt einen Pointer auf diesen
   Buffer zurck.  Der Buffer ist mindestens initlispfilebufsize gro und
   wird entsprechend der Programmgre angepat.  loadlispprog benutzt
   getlisppath um den Pfad fr ein Lispprogramm zufinden. */
char *
loadlispprog (char *name, int *len, int *file_enc)
{
  FILE *iofile = NULL;
  
  iofile = openlispfile (name);

  if (!iofile)
    return NULL;
  
  if (!feof(iofile)) {
    char linebuffer[1025];
    if (fgets (linebuffer, 1024, iofile) == NULL) 
      goto eofhd;

    *file_enc = MAGIC_CHECK(linebuffer);
    if (*file_enc == RAW1_CODE) {
      int bytesread = 0;
      char *buffer = NULL;
      /* at first get length of file */
      fseek (iofile, 0, SEEK_END);
      *len = ftell (iofile);
      fseek (iofile, 0, SEEK_SET);
      
      buffer = (char *) smalloc (*len);
      bytesread = fread (buffer, 1, *len, iofile);
      if (bytesread < *len) {
	free (buffer);
	SETERR(0, _("error reading binary file"));
      }
      return buffer;
    }
    else {
      char 
	*buffer = (char*) smalloc (initlispfilebufsize);
      int 
	bp = 0, l,
	buflen = initlispfilebufsize;
      
      buffer[0] = '\0';		/* buffer auf null setzen */
      fseek (iofile, 0, SEEK_SET); /* reset file! */
      
      while (!feof(iofile)) {
	if (fgets (linebuffer, 1024, iofile) == NULL) 
	  goto eofhd;
	
	if (linebuffer) {
	  l = strlen(linebuffer);
	  if (l + bp >= buflen) {
	    buflen += l;
	    buffer = (char *) realloc (buffer, buflen);
	    if (!buffer) {
	      goto errhd;
	    }
	  }
	  strcpy(&buffer[bp], linebuffer); 
	  bp += l;
	}
      }
    eofhd:
      buffer[bp++] = '\0';		/* force end */
      *len = bp;
      
      fclose (iofile);
      return buffer;
      
    errhd:
      *len = 0;
      if (buffer)
	free (buffer);
      return NULL;
    }
  }
  return NULL;
}


/* ----------------------------------------------------------------------
   Execute a (compiled) string buffer
   ---------------------------------------------------------------------- */
int
execute_buffer (char *buffer, int fenc)
{
/*   clock_t clock_start = 0, clock_end = 0; */
/*   double elapsed; */
  
  if (buffer) {
    
/*     if (verbose) { */
/*       printf("Running ...\n"); */
/*       clock_start = clock(); */
/*     } */
    
    /* and go ... */
    switch (fenc) {
    case RAW1_CODE:
      printf (_("This buffer is unsupported raw 1 code!\n"));
      return -1;
    case PLAIN_CODE:
      runprog_plain (buffer);
    }
    
/*     if (verbose) { */
/*       clock_end = clock (); */
/*       elapsed = ((double) (clock_end - clock_start)) / CLOCKS_PER_SEC; */
/*       printf ("\nTIME MARKED: %f\n", elapsed); */
/*     } */
    return 1;
  }
  return -1;
}

Atom
run_file (char *name, Atom mode)
{
  int ret, tmplen, fenc;
  char *tmp = NULL;
  
  tmp = loadlispprog (name, &tmplen, &fenc); /* load the programm */
  if (tmp == NULL) {
    if (mode) {
      SETERR (0, _("no file"));
    }
    else 
      return 0;
  }
  else {
    /* this construction is vital: if the is an setjmp error in
       execute_buffer a malloced buffer won't be freed! So we have to make
       an dynamically allocated buffer on the stack, and free the malloc
       space from loadlispprog! */
    char buffer[tmplen];
    memcpy (buffer, tmp, tmplen);
    free (tmp);
    
    ret = execute_buffer (buffer, fenc);
    
    if (ret >= 0)
      return 1;
  }
  return 0;
}
