/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "proto.h"
#include "hc.h"

static char s_num_equal[] = "=";
static char a_num_equal[] = ".>n";
Atom
f_num_equal (Atom list)
{
  Atom s = list;
  int lasti = numv(CAR(list)), t = 0;
  while (TYP(s) == CELL_P) {
    t = numv(CAR(s));
    if (t != lasti)
      return FALSE;
    s = CDR(s);
  }
  return TRUE;
}

static char s_num_less[] = "<";
static char a_num_less[] = ".>n";
Atom
f_num_less (Atom list)
{
  Atom s = list;
  int lasti = numv(CAR(s))-1, t = 0;
  while (TYP(s) == CELL_P) {
    t = numv(CAR(s));
    if (t <= lasti)
      return FALSE;
    lasti = t;
    s = CDR(s);
  }
  return TRUE;
}

static char s_num_great[] = ">";
static char a_num_great[] = ".>n";
Atom
f_num_great (Atom list)
{
  Atom s = list;
  int lasti = numv(CAR(list)) + 1, t = 0;
  while (TYP(s) == CELL_P) {
    t = numv(CAR(s));
    if (t >= lasti)
      return FALSE;
    lasti = t;
    s = CDR(s);
  }
  return TRUE;
}

static char s_num_lesseq[] = "<=";
static char a_num_lesseq[] = ".>n";
Atom
f_num_lesseq (Atom list)
{
  Atom s = list;
  int lasti = numv(CAR(list)), t = 0;
  while (TYP(s) == CELL_P) {
    t = numv(CAR(s));
    if (t < lasti)
      return FALSE;
    lasti = t;
    s = CDR(s);
  }
  return TRUE;
}

static char s_num_greateq[] = ">=";
static char a_num_greateq[] = ".>n";
Atom
f_num_greateq (Atom list)
{
  Atom s = list;
  int lasti = numv(CAR(list)), t = 0;
  while (TYP(s) == CELL_P) {
    t = numv(CAR(s));
    if (t > lasti)
      return FALSE;
    lasti = t;
    s = CDR(s);
  }
  return TRUE;
}

static char s_num_plus[] = "+";
static char a_num_plus[] = ".>n";
Atom
f_num_plus (Atom list)
{
  Atom s = list;
  int sum = 0;
  while (TYP(s) == CELL_P) {
    sum += numv(CAR(s));
    s = CDR(s);
  }
  return NEWINT(sum);
}

static char s_num_minus[] = "-";
static char a_num_minus[] = "n.>n";
Atom
f_num_minus (Atom list)
{
  Atom s = list;
  int diff = 0, llen = LENGTH(list);
  
  switch (llen) {
  case 0:
    SETERR(list, LEARGNUM);
  case 1:
    return NEWINT((numv(CAR(s))*-1));
  default:
    diff = numv(CAR(s));
    s = CDR(s);
    while (TYP(s) == CELL_P) {
      diff -= numv(CAR(s));
      s = CDR(s);
    }
    return NEWINT(diff);
  }
  return UNSPECIFIED;
}

static char s_num_times[] = "*";
static char a_num_times[] = ".>n";
Atom
f_num_times (Atom list)
{
  Atom s = list;
  int mul = 1;
  while (TYP(s) == CELL_P) {
    mul *= numv(CAR(s));
    s = CDR(s);
  }
  return NEWINT(mul);
}

static char s_num_div[] = "/";
static char a_num_div[] = "n.>n";
Atom
f_num_div (Atom list)
{
  Atom s = list;
  int dval = 0, d = 1, llen = LENGTH(list);
  
  if (llen == 1) {
    return NEWINT(0);
  }
  
  dval = numv(CAR(s));
  s = CDR(s);
  
  while (TYP(s) == CELL_P) {
    d = numv(CAR(s));
    if (d == 0)
      SETERR(list, LENUMOVERFLOW);
    
    dval /= d;
    s = CDR(s);
  }
  return NEWINT(dval);
}

static char s_num_mod[] = "modulo";
static char a_num_mod[] = "nn";
Atom
f_num_mod (Atom list)
{
  int d = 1;
  
  d = numv(CAR(CDR(list)));
  if (d == 0)
    SETERR(list, LENUMOVERFLOW);
  
  return NEWINT(numv(CAR(list)) % d);
}

static char s_num_zeroq[] = "zero?";
static char a_num_zeroq[] = "n";
Atom
f_num_zeroq (Atom list)
{
  return ZEROQ(numv(CAR(list)));
}

static char s_num_oddq[] = "odd?";
static char a_num_oddq[] = "n";
Atom
f_num_oddq (Atom list)
{
  return ODDQ(numv(CAR(list)));
}

static char s_num_evenq[] = "even?";
static char a_num_evenq[] = "n";
Atom
f_num_evenq (Atom list)
{
  return EVENQ(numv(CAR(list)));
}

static char s_numberq[] = "number?";
static char a_numberq[] = "*";
Atom
f_numberq (Atom list)
{
  return number(CAR(list));
}

static char s_integerq[] = "integer?";
static char a_integerq[] = "*";
Atom
f_integerq (Atom list)
{
  if ((TYP(CAR(list)) == INT_P) 
      || (OBL_TYP(CAR(list)) == EXTINT_T))
    return TRUE;
  return FALSE;
}

static char s_oid2int[] = "oid->integer";
static char a_oid2int[] = "i";
Atom
f_oid2int (Atom list)
{
  return NEWINT (oidv(CAR(list)));
}

static char s_int2oid[] = "integer->oid";
static char a_int2oid[] = "n";
Atom
f_int2oid (Atom list)
{
  return NEWOID (numv(CAR(list)));
}

static char s_inc[] = "increase";
static char a_inc[] = "n";
Atom
f_inc (Atom list)
{
  return NEWINT(numv(CAR(list)) + 1);
}

static char s_dec[] = "decrease";
static char a_dec[] = "n";
Atom
f_dec (Atom list)
{
  return NEWINT(numv(CAR(list)) - 1);
}

void
init_numeric()
{
  Functiontable functbl[] = {
    {s_num_equal, h_NUM_EQUAL, a_num_equal, f_num_equal},
    {s_num_less, h_NUM_LESS, a_num_less, f_num_less},
    {s_num_great, h_NUM_GREAT, a_num_great, f_num_great},
    {s_num_lesseq, h_NUM_LESSEQ, a_num_lesseq, f_num_lesseq},
    {s_num_greateq, h_NUM_GREATEQ, a_num_greateq, f_num_greateq},
    {s_num_plus, h_NUM_PLUS, a_num_plus, f_num_plus},
    {s_num_minus, h_NUM_MINUS, a_num_minus, f_num_minus},
    {s_num_times, h_NUM_TIMES, a_num_times, f_num_times},
    {s_num_div, h_NUM_DIV, a_num_div, f_num_div},
    {s_num_mod, h_NUM_MOD, a_num_mod, f_num_mod},
    {s_num_zeroq, h_NUM_ZEROQ, a_num_zeroq, f_num_zeroq},
    {s_num_oddq, h_NUM_ODDQ, a_num_oddq, f_num_oddq},
    {s_num_evenq, h_NUM_EVENQ, a_num_evenq, f_num_evenq},
    {s_numberq, h_NUMBERQ, a_numberq, f_numberq},
    {s_integerq, h_INTEGERQ, a_integerq, f_integerq},
    {s_inc, h_INCREASE, a_inc, f_inc},
    {s_dec, h_DECREASE, a_dec, f_dec},
    {s_oid2int, h_OID2INT, a_oid2int, f_oid2int},
    {s_int2oid, h_INT2OID, a_int2oid, f_int2oid},
    {NULL, -1, NULL, NULL}
  };
  Functiontable *ft = functbl;
  int i;
  
  for (i = 0; (ft+i)->fname != NULL ; i++)
    make_fset ((ft+i)->fname, (ft+i)->hashcode, (ft+i)->args, (ft+i)->fnc);
}
