/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#include "proto.h"

/* ----------------------------------------------------------------------
   Size values.
   ---------------------------------------------------------------------- */
int gc_stackpagesize = 128;
int gc_initstacksize = 512;
int heappagesize = 128;
int maxheapsize = 16384;
int oblistpagesize = 32;
int maxoblistsize = 4096;
int strlistpagesize = 32;
int maxstrlistsize = 4096;
int minfreecell = 32;
int minfreeobj = 16;
int minfreestr = 16;
int stackpagesize = 128;

Atom *stack;
int sp, stacksize;
Atom activl[MAXACTIVL];         /* die activ-sexp-list */
int activlptrl, activlptrr;
Atom *gc_stack = NULL;
int gc_sp, gc_stacksize;
int free_gc_stack = false;

/* prototyp */
int mark_tree (Atom x, char *cellbf, char *objbf, char *strbf);

/* ----------------------------------------------------------------------
   THE GARBAGE COLLECTOR
   ---------------------------------------------------------------------- */
#define MARK_CELL(cell, bitfield) \
  (bitfield[PTR(cell) / 8] |= 1 << (PTR(cell) % 8))
#define MARK_OBL MARK_CELL
#define MARK_STR MARK_CELL
#define ISMARKED(cell,bitfield)\
  (((bitfield[PTR(cell) / 8] & (1 << (PTR(cell) % 8))) ? 1 : 0))

/* add a pointer to the free list */
#define ADD2FL(cell)				\
({						\
  Atom _cell = (cell);				\
  CAR(_cell) = NIL;				\
  CDR(_cell) = CDR(FL);				\
  CDR(FL) = _cell;				\
})

/* add a pointer to the oblist free list */
#define ADD2OFL(obj)				\
({						\
  Atom _o = (obj);				\
  OBL_TYP(_o) = c_NIL;				\
  OBL_DATA(_o) = (void*)OBL_CDR(OFL);		\
  OBL_CDR(OFL) = AOBL(_o);			\
}) 

/* add a pointer to the strings free list */
#define ADD2STRFL(strp)				\
({						\
  Atom _rp = (strp);				\
  STR_TYP(_rp) = STR_fl;			\
  STR_NEXT(_rp) = ASTR(STR_NEXT(STRFL));	\
  STR_NEXT(STRFL) = ASTR(_rp);			\
})

#define FREE_STR(strp)				\
({						\
  Atom _p = (strp);				\
  if (STR_TYP(_p) == STR_alloc) {		\
    free (STR_STR(_p));				\
  }						\
  else if (STR_TYP (_p) == STR_cons) {          \
    printf ("Arrgh: %s\n", STR_STR(_p));        \
  }                                             \
  ADD2STRFL (_p);				\
})

/* prototypes */
#define PUT2ACTIVL(x)				\
({						\
  activl[activlptrr] = (x);			\
  activlptrr++;					\
  if (activlptrr >= MAXACTIVL)			\
    activlptrr = 0;				\
  if (activlptrr == activlptrl) {		\
    activlptrl++; 				\
    if (activlptrl >= MAXACTIVL)		\
      activlptrl = 0;				\
  }						\
})

#define FREE_OBL(obj)				\
({						\
  Atom _obj = (obj);				\
  switch (OBL_TYP(_obj)) {			\
  case VECTOR_T:				\
    if (VECTOR_ALLOC_SIZE(_obj) > 0) {		\
      free (VECTOR_ARRAY(_obj));		\
      free (OBL_VECTOR(_obj));			\
    }						\
    break;					\
  case DFIELD_T:				\
    free_blob (DFIELD_BLOBID (_obj));		\
    if (DFIELD_SIZE(_obj) > 0)		        \
      free (DFIELD_DATA(_obj));			\
    free (OBL_DFIELD(_obj));			\
    break;					\
  }						\
  ADD2OFL (_obj);				\
})
 
#define PUSH_GC(x)						\
({								\
  if (gc_sp >= gc_stacksize) {					\
    gc_stacksize += gc_stackpagesize;				\
    gc_stack = (Atom*) realloc ((Atom*) gc_stack,		\
				gc_stacksize * sizeof (Atom));	\
  }								\
  gc_stack[gc_sp] = (x);					\
  gc_sp++;							\
})

#define POP_GC()				\
({						\
  if (gc_sp > 0)				\
    gc_sp--;					\
  gc_stack[gc_sp];				\
})

#define MARK(obj,cbf,obf,sbf)			\
({						\
  Atom _om = (obj);				\
  gc_sp = 0;                                    \
  switch (TYP(_om)) {				\
  case CELL_P:					\
    mark_tree (_om, cbf, obf, sbf);		\
    break;					\
  case OBL_P:					\
    MARK_OBL (_om, obf);		        \
    if (OBL_TYP(_om) == VECTOR_T)               \
      mark_vec(_om, cbf, obf, sbf);             \
    break;					\
  case STR_P:					\
    MARK_STR (_om, sbf);			\
    break;					\
  }						\
})

void
mark_vec (Atom obj, char *cellbf, char *objbf, char *strbf)
{
  int i, sov = VECTOR_SIZE(obj);
  Atom atom;
  for (i = 0; i < sov; i++) {
    atom = VECTOR_ARRAY(obj)[i];
    switch (TYP(atom)) {
    case CELL_P:
      mark_tree (atom, cellbf, objbf, strbf);
      break;
    case OBL_P:
      MARK_OBL (atom, objbf);
      if (OBL_TYP(atom) == VECTOR_T)
	mark_vec(atom, cellbf, objbf, strbf);
      break;
    case STR_P:
      MARK_STR (atom, strbf);
      break;
    }
  }
}

int
mark_tree (Atom x, char *cellbf, char *objbf, char *strbf)
{
  int old_gc_sp = gc_sp;
  
  while (1) {
    if ((x == NIL) || (ISMARKED(x, cellbf))) {
      if (gc_sp > old_gc_sp)
	x = POP_GC();
      else
	goto mark_tree_done;
    }
    
    MARK_CELL (x, cellbf);
    if (TYP(CAR(x)) == CELL_P) {
      if (TYP(CDR(x)) == CELL_P) {
	if (CDR(x) != NIL) {
	  PUSH_GC (CDR(x));
	}
      }
      else {
	if (TYP(CDR(x)) == OBL_P) {
	  MARK_OBL (CDR(x), objbf);
	  if (OBL_TYP(CDR(x)) == VECTOR_T)
	    mark_vec(CDR(x), cellbf, objbf, strbf);
	}
	else if (TYP(CDR(x)) == STR_P) {
	  MARK_STR (CDR(x), strbf);
	}
      }
      x = CAR(x);
    }
    else {
      if (TYP(CAR(x)) == OBL_P) {
	MARK_OBL (CAR(x), objbf);
	if (OBL_TYP(CAR(x)) == VECTOR_T)
	  mark_vec(CAR(x), cellbf, objbf, strbf);
      }
      else if (TYP(CAR(x)) == STR_P) {
	MARK_STR (CAR(x), strbf);
      }
      
      if (TYP(CDR(x)) == CELL_P) {
	x = CDR(x);
      }
      else {
	if (TYP(CDR(x)) == OBL_P) {
	  MARK_OBL (CDR(x), objbf);
	  if (OBL_TYP(CDR(x)) == VECTOR_T)
	    mark_vec(CDR(x), cellbf, objbf, strbf);
	}
	else if (TYP(CDR(x)) == STR_P) {
	  MARK_STR (CDR(x), strbf);
	}
	x = c_NIL;
      }
    }
  }
 mark_tree_done:
  return 0;
}

#define MARK_FL(bitfield)			\
({						\
  Atom _x = FL;					\
  while (_x != NIL) {				\
    MARK_CELL (_x, bitfield);			\
    _x = CDR(_x);				\
  }						\
})						\

#define MARK_OBLFL(obf)			        \
({						\
  Atom _x = OFL;				\
  while (PTR(_x)) {				\
    MARK_OBL (_x, obf);		                \
    _x = OBL_CDR(_x);				\
  }						\
})

#define MARK_STRFL(bitfield)			\
({						\
  Atom _x = STRFL;				\
  MARK_STR (0, bitfield);                       \
  while (PTR(STR_NEXT(_x))) {			\
    MARK_STR (PTR(_x), bitfield);		\
    _x = STR_NEXT(_x);				\
  }						\
  MARK_STR (PTR(_x), bitfield);			\
})

#define MARK_STACK(cellbf, objbf, strbf)		\
({							\
  register int i;					\
  if (sp > 0) {						\
    for (i = 0; i < sp; i++)                            \
      MARK(stack[i], cellbf, objbf, strbf);             \
  }							\
}) 

/* Setze das Bitfeld komplet auf 0 */
#define CLEARBITFIELD(len,bitfield)(memset (bitfield, 0, (len / 8) + 1))

/* just for testing the bitfield */
void
printbitfeld (int len, char *rvc)
{
  register int i;
  
  for (i = 0; i < len; i++) {
    putchar((rvc[i/8] & (1 << (i%8))) ? '1' : '0');
    if ((i % 8) == 7) putchar(' ');
  }
  printf("\n");
}

/* der garbage collector sammelt allen ungebrauchten Schrott wieder ein;
   gab es einen Fehler, so gibt es -1 zurck.  Der Garbagecollector
   berprft zuerst einige feste Adressen (Freiliste, Akku, Activp etc) auf
   belegte Pltze (die freilist mu ja nicht jedesmal komplett neu
   aufgebaut werden).  minfree gibt an wieviele Heappltze mindestens
   befreit werden mssen.  Konnten nur weniger als minfree Sexps
   zurckgeholt werden, so meldet garbage_collector einen Fehler.  Ist
   minfree = 0, so wird nie ein Fehler gemeldet, default sollte 1 sein. */
int
gc (int minfree, int *cell, int *obj, int *str)
{
  unsigned char
    cellbf[(heapsize / 8) + 1],	  /* heap bitfield */
    objbf[(oblistsize / 8) + 1],  /* oblist bitfield */
    strbf[(strlistsize / 8) + 1]; /* stringlist bitfield */
  register Atom i;		/* counter */
  int
    freed = 0,			/* number of freed cells */
    obfreed = 0,		/* number of freed objects */
    strfreed = 0;		/* number of freed strings */
  
  /* allocate a stack */
  if (!gc_stack) {
    gc_stacksize = gc_initstacksize;
    gc_stack = (Atom *) smalloc (gc_stacksize * sizeof (Atom));
  }
  
  /* clear the bitfields */
  CLEARBITFIELD (heapsize, cellbf);
  CLEARBITFIELD (oblistsize, objbf);
  CLEARBITFIELD (strlistsize, strbf);
  
  /* now mark all allocated places */
  MARK (ENVL, cellbf, objbf, strbf); /* step through local ENVList */
  MARK (GENVL, cellbf, objbf, strbf); /* step through global ENVList */
  MARK (SCANL, cellbf, objbf, strbf); /* step through functionlist */
  
  MARK_FL (cellbf);		/* mark cell free list */
  MARK_OBLFL (objbf);		/* mark oblist free list */
  MARK_STRFL (strbf);		/* mark string free list */
  MARK_STACK (cellbf, objbf, strbf); /* mark lists in stack */

  MARK_CELL (c_NIL, cellbf);
  
  /* now step through the activ list (like incubator list in smalltalk) to
     save all sexps and objects which aren't attached to one of the above
     lists */
  i = activlptrl;
  for ( ; ; ) {
    MARK(activl[i], cellbf, objbf, strbf);
    if (i == activlptrr)
      goto actldone;
    i++;
    if (i >= MAXACTIVL)
      i = 0;
  }
actldone:
  
  /*printbitfeld (heapsize, cellbf);*/ /* for testing bitfield */
  /*printbitfeld (oblistsize, objbf);*/ /* for testing bitfield */ 
  /*printbitfeld (strlistsize, strbf);*/ /* for testing bitfield */

  /* now we are at the point : free the unmarked cells! */
  for (i = 0; i < heapsize; i++) {
    if (!(ISMARKED (i, cellbf))) {
      ADD2FL (ACELL(i));
      freed++;
    }
  }
  for (i = 0; i < oblistsize; i++) {
    if (!(ISMARKED (i, objbf))) {
      FREE_OBL (i); 
      obfreed++;
    }
  }
  for (i = 0; i < strlistsize; i++) {
    if (!(ISMARKED (i, strbf))) {
      FREE_STR (i); 
      strfreed++;
    }
  }

  if (free_gc_stack) {
    gc_stacksize = 0;
    free (gc_stack);
    gc_stack = NULL;
  }
  *cell = freed;
  *obj = obfreed;
  *str = strfreed;
  if (freed < minfree)		/* is number of freed nodes < minfree */
    return -1;			/* return error */
  return freed;			/* else return number of freed nodes */
}

/* ----------------------------------------------------------------------
   ALLOCATION AND REALLOCATION
   ---------------------------------------------------------------------- */
/* reallocheap vergrert den Heap um den Wert von heappagesize.  Es
   verndert den Wert von heapsize.  Bei einem Fehler gibt's -1 zurck,
   ansonsten 1*/
#define REALLOC_HEAP()					\
({							\
  __label__ done;					\
  int _heapoldp, _retval = 0;				\
  register int _i;					\
  if ((heapsize + heappagesize) > maxheapsize) {	\
    _retval = -1;					\
    goto done;						\
  }							\
  _heapoldp = heapsize;					\
  heapsize += heappagesize;				\
  heap = (Cell*) realloc ((Cell*) heap,			\
			  heapsize * sizeof (Cell));	\
  if (!heap) {						\
    SETERR (0, LEMEM);					\
  }							\
  for (_i = _heapoldp; _i < heapsize; _i++) {		\
    ADD2FL(ACELL(_i));					\
  }							\
done:							\
  _retval;						\
})

/* reallocates new space for the oblist.  If it fails it returns -1,
   otherwise 0 */
#define REALLOC_OBLIST()					\
({								\
  __label__ done;						\
  int _obloldp, _retval = 0;					\
  register int _i;						\
  if ((oblistsize + oblistpagesize) > maxoblistsize) {		\
    _retval = -1;						\
    goto done;							\
  }								\
  _obloldp = oblistsize;		/* save last size */	\
  oblistsize += oblistpagesize; /* new oblistsize */		\
  oblist = (Oblist*) realloc ((Oblist*) oblist,			\
			      oblistsize * sizeof (Oblist));	\
  if (!oblist) {						\
    SETERR (0, LEMEM);						\
  }								\
  for (_i = _obloldp; _i < oblistsize; _i++) {			\
    ADD2OFL (_i);						\
  }								\
done:								\
  _retval;							\
})

/* reallocates new space for the stringlist.  if realloc_strlist fails it
   returns -1, otherwise 0 */
#define REALLOC_STRLIST()						\
({									\
  __label__ done;							\
  int _strloldp, _retval = 0;						\
  register int _i;							\
  if ((strlistsize + strlistpagesize) > maxstrlistsize) {		\
    _retval = -1;							\
    goto done;								\
  }									\
  _strloldp = strlistsize;	/* save last size */			\
  strlistsize += strlistpagesize; /* new oblistsize */			\
  strlist = (Strlist*) realloc ((Strlist*) strlist,			\
				strlistsize * sizeof (Strlist));	\
  if (!strlist) {							\
    SETERR (0, LEMEM);							\
  }									\
  for (_i = _strloldp; _i < strlistsize; _i++) {			\
    ADD2STRFL (_i);							\
  }									\
done:									\
  _retval;								\
})

/* get a new cell from the free list.  Is the freelist is empty
   (PTR(CDR(FL)) == NIL), allocsexp calls the garbage_collector.  If this
   returns -1 alloc_cell tries to call realloc_heap.  If this fails too,
   alloc_cell terminates the interpreter with the errormessage LEMEM.  The
   two arguments car and cdr are set to the car and cdr branch of the new
   allocates cell */
Atom
alloc_cell (Atom car, Atom cdr)
{
  Atom p;
  int fsexps, fobjs, fstr;
  
  if (TYP(CDR(FL)) != CELL_P) {
    if (gc (1, &fsexps, &fobjs, &fstr) < minfreecell) {
      if (REALLOC_HEAP() < 0)
	goto errhd;
    }
  }
  
  p = ACELL(CDR(FL));		/* next freelist! */
  CDR(FL) = CDR(p);		/* manage freelist */
  CAR(p) = car;			/* init cons-cell */
  CDR(p) = cdr;
  
  PUT2ACTIVL (p);		/* p to activ list */
  
  return p;
  
errhd:
  SETERR (0, LEMEM);
}

/* allocates a new object in the oblist.  If the oblist is full it tries to
   allocate new memory.  If this fails, it stops the interpreter.  The new
   allocates Object is set to TYP typ and DATA value */
Atom
alloc_obl (int typ, void *value)
{
  Atom p;
  int fsexps, fobjs, fstr;
  
  if (!PTR(OBL_CDR(OFL))) {
    
    gc (1, &fsexps, &fobjs, &fstr);
    if (fobjs <= minfreeobj) {
      if (REALLOC_OBLIST() < 0)
	goto errhd;
    }
  }
  
  p = AOBL(OBL_CDR(OFL));	/* next obj freelist entry */
  OBL_CDR(OFL) = OBL_CDR(p);	/* manage freelist */
  
  OBL_TYP(p) = typ;
  OBL_DATA(p) = value;
  
  PUT2ACTIVL (p);		/* p to activ list */
  
  return p;
  
errhd:
  SETERR (0, LEMEM);
}

/* allocates a new string slot in the string list.  If the string list is
   full it tries to allocate new memory.  If this fails, it stops the
   interpreter.  The new allocated string is set to the STRING str.  if
   (str != NULL) alloc_str takes a copy of the string (strdup). */
Atom
alloc_str (char *str, int mode)
{
  Atom p;
  int fsexps, fobjs, fstr;

  if (!PTR(STR_NEXT(STRFL))) {
    
    gc (1, &fsexps, &fobjs, &fstr);
    if (fstr <= minfreestr) {
      if (REALLOC_STRLIST() < 0)
	goto errhd;
    }
  }
  
  p = ASTR(STR_NEXT(STRFL));	/* next str freelist entry */
  STR_NEXT(STRFL) = STR_NEXT(p); /* manage freelist */
  
  switch (mode) {
  case STR_alloc:
    STR_TYP(p) = STR_alloc;
    STR_STR(p) = strdup (str);
    break;
  case STR_ptr:
    STR_TYP(p) = STR_alloc;
    STR_STR(p) = str;
    break;
  case STR_cons:
    STR_TYP(p) = STR_cons;
    STR_STR(p) = str;
    break;
  }
  
  PUT2ACTIVL (p);		/* p to activ list */
  
  return p;
  
 errhd:
  SETERR (0, LEMEM);
}


/* ----------------------------------------------------------------------
   The global stack.  Dynamic Stack!
   ---------------------------------------------------------------------- */
void
reset_scm ()
{
  /* secure_envl.cdr = CDR(ENVL);
     secure_envl.car = CAR(ENVL); */
  sp = 0;
  CDR(ENVL) = NIL;
  CAR(ENVL) = NIL;
  DESCERR (0, NULL);		/* setzte Fehlermeldung zurck */
}

void
initstack (int initstacksize)
{
  stacksize = initstacksize;
  stack = (Atom*) smalloc (stacksize * sizeof (Atom));
  sp = 0;
}

void
pushstack (Atom x)
{
  if (sp >= stacksize) {
    stacksize += stackpagesize;
    stack = (Atom*) realloc ((Atom*) stack, stacksize * sizeof (Atom));
  }
  stack[sp] = x;
  sp++;
}

Atom
popstack ()
{
  if (sp > 0) 
    sp--;
  return stack[sp];
}

/* ----------------------------------------------------------------------
   The Activ (Incubator) List
   ---------------------------------------------------------------------- */
void
initactivl ()
{
  int i;
  
  for (i = 0; i < MAXACTIVL; i++)
    activl[i] =  NIL;
  activlptrl = 0;
  activlptrr = 0;
}



