/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#define PROG_NAME "kaenguru"

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <locale.h>

#include "server.h"
#include "path.h"

int connectid = 0;		/* the connection id */
char *connkey = NULL;		/* the connection key */
char *client_usrname = NULL;	/* the client user name */
Uid clientuid = 0;		/* die User ID des clients */
Gid clientgid = 0;		/* die (default) Group ID des clients */
ACode accmask = A_HIER;		/* als default hierachical access code */
bool trace = false;		/* sollen Meldungen geloggt werden? */
int clientsock, daemonsock;	/* the communication streams */
char configpath[FILENAME_MAX] = CONFIG_DEFAULT;
int class_outdate_period = 120;	/* classlist cleanup period, default 2 min. */
int class_check_period = 30;	/* check for cleanup, default 30 sec. */
int timeoutval = 30;		/* default 10 seconds timeout */

void manual_start ();		/* prototyp */

void
sigpipe_handler (int sig)
{
  /* okay, we have to block the broken pipe signal, so we can handle the
     communication in a save way */
  signal (SIGPIPE, sigpipe_handler);
}


void
main (int argc, char **argv)
{
  char c;
  int tmpsize = 1024;
  char *tail, *tmp = (char *) malloc (tmpsize);

  /* set up signal-handlers */
  signal (SIGPIPE, sigpipe_handler); /* broken pipe */

  setlocale (LC_MESSAGES, "");
  textdomain (PACKAGE);

  /* at first be sure to be started by the daemon! */
  if (strcmp (argv[0], "*kaenguru*") != 0) {
    manual_start ();
    exit (1);
  }

  /* now parse the arguments */
  opterr = 0;
  
  while ((c = getopt (argc, argv, "c:d:tE")) != -1) { /* get options */
    switch (c) {
    case 'c':
      clientsock = strtol(optarg, &tail, 0);
      break;
    case 'd':
      daemonsock = strtol(optarg, &tail, 0);
      break;
    case 't':
      trace = true;
      break;

    case '?':			/* wrong option */
      fprintf(stderr, _("unknown option -%c\n"), optopt);
      goto starterrhd;
      
    default:			/* other options */
      fprintf(stderr, _("unsupportet option!\n"));
    }
  }
  
  if (read_from_sock (daemonsock, &tmp, &tmpsize) < 0)
    goto starterrhd;
  if (parse_login_data (tmp, &connectid, &connkey, &client_usrname, 
			&clientuid, &clientgid) < 0)
    goto connerrorhd;
  
  /* read the global configuration file */
  if (read_configuration (NULL) < 0)
    goto starterrhd;
  
  /* now tell the daemon, we are ok. */
  if (writesockf (daemonsock, "%s %d %s %c%s\n", 
		  KCP_SSTARTED,
		  connectid, connkey,
		  KCP_OKC, KCP_OK) <= 0)
    goto no_daemon_hd;
  
  if (writesockf (clientsock, "%c%s Welcome %s\n", 
		  KCP_OKC, KCP_OK, client_usrname) <= 0)
    goto starterrhd;

  free (tmp);

  servercycle ();
  return;
  
 connerrorhd:
  free (tmp);
  PRINTERR (_("Connection error in backend."));

 starterrhd:
  PRINTERR (errstr);
  /* the machine is killed a moment later, so we don't need the error code */
  writesockf (daemonsock, "%s %d %s %c%s.\n", 
	      KCP_CLOSE, connectid, connkey,
	      KCP_ERRC, KCP_ERR);

 no_daemon_hd:
  free (tmp);
  return;
}

/* fr den Server ein eigenes tracing: der Server schickt alle trace-infos
   via daemon-socket an den daemon, der dann fr ihn loggt!  */
int
tracing (const char *message)
{
  if (trace) {
    return writesockf (daemonsock, "%s %d %s %s\n", 
		       KCP_TRACE,
		       connectid, connkey, (char*) message);
  }
  return 1;
}

/* ----------------------------------------------------------------------
   Provide an error message if the backends is started manual.
   ---------------------------------------------------------------------- */
void
manual_start ()
{
  printf (_(
"This is Kaenguru %s - The Kaenguru Database System (%s)\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"
"You can't run %s manualiter.  The kaenguru database server is called\n"
"automatically by the kaenguru database daemon.\n"
"\n"), VERSION, PROG_NAME, COPYRIGHTYEAR, PROG_NAME);
}
