/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#ifndef _KDBS_KSCM_H
#define _KDBS_KSCM_H

#include <setjmp.h>		/* for jmp_buf */

#include "kdbs.h"

/* ----------------------------------------------------------------------
   DEFINITION OF GLOBAL TYPES
   ---------------------------------------------------------------------- */
/*
  Die oberen 4 Bits geben den Typ an
    0   cell    "immediate"
    1   int     immediate
    2   string  ->strlist
    3   hash    ->hashlist/immediate
    4   char    immediate
    5   bool    immediate
    6   others  ->oblist
 */

#define BITS    24

#define TYP_P   (15 << BITS)	/* zum Austesten if (ATOM_P & x) */
#define PTR_P   ((1 << BITS)-1)   /* 2^0+...2^23, */
#define NEG_P   (128 << BITS)	/* the Negativ flag */
#define NEG0 (1 << BITS)
#define CELL_P  (0 << BITS)
#define INT_P   (1 << BITS)
#define STR_P   (2 << BITS)
#define HASH_P  (3 << BITS)
#define CHAR_P  (4 << BITS)
#define BOOL_P  (5 << BITS)
#define OBL_P   (6 << BITS)
#define NIL_P   (7 << BITS)
#define OID_P   (8 << BITS)	/* interne OID pointer BITS breit */

#define NIL     (0 | NIL_P)
#define o_NIL   (0 | OBL_P)

#define CELL_T     0
#define INT_T      1
#define STR_T      2
#define HASH_T     3
#define CHAR_T     4
#define BOOL_T     5
#define DATE_T     7
#define VECTOR_T   8
#define DFIELD_T   9
#define OID_T     10		/* Oblist OID pointer 32 bits breit */
#define SUBR_T    11
#define EXTINT_T  12
#define EXTSTR_T  13
#define NONE_T    99

/* Heap-Offsets der Systemlisten */
#define c_NIL       (0 | CELL_P)
#define FL          (1 | CELL_P) /* free list */
#define ENVL        (2 | CELL_P) /* the current environment */
#define GENVL       (3 | CELL_P) /* the global environment */
#define SCANL       (4 | CELL_P) /* the scan akku */

/* Oblist-Offset der Oblist */
#define OFL         (0 | OBL_P)

/* Strlist-Offset der Stringlist */
#define STRFL       (0 | STR_P)
#define STR_alloc   1
#define STR_ptr     2
#define STR_cons    3
#define STR_fl      0

#define PTR(x)((x & PTR_P))
#define TYP(x)((x & TYP_P))
#define CAR(x)((heap + PTR(x))->car)
#define CDR(x)((heap + PTR(x))->cdr)
#define CAAR(x)(CAR(CAR(x)))
#define CDAR(x)(CDR(CAR(x)))
#define CADR(x)(CAR(CDR(x)))
#define CDDR(x)(CDR(CDR(x)))
#define CAAAR(x)(CAR(CAR(CAR(x))))
#define CDAAR(x)(CDR(CAR(CAR(x))))
#define CADAR(x)(CAR(CDR(CAR(x))))
#define CDDAR(x)(CDR(CDR(CAR(x))))
#define CAADR(x)(CAR(CAR(CDR(x))))
#define CDADR(x)(CDR(CAR(CDR(x))))
#define CADDR(x)(CAR(CDR(CDR(x))))
#define CDDDR(x)(CDR(CDR(CDR(x))))
#define CADDDR(x)(CAR(CDR(CDR(CDR(x)))))


/* set type bits */
#define ACELL(x)(x & PTR_P)
#define AINT(x)								\
({									\
  int _i = (x);								\
  Atom _f = ((_i<0)?((_i&PTR_P)|INT_P|NEG_P):((_i&PTR_P)|INT_P));	\
  _f;									\
})
#define ASTR(x)(((x) & PTR_P) | STR_P)
#define AHASH(x)(((x) & PTR_P) | HASH_P)
#define ACHAR(x)(((x) & PTR_P) | CHAR_P)
#define AOBL(x)(((x) & PTR_P) | OBL_P)
#define ABOOL(x)(((x) & PTR_P) | BOOL_P)
#define AOID(x)(((x) & PTR_P) | OID_P)

#define INTV(x)(((x) & NEG_P) ? (int)(PTR(x)-NEG0) : (int)(PTR(x)))
#define CHARV(x)((char)PTR(x))
#define HASHV(x)(PTR(x))
#define BOOLV(x)(PTR(x))
#define OIDV(x)(PTR(x))

#define UNSPECIFIED (ABOOL(2))
#define TRUE   (ABOOL(1))
#define FALSE  (ABOOL(0))
     
#define CONS(x,y)(alloc_cell(x,y))
#define ACONS(ob1,ob2,ob3) (CONS (CONS (ob1, ob2), ob3))
#define ALLOC()(alloc_cell(NIL,NIL))
#define INS2LIST(l,p) ({ CDR(p) = CDR(l); CDR(l) = p; })

#define NEWOBL(t,x)(alloc_obl(t,x))
#define NEWSTR(s)(alloc_str(s,STR_alloc))
#define PTRSTR(s)(alloc_str(s,STR_ptr))
#define CONSSTR(s)(alloc_str(s,STR_cons))
#define NEWINT(i)(AINT(i))
#define NEWHASH(h)(addhash(h))
#define NEWCHAR(c)(ACHAR(decode_char(c)))
#define NEWDATE(d)(alloc_obl(DATE_T,(void*)d))
#define NEWEXTINT(i)(alloc_obl(EXTINT_T,(void*)i))
#define NEWSUBR(s)(alloc_obl(SUBR_T,(void*)s))
#define NEWOID(o)							\
({ 									\
  ((o) > (1 << BITS) ? (alloc_obl(OID_T, (void*)o)) : AOID(PTR(o)));	\
})
#define NEWVECTOR(v)(alloc_obl(VECTOR_T,(void*)v))
#define NEWDFIELD(d)					\
({							\
  Atom _retval = alloc_obl(DFIELD_T, (void*)d);		\
  record_blob (_retval, DFIELD_BLOBID(_retval));	\
  _retval;						\
})
#define OBL_TYP(x)((oblist + PTR(x))->typ)
#define OBL_DATA(x)((oblist + PTR(x))->data.val)
#define OBL_CDR(x)((oblist + PTR(x))->data.next)

#define STR_DATA(x)((strlist + PTR(x))->data.val)
#define STR_NEXT(x)((strlist + PTR(x))->data.next)
#define STR_TYP(x)((strlist + PTR(x))->typ)

#define HASH_STR(x)((char*)(hashlist + PTR(x))->string)

#define STR_STR(x)((strlist + PTR(x))->data.string)
#define OBL_DATE(x)((oblist + PTR(x))->data.date)
#define OBL_VECTOR(x)((oblist + PTR(x))->data.vector)
#define OBL_DFIELD(x)((oblist + PTR(x))->data.dfield)
#define OBL_OID(x)((oblist + PTR(x))->data.oid)
#define OBL_INT(x)((oblist + PTR(x))->data.num)
#define OBL_STR(x)((oblist + PTR(x))->data.str)
#define OBL_CODE(x)((oblist + PTR(x))->data.subr)

#define VECTOR_SIZE(v)((OBL_VECTOR(v))->size)
#define VECTOR_ALLOC_SIZE(v)((OBL_VECTOR(v))->alloc_size)
#define VECTOR_REF(v,r)((OBL_VECTOR(v))->array[r])
#define VECTOR_ARRAY(v)((OBL_VECTOR(v))->array)

#define DFIELD_SIZE(v)((OBL_DFIELD(v))->size)
#define DFIELD_DATA(v)((OBL_DFIELD(v))->data)
#define DFIELD_BLOBID(v)((OBL_DFIELD(v))->blobid)

#define oidv(o)					\
({						\
  Oid _oid = 0;					\
  Atom _o = (o);				\
  if (TYP(_o) == OID_P)				\
    _oid = OIDV(_o);				\
  else if ((TYP(_o) == OBL_P)			\
	   && (OBL_TYP(_o) == OID_T))		\
    _oid = OBL_OID(_o);				\
  _oid;						\
})

#define PUSH_SCANL()				\
{						\
  pushstack(CAR(SCANL));			\
  pushstack(CDR(SCANL));			\
}

#define POP_SCANL()				\
{						\
  CDR(SCANL) = popstack(); 			\
  CAR(SCANL) = popstack();			\
}

#define SAVE_ROOT()				\
({						\
  CAR(SCANL) = NIL;				\
  CDR(SCANL) = NIL;				\
  SCANL;					\
})
     
#define PUSH_ENVL()				\
{						\
  pushstack(CAR(ENVL));				\
  pushstack(CDR(ENVL));				\
}

#define POP_ENVL()				\
{						\
  CDR(ENVL) = popstack();			\
  CAR(ENVL) = popstack();			\
}

/* ----------------------------------------------------------------------
   CONVENIENTFUNCTIONS UM EINFACH UND STANDARDISIERT AUF VARIABLEN UND
   FUNKTIONEN IN DER VARIABLENLISTE ZUGREIFEN ZU KNNEN.
   ---------------------------------------------------------------------- */

/* ----------------------------------------------------------------------
  VARIABLES

  ->[name|.]->[ro|.]->[value|/]

   name:  hash
   ro:    'ro, 'rw

  FUNCTIONS

  ->[name|.]->[ro|.]->[func|/]

   name:  hash
   func:  closure oder fset
   ro: 'ro (function can't be redefined)
       'rw (function can be redefined)

  als autoloaded definierte Label:

  ->[name|.]->[autoload|.]->["filename"|/]

  autoload : h_AUTOLOAD (!)

   Wie versucht auf eine solche Variable/Referenz zuzugreifen, so wird
   erst die entsprechende Datei geladen und evaluiert. 


   CLOSURE:     [clos|.]->[formals|.]->[env|.]->[begin|.]->[code|/]

   env: das enclosed environment; eine alist
                  ->[.|.]->[.|.]-> ... [...|/]
                     |
		     v
		    [VAR/LAMBDA/FSET/SPEC|.]->...

   args:  alist:  ->[x1|.]->[x2|.]->[x3|/]
                  ->[x1|.]->[x2|.]->[x3|x4]
                  ->x1
 
   FSET:  [fset|.]->[argstr|.]->[fset-ofs|/]

  argstr: ein string, der die Typen und Anzahl der erforderten Parameter
          angibt.  Bei jedem FSET-Call werden alle evaluierten Parameter an
          dieser Liste ausgetestet, so da in den Funktionen (Callbacks)
          keine grundstzliche berprfung mehr auf Typen geschehen mu
          (abgesehen von mehrdeutigen Parameterbergaben).  Erlaubte Code:
          a : hier mu ein Atom stehen (alles aus einer liste)
          l : hier mu eine Liste stehen
	  h : hier mu ein symbol stehen (Hash)
          s : hier mu ein String stehen
	  n : hier mu ein Zahlwert stehen (extern, intern, oid)
	  o : hier mu ein object stehen
	  x : hier mu eine Funktion stehen (exec: clos, fset, spec, object)
	  c : hier mu ein Char stehen
	  * : hier kann irgendwas stehen
	  d : hier mu ein datum stehen
	  v : hier mu ein vector stehen
	  f : hier mu ein datenfeld stehen (field)
	  i : hier mu ein oid stehen (id)
	  b : hier mu ein booleanausdruck stehen.
	  . : ab dieser Stelle sind angegebene Parameter optional
	  > : der nachfolgende Typ gilt fr alle weiteren Parameter gelten.
              In der Regel mit dem Schalter '.' verwendet.

	  ACHTUNG: auf '.' mu irgendein weiteres Zeichen auer '.' folgen
                   auf '>' mu irgendein Typenschalter folgen
                   (kein '.', kein '>'). Ansonsten kann das System abstrzen.

	  Beispiele:
	  "*"  genau ein Parameter beliebigen Typs
	  "s"  genau ein Parameter vom Typ String
	  "hs*" ein symbol, ein string und einen weiteren Parameter 
	        beliebigen Typs
          "n.*" ein num. Wert und optional beliebige weitere Parameter
	  "s.n"  erfordert einen string mit einem optionalen num. Wert
	  "s.nn" erfordert einen string mit zwei optionalen num. Werten
	  "s.>n" erfordert einen string mit einen beliebigen Anzahl
	        an optionalen num. Parametern  
          "sn.>n" ditto, es mu wenigstens 1 num. Parameter stehen
	  "xl.>l" ein closure (Funktion) und mindestens eine Liste

   ---------------------------------------------------------------------- */

#define SET_VAR(p,name,val,ro)			\
{						\
  Atom _p = (p);				\
  CAR(_p) = name;				\
  CAR(CDR(_p)) = ro;				\
  CAR(CDR(CDR(_p))) = val;			\
}
#define VAR_NAME(p)(CAR(p))
#define VAR_DATA(p)(CAR(CDR(CDR(p))))
#define VAR_RO(p)(CAR(CDR(p)))

#define CLOS_ARGS(x)(CADR(x))
#define CLOS_CODE(x)(CDDR(x))
#define FSET_ARGS(x)(STR_STR(CADR(x)))
#define FSET_CODE(x)(CDDR(x))

#define SET_VAR_DATA(p,val)			\
({						\
  Atom _p = (p), _val = (val);			\
  if (HASHV(VAR_RO(_p)) == h_RO) {		\
    SETERR(_p, LEVARRO);			\
  }						\
  VAR_DATA(_p) = _val;				\
  _val;						\
})

#define MAKE_VAR(hash, modus, val)		\
({						\
  Atom _rx, _ry;				\
  PUSH_SCANL();					\
  _rx = SAVE_ROOT();				\
  _ry = CDR(_rx) = CONS (val, NIL);		\
  _ry = CDR(_rx) = CONS (AHASH(modus), _ry);	\
  _ry = CDR(_rx) = CONS (AHASH(hash), _ry);	\
  POP_SCANL();					\
  _ry;						\
})
/* (CONS(AHASH(hash), CONS(AHASH(modus), CONS(val, NIL)))) */
#define PUTSYM(list, hash, modus, cell)\
({ CDR(list) = CONS(MAKE_VAR(hash, modus, cell), CDR(list)); })


/* ----------------------------------------------------------------------
   COMPLETE EVAL-FUNCTIONS

   ---------------------------------------------------------------------- */

#define LENGTH(list)					\
({							\
  Atom _s = (list);					\
  int _count = 0;					\
  while (TYP(_s) == CELL_P) {				\
    _s = CDR(_s);					\
    _count++;						\
  }							\
  _count;						\
})

#define LIST_TAIL(list,k)			\
({						\
  Atom _s = (list), _retval = NIL;		\
  int _k = (k), _count = 0;			\
  while ((TYP(_s) == CELL_P)			\
	 && (_count < _k)) {			\
    _s = CDR(_s);				\
    _count++;					\
  }						\
  if ((_count == _k)				\
      && (TYP(_s) == CELL_P))			\
    _retval = _s;				\
  _retval;					\
})

#define LIST_REF(list,k)			\
({						\
  Atom _l = (list), _retval = NIL;		\
  Atom _ret = LIST_TAIL(_l, k);			\
  if (TYP(_ret) == CELL_P)			\
    _retval = CAR(_ret);			\
  _retval;					\
})

/* for the hashfunctions */
#define SETHASH(name,hc)			\
({						\
  Atom _hc = (hc);				\
  if (_hc + 1 > hashsize)			\
    hashsize = _hc + 1;				\
  if (_hc + 1 > systemhash)			\
    systemhash = _hc + 1;			\
  HASH_STR(_hc) = name;				\
})

#define NCONC(l,n)				\
({						\
  Atom _r = (l), _s = _r, _retval = NIL;	\
  if (TYP(_s) == CELL_P) {			\
    while (TYP(CDR(_s)) == CELL_P)		\
      _s = CDR(_s);				\
    CDR(_s) = n;				\
    _retval = _r;				\
  }						\
  else {					\
    SETERR(_s, LEWRONGTYP);			\
  }						\
  _retval;					\
})

#define APPEND(l,n)				\
({						\
  Atom _s = (l), _retval = NIL, _acts;		\
  if (TYP(_s) == CELL_P) {			\
    PUSH_SCANL ();				\
    _acts = SAVE_ROOT();			\
						\
    while (TYP(CDR(_s)) == CELL_P) {		\
      _acts = CDR(_acts) = CONS(CAR(_s), NIL);	\
      _s = CDR(_s);				\
    }						\
    _acts = CDR(_acts) = CONS(CAR(_s), NIL);	\
    if (CDR(_acts) != NIL) {			\
      SETERR(_acts, LEWRONGTYP);		\
    }						\
    CDR(_acts) = n;				\
    _retval = CDR(SCANL);			\
    POP_SCANL ();				\
  }						\
  else {					\
    SETERR(_s, LEWRONGTYP);			\
  }						\
  _retval;					\
})

#define ATOMQ(x)				\
({						\
  Atom _retval = TRUE;				\
  if (TYP(x) == CELL_P)				\
    _retval = FALSE;				\
  _retval;					\
})

#define LISTQ(x)				\
({						\
  Atom _s = (x), _retval = TRUE;		\
  while (TYP(_s) == CELL_P)			\
    _s = CDR(_s);				\
  if (_s != NIL)				\
    _retval = FALSE; 				\
  _retval; 					\
})

#define NULLQ(x)				\
({						\
  Atom _retval = FALSE;				\
  if ((x) == NIL)				\
    _retval = TRUE;				\
  _retval;					\
})

#define VECTORQ(x)				\
({						\
  Atom _retval = FALSE, _x = (x);		\
  if ((TYP(_x) == OBL_P)			\
      && (OBL_TYP(_x) == VECTOR_T))		\
    _retval = TRUE;				\
  _retval;					\
})

#define OIDQ(x)					\
({						\
  Atom _retval = FALSE, _x = (x);		\
  if ((TYP(_x) == OID_P)			\
      || ((TYP(_x) == OBL_P)			\
	  && (OBL_TYP(_x) == OID_T)))		\
    _retval = TRUE;				\
  _retval;					\
})

#define DATEQ(x)				\
({						\
  Atom _retval = FALSE, _x = (x);		\
  if ((TYP(_x) == OBL_P) 			\
      && (OBL_TYP(_x) == DATE_T))		\
    _retval = TRUE;				\
  _retval;					\
})

#define DFIELDQ(x)				\
({						\
  Atom _retval = FALSE, _x = (x);		\
  if ((TYP(_x) == OBL_P)			\
      && (OBL_TYP(_x) == DFIELD_T))		\
    _retval = TRUE;				\
  _retval;					\
})

#define OBJECTQ(x)						\
({								\
  Atom _retval = FALSE, _x = (x);				\
  if ((TYP(_x) == OBL_P)					\
      && (OBL_TYP(_x) == VECTOR_T)				\
      && (TYP(VECTOR_REF(_x, OBJECT_ID_SLOT)) == HASH_P)	\
      && (HASHV(VECTOR_REF(_x, OBJECT_ID_SLOT)) == h_OBJECT))	\
    _retval = TRUE;						\
  _retval;							\
})

#define NOT(x)					\
({						\
  Atom _retval = TRUE;				\
  if ((x) != FALSE)				\
    _retval = FALSE;				\
  _retval;					\
})

#define ASSQ(alist,val)					\
({							\
  __label__ done;					\
  Atom _s = (alist), _v = (val), _retval = FALSE;	\
  while (TYP(_s) == CELL_P) {				\
    if (TYP(CAR(_s)) == CELL_P) {			\
      if (eq (CAR(CAR(_s)), _v) != FALSE) {		\
	_retval = CAR(_s);				\
	goto done;					\
      }							\
    }							\
    _s = CDR (_s);					\
  }							\
done:							\
  _retval;						\
})

#define ASSOC(alist,val)				\
({							\
  __label__ done;					\
  Atom _s = (alist), _v = (val), _retval = FALSE;	\
  while (TYP(_s) == CELL_P) {				\
    if (TYP(CAR(_s)) == CELL_P) {			\
      if (equal (CAR(CAR(_s)), _v) != FALSE) {		\
	_retval = CAR(_s);				\
	goto done;					\
      }						        \
    }							\
    _s = CDR (_s);					\
  }							\
done:							\
  _retval;						\
})

#define MEMQ(llist,val)					\
({							\
  __label__ done;					\
  Atom _s = (llist), _val = (val), _retval = FALSE;	\
  while (TYP(_s) == CELL_P) {				\
    if (eq (CAR(_s), _val) != FALSE) {			\
      _retval = _s;					\
      goto done;					\
    }							\
    _s = CDR (_s);					\
  }							\
done:							\
  _retval;						\
})

#define MEMBER(llist,val)				\
({							\
  __label__ done;					\
  Atom _s = (llist), _val = (val), _retval = FALSE;	\
  while (TYP(_s) == CELL_P) {				\
    if (equal (CAR(_s), _val) != FALSE) {		\
      _retval = _s;					\
      goto done;					\
    }							\
    _s = CDR (_s);					\
  }							\
done:							\
  _retval;						\
})

#define ODDQ(x)(((x) % 2) > 0 ? TRUE : FALSE)
#define EVENQ(x)(((x) % 2) > 0 ? FALSE : TRUE)
#define ZEROQ(x)((x) == 0 ? TRUE : FALSE)

#define LIST2STR(list)					\
({							\
  Atom _s = (list), _retval = NIL;			\
  int _tmpsize = 256;					\
  int _tmpp = 0;					\
  char *_tmp = (char *) smalloc (_tmpsize);		\
  _tmp[0] = '\0';					\
  while (TYP(_s) == CELL_P) {				\
    if (TYP(CAR(_s)) == CHAR_P) {			\
      char _c = CHARV(CAR(_s));				\
      if (_tmpp + 2 > _tmpsize) {			\
	_tmpsize += + 256;				\
	_tmp = (char *) realloc (_tmp, _tmpsize);	\
	if (!_tmp) {					\
	  SETERR (0, LEMEM);				\
	}						\
      }							\
      _tmp[_tmpp] = _c;					\
      _tmpp++;						\
    }							\
    else {						\
      SETERR (CAR(_s), LEWRONGTYP);			\
    }							\
    _s = CDR(_s);					\
  }							\
  _tmp[_tmpp] = '\0';                                   \
  _retval = NEWSTR(_tmp);				\
  free (_tmp);						\
  _retval;						\
})

/* ----------------------------------------------------------------------
   DEFINITIONS FOR THE OBJECTSYSTEM
   ---------------------------------------------------------------------- */
/* slots   
   0 :object         a symbol allways = :object  (h_OBJECT)
   1 Class           a symbol, string, or OID (the super class).  If NIL 
                     this class has no superclass (only for Core)
   2 Oid             the OID of the class (or @0 for Superclasses)
   3 method list     alist of all methods
   4 desc            the attribut encoding.  An vector, parallel to slot 5
   5 data            the data field.  An vector parallel to slot 4
   6 fid             die Fild Id, sofern ntig (in welcher Datei wird dieses
                     Object abgelegt
   7 uid             Supervisor des Objects
   8 gid             welcher Gruppe gehrt dieses Object?

   Achtung: Es reicht nicht diese Daten an dieser Stelle zu ndern, sondern
   entsprechende nderungen und Nachtrge mssen auch in init.k4
   nachgetragen werden! */

#define SLOT_SIZE        9
#define OBJECT_ID_SLOT   0
#define CLASS_SLOT       1
#define OID_SLOT         2
#define METHL_SLOT       3
#define DESC_SLOT        4
#define DATA_SLOT        5
#define FID_SLOT         6
#define UID_SLOT         7
#define GID_SLOT         8

#define ATTR_LABEL_TYPE        'l'
#define ATTR_STRING_TYPE       's'
#define ATTR_OID_TYPE          'o'
#define ATTR_INT_TYPE          'i'
#define ATTR_DFIELD_TYPE       'f'
#define ATTR_DATE_TYPE         'd'
#define ATTR_VECTOR_TYPE       'v'
#define ATTR_CODE_TYPE         'x'
#define ATTR_START_ATTR        '('
#define ATTR_LAST_ATTR         ')'
#define ATTR_NULL_TYPE         '-'
#define ATTR_TRUE_TYPE         't'
#define ATTR_FALSE_TYPE        'n'
#define ATTR_UNSPECIFIED_TYPE  'u'

/* ----------------------------------------------------------------------
   DIE EIGENTLICHEN TYPENDEFINITIONEN
   ---------------------------------------------------------------------- */
typedef unsigned long Atom;
typedef struct {
  Atom car, cdr;
} Cell;

typedef struct {
  int size;
  int alloc_size;
  Atom *array;
} Vector;

typedef struct {
  int size;
  int blobid;
  char *data;
} Dfield;


typedef struct {
  int typ;
  union {
    void *val;
    Date date;
    Vector *vector;
    Dfield *dfield;
    Oid oid;
    Atom (*subr)();		/* external Function */
    Atom *addr;			/* externe Variablen */
    int *num;
    char *str;
    Atom next;			/* used for the freelist */
  } data;
} Oblist;

typedef struct {
  char *string;
} Hashlist;

typedef struct {
  unsigned char typ;
  union {
    void *val;
    char *string;
    Atom next;
  } data;
} Strlist;

/* fr die Binary Large Object list */
typedef struct t_Blobrec Blobrec;
struct t_Blobrec {
  int id;			/* the blob-id */
  Atom ptr;			/* a pointer to the k4-dfield-object */
  Blobrec *next;		/* the next element in bloblist */
};

/* for the inital functiontable */
typedef struct {
  char *fname;			/* name der Function */
  int hashcode;			/* a perdefined hash code (-1 for none) */
  char *args;			/* a string for the arg list */
  Atom (*fnc)();		/* die callbackfunction */
} Functiontable;		/* Table fr FuncInit */

typedef struct {
  char *name;			/* name der Variablen */
  int hashcode;			/* the hashcode */
  int modus;			/* readonly, readwrite */
  int typ;			/* erlaubter Typ */
  int intrn;			/* true oder false */
  void *deflt;  		/* default values */
  void *addr;			/* the addr for external vars */
} Vartable;			/* Table fr VarInit */

typedef struct {
  int hash;
  char *string;
} Hashpredefined;

/* ----------------------------------------------------------------------
   Typ fr Index Such Rckgaben
   ---------------------------------------------------------------------- */
typedef struct {
  Long size;
  union {
    Oid oid;
    Atom find_list;
  } data;
} Searchinfo;


/* ----------------------------------------------------------------------
   ERROR MANAGAMENT
   ---------------------------------------------------------------------- */
#define LISPERR(errdsc)(errstr = (errdsc))
#define LASTEXP(lexp)(lastexp = (lexp))
#define DESCERR(lexp, errc)			\
({						\
  LISPERR(errc);				\
  LASTEXP(lexp);				\
})
#define JMPERR()({longjmp (interpreter_error, 1);})
#define SETERR(lastexp, errdesc)		\
({						\
  DESCERR(lastexp, errdesc);			\
  JMPERR();					\
})

#define LEVARRO           _("variable readonly")
#define LEUNBOUND         _("unbound variable")
#define LEWRONGTYP        _("wrong type to apply")
#define LEARGNUM          _("wrong number of arguments")
#define LEMEM             _("out of memory")
#define LEESCSEQ          _("bad escape sequence")
#define LEUNTERMSTR       _("unterminated string")
#define LEPARAN           _("unbalanced paranthesis")
#define LEINCLDEPTH       _("error include depth")
#define LESTACKINCL       _("stack include error")
#define LESTACKPARSE      _("error in parser stack")
#define LESTACKUFPARSE    _("parser stack under flow")
#define LESYNTAX          _("syntax error")
#define LEPARSER          _("parser error")
#define LENUMOVERFLOW     _("numerical overflow")
#define LEBADFORMAL       _("bad formals")
#define LERANGE           _("out of range")
#define LERADIX           _("unsupported radix")
#define LESYMDEF          _("symbol undefined")

/* ----------------------------------------------------------------------
   Voreinstellungen, Pfade, etc.
   ---------------------------------------------------------------------- */
#define MAXACTIVL   16
#define MAXENVSTACK         512


/* MAGICNUMBERS fr codierte und compilierte Programme.  Ein Magic-key mu
   mit \n aufhren, da loadlispprog sonst Fehler machen kann!  The Code-Ids
   are arbitrary */
#define MAGIC_RAW1    "!k4raw\001\n"
#define RAW1_CODE     123
#define PLAIN_CODE    32

#define MAGIC_CHECK(buf)			\
({						\
  int _a = PLAIN_CODE;				\
  if (memcmp(buf, MAGIC_RAW1, 8) == 0)		\
    _a = RAW1_CODE;				\
  _a;						\
})

#endif




