/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif

#if defined HAVE_REGEX_H
# include <regex.h>
#endif

#include "proto.h"
#include "hc.h"
#include "sm.h"

/* ----------------------------------------------------------------------
   Routinen fr einen btree.
   ---------------------------------------------------------------------- */
int
scan_page (Database *db, Fid iid, Node wp, Vector *vector)
{
  int i;
  Workpage workpage;
  
  if (wp == i_NOLINK)
    return 0;
  
  if (load_index_page (db, iid, wp, &workpage) <= 0)
    goto errhd;
  
  if (workpage.itm[0].link > i_NOLINK) {
    if (scan_page (db, iid, workpage.itm[0].link, vector) < 0)
      goto errhd;
  }
  for (i = 1; i <= workpage.items; i++) {
    if (workpage.pagetype == i_KEYTREE) {
/*       printf ("%s\n", workpage.itm[i].key); */
    }
    else if (workpage.pagetype == i_REFTREE) {
/*       printf ("%ld\n", workpage.itm[i].info); */
      
      /* add this item to the find-vector */
      vector_set (vector, NEWOID(workpage.itm[i].info), vector->size);
    }
    
    if (workpage.itm[i].link > i_NOLINK) {
      if (scan_page (db, iid, workpage.itm[i].link, vector) < 0)
	goto errhd;
    }
  }
  return 1;
  
 errhd:
  return -1;
}

Atom
list_btree (Database *db,	/* the database */
	    Fid iid,		/* the index */
	    Node startwp,	/* last read workpage */
	    int *size)		/* the number of the found item in wp */
{
  Vector *vector = make_vector (0, 0, FALSE);
  Atom retval, rx;
  
  PUSH_SCANL ();		/* secure the coming NEWOIDs! */
  rx = SAVE_ROOT();
  CDR(rx) = NEWVECTOR(vector);
  
  if (vector == NULL)
    goto errhd;
  if (scan_page (db, iid, startwp, vector) < 0)
    goto errhd;
  *size = vector->size;
  retval = CDR(rx);
  POP_SCANL();
  
  return retval;
  
 errhd:
  POP_SCANL();
  
  *size = 0;
  return FALSE;
}


/* ----------------------------------------------------------------------
   Fr die Rckgabe von Index werten
   ---------------------------------------------------------------------- */
int
search_index (Database *db, Fid iid, char *find_key, Searchinfo **sinfo)
{
  int found;
  int find_list_size;
  Keyinfo keyinfo;
  Atom find_list;
  
  found = retrieve_index (db, iid, find_key, &keyinfo);
  
  switch (found) {
  case 1:
    if (keyinfo.reftype == 'r') {
      *sinfo = (Searchinfo *) smalloc (sizeof (Searchinfo));
      (*sinfo)->data.oid = keyinfo.info;
      (*sinfo)->size = 1;
    }
    else {
      find_list = list_btree (db, iid, keyinfo.info, &find_list_size);
      if (find_list != FALSE) {
	*sinfo = (Searchinfo *) smalloc (sizeof (Searchinfo));
	(*sinfo)->data.find_list = find_list;
	(*sinfo)->size = find_list_size;
      }
      else {
	*sinfo = NULL;
	return -1;
      }
    }
    return 1;
  case 0:
    *sinfo = NULL;
    return 0;
  case -1:
    goto errhd;
  }
  *sinfo = NULL;
  return 0;

 errhd:
  *sinfo = NULL;
  return -1;
}


#if defined HAVE_REGEX_H
/* ----------------------------------------------------------------------
   Routinen um einen btree mit regexp durchzusuchen. btrees sind dafr
   denkbar dumm (da man sie rekursiv Seite fr Seite durchlaufen mu!)
   ---------------------------------------------------------------------- */
int
scan_page_regexp (Database *db, Fid iid, Node wp, regex_t *rexp, 
		  Vector *vector, bool scan_oids_q, bool no_string_q)
{
  int i;
  Workpage workpage;
  
  if (wp == i_NOLINK)
    return 0;
  
  if (load_index_page (db, iid, wp, &workpage) <= 0)
    goto errhd;
  
  if (workpage.itm[0].link > i_NOLINK) {
    if (scan_page_regexp (db, iid, workpage.itm[0].link,
			  rexp, vector, scan_oids_q, no_string_q) < 0)
      goto errhd;
  }
  for (i = 1; i <= workpage.items; i++) {
    if (workpage.pagetype == i_KEYTREE) {

      if (!regexec (rexp, workpage.itm[i].key, 0, NULL, 0)) {
	Vector *ref_vector = NULL;
	Atom retval, rx;
	
	if (scan_oids_q) {
	  ref_vector = make_vector (0, 0, NIL);
	  PUSH_SCANL ();		/* secure the coming NEWOIDs! */
	  rx = SAVE_ROOT();
	  CDR(rx) = NEWVECTOR(ref_vector);
	  
	  if (workpage.itm[i].reftype == i_PAGELINK) {
	    /* here we have to scan the total oid sub tree */
	    scan_page (db, iid, workpage.itm[i].info, ref_vector);
	  }
	  else if (workpage.itm[i].reftype == i_REFLINK) {
	    /* this is only one item, so store it direct to the vector */
	    vector_set (ref_vector, NEWOID (workpage.itm[i].info),
			ref_vector->size);
	  }
	  
	  retval = CDR(rx);
	  POP_SCANL();
	}
	else {
	  retval = NIL;
	}

	if (!no_string_q) {
	  /* add the key to the vector */
	  vector_set (vector, 
		      CONS(NEWSTR(workpage.itm[i].key), retval),
		      vector->size);
	}
	else if (ref_vector) {
	  int j, size = ref_vector->size;
	  
	  for (j = 0; j < size; j++) {
	    vector_set (vector, vector_ref (ref_vector, j),
			vector->size);
	  }
	}
      }
    }
    
    if (workpage.itm[i].link > i_NOLINK) {
      if (scan_page_regexp (db, iid, workpage.itm[i].link,
			    rexp, vector, scan_oids_q, no_string_q) < 0)
	goto errhd;
    }
  }
  return 1;
  
 errhd:
  return -1;
}

Atom
list_btree_regexp (Database *db,	/* the database */
		   Fid iid,		/* the index */
		   Node startwp,	/* last read workpage */
		   regex_t *rexp,     /* the compiled reg exp */
		   int *size,	        /* number of found item */
		   bool scan_oids_q, /* complete oid ref links? */
		   bool no_string_q) /* don#t notice the real data strings */
{
  Vector *vector = make_vector (0, 0, NIL);
  Atom retval, rx;
  
  PUSH_SCANL ();		/* secure the coming NEWOIDs! */
  rx = SAVE_ROOT();
  CDR(rx) = NEWVECTOR(vector);
  
  if (vector == NULL)
    goto errhd;
  if (scan_page_regexp (db, iid, startwp, rexp, vector, 
			scan_oids_q, no_string_q) < 0)
    goto errhd;
  *size = vector->size;
  retval = CDR(rx);
  POP_SCANL();
  
  return retval;
  
 errhd:
  POP_SCANL();
  
  *size = 0;
  return FALSE;
}

#endif /* HAVE_REGEX_H */

int
search_index_regexp (Database *db, Fid iid, char *regexp,
		     Searchinfo **sinfo, bool scan_oids_q, bool no_string_q)
{
#ifndef HAVE_REGEX_H
  SM_DESCERR ("regex are not supported");
  return -3;  
#else
  int find_list_size;
  Atom find_list;
  Node root_node;
  regex_t rexp;
  
  if (regcomp (&rexp, regexp, REG_NOSUB))
    goto errhd;
  
  root_node = get_index_root (db, iid);
  
  if (root_node != i_NOLINK) {
    find_list = list_btree_regexp (db, iid, root_node, 
				   &rexp, &find_list_size, 
				   scan_oids_q, no_string_q);
    if (find_list != FALSE) {
      *sinfo = (Searchinfo *) smalloc (sizeof (Searchinfo));
      (*sinfo)->data.find_list = find_list;
      (*sinfo)->size = 2; /* find_list_size; */
    }
    else {
      regfree (&rexp);
      *sinfo = NULL;
      return 0;
    }
  }
  regfree (&rexp);
  return 1;

 errhd:
  *sinfo = NULL;
  return -2;
#endif /* ifndef HAVE_REGEX_H */
}



