/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif

#include "proto.h"
#include "hc.h"

#define CFATYP(que)((que) ? true : false)
int
check_fset_args(char *argstr, Atom params)
{
  Atom p = params;
  char *s = argstr;
  int forceargnum = true;
  int stepon = true;
  int cor;

  while (*s) {
    if (*s == '.') {
      forceargnum = false;
      s++;
    }
    if (*s == '>') {
      stepon = false;
      s++;
    }
    
    if (TYP(p) != CELL_P) {
      if (forceargnum) {
	SETERR(p, LEARGNUM);
      }
      else
	goto done;
    }

    cor = false;
    switch (*s) {
    case '*': cor = true; break;
    case 's': cor = CFATYP(TYP(CAR(p)) == STR_P); break;
    case 'n': cor = CFATYP(number(CAR(p)) == TRUE); break;
    case 'c': cor = CFATYP(TYP(CAR(p)) == CHAR_P); break;
    case 'a': cor = CFATYP((TYP(CAR(p)) != CELL_P)
			   && (CAR(p) != NIL)); break;
    case 'l': cor = CFATYP((TYP(CAR(p)) == CELL_P) 
			   || (CAR(p) == NIL)); break;
    case 'h': cor = CFATYP(TYP(CAR(p)) == HASH_P); break;
    case 'b': cor = CFATYP(TYP(CAR(p)) == BOOL_P); break;
    case 'x':
      if ((TYP(CAR(p)) == CELL_P) 
	  && (TYP(CAAR(p)) == HASH_P) 
	  && ((HASHV(CAAR(p)) == h_CLOS) 
	      || (HASHV(CAAR(p)) == h_FSET)))
	cor = true;
      break;
    case 'd':
      if ((TYP(CAR(p)) == OBL_P) 
	  && (OBL_TYP(CAR(p)) == DATE_T))
	cor = true;
      break;
    case 'i':
      if ((TYP(CAR(p)) == OID_P)
	  || ( (TYP(CAR(p)) == OBL_P) 
	       && (OBL_TYP(CAR(p)) == OID_T)))
	cor = true;
      break;
    case 'v':
      if ((TYP(CAR(p)) == OBL_P) 
	  && (OBL_TYP(CAR(p)) == VECTOR_T))
	cor = true;
      break;
    case 'f':
      if ((TYP(CAR(p)) == OBL_P) 
	  && (OBL_TYP(CAR(p)) == DFIELD_T))
	cor = true;
      break;
    case 'o':
      if ((TYP(CAR(p)) == OBL_P) 
	  && (OBL_TYP(CAR(p)) == VECTOR_T) 
	  && (TYP(VECTOR_REF(CAR(p), OBJECT_ID_SLOT)) == HASH_P) 
	  && (HASHV(VECTOR_REF(CAR(p), OBJECT_ID_SLOT)) == h_OBJECT))
	cor = true;
      break;
    default:
      SETERR(0, _("internal error. Check list.c"));
    }
    if (!cor) {
      SETERR(p, LEWRONGTYP);
    }
    if (stepon)
      s++;
    p = CDR(p);
  }
  if (TYP(p) == CELL_P) {
    SETERR(p, LEARGNUM);
  }
done:
  return true;
}


char *actautoloadfile = "";

Atom
autoload_sym (char *filename)
{
  char *bualfn = NULL;
  if (strcmp (filename, actautoloadfile) != 0) {
    Atom retval;

    bualfn = actautoloadfile;
    actautoloadfile = filename;
    
    retval = run_file (filename, 0);
    
    actautoloadfile = bualfn;
    if (retval)
      return 0;
    else
      SETERR(0, _("no file"));
  }
  return 1;
}

Atom
update (Atom var, Atom val)
{
  Atom p;
  
  p = lookup (var);
  if (PTR(p)) {
    if (HASHV(VAR_RO(p)) == h_AUTOLOAD) {
      pushstack(val);		/* save val from autoload */
      if (autoload_sym (STR_STR(VAR_DATA(p))))
	SETERR (var, _("autoload failed to define symbol"));
      popstack();
      if (HASHV(VAR_RO(p)) == h_AUTOLOAD)
	SETERR(var, _("autoload failed to define symbol"));
    }
    SET_VAR_DATA (p, val);
    return UNSPECIFIED;
  }
  SETERR (var, LEUNBOUND);
}

Atom
lookup_sym (Atom atom)
{
  Atom p = lookup (atom);
  if (PTR(p)) {
    if (HASHV(VAR_RO(p)) == h_AUTOLOAD) {
      if (autoload_sym (STR_STR(VAR_DATA(p))))
	SETERR(atom, _("autoload failed to define symbol"));
      if (HASHV(VAR_RO(p)) == h_AUTOLOAD)
	SETERR(atom, _("autoload failed to define symbol"));
    }
    return VAR_DATA(p);
  }
  SETERR (atom, LEUNBOUND);
}

Atom
define_sym (Atom var, Atom val)
{
  Atom p;
  
  p = lookup (var);
  if (PTR(p)) {
    if (HASHV(VAR_RO(p)) == h_AUTOLOAD) {
      Atom rx;
      PUSH_SCANL();		/* save val from autoload */
      rx = SAVE_ROOT();
      CDR(rx) = val;
      if (autoload_sym (STR_STR(VAR_DATA(p)))) {
	VAR_RO(p) = AHASH(h_RW);
	SET_VAR_DATA (p, val);
      }
      POP_SCANL();
      if (HASHV(VAR_RO(p)) == h_AUTOLOAD)
	SETERR(p, _("autoload failed to define symbol"));
    }
    SET_VAR_DATA (p, val);
  }
  else
    PUTSYM (GENVL, var, h_RW, val);
  return UNSPECIFIED;
}

#define MAKE_FUNCTION(formals,code)				\
({								\
  Atom _rootx, _s = (formals), _f = _s;				\
  /* check the formals for consistency */			\
  while (TYP(_s) == CELL_P) {					\
    if (TYP(CAR(_s)) != HASH_P)					\
      SETERR(CAR(_s), LEBADFORMAL);				\
    _s = CDR(_s);						\
  }								\
  if ((_s != NIL) && (TYP(_s) != HASH_P))		        \
    SETERR(_s, LEBADFORMAL);					\
  								\
  PUSH_SCANL();							\
  _s = _rootx = SAVE_ROOT();					\
  _s = CDR(_s) = CONS (AHASH(h_CLOS), NIL); /* a closure */	\
  _s = CDR(_s) = CONS(_f, (code)); /* the formals + code */	\
  _s = CDR(_rootx);						\
  POP_SCANL();							\
  _s;								\
})

#define BIND_ARGS(formals, args)				\
({								\
  Atom _f = (formals), _a = (args);				\
  while (TYP(_f) == CELL_P) {					\
    if (_a != NIL)						\
      PUTSYM (ENVL, CAR(_f), h_RW, CAR(_a));			\
    else							\
      SETERR((args), LEARGNUM);					\
    _f = CDR(_f);						\
    _a = CDR(_a);						\
  }								\
  if (TYP(_f) == HASH_P) {					\
    PUTSYM (ENVL, _f, h_RW, _a);				\
    _a = NIL;							\
  }								\
  if (_a != NIL)						\
    SETERR(_a, LEARGNUM);					\
})

#define RETURN(last)				\
({						\
  Atom _lst = (last);				\
  POP_ENVL(); 					\
  POP_SCANL(); 					\
  return _lst;					\
})

#define TAIL_RECURS ({tail = true; goto loop;})

Atom
evaluate (Atom list)
{
  Atom s = list, rootx, r, rx;
  int tail = false;
  
  PUSH_SCANL();
  rootx = SAVE_ROOT(); 
  CDR(rootx) = list;		/* save list */
  PUSH_ENVL();
  
 loop:				/* for tail recursion */
  switch (TYP(s)) {
  case INT_P: case STR_P: case BOOL_P: case CHAR_P: case NIL_P: case OID_P:
  case OBL_P:
    RETURN(s);
  case HASH_P:
    RETURN(lookup_sym (s));
  case CELL_P:
    if (TYP(CAR(s)) == HASH_P) {
      switch (HASHV(CAR(s))) {
      case h_QUOTE:
	RETURN(CADR(s));
      case h_LAMBDA:
	RETURN(MAKE_FUNCTION (CADR(s), CDDR(s)));
      case h_DEFINE:
	if (TYP(CADR(s)) != HASH_P)
	  SETERR(CADR(s), _("bad variable"));
	if (TYP(CDDR(s)) == CELL_P)
	  r = evaluate (CADDR(s));
	else
	  r = UNSPECIFIED;
	define_sym (CADR(s), r);
	RETURN(UNSPECIFIED);
      case h_SET:
	if (TYP(CADR(s)) != HASH_P)
	  SETERR(CADR(s), _("bad variable"));
	if (TYP(CDDR(s)) != CELL_P)
	  SETERR(s, _("missing expression"));
	update (CADR(s), evaluate (CADDR(s)));
	RETURN(UNSPECIFIED);
      case h_DEFINES:
	s = CDR(s);
	if (TYP(s) != CELL_P)
	  SETERR(s, _("wrong number of args"));
	r = evaluate (CAR(s));
	if (TYP(r) != HASH_P)
	  SETERR(r, _("bad variable"));
	PUSH_SCANL();
	rx = SAVE_ROOT();
	CDR(rx) = r;
	if (TYP(CDR(s)) == CELL_P)
	  r = evaluate (CADR(s));
	else
	  r = UNSPECIFIED;
	define_sym (CDR(rx), r);
	POP_SCANL();
	RETURN(UNSPECIFIED);
      case h_SETS:
	s = CDR(s);
	if (TYP(s) != CELL_P)
	  SETERR(s, _("wrong number of args"));
	r = evaluate (CAR(s));
	if (TYP(r) != HASH_P)
	  SETERR(r, _("bad variable"));
	PUSH_SCANL();
	rx = SAVE_ROOT();
	  CDR(rx) = r;
	  if (TYP(CDR(s)) != CELL_P)
	    SETERR(s, _("missing expression"));
	  update (CDR(rx), evaluate (CADR(s)));
	  POP_SCANL();
	  RETURN(UNSPECIFIED);
      case h_IF:
	s = CDR(s);
	if (TYP(s) != CELL_P) 
	  SETERR(s, _("bad IF test"));
	if (evaluate (CAR(s)) != FALSE) {
	  if (TYP(CDR(s)) != CELL_P)
	    SETERR(s, _("bad IF body"));
	  s = CDR(rootx) = CADR(s);
	  TAIL_RECURS;		/* tail recursion */
	}
	else if (CDR(s) != NIL) {
	  s = CDR(rootx) = CADDR(s);
	  TAIL_RECURS;		/* tail recursion */
	}
	else
	  RETURN(UNSPECIFIED);
      case h_AND:
	s = CDR(s);
	if (CDR(s) != NIL) {
	  while (TYP(CDR(s)) == CELL_P) {
	    if (evaluate(CAR(s)) == FALSE)
	      RETURN(FALSE);
	    s = CDR(rootx) = CDR(s);
	  }
	  s = CDR(rootx) = CAR(s); /* discard the rest of list! */
	}
	else
	  s = CDR(rootx) = TRUE;
	TAIL_RECURS;		/* tail recursion */
      case h_OR:
	s = CDR(s);
	if (CDR(s) != NIL) {
	  while (TYP(CDR(s)) == CELL_P) {
	    r = evaluate(CAR(s));
	    if (r != FALSE)
	      RETURN(r);
	    s = CDR(rootx) = CDR(s);
	  }
	  s = CDR(rootx) = CAR(s); /* discard the rest of list! */
	}
	else
	  s = CDR(rootx) = FALSE;
	TAIL_RECURS;		/* tail recursion */
      case h_BEGIN:
	s = CDR(s);
	if (TYP(s) != CELL_P)
	  SETERR(s, _("bad sequence"));
seq_loop:
	while (TYP(CDR(s)) == CELL_P) {
	  evaluate(CAR(s));
	  s = CDR(rootx) = CDR(s);
	}
	s = CDR(rootx) = CAR(s); /* discard the rest of list! */
	TAIL_RECURS;		/* tail recursion */
      case h_COND:
	s = CDR(s);
	if (TYP(s) != CELL_P)
	  SETERR(s, _("bad or missing COND clauses"));
	while (TYP(CDR(s)) == CELL_P) {
cond_loop:
	  r = evaluate(CAAR(s));
	  if (r != FALSE) {
	    if (CDAR(s) != NIL) {
	      s = CDAR(s);
	      goto seq_loop;	/* it follows a sequence */
	    }
	    RETURN(r);
	  }
	  s = CDR(rootx) = CDR(s);
	}
	if (s != NIL) {
	  if ((TYP(CAR(s)) == CELL_P) /* is this clause a else-clause? */
	      && (TYP(CAAR(s)) == HASH_P)
	      && (HASHV(CAAR(s)) == h_ELSE)) {
	    if (CDAR(s) == NIL)
	      SETERR(CAR(s), _("bad ELSE clause in COND"));
	    s = CDAR(s);
	    goto seq_loop;	/* sequence */
	  }
	  goto cond_loop;	/* another normal clause */
	}
	RETURN(UNSPECIFIED);
      case h_CASE:
	s = CDR(s);
	if (TYP(s) != CELL_P)
	  SETERR(s, _("bad or missing CASE key"));
	r = evaluate (CAR(s));
	s = CDR(s);
	while (TYP(CDR(s)) == CELL_P) {
case_loop:
	  if (TYP(CAAR(s)) != CELL_P)
	    SETERR(CAAR(s), _("bad CASE datum"));
	  if (equal(r, CAAAR(s)) != FALSE) {
	    if (CDAR(s) == NIL)
	      SETERR(CAR(s), _("bad or missing CASE clause"));
	    s = CDAR(s);
	    goto seq_loop;	/* sequence */
	  }
	  s = CDR(s);
	}
	if (s != NIL) {
	  if ((TYP(CAR(s)) == CELL_P) /* is this clause a else clause? */
	      && (TYP(CAAR(s)) == HASH_P)
	      && (HASHV(CAAR(s)) == h_ELSE)) {
	    if (CDAR(s) == NIL)
	      SETERR(CAR(s), _("bad ELSE clause in CASE"));
	    s = CDAR(s);
	    goto seq_loop;	/* sequence */
	  }
	  goto case_loop;
	}
	RETURN(UNSPECIFIED);
      case h_LET:
	s = CDR(s);
	if ((TYP(s) != CELL_P) && (s != NIL))
	  SETERR(s, _("bad formals in LET"));
	
	r = CAR(s);
	while (TYP(r) == CELL_P) {
	  if (TYP(CAR(r)) != CELL_P)
	    SETERR(r, _("bad formals in LET"));
	  if (TYP(CAAR(r)) == HASH_P) {
	    Atom f = UNSPECIFIED;
	    if (CDAR(r) != NIL)
	      f = evaluate(CADAR(r));
	    PUTSYM (ENVL, CAAR(r), h_RW, f);
	  }
	  else
	    SETERR(r, _("bad formal in LET"));
	  r = CDR(r);
	}
	
	s = CDR(s);
let_loop:
	while (TYP(s) == CELL_P) { /* this can't be done tail recursive! */
	  rx = evaluate (CAR(s));
	  s = CDR(rootx) = CDR(s);
	}
	RETURN(rx);
      case h_LETS:
	s = CDR(s);
	if ((TYP(s) != CELL_P) && (s != NIL))
	  SETERR(s, _("bad formals in LET*"));
	
	PUSH_SCANL();
	rx = SAVE_ROOT();
  
	r = CAR(s);
	while (TYP(r) == CELL_P) {
	  if (TYP(CAR(r)) != CELL_P)
	    SETERR(r, _("bad formals in LET*"));
	  
	  if (TYP(CAAR(r)) == HASH_P) {
	    Atom f = UNSPECIFIED;
	    if (CDAR(r) != NIL)
	      f = evaluate(CADAR(r));
	    PUTSYM(rx, CAAR(r), h_RW, f);
	  }
	  else
	    SETERR(r, _("bad formal in LET*"));
	  r = CDR(r);
	}
	
	r = CDR(rx);
	while (TYP(CDR(r)) == CELL_P)
	  r = CDR(r);
	CDR(r) = CDR(ENVL);	/* bind the new values to he env.list */
	CDR(ENVL) = CDR(rx);

	POP_SCANL();		/* now we can pop the stack */
	
	s = CDR(s);
	goto let_loop;
      }
    }
    else if (TYP(CAR(s)) == CELL_P) { /* if the func is evaluated yet */
      switch (HASHV(CAAR(s))) {	/* especially for map and for-each! */
      case h_CLOS: 
	goto clos_invoke;
      case h_FSET: 
	goto fset_invoke;
      }
    }
    
    /* No keyword found.  We have to evaluate the application */    
    {
      Atom f, fx;
      PUSH_SCANL();
      f = fx = SAVE_ROOT();
      
      f = CDR(f) = CONS(evaluate (CAR(s)), NIL);
      if ((TYP(CAR(f)) != CELL_P) 
	  || (TYP(CAAR(f)) != HASH_P)) {
	SETERR(CAR(f), _("wrong type to apply"));
      }
      s = CDR(s);		/* eval arguments */
      while (TYP(s) == CELL_P) {
	f = CDR(f) = CONS(evaluate (CAR(s)), NIL);
	s = CDR(s);
      }
      f = CDR(fx);
      POP_SCANL();		/* no pop the scanl */
      s = CDR(rootx) = f;		/* discard the old list! */
    }

    switch (HASHV(CAAR(s))) {
    case h_CLOS:		/* lambda */
clos_invoke:
      if (tail) {		/* give up the current environment */
	POP_ENVL();		/* restore old environment */
	PUSH_ENVL();		/* and save it at once! */
      }
      { 
	Atom fx;
	PUSH_SCANL();
	fx = SAVE_ROOT();
	CDR(fx) = s;
	BIND_ARGS(CLOS_ARGS(CAR(s)), CDR(s));
	POP_SCANL();
      }
      
      s = CDR(rootx) = CLOS_CODE(CAR(s)); /* discard the old list */
      if (TYP(s) != CELL_P)
 	SETERR(s, _("bad LAMBDA body")); 
      goto seq_loop;		/* a sequence */

    case h_FSET:		/* builtin fset function */
fset_invoke:
      check_fset_args(FSET_ARGS(CAR(s)), CDR(s));
      if (OBL_DATA(FSET_CODE(CAR(s))))
	RETURN((Atom)OBL_CODE(FSET_CODE(CAR(s)))(CDR(s)));
      else
	SETERR(s, _("unset fset function"));
    }
  }
  SETERR(CAR(s), _("wrong typ to apply"));
}




