/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif

#include "proto.h"
#include "hc.h"
#include "dbfunc.h"

Database database;
int func_status = func_NONE;
int forceindex = false;

/* ----------------------------------------------------------------------
   DATENBASE CONNECTION FUNCTIONS
   ---------------------------------------------------------------------- */
void
initdatabase ()
{
  initdbvar (&database);
}

Atom
set_database_data (Database *db, int mode)
{
  Vector *v;
  int i;
  Atom p;
  
  SET_VAR_DATA (lookup (h_UPDATEOBJ), NIL);
  
  p = lookup (AHASH(h_DBNAME));
  if (!PTR(p))
    SETERR (AHASH(h_DBNAME), LESYMDEF);
  if (mode)
    SET_VAR_DATA (p, NEWSTR (db->basename));
  else
    SET_VAR_DATA (p, NIL);
  
  p = lookup (AHASH(h_DBPATH));
  if (!PTR(p))
    SETERR(AHASH(h_DBPATH), LESYMDEF);
  if (mode)
    SET_VAR_DATA (p, NEWSTR (db->path));
  else
    SET_VAR_DATA (p, NIL);

  if (mode) {
    char *iname;
    int ifno = get_indexfileno (db);
    
    v = make_vector (ifno, ifno, FALSE);
    for (i = 0; i < ifno; i++) {
      iname = get_index_name (db, i);
      if (iname)
	vector_set (v, NEWSTR (iname), get_index_id(db, i));
    }
  }
  else {
    v = make_vector (0, 0, FALSE);
  }
  
  p = lookup (AHASH(h_INDICES));
  if (!PTR(p))
    SETERR (AHASH(h_INDICES), LESYMDEF);
  if (mode)
    SET_VAR_DATA (p, NEWVECTOR(v));
  else
    SET_VAR_DATA (p, NEWVECTOR(v));
  
  if (mode) {
    char *dname;
    int dfno = get_datafileno (db);

    v = make_vector (dfno, dfno, FALSE);
    for (i = 0; i < dfno; i++) {
      dname = get_datafile_name (db, i);
      if (dname)
	vector_set (v, NEWSTR(dname), get_datafile_id (db, i));
    }
  }
  else {
    v = make_vector (0, 0, FALSE);
  }
  
  p = lookup (AHASH(h_DATAFILES));
  if (!PTR(p))
    SETERR (AHASH(h_DATAFILES), LESYMDEF);
  if (mode)
    SET_VAR_DATA (p, NEWVECTOR(v));
  else
    SET_VAR_DATA (p, NEWVECTOR(v));
  return p;
}


/* ----------------------------------------------------------------------
   Converter functions
   obj_to_buffer    converts a object to a buffer (for saving)
   buf_to_obj       converts a buffer to a object (after loading)
   ---------------------------------------------------------------------- */
int convbufsize;
char *convbuf;
char *cbp;


int cv_bufferinitsize = 256;
int cv_bufferpagesize = 256;

void
cv_initbuf ()
{
  convbufsize = cv_bufferinitsize;
  convbuf = (char*) smalloc (convbufsize);
  cbp = convbuf;
}

int
cv_spaceavail (int len)
{
  int oldpos;
  
  if ((cbp - convbuf) + len < convbufsize) {
    return 1;
  }
  else {
    oldpos = cbp - convbuf;	/* compute old offset */

    if (len > cv_bufferpagesize)
      convbufsize += cv_bufferpagesize + len;
    else
      convbufsize += cv_bufferpagesize; /* add bytes */
    convbuf = (char *) realloc (convbuf, convbufsize); /* realloc new space */
    
    if (!convbuf)
      SETERR (0, _("out of memory"));
    cbp = convbuf + oldpos;	/* recompute old ptr */
    
    return 1;
  }
  return 0;
}

char *
cv_addchar (char data)
{
  cv_spaceavail (1);
  *cbp = data;
  cbp++;
  return cbp;
}

/* cv_addlng implementiert ein mehrstufiges Lngensystem.  
   
   0 < x < 254         253 B    1byte 0..253
   254 <= x <= 65535   65 KB    1byte =254, 2byte = 254..65535
   65535 < x ...       4 GB     1byte =255, 4byte = 0..4294967295

   */
char *
cv_addlng (int lng)
{
  if (lng >= WORDLNG) {
    if (lng >= LONGLNG) {	/* 65536..4294967295 */
      int l = lng;
      cv_spaceavail (5);
      *cbp = LONGFLAG;
      cbp++;
      memcpy (cbp, &l, 4);
      cbp += 4;
    }
    else {			/* 254..65535 */
      unsigned short l = (unsigned short)lng;
      cv_spaceavail (3);
      *cbp = WORDFLAG; 
      cbp++;
      memcpy (cbp, &l, 2);
      cbp += 2;
    }
  }
  else {			/* 0..253 */
    cv_spaceavail (1);
    *cbp = (unsigned char) lng;
    cbp ++;
  }
  return cbp;
}

char *
cv_addstring (char *string)
{
  int l = strlen (string);
  
  cv_addlng (l);

  cv_spaceavail (l);
  strcpy(cbp, string);
  cbp += l;
  return cbp;
}

char *
cv_addlong (long data)
{
  cv_spaceavail (4);
  memcpy (cbp, &data, 4);
  cbp += 4;
  return cbp;
}

char *
cv_addbuffer (char *data, int lng)
{
  cv_addlng (lng);
  
  cv_spaceavail (lng);
  memcpy(cbp, data, lng);
  cbp += lng;
  return cbp;
}

void
cv_setptr (int ofs, short addr)
{
  memcpy (&convbuf[ofs], &addr, 2);
}

/* converts an object from k4 internal representation (vector) to the real
   database format.  Informations like UID, GID, DATE, etc. are not
   included in this dataformat, since their representation is covered by
   the storage manager functions. */
char *
convert_object_to_buffer (Vector *objv, int *length)
{
  int sectionptr, i;
  long desc_sec = 0, class_sec = 0, meth_sec = 0, data_sec = 0;
  Vector *descv, *datav;
  Atom p;
  
  cv_initbuf();
  
  if (!objv)
    goto errhd;
  
  cv_addlong (0);		/* ptr for the pointer table */
  
  /* DESC SECTION */
  sectionptr = cbp - convbuf;
  p = vector_ref (objv, DESC_SLOT);
  if (!VECTORQ(p))
    goto errhd;
  descv = OBL_VECTOR (p);
  
  for (i = 0; i < descv->size; i++) {
    if (TYP(descv->array[i]) == HASH_P) {
      cv_addstring (HASH_STR(descv->array[i]));
      cv_addlng (i);
    }
  }
  
  if ((cbp-convbuf) > sectionptr) {
    cv_addchar (0);		/* zerolng of list -> end */
    desc_sec = sectionptr;
  }

  /* CLASS SECTION */
  sectionptr = cbp - convbuf;
  p = vector_ref (objv, CLASS_SLOT);

  if (p != NIL) {
    switch (TYP(p)) {
    case HASH_P:
      cv_addchar (ATTR_LABEL_TYPE);
      cv_addstring (HASH_STR(p));
      break;
    case STR_P:
      cv_addchar (ATTR_STRING_TYPE);
      cv_addstring (STR_STR(p));
      break;
    case OID_P:
      cv_addchar (ATTR_OID_TYPE);
      cv_addlong (OIDV(p));
      break;
    case OBL_P:
      if (OBL_TYP(p) == OID_T) {
	cv_addchar (ATTR_OID_TYPE);
	cv_addlong (OBL_OID(p));
      }
      else
	SETERR (0, _("wrong type in object class slot"));
      break;
    default:
      SETERR (0, _("wrong type in object class slot"));
    }
  }
  if ((cbp-convbuf) > sectionptr) {
    class_sec = sectionptr;
  }
  
  /* METHODS SECTION */
  sectionptr = cbp - convbuf;
  p = vector_ref (objv, METHL_SLOT);

  if (p != NIL) {
    char *buf;
    int blng;
    
    buf = list2code (p, &blng, false);
    cv_addbuffer (buf, blng);
    free (buf);
  }
  if ((cbp-convbuf) > sectionptr) {
    meth_sec = sectionptr;
  }

  /* DATA SECTION */
  sectionptr = cbp - convbuf;
  p = vector_ref (objv, DATA_SLOT);
  if (!VECTORQ(p))
    goto errhd;
  datav = OBL_VECTOR (p);
  
  for (i = 0; i < datav->size; i++) {
    if (datav->array[i] != NIL) {
      cv_addchar (ATTR_START_ATTR);
      cv_addlng (i);
      switch (TYP(datav->array[i])) {
      case HASH_P:
	cv_addchar (ATTR_LABEL_TYPE);
	cv_addstring (STR_STR(datav->array[i]));
	break;
      case STR_P:
	cv_addchar (ATTR_STRING_TYPE);
	cv_addstring (STR_STR(datav->array[i]));
	break;
      case INT_P:
	cv_addchar (ATTR_INT_TYPE);
	cv_addlong (numv(datav->array[i]));
	break;
      case OID_P:
	cv_addchar (ATTR_OID_TYPE);
	cv_addlong (OIDV(datav->array[i]));
      case BOOL_P:
	switch (BOOLV(datav->array[i])) {
	case FALSE:
	  cv_addchar (ATTR_FALSE_TYPE); 
	  break;
	case TRUE:
	  cv_addchar (ATTR_TRUE_TYPE); 
	  break;
	case UNSPECIFIED:
	  cv_addchar (ATTR_UNSPECIFIED_TYPE); 
	  break;
	}
	break;
      case OBL_P:
	switch (OBL_TYP(datav->array[i])) {
	case EXTINT_T:
	case INT_T:
	  cv_addchar (ATTR_INT_TYPE);
	  cv_addlong (numv(datav->array[i]));
	  break;
	case EXTSTR_T:
	  cv_addchar (ATTR_STRING_TYPE);
	  cv_addstring (OBL_STR(datav->array[i]));
	  break;
	case OID_T:
	  cv_addchar (ATTR_OID_TYPE);
	  cv_addlong (OBL_OID(datav->array[i]));
	  break;
	case DFIELD_T:
	  cv_addchar (ATTR_DFIELD_TYPE);
	  cv_addbuffer (OBL_DFIELD(datav->array[i])->data,
			OBL_DFIELD(datav->array[i])->size);
	  break;
	case DATE_T:
	  cv_addchar (ATTR_DATE_TYPE);
	  cv_addlong (OBL_DATE(datav->array[i]));
	  break;
	case VECTOR_T:
	  printf (_("object save: Vectors are not supported yet.\n"));
	  break;
	}
	break;
      case CELL_P:
	{
	  char *buf;
	  int blng;
    
	  buf = list2code (datav->array[i], &blng, false);
	  if (buf) {
	    cv_addchar (ATTR_CODE_TYPE);
	    cv_addbuffer (buf, blng);
	    free (buf);
	  }
	  break;
	}
      }
    }
  }
  cv_addchar (ATTR_LAST_ATTR);	/* last attribut */

  if ((cbp-convbuf) > sectionptr)
    data_sec = sectionptr;
  
  cv_setptr (0, cbp - convbuf);	/* set ptr to the pointer table */

  cv_addlng (desc_sec);		/* add ptr to the desc section */
  cv_addlng (class_sec);	/* add ptr to the class section */
  cv_addlng (meth_sec);		/* add ptr to the method section */
  cv_addlng (data_sec);		/* add ptr to the data section */
  
  *length = cbp-convbuf;
  return convbuf;
  
errhd:
  if (convbuf) {
    free (convbuf);
    convbuf = NULL;
  }
  *length = 0;
  return NULL;
}

/* converts a object from real database format to k4 internal
   representation (vector).  len must specify the byte length of the buffer
   buf.  The function returns the object as k4 internal Atom type. */
Atom
convert_buffer_to_object (Oid oid, char *buf, int len, Objstat *objstat)
{
  int ptr_tbl, desc_sec, class_sec, meth_sec, data_sec;
  char *bp;
  Vector *objv = NULL;
  Atom rx, retval, rootx;
  int slng;

#define STORE_SCAN_DATA_SEC(DATAV, RX, ROOTX, AID)		\
({								\
  Atom _rootx = (ROOTX), _rx = (RX);				\
  Vector *_datav = (DATAV);					\
  int _aid = (AID);						\
  if (CDR(_rootx) != NIL) {					\
    if (CDDR(_rootx) != NIL) {	/* one or more components? */	\
      vector_set (_datav, CDR(_rootx), _aid);			\
    }								\
    else {							\
      vector_set (_datav, CADR(_rootx), _aid);			\
    }								\
    CDR(_rootx) = NIL; 					        \
    _rx = _rootx;						\
  }								\
  _rx;								\
})
#define read_OK     0
#define read_ATTR   1
#define read_AID    2
#define read_TYP    3
#define read_CODE   4


  objv = empty_object (NIL, NIL);

  PUSH_SCANL();
  rx = rootx = SAVE_ROOT();
  CAR(rootx) = NEWVECTOR(objv);	/* save object from deletion */
  
  /* first load the pointertable to the four sections */    
  memcpy (&ptr_tbl, buf, 4);	/* get the ofs of the ptr section */
  bp = buf + ptr_tbl;
  /* now read the pointer */
  bp = rb_getln (bp, &desc_sec);/* ptr to the desc_sec */
  bp = rb_getln (bp, &class_sec); /* ptr to the class_sec */
  bp = rb_getln (bp, &meth_sec); /* ptr to the meth_sec */
  bp = rb_getln (bp, &data_sec); /* ptr to the data_sec */

  /* scan the desc section and put all the values as symbols to the
     description table */
  if (desc_sec) {
    char *val_str;
    int aid;
    Vector *descv = OBL_VECTOR(vector_ref (objv, DESC_SLOT));
    
    bp = buf + desc_sec;

    while (*bp) {
      bp = rb_getln (bp, &slng); /* get string length */
      val_str = (char *) smalloc (slng+1);
      memcpy (val_str, bp, slng);	/* copy string! */
      *(val_str + slng) = '\0';
      bp += slng;
      
      bp = rb_getln (bp, &aid);	/* get the aid */
      vector_set (descv, NEWHASH(val_str), aid);
      
      free (val_str);
    }
  }
  
  /* Scan the class entry and put the value to the class slot */
  if (class_sec) {
    char *val_str;
    Oid val_oid;
    
    bp = buf + class_sec;
    
    switch (*bp) {
    case ATTR_LABEL_TYPE:
      bp++;
      bp = rb_getln (bp, &slng); /* get string length */
      val_str = (char *) smalloc (slng + 1);
      memcpy (val_str, bp, slng);	/* copy string! */
      *(val_str + slng) = '\0';
      bp += slng;
      vector_set (objv, NEWHASH(val_str), CLASS_SLOT);
      free (val_str);
      break;
    case ATTR_STRING_TYPE:
      bp++;
      bp = rb_getln (bp, &slng); /* get string length */
      val_str = (char *) smalloc (slng + 1);
      memcpy (val_str, bp, slng);	/* copy string! */
      *(val_str + slng) = '\0';
      bp += slng;
      vector_set (objv, PTRSTR(val_str), CLASS_SLOT);
      break;
    case ATTR_OID_TYPE:
      bp++;
      memcpy (&val_oid, bp, sizeof (Oid));
      bp += sizeof (Oid);
      vector_set (objv, NEWOID(val_oid), CLASS_SLOT);
      break;
    default:
      break;
    }
  }
  else
    vector_set (objv, NEWOID(0), CLASS_SLOT);
  
  /* Scan the method section */
  if (meth_sec) {
    bp = buf + meth_sec;
    bp = rb_getln (bp, &slng);	/* get section length */

    if (slng > 0) {
      Atom p;
      char *prog = (char *) smalloc (slng);

      memcpy (prog, bp, slng);
      bp += slng;
      
      p = code2list (prog);
      vector_set (objv, p, METHL_SLOT);
      free (prog);
    }
  }

  /* Scan the data section and build a correct (of course) lisp-alist */
  if (data_sec) {
    int aid;			/* a attr ID */
    int val_int;
    long val_long;
    char *val_str;
    char *val_buf;
    Vector *datav = NULL;
    Dfield *dfield;

    bp = buf + data_sec;

    datav = OBL_VECTOR(vector_ref (objv, DATA_SLOT));
    CDR(rootx) = NIL;
  loop:
    switch (*bp) {
    case ATTR_START_ATTR:
      rx = STORE_SCAN_DATA_SEC(datav, rx, rootx, aid);
      bp++;
      bp = rb_getln (bp, &aid);	/* the aid */
      goto loop;
    case ATTR_LAST_ATTR:
      rx = STORE_SCAN_DATA_SEC (datav, rx, rootx, aid);
      bp++;
      goto done;

    case ATTR_TRUE_TYPE:
      bp++;
      rx = CDR(rx) = CONS (TRUE, NIL);
      goto loop;
    case ATTR_FALSE_TYPE:
      bp++;
      rx = CDR(rx) = CONS (FALSE, NIL);
      goto loop;
    case ATTR_UNSPECIFIED_TYPE:
      bp++;
      rx = CDR(rx) = CONS (UNSPECIFIED, NIL);
      goto loop;

    case ATTR_STRING_TYPE:	/* a string (dyn length) */
      bp++;
      bp = rb_getln (bp, &slng); /* get the length of the string */
      val_str = (char*) smalloc (slng+1); /* alloc memory */
      memcpy (val_str, bp, slng); /* copy the string */
      *(val_str + slng) = '\0'; /* terminate string */
      bp += slng;
      
      rx = CDR(rx) = CONS (PTRSTR(val_str), NIL);
      goto loop;

    case ATTR_LABEL_TYPE:	/* a label (dyn length) */
      bp++;
      bp = rb_getln (bp, &slng); /* get the length of the string */
      val_str = (char*) smalloc (slng+1); /* alloc memory */
      memcpy (val_str, bp, slng); /* copy the string */
      *(val_str + slng) = '\0'; /* terminate string */
      bp += slng;
      
      rx = CDR(rx) = CONS (NEWHASH(val_str), NIL);
      free (val_str);
      goto loop;

    case ATTR_INT_TYPE:		/* a 4byte integer */
      bp++;
      memcpy (&val_int, bp, 4);
      bp += 4;

      rx = CDR(rx) = CONS (NEWINT(val_int), NIL);
      goto loop;

    case ATTR_DATE_TYPE:	/* a date type (4byte) */
      bp++;
      memcpy (&val_int, bp, 4);
      bp += 4;
      
      rx = CDR(rx) = CONS (NEWDATE(val_int), NIL);
      goto loop;

    case ATTR_OID_TYPE:	/* a oid type (4byte) */
      bp++;
      memcpy (&val_long, bp, 4);
      bp += 4;
      
      rx = CDR(rx) = CONS (NEWOID(val_long), NIL);
      goto loop;
      
    case ATTR_DFIELD_TYPE:	/* a data field type (dyn length) */
      bp++;
      bp = rb_getln (bp, &slng); /* get the length of the data */
      val_buf = (char*) smalloc (slng); /* alloc memory */
      memcpy (val_buf, bp, slng); /* copy the data */
      bp += slng;

      dfield = make_dfield (val_buf, slng); 

      rx = CDR(rx) = CONS(NEWDFIELD(dfield), NIL);
      goto loop;

    case ATTR_CODE_TYPE:	/* inactiv lisp code */
      bp++;
      bp = rb_getln (bp, &slng); /* get code length */    
      
      if (slng > 0) {
	Atom p;
	val_buf = (char *) smalloc (slng); /* alloc memory */
	memcpy (val_buf, bp, slng);	/* copy the binary data */
	bp += slng;
	
	p = code2list (val_buf); /* de-compile & install the data */
	free (val_buf);

	rx = CDR(rx) = CONS (p, NIL);
      }
      goto loop;

    case ATTR_VECTOR_TYPE:
      bp++;
      printf (_("vectors can't be loaded from database yet.\n"));
      goto loop;
    }
  }

 done:
  vector_set (objv, NEWOID(oid), OID_SLOT);
  /*   vector_set (objv, NEWDATE(objstat->creat), CREAT_SLOT); */
  /*   vector_set (objv, NEWINT(objstat->uid), UID_SLOT); */
  /*   vector_set (objv, NEWINT(objstat->gid), GID_SLOT); */
  
  
  retval = CAR(rootx);
  POP_SCANL();
  return retval;
}


/* ----------------------------------------------------------------------
   Grundlegende Zugriffsroutinen auf Datenstze.  Automatisierte
   Grundfunktionen (insert-object, ...) werden in k4 selbst implementiert.
   ---------------------------------------------------------------------- */
void
merge_iid (Atom list, int keylist[], int *keylistsize)
{
  Atom s = list;
  int i;

  while (TYP(s) == CELL_P) {
    if ((TYP(CAR(s)) == CELL_P) 
	&& (TYP(CAAR(s)) == STR_P) 
	&& (number(CDAR(s)) != FALSE)) {
      int addkey = numv (CDAR(s));
      
      for (i = 0; i < *keylistsize; i++) {
        if (keylist[i] == addkey)
          goto addkeyfound;
      }
      keylist[*keylistsize] = addkey;
      (*keylistsize)++;
    }
  addkeyfound:
    s = CDR(s);
  }
}

/* ----------------------------------------------------------------------
   Index updating routines
   ---------------------------------------------------------------------- */
int
insert_keylist (Atom newkeys, Oid oid)
{
  int inserted = 0;
  Atom s = newkeys;
  
  while (TYP(s) == CELL_P) {
    if ((TYP(CAR(s)) == CELL_P)
	&& (TYP(CAAR(s)) == STR_P)
	&& (number(TYP(CDAR(s))) != FALSE)) {
      char *addkey = STR_STR(CAAR(s));
      int addiid = numv(CDAR(s));
      
      if (update_index (&database, addiid, addkey, oid) < 0)
	return -1;
      else
	inserted ++;
    }
    s = CDR(s);
  }
  return inserted;
}

int
delete_keylist (Atom oldkeys, Oid oid)
{
  int deleted = 0;
  Atom s = oldkeys;
  
  while (TYP(s) == CELL_P) {
    if ((TYP(CAR(s)) == CELL_P)
	&& (TYP(CAAR(s)) == STR_P)
	&& (number(TYP(CDAR(s))) != FALSE)) {
      char *delkey = STR_STR(CAAR(s));
      int deliid = numv(CDAR(s));
      
      if (delete_index (&database, deliid, delkey, oid) < 0)
	return -1;
      else
	deleted ++;
    }
    s = CDR(s);
  }
  return deleted;
}

/* looks up a index id for a given index label from the system var
   `indices'.  Indices are system wide and not bound to a specifiv
   object. */
int
look_up_index (char *indxname)
{
  Atom p;

  p = lookup (h_INDICES);
  if (!PTR(p))
    SETERR (AHASH(h_INDICES), LESYMDEF);
  if (!VECTORQ(VAR_DATA(p)))
    SETERR (p, _("bad vector type in system variable db-indices"));
  
  return vector_lookup (OBL_VECTOR(VAR_DATA(p)), NEWSTR(indxname));
}

Oid
get_oid_by_index (char *key, int indxno)
{
  Searchinfo *searchinfo;
  int lockid, retval;
  
  if (database.status) {	/* datenbank opened? */
    
    lockid = begin_read_index (indxno, database.dbid);
    
    if (lockid > 0) {
      retval = search_index (&database, indxno, key, &searchinfo);
      end_read_index (lockid);
      if (retval < 0)
	return UNKNOWNOID;
      
      if ((searchinfo) 
	  && (searchinfo->size == 1))
	return searchinfo->data.oid;
    }
  }
  return UNKNOWNOID;		/* nothing found */
}

/* ----------------------------------------------------------------------
   READ OBJECT
   ---------------------------------------------------------------------- */
/* liest einen Datensatz oid aus der Datenbank database (globale Variable).
   Funktioniert alles wunderbar, so liefert read_object ein neues Atom
   zurck.  Geht irgendwas schief liefert die Funktion FALSE */
Atom
read_object (Oid oid)
{
  char *buf = NULL;
  int readsize, retval;
  Objstat objstat;
  int lockid;
  
  /* get the access rights */
  lockid = begin_read (oid, database.dbid); /* lock your resources! */
  if (lockid <= 0)
    return FALSE;
  
  /* At first read the CtrlData of the item an verify that we have the
     permission to read the item! */
  if (readItemCtrlData (&database, oid, &objstat) < 0) {
    end_read (lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      func_READ);
  if (retval == 0) {
    end_read (lockid);
    DESCERR (0, _("no permissions to read object"));
    return FALSE; 
  }
  else if (retval < 0) {
    end_read (lockid);
    return FALSE;
  }
  
  readsize = readItem (&database, /* database */
		       oid, /* object id */
		       &buf,	/* the buffer */
		       &objstat); /* the itemstatus */
  retval = end_read (lockid);	/* free your resources */
  
  if (readsize < 0) {
    JMPERR();
  }
  else if (readsize > 0) {
    if (buf) {
      Atom newobj;
      newobj = convert_buffer_to_object (oid, buf, readsize, &objstat);
      free (buf); 
      return newobj;
    }
  }
  return FALSE;
}


/* ----------------------------------------------------------------------
   INSERT (NEW) OBJECT
   ---------------------------------------------------------------------- */
/* dies ist das lowlevel pendant zur evaluierung auf Hochebene.  db-insert
   takes 3 arguments: the object (a vector), the file id (number) and a
   alist of all keywords (list, form: ((word iid) (word iid) (word iid)) */
Atom
insert_object (Atom target_obj, int target_fid, Atom keylist)
{
  char *buf;
  int buflen;
  
  /* convert the lisp-oriented data to buffer-format. */  
  buf = convert_object_to_buffer (OBL_VECTOR(target_obj), &buflen); 
  
  if (buf) {
    int retval;
    Oid newoid;
    int indxlistsize = 0;
    int indxlist[MAXINDEX];
    int inserted;
    int lockid = 0;
    
    /* merge the keywordlist to the indexlist */
    merge_iid (keylist, indxlist, &indxlistsize);
    
    /* no get the resources */
    lockid = begin_insert (0, target_fid, indxlist, indxlistsize,
			   database.dbid);
    if (lockid <= 0) {
      free (buf);
      return FALSE;
    }

    /* Verify that we have the permission to insert an Item! */
    retval = verify_permission (clientuid, database.asowner,
				0, /* itemuid */
				0, /* itemgid */
				0, /* accesscode */
				func_INSERT);
    if (retval == 0) {
      end_insert (lockid);
      free (buf);
      DESCERR (0, _("no permissions to insert object"));
      return FALSE;
    }
    else if (retval < 0) {
      end_insert (lockid);
      free (buf);
      return FALSE;
    }
    
    newoid = newItem (&database, /* database */
		      target_fid, /* actuell file */
		      clientuid, /* default uid */
		      clientgid, /* default gid */
		      accmask,	/* default accesscode */
		      buf,	/* buffer */
		      buflen);	/* len of buffer */
    
    if (newoid == 0) {
      retval = end_insert (lockid); /* free the resources */      
      goto errhd;
    }
    
    /* now insert the index */
    if ((inserted = insert_keylist (keylist, newoid)) < 0) {
      retval = end_insert (lockid);
      goto errhd;
    }
    
    retval = end_insert (lockid); /* free the resources */
    if (retval < 0)
      goto errhd;

    free (buf);
    return NEWINT (newoid);
  }
  
  return FALSE;

errhd:
  if (buf)
    free (buf);
  
  JMPERR();
}


/* ----------------------------------------------------------------------
   UPDATE OBJECT
   ---------------------------------------------------------------------- */
struct {
  Atom var;
  Fid fid;
  Oid oid;
  int lockid;
} updobj_data;

/* begin_update_object starts the update of a served data object.  It just
   gets the permissions and loads the object into the system wide variable
   *db:update-obj*.  If succesful it returns the loaded object (in true a
   pointer to the data of *db:update-obj*), otherwise FALSE.  Please note,
   that begin_update_object don't take any action in generating an index
   keylist!  begin_update_object sets the data in updobj_data, and sets the
   global (c-internal) variable func_status to func_UPDATE. */
Atom
begin_update_object (Oid target_oid)
{
  char *buf;
  int readsize, retval;
  Objstat objstat;
  Atom updateobj;

  if (func_status != func_NONE)
    SETERR (0, _("multi function access not supported"));
  
  /* get the access rights */
  updobj_data.lockid = begin_update (target_oid, database.dbid);
  if (updobj_data.lockid <= 0)
    return FALSE;

  /* At first read the CtrlData of the item an verify that we have the
     permission to read the item! */
  if (readItemCtrlData (&database, target_oid, &objstat) < 0) {
    end_update (updobj_data.lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      func_UPDATE);
  if (retval == 0) {
    end_update (updobj_data.lockid);
    DESCERR (0, _("Access denied to update"));
    return FALSE;
  }
  else if (retval < 0) {
    end_update (updobj_data.lockid);
    return FALSE;
  }
  
  readsize = readItem (&database, /* database */
		       target_oid, /* object id */
		       &buf,	/* the buffer */
		       &objstat); /* the object statistics */
  
  if (readsize < 0) {
    retval = end_update (updobj_data.lockid); /* free your resources */
    JMPERR();
  }
  else if (readsize > 0) {
    if (buf) {
      /* get the backstore object 'update-obj' to save the data to */
      updateobj = lookup (h_UPDATEOBJ);
      SET_VAR_DATA (updateobj,
		    convert_buffer_to_object (target_oid, buf,
					      readsize, &objstat));

      free (buf);
      if (VAR_DATA(updateobj) == FALSE)
	return FALSE;

      /* now save the secure object data: oid and fid.  Everything else is
         less important (the keylist is a special problem, I can't solve in
         the moment.  */
      
      updobj_data.var = updateobj;
      updobj_data.fid = objstat.fid;
      updobj_data.oid = target_oid;
      
      func_status = func_UPDATE;
      return VAR_DATA(updateobj);
    }
  }
  
  return FALSE;
}

#define CMPKEYS(str1, str2, iid1, iid2)		\
({						\
  int _retval = false;				\
  if ((iid1 == iid2) 				\
      && (strcmp (str1, str2) == 0))		\
    _retval = true;				\
  _retval;					\
})

Atom
update_object (Atom old_keylist, Atom new_keylist)
{
  Atom updateobj;
  char *buf;
  int len;
  int indxlistsize = 0;
  int indxlist[MAXINDEX];

  if (func_status != func_UPDATE)
    SETERR (0, _("Not in update mode"));
  
  updateobj = lookup (h_UPDATEOBJ);
  if (OBJECTQ(VAR_DATA (updateobj)) == FALSE)
    SETERR (VAR_DATA(updateobj), _("bad object"));

  /* convert the k4-oriented data to buffer-format. */  
  buf = convert_object_to_buffer (OBL_VECTOR(VAR_DATA(updateobj)), &len);
  
  if (buf) {
    int retval, wriretval;
    int deleted, inserted;
    
    /* get the keylist of the object */
    indxlistsize = 0;
    merge_iid (old_keylist, indxlist, &indxlistsize);
    /* now merge the iids of the newkeys-list to the keylist-array.  This
       are the indices which must be locked. */
    merge_iid (new_keylist, indxlist, &indxlistsize);
    
    /* now go through the old- and new-keylist and kill every key (in both
       lists) that is identical in both lists.  The remaining keys in the
       old-keylist have to be removed from the indices, the remaining keys
       in the new-keylist have to be inserted (with the actual oid as
       object id) */

    if (!forceindex) {
      Atom s = old_keylist;

      while (TYP(s) == CELL_P) {
	if ((TYP(CAR(s)) == CELL_P)
	    && (TYP(CAAR(s)) == STR_P)
	    && (number(TYP(CDAR(s))) != FALSE)) {
	  char *delkey = STR_STR(CAAR(s));
	  int deliid = numv(CDAR(s));
	  Atom t = new_keylist;
	  int deletd = false;
	  
	  while (TYP(t) == CELL_P) {
	    if ((TYP(CAR(t)) == CELL_P)
		&& (TYP(CAAR(t)) == STR_P)
		&& (number(CDAR(t)) != FALSE)) {
	      char *addkey = STR_STR(CAAR(t));
	      int addiid = numv(CDAR(t));
	      
	      if (CMPKEYS (delkey, addkey, deliid, addiid)) { 
		CAR(t) = FALSE;	/* kill this one from the newkeylist */
		deletd = true; 
	      } 
	    }
	    t = CDR(t);
	  }
	  if (deletd) {
	    CAR(s) = FALSE;	/* kill this one from the oldkeylist */
	  }
	}
	s = CDR(s);
      }
    }

    /* now get the resources */
    retval = lock_update (updobj_data.oid, updobj_data.fid,
			  indxlist, indxlistsize,
			  database.dbid, updobj_data.lockid);
    if (retval <= 0) {
      end_update (updobj_data.lockid); /* restore locks from begin_update! */
      free (buf);
      SET_VAR_DATA (updateobj, FALSE); /* kill the update data */
      func_status = func_NONE;
      return FALSE;
    }
    
    wriretval = rewriteItem (&database,	/* database */
			     updobj_data.oid, /* the actual oid */
			     buf,       /* buffer */
			     len);	/* len of buffer */
    
    if (wriretval < 0) {
      retval = end_update (updobj_data.lockid); /* free the resources */      
      goto errhd;
    }
    
    deleted = delete_keylist (old_keylist, updobj_data.oid);
    if (deleted < 0) {
      retval = end_update (updobj_data.lockid);
      goto errhd;
    }
    inserted = insert_keylist (new_keylist, updobj_data.oid);
    if (inserted < 0) {
      retval = end_update (updobj_data.lockid);
      goto errhd;
    }

    /* So, und jetzt mte wirklich alles getan sein. */
    
    retval = end_update (updobj_data.lockid); /* free the resources */      
    if (retval < 0)
      goto errhd;
  }
  
  if (buf)
    free (buf);

  SET_VAR_DATA (updateobj, FALSE);
  func_status = func_NONE;
  return NEWINT (len);
  
errhd:
  if (buf)
    free (buf);
  
  SET_VAR_DATA (updateobj, FALSE);
  func_status = func_NONE;
  JMPERR();
}

#define esc_update_object()						\
({									\
  int _retval = UNSPECIFIED, _retv;					\
  Atom _updateobj;							\
									\
  if (func_status != func_UPDATE)					\
    SETERR (0, _("not in update mode"));				\
  									\
  _updateobj = lookup (h_UPDATEOBJ);					\
  SET_VAR_DATA (_updateobj, FALSE); /* free the loaded update data */	\
  									\
  _retv = end_update (updobj_data.lockid);				\
  									\
  func_status = func_NONE;						\
  if (_retv <= 0)							\
    _retval = FALSE;							\
  _retval;								\
})


/* ----------------------------------------------------------------------
   DELETE / UNDELETE / KILL OBJECT

   Anders als ursprnglich im SM vorgesehen, mu in diesen high-level
   routinen ein Datensatz dummerweise erst gelesen werden, bevor er
   gelscht werden kann -- denn wir mssen ja schlielich auch die
   Indexeintrge eliminieren.  Das gilt fr alle drei Funktionen: Delete
   lscht alle Indexeintrge; Undelete fgt die Indexeintrge allesamt
   wieder ein! Kill lscht sie ebenfalls wieder. ACHTUNG: Wenn ein
   Datensatz mit delete `gelscht' wird, kann er ber keinen einzigen
   Indexeintrag mehr auf ihn zugegriffen werden, OID notieren!

   ---------------------------------------------------------------------- */
/* ----------------------------------------------------------------------
   DELETE
   ---------------------------------------------------------------------- */
int delete_lockid;
Oid delete_oid;
Atom
begin_delete_object (Oid target_oid)
{
  char *buf;
  int readsize, retval;
  Objstat objstat;

  if (func_status != func_NONE)
    SETERR (0, _("multi function access not supported"));

  /* get the access rights */
  delete_lockid = begin_delete (target_oid, database.dbid);
  if (delete_lockid <= 0)
    return FALSE;
  
  /* At first read the CtrlData of the item an verify that we have the
     permission to delete the item! */
  if (readItemCtrlData (&database, target_oid, &objstat) < 0) {
    end_delete (delete_lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      func_DELETE);
  if (retval == 0) {
    end_delete (delete_lockid);
    DESCERR(0, _("access denied for delete"));
    return FALSE;
  }
  else if (retval < 0) {
    end_delete (delete_lockid);
    return FALSE;
  }

  /* erst den Datensatz lesen, um die index keys zu bekommen */
  readsize = readItem (&database, /* database */
		       target_oid, /* object id */
		       &buf,	/* the buffer */
		       &objstat); /* the object statistics */
  if (readsize < 0) {
    retval = end_delete (delete_lockid); /* free your resources */
    JMPERR();
  }
  else if (readsize > 0) {
    if (buf) {
      Atom readobj = convert_buffer_to_object (target_oid, buf, readsize, 
					       &objstat);
      free (buf);
      func_status = func_DELETE;
      delete_oid = target_oid;
      return readobj;
    }
  }
  return FALSE;
}

Atom
delete_object (Atom keylist)
{
  int indxlist[MAXINDEX];
  int indxlistsize; 
  int retval, delretval, deleted;

  if (func_status != func_DELETE)
    SETERR (0, _("not in delete mode"));

  func_status = func_NONE;
      
  /* merge the iids of the keylist to the keylist-array.  This are the
     indices which must be locked. */
  indxlistsize = 0;
  merge_iid (keylist, indxlist, &indxlistsize);
  
  /* now get the resources */
  retval = lock_delete (indxlist, indxlistsize, database.dbid, delete_lockid);
  if (retval <= 0) {
    return FALSE;
  }
  
  delretval = delItem (&database, delete_oid); /* the actual oid */
  
  if (delretval < 0) {
    retval = end_delete (delete_lockid); /* free the resources */      
    JMPERR();
  }

  deleted = delete_keylist (keylist, delete_oid);
  if (deleted < 0) {
    retval = end_delete (delete_lockid);
    JMPERR();
  }
  
  /* So, und jetzt mte wirklich alles getan sein.  Hoffe ich
     jedenfalls. */
  retval = end_delete (delete_lockid); /* free the resources */      
  if (retval < 0)
    JMPERR();
  
  return TRUE;
}

#define esc_delete_object()			\
({						\
  int _retval = UNSPECIFIED, _retv;		\
  						\
  if (func_status != func_DELETE)		\
    SETERR (0, _("not in delete mode"));	\
  						\
  _retv = end_delete (delete_lockid);		\
  						\
  func_status = func_NONE;			\
  if (_retv <= 0)				\
    _retval = FALSE;				\
  _retval;					\
})

/* ----------------------------------------------------------------------
   UNDELETE
   ---------------------------------------------------------------------- */
int undelete_lockid;
Oid undelete_oid;
Atom
begin_undelete_object (Oid target_oid)
{
  char *buf;
  int readsize, retval;
  Objstat objstat;

  if (func_status != func_NONE)
    SETERR (0, _("multi function access not supported"));

  /* get the access rights */
  undelete_lockid = begin_undelete (target_oid, database.dbid);
  if (undelete_lockid <= 0)
    return FALSE;

  /* At first read the CtrlData of the item an verify that we have the
     permission to delete the item! */
  if (readItemCtrlData (&database, target_oid, &objstat) < 0) {
    end_undelete (undelete_lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      func_UNDELETE);
  if (retval == 0) {
    end_undelete (undelete_lockid);
    DESCERR(0, _("access denied for undelete"));
    return FALSE;
  }
  else if (retval < 0) {
    end_undelete (undelete_lockid);
    return FALSE;
  }
  
  /* now undelete the object */
  retval = undelItem (&database, target_oid); /* the oid */
  if (retval < 0) {
    retval = end_undelete (undelete_lockid); /* free the resources */      
    JMPERR ();
  }
  
  /* and now read the object, cause we have to generate the index list */
  readsize = readItem (&database, /* database */
		       target_oid, /* object id */
		       &buf,	/* the buffer */
		       &objstat); /* the object statistics */
  
  if (readsize < 0) {
    retval = end_undelete (undelete_lockid); /* free your resources */
    JMPERR();
  }
  else if (readsize > 0) {
    if (buf != NULL) {
      Atom readobj = convert_buffer_to_object (target_oid, buf,
					       readsize, &objstat);
      free (buf);
      func_status = func_UNDELETE;
      undelete_oid = target_oid;
      return readobj;
    }
  }
  return FALSE;
}

Atom
undelete_object (Atom keylist)
{
  int indxlist[MAXINDEX];
  int indxlistsize = 0;  
  int retval, inserted;
  
  if (func_status != func_UNDELETE)
    SETERR (0, _("not in undelete mode"));
  
  func_status = func_NONE;
  
  /* now merge the iids of the keylist to the keylist-array.  This are
     the indices which must be locked. */
  merge_iid (keylist, indxlist, &indxlistsize);
    
  /* now get the resources */
  retval = lock_undelete (indxlist, indxlistsize, database.dbid, 
			  undelete_lockid);
  if (retval <= 0) {
    return FALSE;
  }
  
  inserted = insert_keylist (keylist, undelete_oid);
  if (inserted < 0) {
    retval = end_undelete (undelete_lockid);
    JMPERR();
  }
    
  /* So, und jetzt mte wirklich alles getan sein.  Hoffe ich
     jedenfalls. */
  retval = end_undelete (undelete_lockid); /* free the resources */      
  if (retval < 0)
    JMPERR();

  return TRUE;
}

#define esc_undelete_object()			\
({						\
  int _retval = UNSPECIFIED, _retv;		\
  						\
  if (func_status != func_UNDELETE)		\
    SETERR (0, _("not in undelete mode"));	\
  						\
  _retv = end_undelete (undelete_lockid);	\
  						\
  func_status = func_NONE;			\
  if (_retv <= 0)				\
    _retval = FALSE;				\
  _retval;					\
})


/* ----------------------------------------------------------------------
   KILL
   ---------------------------------------------------------------------- */
int kill_lockid;
Oid kill_oid;
Fid kill_fid;
Atom
begin_kill_object (Oid target_oid)
{
  char *buf;
  int readsize, retval;
  Objstat objstat;
  
  if (func_status != func_NONE)
    SETERR (0, _("multi function access not supported"));

  /* get the access rights */
  kill_lockid = begin_kill (target_oid, database.dbid);
  if (kill_lockid <= 0) {
    DESCERR (0, _("got no kill lock id"));
    return FALSE;
  }

  /* At first read the CtrlData of the item an verify that we have the
     permission to delete the item! */
  if (readItemCtrlData (&database, target_oid, &objstat) < 0) {
    end_kill (kill_lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      func_KILL);
  if (retval == 0) {
    end_kill (kill_lockid);
    DESCERR(0, _("access denied for kill object"));
    return FALSE;
  }
  else if (retval < 0) {
    end_kill (kill_lockid);
    DESCERR(0, _("error in get permission process"));
    return FALSE;
  }
  
  /* erst den Datensatz lesen, um die index keys zu bekommen */
  readsize = readItem (&database, /* database */
		       target_oid, /* object id */
		       &buf,	/* the buffer */
		       &objstat); /* the object statistics */
  
  if (readsize < 0) {
    retval = end_kill (kill_lockid); /* free your resources */
    JMPERR();
  }
  else if (readsize > 0) {
    if (buf) {
      Atom readobj = convert_buffer_to_object (target_oid, buf, 
					       readsize, &objstat);
      free (buf);
      func_status = func_KILL;
      kill_oid = target_oid;
      kill_fid = objstat.fid;
      return readobj;
    }
  }
  return FALSE;
}

Atom
kill_object (Atom keylist)
{
  int indxlist[MAXINDEX];
  int indxlistsize = 0;  
  int retval, delretval, deleted;
    
  if (func_status != func_KILL)
    SETERR(0, _("not in kill mode"));

  /* now merge the iids of the keylist to the keylist-array.  This are
     the indices which must be locked. */
  merge_iid (keylist, indxlist, &indxlistsize);

  /* now get the resources */
  retval = lock_kill (kill_fid, indxlist, indxlistsize, database.dbid,
		      kill_lockid);
  if (retval <= 0) {
    DESCERR (0, _("some error in lock kill"));
    return FALSE;
  }
  
  delretval = killItem (&database, kill_oid);
    
  if (delretval < 0) {
    retval = end_kill (kill_lockid); /* free the resources */      
    JMPERR ();
  }

  deleted = delete_keylist (keylist, kill_oid);
  if (deleted < 0) {
    retval = end_kill (kill_lockid);
    JMPERR ();
  }
    
  /* So, und jetzt mte wirklich alles getan sein.  Hoffe ich
     jedenfalls. */
  retval = end_kill (kill_lockid); /* free the resources */      
  if (retval < 0) {
    JMPERR ();
  }

  return TRUE;
}

#define esc_kill_object()			\
({						\
  int _retval = UNSPECIFIED, _retv;		\
						\
  if (func_status != func_KILL)			\
    SETERR (0, _("not in kill mode"));		\
						\
  _retv = end_kill (kill_lockid);		\
						\
  func_status = func_NONE;			\
  if (_retv <= 0)				\
    _retval = FALSE;				\
  _retval;					\
})


/* ----------------------------------------------------------------------
   ZUGRIFFSRECHTE
   ---------------------------------------------------------------------- */
Atom 
change_object (Oid target_oid, int data, int chmode)
{
  int lockid; 
  int retval;
  Objstat objstat;
  
  /* get the access rights */
  lockid = begin_change_flags (target_oid, database.dbid);
  if (lockid <= 0)
    return FALSE;
  
  /* At first read the CtrlData of the item an verify that we have the
     permission to delete the item! */
  if (readItemCtrlData (&database, target_oid, &objstat) < 0) {
    end_change_flags (lockid);
    return FALSE;
  }
  
  retval = verify_permission (clientuid, database.asowner,
			      objstat.uid, /* itemuid */
			      objstat.gid, /* itemgid */
			      objstat.accesscode, /* accesscode */
			      chmode); /* func_CHUSR, func_CHGRP, func_CHACCESS */
  if (retval == 0) {
    end_change_flags (lockid);
    DESCERR(0, _("access denied for change object"));
    return FALSE;
  }
  else if (retval < 0) {
    end_change_flags (lockid);
    return FALSE;
  }
  
  /* erst den Datensatz lesen, um die index keys zu bekommen */
  switch (chmode) {
  case func_CHUSR:
    retval = changeItemUid (&database, target_oid, data);
    break;
  case func_CHGRP:
    retval = changeItemGid (&database, target_oid, data);
    break;
  case func_CHACCESS:
    retval = changeItemAcode (&database, target_oid, data);
    break;
  }

  if (retval < 0) {
    retval = end_change_flags (lockid); /* free your resources */
    JMPERR();
  }
  end_change_flags (lockid);
  return TRUE;
}

Atom
read_object_flags (Oid oid)
{
  Objstat objstat;
  int lockid;
  Atom rx, rootx;

  /* get the access rights */
  lockid = begin_read (oid, database.dbid); /* lock your resources! */
  if (lockid <= 0)
    return FALSE;
  
  /* Now read the CtrlData! */
  if (readItemCtrlData (&database, oid, &objstat) < 0) {
    end_read (lockid);
    return FALSE;
  }
  end_read (lockid);

  PUSH_SCANL();
  rx = rootx = SAVE_ROOT();
  rx = CDR(rx) = CONS (CONS(AHASH(h_OID), NEWOID(objstat.oid)), NIL);
  rx = CDR(rx) = CONS (CONS(AHASH(h_FID), NEWINT(objstat.fid)), NIL);
  rx = CDR(rx) = CONS (CONS(AHASH(h_UID), NEWINT(objstat.uid)), NIL);
  rx = CDR(rx) = CONS (CONS(AHASH(h_GID), NEWINT(objstat.gid)), NIL);
  rx = CDR(rx) = CONS (CONS(AHASH(h_MOD), NEWINT(objstat.accesscode)), NIL);
  rx = CDR(rx) = CONS (CONS(AHASH(h_CREAT), NEWDATE(objstat.creat)), NIL);
  rx = CDR(rootx);
  POP_SCANL();

  return rx;
}

/* ----------------------------------------------------------------------
   k4 functions
   ---------------------------------------------------------------------- */
static char s_db_open[] = "db-open";
static char a_db_open[] = "s";
Atom
f_db_open (Atom list)
{
  char *dbname;
  int retval;
  VOretval *voretval;

  dbname = STR_STR(CAR(list));
  
  if (dbname == NULL)
    SETERR (CAR(list), _("internal error: Database name"));
  
  retval = verify_opendb (dbname, &voretval);
  if (retval < 0)
    SETERR (0, _("can't open database")); /* connection error! */
  if (retval == 0)		/* permission denied! */
    SETERR (0, _("open database: Permission denied"));

  if (database.status) {	/* datenbank bereits geffnet? Schlieen! */
    free_all_resources ();	/* erst alle resourcen freigeben! */
    closedb (&database);
  }
  
  killdbvar (&database);	/* at first kill the database var */
  
  database.lockid = voretval->lockid;
  
  if (mountdb (&database, voretval->path) < 0)
    SETERR (0, _("error mounting database"));

  database.dbid = voretval->dbid;
  database.asowner = voretval->dbowner;
  
  set_database_data (&database, 1);
  
  if (opendb (&database) < 0)
    SETERR (0, _("can't open database"));
  
  return UNSPECIFIED;
}

static char s_db_close[] = "db-close";
static char a_db_close[] = "";
Atom
f_db_close (Atom list)
{
  if (database.status) {
    free_all_resources ();
    closedb (&database);
  }
  else
    SETERR (0, _("database was not opened"));
  
  set_database_data (&database, 0);
  
  return UNSPECIFIED;
}


static char s_db_get_iid[] = "db-get-index-id";
static char a_db_get_iid[] = "s";
Atom
f_db_get_iid (Atom list)
{
  if (database.status) {
    int retval = look_up_index (STR_STR(CAR(list)));
    if (retval >= 0)
      return NEWINT(retval);
  }
  return FALSE;
}

/* schlgt den Suchbegriff s (string) im Index h (hash) nach.  Liefert das
   Ergebnis -- unabhngig vom Umfang der Ergebnismenge als Vector (auch bei
   einem oder keinem Element!).  Mit dem letzten (optionalen) Parameter
   lt sich die Suchvariante bestimmen: 'key : sucht nach kompletten
   Schlsseln, 'regexp : betrachtet den zweiten Parameter (string) als
   Regular expression, im Ergebnis finden sich nur die Schlsselworte;
   'regexp-full : dto., allerdings sind zu jedem Schlsselwort auch alle
   Oids aufgelistet; 'regexp-oid : analog liefert jedoch lediglich einen
   (linearen) Vector der gefundenen oids zurck (unterschiedslos und ohne
   Abstufung) */
static char s_db_search[] = "db-search";
static char a_db_search[] = "ns.h";
Atom
f_db_search (Atom list)
{
  Atom retvec;
  int indxno;
  Searchinfo *sinfo;
  int lockid, retval;
  int style = h_KEY;

  if (CDDR(list) != NIL)
    style = HASHV(CADDR(list));

  indxno = numv (CAR(list));
  if ((indxno < 0)
      || (indxno > database.indexno))
    SETERR (list, _("unknown index or bad iid"));

  lockid = begin_read_index (indxno, database.dbid);
  
  if (lockid > 0) {
    switch (style) {
    case h_KEY:
      retval = search_index (&database, indxno, STR_STR(CADR(list)), &sinfo);
      break;
    case h_REGEXP_FULL:
      retval = search_index_regexp (&database, indxno, 
				    STR_STR(CADR(list)), &sinfo, true, false);
      break;
    case h_REGEXP:
      retval = search_index_regexp (&database, indxno, 
				    STR_STR(CADR(list)), &sinfo, false, false);
      break;
    case h_REGEXP_OID:
      retval = search_index_regexp (&database, indxno,
				    STR_STR(CADR(list)), &sinfo, true, true);
      break;
      /*     case h_TRAVERSE: */
      /*       retval = traverse_index (&database, indxno, &sinfo); */
      /*       break; */
    }
    end_read_index (lockid);
    
    if (retval < 0)
      SETERR (list, _("error reading index"));
    if (!sinfo)			/* nothing found */
      goto return_null_vector;
    
    if (sinfo->size == 1)
      retvec = NEWVECTOR(make_vector (1, 1, NEWOID(sinfo->data.oid)));
    else
      retvec = sinfo->data.find_list;
    
    free (sinfo);		/* sinfo was malloc'd in search_index */
    return retvec;
  }
  
 return_null_vector:
  return NEWVECTOR(make_vector (0, 0, NIL));
}

static char s_db_read[] = "db-read";
static char a_db_read[] = "i.b";
Atom
f_db_read (Atom list)
{
  Atom readobj = read_object (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (readobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return readobj;
}

static char s_db_insert[] = "db-insert";
static char a_db_insert[] = "onl.b";
Atom
f_db_insert (Atom list)
{
  Atom newoid = insert_object (CAR(list), CADR(list), CADDR(list));
  if ((CDDDR(list) != NIL)
      && BOOLV(CADDDR(list))
      && (newoid == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return newoid;
}

static char s_db_bupdate[] = "db-begin-update";
static char a_db_bupdate[] = "i.b";
Atom
f_db_bupdate (Atom list)
{
  Atom updobj = begin_update_object (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (updobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return updobj;
}

/* the object : old keylist : new keylist : [ bei einem fehler abbrechen? ] */
static char s_db_update[] = "db-update";
static char a_db_update[] = "ll.b";
Atom
f_db_update (Atom list)
{
  Atom inslen = update_object (CAR(list), CADR(list));
  if ((CDDR(list) != NIL)
      && BOOLV(CADDR(list))
      && (inslen == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return inslen;
}

static char s_db_bdelete[] = "db-begin-delete";
static char a_db_bdelete[] = "i.b";
Atom
f_db_bdelete (Atom list)
{
  Atom readobj = begin_delete_object (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (readobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return readobj;
}

static char s_db_delete[] = "db-delete";
static char a_db_delete[] = "l.b";
Atom
f_db_delete (Atom list)
{
  Atom delobj = delete_object (CAR(list));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (delobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return delobj;
}

static char s_db_bundelete[] = "db-begin-undelete";
static char a_db_bundelete[] = "i.b";
Atom
f_db_bundelete (Atom list)
{
  Atom readobj = begin_undelete_object (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (readobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return readobj;
}

static char s_db_undelete[] = "db-undelete";
static char a_db_undelete[] = "l.b";
Atom
f_db_undelete (Atom list)
{
  Atom delobj = undelete_object (CAR(list));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (delobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return delobj;
}

static char s_db_bkill[] = "db-begin-kill";
static char a_db_bkill[] = "i.b";
Atom
f_db_bkill (Atom list)
{
  Atom readobj = begin_kill_object (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (readobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return readobj;
}

static char s_db_kill[] = "db-kill";
static char a_db_kill[] = "l.b";
Atom
f_db_kill (Atom list)
{
  Atom killobj = kill_object (CAR(list));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (killobj == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return killobj;
}

static char s_db_abort_trans[] = "db-abort-transaction";
static char a_db_abort_trans[] = ".b";
Atom
f_db_abort_trans (Atom list)
{
  Atom retval = FALSE;
  
  switch (func_status) {
  case func_UPDATE:
    retval = esc_update_object ();
    break;
  case func_DELETE:
    retval = esc_delete_object ();
    break;
  case func_UNDELETE:
    retval = esc_undelete_object ();
    break;
  case func_KILL:
    retval = esc_kill_object ();
    break;
  }
  if ((list != NIL)
      && BOOLV(CAR(list))
      && (retval == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return retval;
}

/* ----------------------------------------------------------------------
   ACCESSCODE CHANGE FUNCTION
   ---------------------------------------------------------------------- */
static char s_db_chuid[] = "db-change-uid";
static char a_db_chuid[] = "in.b";
Atom
f_db_chuid (Atom list)
{
  Atom retval = change_object (oidv(CAR(list)), numv(CADR(list)), func_CHUSR);
  if ((CDDR(list) != NIL)
      && BOOLV(CADDR(list))
      && (retval == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return retval;
}

static char s_db_chgid[] = "db-change-gid";
static char a_db_chgid[] = "in.b";
Atom
f_db_chgid (Atom list)
{
  Atom retval = change_object (oidv(CAR(list)), numv(CADR(list)),
			       func_CHGRP);
  if ((CDDR(list) != NIL)
      && BOOLV(CADDR(list))
      && (retval == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return retval;
}

static char s_db_chmod[] = "db-change-mode";
static char a_db_chmod[] = "in.b";
Atom
f_db_chmod (Atom list)
{
  Atom retval = change_object (oidv(CAR(list)), numv(CADR(list)),
			       func_CHACCESS);
  if ((CDDR(list) != NIL)
      && BOOLV(CADDR(list))
      && (retval == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return retval;
}

static char s_db_getmod[] = "db-get-mode";
static char a_db_getmod[] = "i.b";
Atom
f_db_getmod (Atom list)
{
  Atom flags = read_object_flags (oidv(CAR(list)));
  if ((CDR(list) != NIL)
      && BOOLV(CADR(list))
      && (flags == FALSE)) {
    LASTEXP (list);
    JMPERR();
  }
  return flags;
}


/* ----------------------------------------------------------------------
   Initialisierungsfunktion der k4-grundfunktion
   ---------------------------------------------------------------------- */
void
init_db_funcs ()
{
  Functiontable functbl[] = {
    {s_db_open, h_DB_OPEN, a_db_open, f_db_open},
    {s_db_close, h_DB_CLOSE, a_db_close, f_db_close},
    {s_db_insert, h_DB_INSERT, a_db_insert, f_db_insert},
    {s_db_read, h_DB_READ, a_db_read, f_db_read},
    {s_db_bupdate, h_DB_BUPDATE, a_db_bupdate, f_db_bupdate},
    {s_db_update, h_DB_UPDATE, a_db_update, f_db_update},
    {s_db_bdelete, h_DB_BDELETE, a_db_bdelete, f_db_bdelete},
    {s_db_delete, h_DB_DELETE, a_db_delete, f_db_delete},
    {s_db_bundelete, h_DB_BUNDELETE, a_db_bundelete, f_db_bundelete},
    {s_db_undelete, h_DB_UNDELETE, a_db_undelete, f_db_undelete},
    {s_db_bkill, h_DB_BKILL, a_db_bkill, f_db_bkill},
    {s_db_kill, h_DB_KILL, a_db_kill, f_db_kill},
    {s_db_get_iid, h_DB_GET_IID, a_db_get_iid, f_db_get_iid},
    {s_db_search, h_DB_SEARCH, a_db_search, f_db_search},
    {s_db_abort_trans, h_DB_ABORT_TRANS, a_db_abort_trans, f_db_abort_trans},
    {s_db_chuid, h_DB_CHUID, a_db_chuid, f_db_chuid},
    {s_db_chgid, h_DB_CHGID, a_db_chgid, f_db_chgid},
    {s_db_chmod, h_DB_CHMOD, a_db_chmod, f_db_chmod},
    {s_db_getmod, h_DB_GETMOD, a_db_getmod, f_db_getmod},
    {NULL, -1, NULL, NULL}
  };
  Functiontable *ft = functbl;
  int i;
  
  for (i = 0; (ft+i)->fname != NULL ; i++)
    make_fset ((ft+i)->fname, (ft+i)->hashcode, (ft+i)->args, (ft+i)->fnc);
}
