/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif

#include "proto.h"
#include "hc.h"
#include "dbfunc.h"

char *
rb_getln(char *bp, int *stringlength)
{
  short cmp = (unsigned char)*bp;
  if (cmp == WORDFLAG) {
    unsigned short l;
    memcpy (&l, bp+1, 2);
    *stringlength = l;
    bp += 3;
  }
  else if (cmp == LONGFLAG) {
    unsigned int l;
    memcpy (&l, bp+1, 4);
    *stringlength = l;
    bp += 5;
  }
  else {
    *stringlength = cmp;
    bp++;
  }
  return bp;
}


#define CMP_STACKPAGESIZE   128;
#define CMP_INITSTACKSIZE   128;
char *buf = NULL, *bufp;
int bufsize;
Atom *cmp_stack = NULL;
int cmp_sp, cmp_stacksize;

#define INIT_CMP_STACK()					\
({								\
  cmp_sp = 0;							\
  cmp_stacksize = CMP_INITSTACKSIZE;				\
  cmp_stack = (Atom*) smalloc (cmp_stacksize * sizeof (Atom*));	\
})
#define FREE_CMP_STACK()			\
({						\
  if (cmp_stack) {				\
    free(cmp_stack);				\
    cmp_stack = NULL;				\
  }						\
})
#define PUSH_CMP(x)						\
({								\
  if (cmp_sp >= cmp_stacksize) {				\
    cmp_stacksize += CMP_STACKPAGESIZE;				\
    cmp_stack = (Atom*) realloc ((Atom*) cmp_stack,		\
				cmp_stacksize * sizeof (Atom));	\
  }								\
  cmp_stack[cmp_sp] = (x);					\
  cmp_sp++;							\
})
#define POP_CMP()				\
({						\
  if (cmp_sp > 0)				\
    cmp_sp--;					\
  cmp_stack[cmp_sp];				\
})
#define STACK_CMP_BOT  ((cmp_sp > 0) ? 1 : 0)
#define GETSPACE(size)\
({						\
  if ((bufp - buf) + size > bufsize) {		\
    long obufp = bufp - buf;			\
    bufsize += COMPBUFPAGESIZE;			\
    buf = (char *) realloc(buf, bufsize);	\
    bufp = buf + obufp;				\
  }						\
})


/* ======================================================================
   Compiler functions
   ====================================================================== */
#define dir_LEFT    1
#define dir_RIGHT   2
#define MAXCSTACK  128
#define COMPBUFINITSIZE   1024
#define COMPBUFPAGESIZE    256

#define comp_t_INT         'i'
#define comp_t_STR         's'
#define comp_t_CHAR        'c'
#define comp_t_TRUE        't'
#define comp_t_FALSE       'f'
#define comp_t_UNSPECIFIED 'u'
#define comp_t_DATE        'd'
#define comp_t_OID         'o'
#define comp_t_HASH        'h'
#define comp_t_LABEL       'l'
#define comp_t_VECTOR      'v'
#define comp_t_DFIELD      'D'
#define comp_t_NIL         'n'
#define comp_LA            '['
#define comp_LN            '<'
#define comp_RA            ']'
#define comp_RN            '>'
#define comp_OK            '|'
#define comp_S             'S'
#define comp_P             'P'

Atom
code2list (char *prog)
{
  char *ip;
  long val_long;
  int val_int;
  Atom acts, rootx, retval;

#define SET_BOOL(obj,val) ({ ip++; obj = val; })
#define SET_NIL(obj) ({ ip++; obj = NIL; })
#define SET_INT(obj)				\
({						\
  ip++;						\
  memcpy(&val_int, ip, 4);			\
  obj = NEWINT(val_int);			\
  ip += 4;					\
})
#define SET_CHAR(obj)				\
({						\
  ip++;						\
  obj = ACHAR(*ip); /* one byte hashcode */	\
  ip++;						\
})
#define SET_OID(obj)				\
({						\
  ip++;						\
  memcpy(&val_long, ip, 4);			\
  obj = NEWOID(val_long);			\
  ip += 4;					\
})
#define SET_DATE(obj)				\
({						\
  ip++;						\
  memcpy(&val_int, ip, 4);			\
  obj = NEWDATE(val_int);			\
  ip += 4;					\
})
#define SET_STR(obj)				\
({						\
  ip++;						\
  obj = NEWSTR(ip);				\
  ip += strlen(ip) + 1;				\
})
#define SET_HASH(obj)				\
({						\
  int _hc;					\
  ip++;						\
  ip = rb_getln (ip, &_hc);			\
  obj = AHASH(_hc);				\
})
#define SET_LABEL(obj)				\
({						\
  ip++;						\
  obj = NEWHASH(ip);				\
  ip += strlen(ip) + 1;				\
})

#define COMP_ATOM(WHERE)					\
({ 								\
  switch (*ip) {						\
  case comp_t_TRUE: SET_BOOL(WHERE, TRUE); break;		\
  case comp_t_FALSE: SET_BOOL(WHERE, FALSE); break;		\
  case comp_t_UNSPECIFIED: SET_BOOL(WHERE, UNSPECIFIED); break;	\
  case comp_t_INT: SET_INT(WHERE); break;			\
  case comp_t_CHAR: SET_CHAR(WHERE); break;			\
  case comp_t_STR: SET_STR(WHERE); break;			\
  case comp_t_HASH: SET_HASH(WHERE); break;			\
  case comp_t_OID: SET_OID(WHERE); break;			\
  case comp_t_DATE: SET_DATE(WHERE); break;			\
  case comp_t_LABEL: SET_LABEL(WHERE); break;			\
  case comp_t_NIL: SET_NIL(WHERE); break;			\
  }								\
})

  INIT_CMP_STACK();
  
  PUSH_SCANL();
  acts = rootx = SAVE_ROOT();
  
  ip = prog + 8;		/* programmheaderofs */
  ip += strlen(ip)+1;		/* surpase the headcomment! */

  while (*ip != comp_OK) {
    switch (*ip) {
    case comp_LA:
      ip++;
      COMP_ATOM (CAR(acts));
      break;
    case comp_RA:
      ip++;
      COMP_ATOM (CDR(acts));
      break;
    case comp_LN:
      acts = CAR(acts) = ALLOC();
      ip++;
      break;
    case comp_RN:
      acts = CDR(acts) = ALLOC();
      ip++;
      break;
    case comp_S:
      PUSH_CMP(acts);
      ip++;
      break;
    case comp_P:
      acts = POP_CMP();
      ip++;
      break;
    default:
      ip++;
    }
  }
  retval = CDR(rootx);
  POP_SCANL();
  FREE_CMP_STACK ();
  return retval;
}

static char transferdata[8] = "";
void
compatom (Atom atom, char *typ, char **val_data, int *lng)
{
  int i;

  switch (TYP(atom)) {
  case NIL_P:
    *lng = 0;
    *typ = comp_t_NIL;
    *val_data = 0;
    break;
  case BOOL_P:
    *lng = 0;
    switch (atom) {
    case TRUE: *typ = comp_t_TRUE; break;
    case FALSE: *typ = comp_t_FALSE; break;
    case UNSPECIFIED: *typ = comp_t_UNSPECIFIED;
    }
    *val_data = 0;
    break;
  case INT_P:
    i = INTV(atom);
    *lng = 4;
    *typ = comp_t_INT;
    memcpy(transferdata, &i, sizeof (int));
    *val_data = transferdata;
    break;
  case OID_P:
    *lng = 4;
    *typ = comp_t_OID;
    *val_data = (char *)OIDV(atom);
    break;
  case STR_P:
    *lng = strlen(STR_STR(atom)) + 1;
    *typ = comp_t_STR;
    *val_data = STR_STR(atom);
    break;
  case CHAR_P:
    *lng = 1;
    *typ = comp_t_CHAR;
    transferdata[0] = CHARV(atom);
    *val_data = transferdata;
    break;
  case HASH_P:
    if (HASHV(atom) <= systemhash) { /* for predefined hashcodes */
      *lng = 1;			/* LOOK OUT: max. 256 system hashcodes! */
      *typ = comp_t_HASH;
      transferdata[0] = (char)HASHV(atom);
      *val_data = transferdata;
    } 
    else {			/* for runtime hashcodes (labels) */
      *lng = strlen(HASH_STR(atom)) + 1;
      *typ = comp_t_LABEL;
      *val_data = HASH_STR(atom);
    }
    break;
  case OBL_P:
    switch (OBL_TYP(atom)) {
    case OID_T:
      *lng = 4;
      *typ = comp_t_OID;
      *val_data = (char *)OBL_OID(atom);
      break;
    case DATE_T:
      *lng = 4;
      *typ = comp_t_DATE;
      *val_data = (char *)OBL_DATE(atom);
      break;
    case INT_T:
      *lng = -1;
      printf (_("external Integer is uncodeable"));
      break;
    case STR_T:
      *lng = -1;
      printf (_("external Strings are uncodeable"));
      break;
    case VECTOR_T:
      *lng = -1;
      printf (_("vector compilation not supported yet!"));
      break;
    case DFIELD_T:
      *lng = -1;
      printf (_("dfield compilation not supported yet!"));
      break;
    }
    break;
  }
}

char *
list2code (Atom list, int *blng, int commentp)
{
  Atom next;
  int dir;
  char typ;
  char *val_str;
  int lng;
  char *headcomment = "This is raw 1 coded file";

#define SET_BUFP(VAL) ({ *bufp = VAL; bufp++; })

  INIT_CMP_STACK();

  bufsize = COMPBUFINITSIZE;
  bufp = buf = (char *) smalloc (bufsize);
  *bufp = '\0';
  
  GETSPACE (8);			/* programm header */
  memcpy (bufp, MAGIC_RAW1, 8);
  bufp += 8;
  if (commentp) {
    GETSPACE (strlen(headcomment)+1); /* headercomment */
    strcpy (bufp, headcomment);
    bufp += strlen(headcomment)+1;
  }
  else {
    GETSPACE (1);		/* a \0! */
    SET_BUFP('\0');
  }
  
  next = list;
  if (TYP(next) != CELL_P)
    dir = dir_RIGHT;
  else {
    GETSPACE (1);
    SET_BUFP(comp_RN);
    dir = dir_LEFT;
  }
    
  while (1) {
    switch (dir) {
    case dir_LEFT:
      if (TYP(CAR(next)) == CELL_P) {
	GETSPACE (2);
	SET_BUFP (comp_S);
	SET_BUFP (comp_LN);
	PUSH_CMP(next);
	dir = dir_LEFT;
	next = CAR(next);
      }
      else {
	compatom (CAR(next), &typ, &val_str, &lng);
	if (lng >= 0) {
	  GETSPACE (lng+2);
	  SET_BUFP (comp_LA);	/* store left atom code */
	  SET_BUFP (typ);	/* the atom typ */
	  if (lng > 0) {
	    memcpy (bufp, val_str, lng); 
	    bufp += lng;
	  }
	}
	dir = dir_RIGHT;
      }
      break;
      
    case dir_RIGHT:
      if (TYP(CDR(next)) == CELL_P) {
	GETSPACE (1);
	SET_BUFP (comp_RN);
	dir = dir_LEFT;
	next = CDR(next);
      }
      else {
	compatom (CDR(next), &typ, &val_str, &lng);
	if (lng >= 0) {
	  GETSPACE (lng+3);	/* +3: code, type ..., P or OK */
	  SET_BUFP (comp_RA); /* store left atom code */
	  SET_BUFP (typ);	/* the atom typ */
	  if (lng > 0) {
	    memcpy (bufp, val_str, lng); 
	    bufp += lng;
	  }
	}
	if (STACK_CMP_BOT) {
	  SET_BUFP (comp_P);
	  next = POP_CMP();
	}
	else {
	  SET_BUFP (comp_OK);
	  goto compok;
	}
      }
    }
  }
 compok:
  FREE_CMP_STACK();
  *blng = bufp - buf;
  return buf;
}


