/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif

#include "server.h"
#include "proto.h"
#include "lock.h"
#include "dbfunc.h"

/* ----------------------------------------------------------------------
   UPDATE
   ---------------------------------------------------------------------- */
int
lock_update (Oid oid, Fid fid, int *indxlist, int indxlistsize, 
	     Dbid databaseid, int lockid)
{
  char tmp[MAXMSG], resstr[MAXMSG];
  int retval;

  /* change the restricted lock for the object entry to a locked lock */
  
  resstrcpy (resstr, lock_LOCKED, lock_DATA, databaseid, oid);
  retval = GETRESOURCES (KCP_CHANGETRANS, lockid, resstr);
  retval = TESTRES(retval);

  /* now add the openlist and the indices to the locklist lockid */
  
  resstrcpy (resstr, lock_LOCKED, lock_OPENLIST, databaseid, fid);
  strcpy (tmp, resstr);
  
  if (indxlistsize > 0) {
    int i;
    
    for (i = 0; i < indxlistsize; i++) {
      resstrcpy (resstr, lock_LOCKED, lock_INDEX, databaseid, indxlist[i]);
      strcat (tmp, resstr);
    }
  }
  
  retval = GETRESOURCES (KCP_ADDTRANS, lockid, tmp); 
  retval = TESTRES(retval);

  return 1;
}

/* ----------------------------------------------------------------------
   INSERT (NEW)
   ---------------------------------------------------------------------- */
int
begin_insert (Oid oid, Fid fid, int *indxlist, int indxlistsize,
	      Dbid databaseid)
{
  char tmp[MAXMSG], resstr[MAXMSG];
  int retval;
  
  resstrcpy (resstr, lock_LOCKED, lock_NEXTOID, databaseid, oid);
  strcpy (tmp, resstr);
  resstrcpy (resstr, lock_LOCKED, lock_OPENLIST, databaseid, fid);
  strcat (tmp, resstr);
  
  if (indxlistsize > 0) {
    int i;
    
    for (i = 0; i < indxlistsize; i++) {
      resstrcpy (resstr, lock_LOCKED, lock_INDEX, databaseid, indxlist[i]);
      strcat (tmp, resstr);
    }
  }
  
  retval = GETRESOURCES (KCP_BEGINTRANS, 0, tmp);
  retval = TESTRES(retval);

  return retval;
}

/* ----------------------------------------------------------------------
   Check user permission & access
   ---------------------------------------------------------------------- */
int
verify_opendb (char *dbname, VOretval **voretval)
{
  char resstr[MAXMSG];
  int retval = verify_database (daemonsock, connectid, connkey, clientuid,
				dbname, voretval);
  if (retval > 0) {
    /* get a shared lock for the database */
    resstrcpy (resstr, lock_SHARED, lock_DATABASE, (*voretval)->dbid, 0);
    retval = GETRESOURCES (KCP_BEGINTRANS, 0, resstr);
    retval = TESTRES (retval);
    (*voretval)->lockid = retval;
    return 1;
  }
  return retval;
}

int
lock_database (Database *db, int lockmode)
{
  char resstr[MAXMSG];
  int retval = 0;

  resstrcpy (resstr, lockmode, lock_DATABASE, db->dbid, 0);
  retval = GETRESOURCES (KCP_CHANGETRANS, db->lockid, resstr);
  switch (retval) {
  case -1:
    return -1;
  case -2:
    resstrcpy (resstr, lock_SHARED, lock_DATABASE, db->dbid, 0);
    retval = GETRESOURCES (KCP_CHANGETRANS, db->lockid, resstr);
    return -1;
  case -3:
    eoserver = true;
    return -1;
  case 0:
    return 0;
  }
  return retval;
}

#define USRINGRP(uid, itemgid)					\
({								\
  __label__ done;						\
  int _retval = 0;						\
  int _tmpsize = 128;						\
  char *_tmp = (char*) malloc (_tmpsize), *_token;		\
  if (writesockf (daemonsock, "%s %d %s %d %d", KCP_USRINGRP,   \
	          connectid, connkey, uid, itemgid) <= 0) {	\
    _retval = -1;						\
    goto done;							\
  }								\
  if (read_from_sock (daemonsock, &_tmp, &_tmpsize) < 0) {	\
    _retval = -1;						\
    goto done;							\
  }								\
								\
  _token = strtok (_tmp, " \r");	/* get +OK or -ERR */	\
  if ((_token)							\
      && (strlen (_token) > 0)					\
      && (_token[0] == '+')) {					\
    _retval = 1;						\
  }								\
 done:								\
  free (_tmp);							\
  _retval;							\
})
	  
int
verify_permission (Uid uid, int owner, 
		   Uid itemuid, Gid itemgid, ACode acode, int func)
{
  int retval;

  /* Fr den Systemroot gelten berhaupt keine Einschrnkungen, wir
     brauchen daher nichts weiter abzufragen */
  if (uid == U_ROOT)
    goto permhd;

  switch (func) {
  case func_KILL:
    /* einen Datesatz darf nur der Datenbankeigentmer lschen; daher
       brauchen wir auch nur noch abzufragen, ob der Datensatz fr den Usr
       (WUSR) schreibeerlaubnis hat. */

    if (!owner)
      goto nopermhd;
    
    if (acode & A_WUSR)
      goto permhd;
    
    break;
    
  case func_INSERT:
    /* Datenstze darf jeder -- bis auf einen Gast -- einfgen. TODO! Hier
       mu etwas verndert werden: Es mu einen Einfgeschutz pro Datenbank
       geben! (D.h. eine Datenbank wird mit den Labeln RW oder RO geffnet!  */
    if (uid != U_GUEST)
      goto permhd;
    break;
    
  case func_READ:
    /* lesen von Datenstzen ist dann schon etwas verzwickter.  Lesen darf
       jeder, fr den eine Flagge in acode gesetzt ist, bzw. auch die Gste
       wenn das Guestbit gesetzt ist. */
    if (uid != U_GUEST) {
      if (uid == itemuid) {	/* selber usr */
	goto permhd;
      }
      else {
	/* ask the daemon, if our uid is in the group */
	retval = USRINGRP(uid, itemgid);
	if (retval > 0) {
	  if (acode & A_RGRP)
	    goto permhd;
	}
	else if (retval == 0) {
	  if (acode & A_ROTH)
	    goto permhd;
	}
	else 
	  goto errhd;
      }
    }
    else {
      if (acode & A_RGUEST)
	goto permhd;
    }

    goto nopermhd;
    break;
    
  case func_UPDATE:
  case func_DELETE:
  case func_UNDELETE:
    /* die writeroutine ist hnlich heikel wie die read routine:
    write/delete/undelete darf jeder fr den das Schreibbit gesetzt ist --
    allerdings mu hier auch fr den usr (Eigentmer) abgefragt werden!
    Ein Gast braucht berhautpt nicht weiter bercksichtigt zu werden, da
    er eh niemals Schreibeerlaubnis besitzt. */
    if (uid != U_GUEST) {
      if (uid == itemuid) {	/* selber usr */
	if (acode & A_WUSR)
	  goto permhd;
      }
      else {
	/* ask the daemon, if our uid is in the group */
	retval = USRINGRP(uid, itemgid);
	if (retval > 0) {
	  if (acode & A_WGRP)
	    goto permhd;
	}
	else if (retval == 0) {
	  if (acode & A_WOTH)
	    goto permhd;
	}
	else 
	  goto errhd;
      }
    }
    
    goto nopermhd;
    break;

  case func_CHUSR:
  case func_CHGRP:
  case func_CHACCESS:
    /* das setzen oder ndern von Zugriffsmasken luft wie die read
    routine: chusr/chgrp/chaccess darf jeder fr den das ACCESSbit gesetzt
    ist.  Der Eigentmer darf dies stets ndern!  Ein Gast braucht
    berhaupt nicht weiter bercksichtigt zu werden, da er eh niemals
    Schreibeerlaubnis besitzt. */
    if (uid != U_GUEST) {
      if (uid == itemuid) {	/* selber usr */
	goto permhd;
      }
      else {
	/* ask the daemon, if our uid is in the group */
	retval = USRINGRP(uid, itemgid);
	if (retval > 0) {
	  if (acode & A_AGRP)
	    goto permhd;
	}
	else if (retval == 0) {
	  if (acode & A_AOTH)
	    goto permhd;
	}
	else 
	  goto errhd;
      }
    }
    
    goto nopermhd;
    break;
  }
  
nopermhd:
  return 0;
  
permhd:
  return 1;
  
errhd:
  return -1;
}


/* ----------------------------------------------------------------------
   Delete, undelete, kill
   ---------------------------------------------------------------------- */
int
lock_delete (int *indxlist, int indxlistsize, Dbid databaseid, int lockid)
{
  char tmp[MAXMSG], resstr[MAXMSG];
  int retval;
  
  tmp[0] = '\0';
  
  /* add the indices to the locklist lockid */
  if (indxlistsize > 0) {
    int i;
    
    for (i = 0; i < indxlistsize; i++) {
      resstrcpy (resstr, lock_LOCKED, lock_INDEX, databaseid, indxlist[i]);
      strcat (tmp, resstr);
    }

    retval = GETRESOURCES (KCP_ADDTRANS, lockid, tmp);
    retval = TESTRES(retval);
  }
  
  return 1;
}

int
lock_undelete (int *indxlist, int indxlistsize, 
	       Dbid databaseid, int lockid)
{
  char tmp[MAXMSG], resstr[MAXMSG];
  int retval;

  tmp[0] = '\0';

  /* now add the indices to the locklist lockid */
  if (indxlistsize > 0) {
    int i;
    
    for (i = 0; i < indxlistsize; i++) {
      resstrcpy (resstr, lock_LOCKED, lock_INDEX, databaseid, indxlist[i]);
      strcat (tmp, resstr);
    }
  
    retval = GETRESOURCES (KCP_ADDTRANS, lockid, tmp);
    retval = TESTRES(retval);
  }
  
  return 1;
}

int
lock_kill (Fid fid, int *indxlist, int indxlistsize, 
	   Dbid databaseid, int lockid)
{
  char tmp[MAXMSG], resstr[MAXMSG];
  int retval;
  
  /* now add the openlist and the indices to the locklist lockid */
  resstrcpy (resstr, lock_LOCKED, lock_OPENLIST, databaseid, fid);
  strcpy (tmp, resstr);
  
  if (indxlistsize > 0) {
    int i;
    
    for (i = 0; i < indxlistsize; i++) {
      resstrcpy (resstr, lock_LOCKED, lock_INDEX, databaseid, indxlist[i]);
      strcat (tmp, resstr);
    }
  }
  
  retval = GETRESOURCES (KCP_ADDTRANS, lockid, tmp);
  retval = TESTRES(retval);
  return 1;
}
