/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#define PROG_NAME "kdmon"

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <locale.h>
#include <ctype.h>
#include <signal.h>

#if defined HAVE_GETTEXT
# include <libintl.h>
# define _(String) gettext (String)
#else
# define _(STRING) STRING
#endif

#define USE_READLINE 1

#if defined HAVE_READLINE_READLINE_H
# include <readline/readline.h>
#endif

#include "kaenguru.h"

KD_conn *connection = NULL;
char *version = "v0.01";
char *copyrightyear = "1997,98";

/* prototypes */

void showhelp (void);
void Welcome();

char *
get_filename (char *prompt)
{
  char *inputline;

#if defined HAVE_READLINE_READLINE_H
  inputline = readline (prompt);
#else
  inputline = (char*) malloc (512);
  fgets (inputline, 512, stdin);
#endif
  if (inputline == NULL)
    return NULL;
  return inputline;
}

int 
get_id (char *prompt)
{
  char *inputline, *tail;
  int num;

#if HAVE_READLINE_READLINE_H
  inputline = readline (prompt);
#else
  inputline = (char*) malloc (512);
  fgets (inputline, 512, stdin);
#endif
  if (inputline == NULL)
    return 0;
  
  num = strtol (inputline, &tail, 0);
  free (inputline);
  return num;
}

/* ----------------------------------------------------------------------
   EINE HAUPTROUTINE
   ---------------------------------------------------------------------- */
void
main (int argc, char **argv)
{
  int indx;			/* for the options */
  int c;
  char *inputline;
  char *portstring = NULL;	/* port argument */
  char *host = NULL;
  char *usrname = NULL;
  unsigned short port = 0;
  int ende;
  int respmode = 0;
  int blobid = 0;
  char *fname = NULL;

  setlocale (LC_MESSAGES, "");
  textdomain ("kaenguru");

  /* ------------------------------------------------------------ 
     OPTIONS PARSING
     ------------------------------------------------------------*/
  opterr = 0;
  while ((c = getopt (argc, argv, "hp:a:u:")) != -1)
    switch (c) {
    case 'h':			/* help */
      showhelp();
      exit (EXIT_SUCCESS);
    case 'p':			/* port to address the daemon */
      portstring = optarg;
      port = atoi (portstring);
      break;
    case 'a':			/* host to connect to */
      host = optarg;
      break;
    case 'u':
      usrname = optarg;
      break;
    case '?':
      if (isprint (optopt))
	fprintf (stderr, _("Unknown option `-%c'.\n"), optopt);
      exit (EXIT_FAILURE);
    default:
      exit (EXIT_FAILURE);
    }
  
  if (!usrname) {
    usrname = getenv ("LOGNAME");
  }

  connection = KD_connect (port, host, usrname);
  if (!connection) {
    fprintf (stderr, _("Can't login.\n"));
    exit (EXIT_FAILURE);
  }

  /* load a file -------------------------------------------------- */

  printf ("Kaenguru monitor %s\n", version);
  ende = 0;

  while (!ende) {
#if HAVE_READLINE_READLINE_H
    inputline = readline ("~");
#else
    fprintf (stdout, "\n~"); fflush (stdout);
    inputline = (char*) malloc (512);
    fgets (inputline, 512, stdin);
#endif
    if (inputline == NULL)

      ende = 1;
    else {
      if (inputline[0] == ':') {
	if (strlen (inputline) > 1) {
	  switch (inputline[1]) {
	  case '?': case 'h':
	    printf (_("Help:\n"
		      ":q or :x      quit\n"
		      ":h or :?      this help\n"
		      ":l            load dfield\n"
		      ":s            store dfield\n"
		      ":t            switch talklisp/data\n"));
	    break;
	  case 't':
	    if (respmode)
	      respmode = 0;
	    else
	      respmode = 1;
	    printf (_("Getvalue mode: %s\n"), 
		    (respmode ? "talklisp" : "data"));
	    break;
	  case 'q': case 'x':
	    ende = 1;
	    break;
	  case 'l':		/* load datafield */
	    printf (_("Read a DFIELD from the database\n"));
	    blobid = get_id (_("Input the dfield ID:"));
	    fname = get_filename (_("The file to write to:"));
	    if (fname && blobid) {
	      FILE *stream;

	      stream = fopen (fname, "w");
	      if (!stream) {
		fprintf (stderr, _("Unknown output file"));
	      }
	      else {
		char *buf;
		int bufsize;
		buf = KD_read_blob (connection, blobid, &bufsize);
		if (buf) {
		  fwrite (buf, 1, bufsize, stream);
		  free (buf);
		  fprintf (stdout, _("%d bytes written to file %s\n"),
			   bufsize, fname);
		}
		fclose (stream);
	      }
	      free (fname);
	    }
	    break;
	  case 's':		/* store datafield */
	    printf (_("Store a file to a database blob\n"));
	    fname = get_filename (_("The file to read:"));
	    blobid = get_id (_("The ID to store data in:"));
	    if (fname && blobid) {
	      FILE *stream;
	      
	      stream = fopen (fname, "r");
	      if (!stream) {
		fprintf (stderr, _("Unknown input file"));
	      }
	      else {
		char *buf;
		long int filesize = 0;
		fseek (stream, 0, SEEK_END);
		filesize = ftell (stream);
		fseek (stream, 0, SEEK_SET);
		buf = (char*) malloc (filesize);
		fread (buf, 1, filesize, stream);
		KD_store_blob (connection, blobid, buf, filesize);
		free (buf);
		fprintf (stdout, _("%ld bytes stored to ID %d"), filesize,
			 blobid);
		fclose (stream);
	      }
	      free (fname);
	    }
	    break;
	  }
	}
      }
      else {
	if (KD_send (connection, inputline) < 0) {
	  printf ("%s\n", KD_geterrmsg ());
	}
	else {
	  if (!respmode) {
	    char * buf = KD_talklisp (connection);
	    if (buf) {
	      printf ("%s\n", buf);
	      free (buf);
	    }
	  }
	  else {
	    KD_data *data = KD_getval (connection);
	    
	    if (data) {
	      KD_data *ptr = data;
	      while (ptr) {
		switch (ptr->typ) {
		case KD_INT: 
		  printf ("Int: %d\n", ptr->data.num); 
		  break; 
		case KD_UINT: 
		  printf ("UInt: %ld\n", ptr->data.unum); 
		  break; 
		case KD_CHAR: 
		  printf ("Char: %d\n", ptr->data.num); 
		  break; 
		case KD_BOOL: 
		  printf ("Bool: %d\n", ptr->data.num); 
		  break; 
		case KD_STR: 
		  printf ("Str: %s\n", ptr->data.str); 
		  break; 
		case KD_SYM:
		  printf ("Sym: %s\n", ptr->data.str); 
		  break;
		case KD_DFIELD: 
		  printf ("Datafield, Size: %d ID: %d\n",  
			  ptr->data.dfield.size, 
			  ptr->data.dfield.blobid); 
		  break; 
		} 
		ptr = ptr->next; 
	      }
	      KD_free (data);
	    }
	  }
	}
      }
      free (inputline);
    }
  }
  
  KD_close (connection);

  exit (EXIT_SUCCESS);
}

/*------------------------------------------------------------
  General routines
  ------------------------------------------------------------*/	
void
Welcome()
{
  fprintf(stdout, _(
"%s - the kaenguru database online monitor\n"
"Use kdline -h for detailed help on syntax and use.\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"), PROG_NAME, copyrightyear);
}


void
showhelp (void)
{
  fprintf(stdout, _(
"%s is a basic frontend (client) for the kaenguru database system.  It\n"
"is a very simple an rudimentary access utility and is thought at first at a\n"
"test and experimentation tool.\n"
"\n"
"General usage:\n"
"  kdline [OPTIONS]\n"
"\n"
"Options:\n"
"  -h         This help\n"
"  -p         The port to connect to\n"
"  -a         The host to connect to\n"
"  -u         The name of the user to be used while connecting\n"
"\n"), PROG_NAME);    
}
