/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#ifndef _KDBS_SM_H
#define _KDBS_SM_H

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include "kdbs.h"

#define MAXDTAFILE       10
#define MAXINDEX         10
#define FIRSTOID          1
#define UNKNOWNOID        0

#define CLASS_INDEX_NAME   "class"
#define TABLE_FILE_NAME    "table"
#define FILES_FILE_NAME    "files"
#define BLOB_DIR_NAME      "blobs"


/* ----------------------------------------------------------------------
   BTREE DEFINES 

   you mustn't play with this values!  This is a great hack, the values
   *are* hardware-dependend!
   ---------------------------------------------------------------------- */
#define i_RIGHT 0		/* fuer rechts oder links suchen */
#define i_LEFT  1

#define i_MAXITEMS       63
#define i_MAXITEMS_KEY   15	/* ACHTUNG: EMPTYPAGEGAP ANGLEICHEN! */
#define i_PAGEMIDTH_KEY   7
#define i_MAXITEMS_REF   63	
#define i_PAGEMIDTH_REF  31
#define i_KEYSIZE        23	/* gilt nur fuer knot_key */
#define i_EMPTYPAGEGAP  504	/* ACHTUNG: ANGLEICHEN! 
				   MAXITEMS*32+kt.gap[12]+kt.init[16]+4 - 4-1*/
#define i_KEYTREE        'k'
#define i_REFTREE        'r'
#define i_EMPTYPAGE      'e'
#define i_PAGELINK       'p'
#define i_REFLINK        'r'

#define i_NOLINK          0	/* to get the code readable */
#define i_NOREF           0

#define i_BSPUNUSED      24

typedef struct {
  int maxitems;
  int pagemidth;
  char pagetype;		/* KEYTREE, REFTREE, EMPTYPAGE */
  int keysize;
  int emptypagegap;
} Indexstatus;

typedef struct {
  char key[i_KEYSIZE];
  Oid info;
  char reftype;
} Keyinfo;

/* Der Typ eines Schluessels (Item_key) und der Referencen (Item_ref) */
typedef struct {
  long link;
  Oid info;
  char reftype;			/* 'r'=reference or 't'=tree */
  char key[i_KEYSIZE];
} Item_key;

typedef struct {
  long link;
  long ref;
} Item_ref;


/* the page structures on the harddisk */
typedef union {
  struct {			/* key page */
    char pagetype;
    char unused;
    short items;
    long nulllink;
    Item_key itm[i_MAXITEMS_KEY - 1];
  } kt;
  struct {			/* reference page */
    char pagetype;
    char unused;
    short items;
    long nulllink;
    Item_ref itm[i_MAXITEMS_REF - 1];
  } ref;
  struct {			/* emptypage */
    char pagetype;
    char unused[3];
    long nextfree;
    char space[i_EMPTYPAGEGAP];
  } ep;
} Backpage;

/* item structures in memory */
typedef struct {
  long link;
  Oid info;
  char reftype;
  char key[i_KEYSIZE];
} Item;

/* page structure in memory */
typedef struct {
  short items;
  char pagetype;
  long nextfree;
  Item itm[i_MAXITEMS];
} Workpage;

/* the basepage (fundamental data of index) */
typedef struct {
  long root;			/* root of tree */
  long nextfree;		/* ptr to next free page */
  char unused[i_BSPUNUSED];
} Basispage;


/* ----------------------------------------------------------------------
   Types for the datafiles etc. 
   ---------------------------------------------------------------------- */
/* this is the struct holding data for each data object */
typedef struct {
  Oid oid;			/* itemnumber */
  char status;			/* status */
  Uid uid;			/* creator id */
  unsigned short unused;
  long length;			/* length of entry */
  Date creat;			/* Date of creation */
} Itemhead;

/* this is the struct building the central table file (a big array of
   statically stored Tableitem-structures */
typedef struct {
  Ofs ofs;			/* ofs in the datafile */
  Fid fid;			/* the data file id */
  ACode accesscode;		/* the accessflags */
  Uid uid;			/* user id */
  Gid gid;			/* group id */
} Tableitem;

/* the struct which are the openlist files build from */
typedef struct {
  Ofs ofs;			/* ofs */
  long length;			/* length of free entry; > 0 if valied */
} Oplitem;

/* struture for blob file headers */
typedef struct {
  Oid oid;
  long length;
  long r1, r2;		/* reserverd for future extensions */
} Blobhead;

/* an internal struct that serves for communication between different parts
   of the code, e.g. between readItem and higher level functions.  It mixes
   up information from both the table and the data object (header) itself */
typedef struct {
  Fid fid;
  Uid uid;
  Gid gid;
  ACode accesscode;
  char status;
  Oid oid;
  int length;
  Date creat;
} Objstat;

/* ----------------------------------------------------------------------
   Types for the database management 
   ---------------------------------------------------------------------- */
typedef struct {
  char *name;			/* path+name of the index */
  FILE *file;			/* filevar */
  bool bp_changed;		/* hat sich die basispage veraendert? */
  Basispage basispage;		/* die Basispage */
  Fid iid;			/* the index id */
} Index;			/* ein Index */

typedef struct {
  char *name;			/* path+name of datafile */
  FILE *file;			/* filevar of datafile */
  FILE *opfile;			/* openlistfile */
  Fid fid;			/* the file id */
} Datafile;			/* ein Datafile */

typedef struct {
  char *basename;		/* name of database */
  char *path;			/* the path of the database */
  FILE *tblfile;		/* tablefile */
  FILE *cfgfile;		/* configfile */
  int datafileno;		/* number of datafiles */
  Datafile data[MAXDTAFILE];	/* the datafiles */
  int indexno;			/* number of indices */
  Index index[MAXINDEX];	/* die Indexlisten */
  int status;			/* 1 open 0 closed */
  bool asowner;			/* is db opened as owner? */
  Dbid dbid;			/* the database id */
  int lockid;			/* a lock id */

#if DISTRIBUTED_DB
  int nettype;			/* type of remote connection; 0 = tcp */
  int remote;			/* a remote database ? */
  char *server;			/* the remote host name */
  unsigned short port;		/* the remote server port */
  int sock;			/* the socket to the datamanager */
  char *connkey;		/* connection key to datamanager */
  int connid;			/* connection id to datamanager */
#endif

} Database;

/* ----------------------------------------------------------------------
   PROTOTYPEN etc.
   ---------------------------------------------------------------------- */
#define status_UNDEF    '*'
#define status_DEL      'd'
#define status_ENTRY    'E'
#define status_FREE     '-'
#define status_KILLED   -1

/* from files.c */
extern void initdbvar (Database *db);
extern void killdbvar (Database *db);
extern int readcfgfile (Database *db);
extern int setdbfilenames (Database *db, char *basename, char *path);
extern int mountdb (Database *db, char *dbpath);
extern int opendatafile (Database *db, int datafileno);
extern int openoplfile (Database *db, int datafileno);
extern int createoplfile (Database *db, int datafileno);
extern int closedatafile (Database *db, int datafileno);
extern int closeoplfile (Database *db, int datafileno);
extern int opendb (Database *db);
extern int closedb (Database *db);
extern int createIndex (char *path, char *filename);
extern int openIndex (Database *db, int indexfileno);
extern int closeIndex (Database *db, int indexfileno);
extern int createdb (Database *db, char *basename, char *path,
		     char *list_of_files, char *list_of_index);

extern int deloplfile (Database *db, int datafileno);
extern int deldatafile (Database *db, int datafileno);

extern int get_fid_index (Database *db, Fid fid);
extern int get_iid_index (Database *db, Fid iid);
extern int get_iname_index (Database *db, char *indexname);
extern int get_dname_index (Database *db, char * datafilename);

extern char kaengurubase[FILENAME_MAX];
extern char tmppath[FILENAME_MAX];

extern char * get_tmp_filename ();
extern char * get_filename (Database *db);
extern FILE * create_file (char *path);

#define datafilename(BUF, DB, FID)				\
({								\
  sprintf (BUF, "%s/%s.data", DB->path, DB->data[FID].name);	\
})

#define oplfilename(BUF, DB, FID)				\
({								\
  sprintf (BUF, "%s/%s.opl", DB->path, DB->data[FID].name);	\
})

#define indxfilename(BUF, DB, IID)				\
({								\
  sprintf (BUF, "%s/%s.indx", DB->path, DB->index[IID].name);	\
})

/* from vacuum.c */
extern int vacuumFile (Database *db, Fid fid, int deldeleted);
extern int vacuum (Database *db, int deldeleted, Fid fid);

/* from basic.c */
extern int forcesync;
extern int blob_limit;
#if defined HAVE_FDATASYNC
extern int pedanticforcesync;
#endif

extern void cleanoutput (FILE *stream);
extern void cleaninput (FILE *stream);
extern int writeItemhead (FILE *iofile, long ofs, Itemhead *ihead); 
extern int readItemhead (FILE *iofile, long ofs, Itemhead *ihead);
extern int writeTableitem (FILE *tblfile, Oid oid, Tableitem *titem);
extern int readTableitem (FILE *tblfile, Oid oid, Tableitem *titem);
extern long put2oplist (Database *db, int file, long ofs, long length);
extern long getnewpos(Database *db, int file, long minlength);
extern Oid getnew_oid (Database *db);
extern Oid gettotal_oid (Database *db);
extern int readItem (Database *db, Oid oid, char **buf, Objstat *);
extern int rewriteItem (Database *db, Oid oid, char *buf, long length);
extern Oid newItem (Database *db, Fid file, Uid uid, Gid gid, ACode acode,
		    char *buf, long length);
extern int delItem (Database *db, Oid oid);
extern int undelItem (Database *db, Oid oid);
extern int killItem (Database *db, Oid oid);
extern int changeItem (Database *db, Oid oid, int func, int data);
#define changeItemUid(DB, OID, UID) ({ changeItem (DB, OID, 0, UID); })
#define changeItemGid(DB, OID, GID) ({ changeItem (DB, OID, 1, GID); })
#define changeItemAcode(DB, OID, ACODE) ({ changeItem (DB, OID, 2, ACODE); })

extern int readItemCtrlData (Database *db, Oid oid, Objstat *objstat);

extern Date getcurrentdate ();

/* remotesite.c */
extern int connect_to_remote_site (Database *db);
extern int disconnect_from_remote_site (Database *db);

/* public.c */
extern char * get_index_name (Database *db, int iindex);
extern char * get_datafile_name (Database *db, int findex);
extern Fid get_index_id (Database *db, int iindex);
extern Fid get_datafile_id (Database *db, int findex);
extern int get_datafileno (Database *db);
extern int get_indexfileno (Database *db);
extern int update_index (Database *db, Fid iid, char *token, Oid oid);
extern int delete_index (Database *db, Fid iid, char *token, Oid oid);
extern int retrieve_index (Database *db, Fid iid, char *token, 
			   Keyinfo *keyinfo);
extern int load_index_page (Database *db, Fid iid, Node node, 
			    Workpage *workpage);
extern Node get_index_root (Database *db, Fid iid);

/* ----------------------------------------------------------------------
   ERROR CODE
   ---------------------------------------------------------------------- */
#define SM_GOERR(estr) ({ errstr = estr; goto errhd; })
#define SM_DESCERR(estr) ({ errstr = estr; })

#define NWEEFID _("unknown file id")
#define NWEGETNEWPOS _("bad new file position")
#define NWERTABLE _("reading table")
#define NWEIDKILL _("reading item (killed")
#define NWEDATACORRUPT _("data corrupt (unknown item status)")
#define NWEEPOS _("bad file position")
#define NWEDNITEM _("data corrupt (unknown item status)")
#define NWEIDEL _("reading item (deleted)")
#define NWERITEMH _("reading item head")
#define NWEUDNDITEM _("undeleting not-deleted item")
#define NWEUPDOPL _("updating openlist")
#define NWEWDATA _("writing item data")
#define NWEWITEMH _("writing item head")
#define NWEWOPL _("writing openlist")
#define NWEWTABLE _("writing table")
#define NWEDBLOB _("removing blob")
#define NWECBLOB _("creating blob")
#define NWEWBLOB _("writing blob")
#define NWERBLOB _("reading blob")
#define NWEVCTMP _("creating temporary file")
#define NWEVDORG _("deleting original file")
#define NWEVMTMP _("renaming temporary file")

#if DISTRIBUTED_DB
#define NWEREMOTELOGIN _("remote login to sm failed")
#else
#define NODISTRDB _("this server doesn't support distributed databases")
#endif

#endif /* _KDBS_SM_H */

