;;; k4.el --- Kaenguru k4 editing mode.

;; Copyright (c) 1997 by Gregor Klinke, based on scheme.el Copyright (C)
;; 1986, 87, 88, 1997 Free Software Foundation, Inc.

;; Author: Bill Rozas <jinx@martigny.ai.mit.edu>
;; modified by Gregor Klinke <gregor.c.klinke@ruhr-uni-bochum.de>
;; Keywords: languages, lisp, k4

;;; Commentary:

;; The major mode for editing Scheme-type Lisp code, very similar to
;; the Lisp mode documented in the Emacs manual.  `dsssl-mode' is a
;; variant of scheme-mode for editing DSSSL specifications for SGML
;; documents.  [As of Apr 1997, some pointers for DSSSL may be found,
;; for instance, at <URL:http://www.sil.org/sgml/related.html#dsssl>.]
;; All these Lisp-ish modes vary basically in details of the language
;; syntax they highlight/indent/index, but dsssl-mode uses "^;;;" as
;; the page-delimiter since ^L isn't normally a legal SGML character.
;;

;;; Code:

(require 'lisp-mode)

(defvar k4-mode-syntax-table nil "")
(if (not k4-mode-syntax-table)
    (let ((i 0))
      (setq k4-mode-syntax-table (make-syntax-table))
      (set-syntax-table k4-mode-syntax-table)

      ;; Default is atom-constituent.
      (while (< i 256)
	(modify-syntax-entry i "_   ")
	(setq i (1+ i)))

      ;; Word components.
      (setq i ?0)
      (while (<= i ?9)
	(modify-syntax-entry i "w   ")
	(setq i (1+ i)))
      (setq i ?A)
      (while (<= i ?Z)
	(modify-syntax-entry i "w   ")
	(setq i (1+ i)))
      (setq i ?a)
      (while (<= i ?z)
	(modify-syntax-entry i "w   ")
	(setq i (1+ i)))

      ;; Whitespace
      (modify-syntax-entry ?\t "    ")
      (modify-syntax-entry ?\n ">   ")
      (modify-syntax-entry ?\f "    ")
      (modify-syntax-entry ?\r "    ")
      (modify-syntax-entry ?  "    ")

      ;; These characters are delimiters but otherwise undefined.
      ;; Brackets and braces balance for editing convenience.
      (modify-syntax-entry ?\[ "(]  ")
      (modify-syntax-entry ?\] ")[  ")
      (modify-syntax-entry ?{ "(}  ")
      (modify-syntax-entry ?} "){  ")
      (modify-syntax-entry ?\| "  23")

      ;; Other atom delimiters
      (modify-syntax-entry ?\( "()  ")
      (modify-syntax-entry ?\) ")(  ")
      (modify-syntax-entry ?\; "<   ")
      (modify-syntax-entry ?\" "\"    ")
      (modify-syntax-entry ?' "  p")
      (modify-syntax-entry ?` "  p")

      ;; Special characters
      (modify-syntax-entry ?, "_ p")
      (modify-syntax-entry ?@ "_ p")
      (modify-syntax-entry ?# "_ p14")
      (modify-syntax-entry ?\\ "\\   ")))

(defvar k4-mode-abbrev-table nil "")
(define-abbrev-table 'k4-mode-abbrev-table ())

(defvar k4-imenu-generic-expression
      '((nil 
	 "^(define\\(\\|-\\(generic\\(\\|-procedure\\)\\|method\\)\\)*\\s-+(?\\(\\(\\sw\\|\\s_\\)+\\)" 4)
	(" Types" 
	 "^(define-class\\s-+(?\\(\\(\\sw\\|\\s_\\)+\\)" 1)
	(" Macros"
	 "^(\\(defmacro\\|define-macro\\|define-syntax\\)\\s-+(?\\(\\(\\sw\\|\\s_\\)+\\)" 2))
  "Imenu generic expression for K4 mode.  See `imenu-generic-expression'.")

(defun k4-mode-variables ()
  (set-syntax-table k4-mode-syntax-table)
  (setq local-abbrev-table k4-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'fill-paragraph-function)
  (setq fill-paragraph-function 'lisp-fill-paragraph)
  ;; Adaptive fill mode gets in the way of auto-fill,
  ;; and should make no difference for explicit fill
  ;; because lisp-fill-paragraph should do the job.
  (make-local-variable 'adaptive-fill-mode)
  (setq adaptive-fill-mode nil)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'lisp-indent-line)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments t)
  (make-local-variable 'outline-regexp)
  (setq outline-regexp ";;; \\|(....")
  (make-local-variable 'comment-start)
  (setq comment-start ";")
  (make-local-variable 'comment-start-skip)
  ;; Look within the line for a ; following an even number of backslashes
  ;; after either a non-backslash or the line beginning.
  (setq comment-start-skip "\\(\\(^\\|[^\\\\\n]\\)\\(\\\\\\\\\\)*\\);+[ \t]*")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'lisp-comment-indent)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments t)
  (make-local-variable 'lisp-indent-function)
  (set lisp-indent-function 'k4-indent-function)
  (setq mode-line-process '("" k4-mode-line-process))
  (make-local-variable 'imenu-generic-expression)
  (setq imenu-generic-expression k4-imenu-generic-expression)
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(k4-font-lock-keywords nil t)))

(defvar k4-mode-line-process "")

(defvar k4-mode-map nil
  "Keymap for K4 mode.
All commands in `shared-lisp-mode-map' are inherited by this map.")

(if k4-mode-map
    ()
  (let ((map (make-sparse-keymap "K4")))
    (setq k4-mode-map
	  (nconc (make-sparse-keymap) shared-lisp-mode-map))
    (define-key k4-mode-map "\e\t" 'lisp-complete-symbol)
    (define-key k4-mode-map [menu-bar] (make-sparse-keymap))
    (define-key k4-mode-map [menu-bar k4]
      (cons "K4" map))
    (define-key map [comment-region] '("Comment Out Region" . comment-region))
    (define-key map [indent-region] '("Indent Region" . indent-region))
    (define-key map [indent-line] '("Indent Line" . lisp-indent-line))
    (put 'comment-region 'menu-enable 'mark-active)
    (put 'indent-region 'menu-enable 'mark-active)))


;;;###autoload
(defun k4-mode ()
  "Major mode for editing K4 code.
Editing commands are similar to those of lisp-mode.

Commands:
Delete converts tabs to spaces as it moves back.
Blank lines separate paragraphs.  Semicolons start comments.
\\{k4-mode-map}
Entry to this mode calls the value of k4-mode-hook
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (k4-mode-initialize)
  (k4-mode-variables)
  (run-hooks 'k4-mode-hook))

(defun k4-mode-initialize ()
  (use-local-map k4-mode-map)
  (setq major-mode 'k4-mode)
  (setq mode-name "K4"))

(defgroup k4 nil
  "Editing K4 code"
  :group 'lisp)


(defvar calculate-lisp-indent-last-sexp)

;; Copied from lisp-indent-function, but with gets of
;; k4-indent-{function,hook}.
(defun k4-indent-function (indent-point state)
  (let ((normal-indent (current-column)))
    (goto-char (1+ (elt state 1)))
    (parse-partial-sexp (point) calculate-lisp-indent-last-sexp 0 t)
    (if (and (elt state 2)
             (not (looking-at "\\sw\\|\\s_")))
        ;; car of form doesn't seem to be a a symbol
        (progn
          (if (not (> (save-excursion (forward-line 1) (point))
                      calculate-lisp-indent-last-sexp))
              (progn (goto-char calculate-lisp-indent-last-sexp)
                     (beginning-of-line)
                     (parse-partial-sexp (point)
					 calculate-lisp-indent-last-sexp 0 t)))
          ;; Indent under the list or under the first sexp on the same
          ;; line as calculate-lisp-indent-last-sexp.  Note that first
          ;; thing on that line has to be complete sexp since we are
          ;; inside the innermost containing sexp.
          (backward-prefix-chars)
          (current-column))
      (let ((function (buffer-substring (point)
					(progn (forward-sexp 1) (point))))
	    method)
	(setq method (or (get (intern-soft function) 'k4-indent-function)
			 (get (intern-soft function) 'k4-indent-hook)))
	(cond ((or (eq method 'defun)
		   (and (null method)
			(> (length function) 3)
			(string-match "\\`def" function)))
	       (lisp-indent-defform state indent-point))
	      ((integerp method)
	       (lisp-indent-specform method state
				     indent-point normal-indent))
	      (method
		(funcall method state indent-point)))))))


;;; Let is different in K4

(defun would-be-symbol (string)
  (not (string-equal (substring string 0 1) "(")))

(defun next-sexp-as-string ()
  ;; Assumes that protected by a save-excursion
  (forward-sexp 1)
  (let ((the-end (point)))
    (backward-sexp 1)
    (buffer-substring (point) the-end)))

;; This is correct but too slow.
;; The one below works almost always.
;;(defun k4-let-indent (state indent-point)
;;  (if (would-be-symbol (next-sexp-as-string))
;;      (lisp-indent-specform 2 state indent-point (current-column))
;;    (lisp-indent-specform 1 state indent-point (current-column))))

(defun k4-let-indent (state indent-point)
  (skip-chars-forward " \t")
  (if (looking-at "[-a-zA-Z0-9+*/?!@$%^&_:~]")
      (lisp-indent-specform 2 state indent-point (current-column))
      (lisp-indent-specform 1 state indent-point (current-column))))

;; (put 'begin 'k4-indent-function 0), say, causes begin to be indented
;; like defun if the first form is placed on the next line, otherwise
;; it is indented like any other form (i.e. forms line up under first).

(put 'begin 'k4-indent-function 0)
(put 'case 'k4-indent-function 1)
(put 'lambda 'k4-indent-function 1)
(put 'let 'k4-indent-function 1)  ;; doesn't work! 'k4-let-indent)
(put 'let* 'k4-indent-function 1)
(put 'letrec 'k4-indent-function 1)


;; Font lock for k4
(defconst k4-font-lock-keywords
  (eval-when-compile
    (list
     (list (concat 
	    "(" 
	    (regexp-opt '("define" "define*"
			  "define-method" "define-class"
			  "define-attribut" "define-variable") t)
	    "\\>"
	    ;; Any whitespace and declared object.
	    "[ \t]+(?"
	    "\\([']*\\(\\sw\\|\\s_\\)+\\)?")
	   '(1 font-lock-keyword-face)
	   '(4 (cond ((match-beginning 2) font-lock-variable-name-face)
		     ((match-beginning 1) font-lock-function-name-face)
		     (t font-lock-type-face))
	       nil t))
     (list (concat 
	    "("
	    (regexp-opt '("and" "begin" "or" "case" "cond" "else"
			  "for-each" "map" "if" "let" "let*" "letrec"
			  "lambda" "eval" "error" "quote") t)
	    "\\>")
	   '(1 font-lock-keyword-face nil t))
     '("[']*@[0-9]+" . font-lock-reference-face)
     ;; Scheme `:' keywords as references.
     '("\\<:\\sw+\\>" . font-lock-type-face)
     ))
  "Default expressions to highlight in K4 modes.")

(provide 'k4)

;;; k4.el ends here
