/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <ctype.h>
#include <locale.h>

#include "kdbs.h"

#ifdef DISTRIBUTED_DB

#include "smserver.h"		/* autoincludes net.h */
#include "error.h"

#define KSMDPORT 8123

int exit_status = 0;
unsigned short port = 0;	/* the port */
char *host = NULL;		/* the host */
bool errormsgs = true;		/* read the errormessages? */
bool trace = false;		/* trace disable by default */

/* prototypes */

void showhelp (void);
void Welcome();
void handle_exit(int exit_status);


/* ----------------------------------------------------------------------
   SIGNAL-Handler
   ---------------------------------------------------------------------- */
void
sigpipe_handler (int sig)
{
  fprintf (stderr, "ksmd: connection brocken - daemon halted\n");
  raise (sig);
}

void
sigint_handler (int sig)
{
  fprintf (stderr, "ksmd: ignored SIGINT\n");
  signal (SIGINT, sigint_handler);
}

void
sigterm_handler (int sig)
{
  fprintf (stderr, "ksmd: received SIGTERM signal, quitting. Bye.\n");
  raise (sig);
}

void
sigquit_handler (int sig)
{
  fprintf (stderr, "ksmd: ignored SIGQUIT\n");
  signal (SIGQUIT, sigquit_handler);
}


/* ----------------------------------------------------------------------
   main, init and startup routines 
   ---------------------------------------------------------------------- */
int
main(int argc, char **argv)
{
  char
    *portstring = NULL;		/* port argument */
  char *tail;
  char c;
  int retval;
  char *configfilename = NULL;
  unsigned short portv = 0;

  /* set up signal-handlers */
  
  signal (SIGPIPE, sigpipe_handler); /* broken pipe */
  signal (SIGINT, sigint_handler); /* interrupt (close) */
  signal (SIGQUIT, sigquit_handler); /* interrupt (close) */
  signal (SIGTERM, sigterm_handler); /* interrupt (close) */

  setlocale (LC_MESSAGES, "");
  textdomain (PACKAGE);

  /* argumentpassing */
  
  while ((c = getopt (argc, argv, "hp:f:E")) != -1) { /* get options */
    switch (c) {
    case 'h':			/* help */
      showhelp();
      exit (EXIT_SUCCESS);
    case 'p':			/* port to establish to */
      portstring = optarg;
      portv = strtol (portstring, &tail, 0);
      break;
    case 'E':
      errormsgs = false;
      break;
    case 'f':
      configfilename = optarg;
      break;
    case '?':			/* wrong option */
      if (isprint (optopt)) { 
	fprintf(stderr, "Unknown option -%c\n", optopt);
	exit (EXIT_FAILURE);
      }
      else {
	fprintf (stderr, "Unknown option character `\\x%x'.", optopt);
	exit (EXIT_FAILURE);
      }
    default:			/* other options */
      exit (EXIT_FAILURE);
    }
  }
  
  read_configuration (configfilename);
  
  if (portv > 0) {  
    port = portv;
  }
  if (port == 0)
    port = KSMDPORT;

  if (errormsgs) {
    readerrorconf (descfile);	/* read the errormessages */
  }
  
  init_lists ();
  
  read_database_conf ();

  retval = storagecycle (port);	/* start to listen for clients */
  
  if (retval < 0) {
    fprintf (stderr, "ksmd (errormsg): %s\n", geterrmsg ());
    exit (EXIT_FAILURE);
  }
  fprintf (stderr, "ksmd (errormsg): %s\n", geterrmsg ());
  exit (EXIT_SUCCESS);
}




/*------------------------------------------------------------
  General routines
  ------------------------------------------------------------*/	
/* just for a clean exit */
void
handle_exit(int exit_status)
{
  exit (exit_status);
}

void
Welcome()
{
  fprintf(stdout,
	  "ksmd - kaenguru database storage manager daemon version 0.1
This is ksmd - the storage manager daemon for the kaenguru database system
Use ksmd -h for detailed help on syntax and use.
"
	  );
}


void
showhelp (void)
{
  fprintf(stdout, "

ksmd is the storage manager daemon which adds distributed database
extensions to the kaenguru database system.  It establishs an adress in the
internet domain listening for clients (servers) to connect.  If a server
logs in it reads the requestdata from it and will do the requested action
on a local database (read, write, etc.)


General usage:
  ksmd [-h] [-p<port>] [-E] &

Options:
  -h         This help
  -p<port>   The port, to which the server will listen
  -E         don't read the error-messages on startup.  Once a system is
             running secure you won't need the errormessage online.  The
             daemon reports - if a runtime error occurs - only a number, which
             you may look up in the error.conf file.  This feature is for
             memory convenience only.

  NOTE: Remember to start ksmd with the bash command '&' in the background.
  ksmd still isn't able to to do this by itself.  But remember that ksmd is
  surely not bugfree and still prints a lot of verbose data and logs to the
  screen (tty).

"
	 );    
}

#else  /* if DISTRIBUTED_DB */

int
main()
{
  printf (_("Remote storage server not supported\n"));
  exit (2);
}
#endif
