/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>

#include "kdbs.h"

#ifdef DISTRIBUTED_DB

#include "smserver.h"
#include "error.h"

#include "sm.h"			/* for storage manager defaults */

char basepath[FILENAME_MAX] = "/var/lib/kaenguru/base";
char systempath[FILENAME_MAX] = "/usr/lib/kaenguru";
char descfile[FILENAME_MAX] = "/usr/lib/kaenguru/etc/error.desc";

#define DATABASEFILE  "databases"

const char fielddelim[] = ":";

Dballoc *dbases;

int
init_lists ()
{
  dbases = NULL;
  return 0;
}

/* ----------------------------------------------------------------------
   Read the database configuration data
   ---------------------------------------------------------------------- */
Dballoc *
alloc_dballoc ()
{
  Dballoc *nde = (Dballoc *) malloc (sizeof (Dballoc));
  nde->name = NULL;
  nde->path = NULL;
  nde->comment = NULL;
  nde->next = NULL;
  return nde;
}

int
read_database_conf ()
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *token;
  Dballoc *deptr, *nde;
  
  sprintf (tmp, "%s/conf/%s", systempath, DATABASEFILE);
  stream = fopen (tmp, "r");
  
  if (!stream)
    smd_seterr (SMDEDBCONFOP);
  
  deptr = dbases;
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	smd_seterr (SMDEDBCONFR);
    }

    nde = alloc_dballoc ();
    
    token = strtok (buffer, fielddelim); /* keywords are divided by : */
    if (!token)
      smd_seterr (SMDEDBCONFNAME);
    nde->name = strdup (token);	/* get the database name */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      smd_seterr (SMDEDBCONFPATH);
    if (strchr (token, '<')) 
      smd_seterr (SMDEDBCONFREC); /* no recursive remote path definition! */
    nde->path = strdup (token);	/* the path */
    
    token = strtok (NULL, fielddelim);
    if (token) {
      if (strlen (token) > 0) {
	nde->comment = strdup (token); /* the comment */
      }
    }
    
    if (deptr) {
      deptr->next = nde;
      deptr = deptr->next;
    }
    else
      dbases = deptr = nde;
  }

eofhd:
  if (fclose (stream) != 0)
    smd_seterr (SMDEDBCONFCL);
  return 1;
  
errhd:
  fprintf (stderr, "ksmd: %s\n", slisperr (lisperr));
  exit (EXIT_FAILURE);
}

Dballoc *
getdbdata_byname (char *dbname)
{
  Dballoc *deptr = dbases;
  while (deptr) {
    if (strcmp (dbname, deptr->name) == 0)
      return deptr;
    deptr = deptr->next;
  }
  return NULL;
}


/* ----------------------------------------------------------------------
   GENERAL CONFIG READ FUNCTION
   ---------------------------------------------------------------------- */
#define CONFIGFILE "/etc/kaenguru.conf"

#define get_boolean(TOKEN, VRF, DEFAULT)	\
{						\
  TOKEN = strtok (NULL, " \r\t\n");		\
  if (TOKEN) {					\
    if (strcasecmp (TOKEN, "yes") == 0)		\
      VRF = true;				\
    else if (strcasecmp (TOKEN, "no") == 0)	\
      VRF = false;				\
    else					\
      smd_seterr (SMDEYESNO);			\
  }						\
  else 						\
    VRF = DEFAULT;				\
}

int
read_configuration (char *name)
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *tail, *token;
  
  if (name) {
    sprintf (tmp, "%s", name);
  }
  else {
    sprintf (tmp, "%s", CONFIGFILE);
  }
  stream = fopen (tmp, "r");
  
  if (!stream)
    smd_seterr (SMDECONFIG);
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	smd_seterr (SMDECONFIGR);
    }
    
    token = strtok (buffer, ": \n");

    if (token) {

      if (strcmp (token, "basepath") == 0) {
	token = strtok(NULL, " \r\t\n");
	if (token)
	  strcpy (basepath, token);
      }
      else if (strcmp (token, "systempath") == 0) {
	token = strtok(NULL, " \r\t\n");
	if (token)
	  strcpy (systempath, token);
      }
      else if (strcmp (token, "error-desc") == 0) {
	token = strtok(NULL, " \r\t\n");
	if (token)
	  strcpy (descfile, token);
      }
      else if (strcmp (token, "sm-msgs") == 0) {
	get_boolean (token, errormsgs, true);
      }
      else if (strcmp (token, "forcesync") == 0) {
	get_boolean (token, forcesync, true);
      }
      else if (strcmp (token, "sm-port") == 0) {
	token = strtok(NULL, " \r\t\n");
	if (token)
	  port = strtol (token, &tail, 0);
	if (!token ||
	    (port == 0))
	  smd_seterr (SMDECONST);
      }
      else if (strcmp (token, "sm-log") == 0) {
	get_boolean (token, trace, true);
      }
      else {
	if (*token == '#') {}	/* comment */
/* 	else  */
/* 	  smd_seterr (SMDEKEYWORD); */
      }
    }
  }
  
eofhd:
  if (fclose (stream) != 0)
    smd_seterr (SMDECONFIGCL);
  return 1;
  
errhd:
  fprintf (stderr, "ksmd: %s\n", slisperr (lisperr));
  exit (EXIT_FAILURE);
}

#endif
