/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#if defined HAVE_SYS_TIME_H
# include <sys/time.h>
#else
# include <time.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <ctype.h>

#include "kdbd.h"
#include "lock.h"
#include "conn.h"		/* connectionlists */

/* globale defines */
#define SELECTTIMEVAL        0	/* 0 seconds */

#define spacedelim " \n\t\r"

#define KILL_CODE         0
#define REREAD_CODE       1
#define CLOSE_CODE        2
#define LOGIN_CODE        3
#define SSTARTED_CODE     4
#define VERIFYDB_CODE     5
#define BEGINTRANS_CODE   6
#define ADDTRANS_CODE     7
#define ENDTRANS_CODE     8
#define CHANGETRANS_CODE  9
#define TRACE_CODE       10
#define USRINGRP_CODE    11
#define SUPERVISOR_CODE  12
#define LISTCONN_CODE    13

int conn_count = 0;

/* globale Typen, nur fr listen.c interessant */
typedef struct {
  char *name;
  int code;
} Searchkey;

/* GLOBALE VARIABLEN */
bool some_server_died = false;
Clist *clist = NULL;

void
showlocklist ()
{
  int i, j;

  if (!verbose)
    return;

  if (lockreqlistcount > 0)
    fprintf(tracefile, "LOCKLIST:\n");
  for (i = 0; i < lockreqlistcount; i++) {
    fprintf(tracefile, "%d. ID %d Server %d NrRes %d Compl %d RES: ",
	    i, 
	    lockreqlist[i]->lockid,
	    lockreqlist[i]->serverid,
	    lockreqlist[i]->maxrid,
	    iscomplete (i)); 
    for (j = 0; j < lockreqlist[i]->maxrid; j++) {
      fprintf(tracefile, "[S%d %c%c:O%ld-%d] ",
	      lockreqlist[i]->reslist[j]->status,
	      lockreqlist[i]->reslist[j]->reqstatus,
	      lockreqlist[i]->reslist[j]->rid.type,
	      lockreqlist[i]->reslist[j]->rid.eid,
	      lockreqlist[i]->reslist[j]->rid.dbid);
      
    }
    fprintf(tracefile, "\n");
  }
  
  for (i = 0; i < MAXRESOURCES; i++)
    if (resourceslist[i].status != lock_NULL) 
      fprintf(tracefile, "<ID%d %c%c:O%ld-%d> ",
	      resourceslist[i].lockid,
	      resourceslist[i].status,
	      resourceslist[i].rid.type,
	      resourceslist[i].rid.eid,
	      resourceslist[i].rid.dbid);
  if (lockreqlistcount > 0)
    fprintf(tracefile, "\n");
}

/* geht alle Server durch, und berprft, ob sie mittlerweile alle
 * Resourcen zugeteilt bekommen haben.  Sollte das sein, so erhalten
 * diese Server eine NW_OK-Mitteilung)
 */
int
check_for_complete_servers ()
{
  int i;
  bool some_locks_changed;

  some_locks_changed = updatelocklist (); /* Lockliste aktualisieren */

  if (lockchanged) {
    showlocklist ();
    lockchanged = false;
  }
  
  if (some_locks_changed) {	/* check only, if something changed */
    for (i = 0; i < lockreqlistcount; i++) {
      if (iscomplete (i)) {
	if (!lockreqlist[i]->announced) {
	  Clist *cl = get_connection (clist, lockreqlist[i]->serverid);
	  if (cl) {
	    int retval;
	    
	    retval = writesockf (cl->serversock, 
				 "%s %c%s %d", KCP_TRANS, KCP_OKC, KCP_OK,
				 lockreqlist[i]->lockid);
	    if (retval == 0) {
	      /* ok, mark this serversock as faded away.  In a second step
                 we will clean the list up.  Zero-Serversockets will be
                 removed by cleanup_clist below */
	      cl->status = C_ZOMBIE;
	      some_server_died = true;
	    }
	    else if (retval < 0) {
	      goto errhd;		/* to the server: go! */
	    }
	    lockreqlist[i]->announced = true;
	  }
	  else {
	    PRINTERR (_("Unknown serverid."));
	  }
	}
      }
    }
  }
  return 1;

 errhd:
  return 0;
}


/* schickt allen Servern eine EXIT-Message */
int
stop_all_servers ()
{
  Clist *cl = clist;
  
  while (cl) {
    if (cl->status == C_OK) {
      /* here we ignore simply any error message. */
      if (writesockf (cl->serversock, "%s", KCP_HALT) > 0) {
	close (cl->serversock);
      }
    }
    cl = cl->next;
  }
  free_total_clist (&clist);
  return 1;
}

int
add_lock_data (char *locklist, int lockid)
{
  Oid oid;
  Dbid dbid;
  char *p, resstr[16], *tail, locktype, lockstatus;
  Resource resource;
  
  p = locklist;

  for (; *p ; ) {
    lockstatus = *p;	/* SHARED, LOCKED */
    
    p++;		/* next char: lock-resource */
    locktype = *p; /* NEXTEID, INDEX, DATA, OPENLIST */
    
    p++;		/* next char: database*/
    strcpy (resstr, "0X"); 
    strncat (resstr, p, 2);
    resstr[4] = '\0';
    dbid = strtol (resstr, &tail, 0);
    p += 2;
    
    p++;		/* next char: number of resource */
    strcpy (resstr, "0X"); /* alles hex! */
    strncat (resstr, p, 8); /* get IDno (8 Zeichen lang) */
    resstr[10] = '\0';
    oid = strtol (resstr, &tail, 0);
    p += 8;
    
    setresource (&resource, locktype, oid, dbid);
    addtolock (lockid, lockstatus, &resource);
  }
  return 1;
}

int
change_lock_data (char *locklist, int lockid)
{
  Oid oid;
  Dbid dbid;
  char *p, resstr[16], *tail, locktype, lockstatus;
  Resource resource;
  
  p = locklist;
  
  for (; *p ; ) {
    lockstatus = *p;	/* SHARED, LOCKED */
    
    p++;		/* next char: lock-resource */
    locktype = *p; /* NEXTEID, INDEX, DATA, OPENLIST */
    
    p++;		/* next char: database*/
    strcpy (resstr, "0X"); 
    strncat (resstr, p, 2);
    resstr[4] = '\0';
    dbid = strtol (resstr, &tail, 0);
    p += 2;
    
    p++;		/* next char: number of resource */
    strcpy (resstr, "0X"); /* alles hex! */
    strncat (resstr, p, 8); /* get IDno (8 Zeichen lang) */
    resstr[10] = '\0';
    oid = strtol (resstr, &tail, 0);
    p += 8;
    
    setresource (&resource, locktype, oid, dbid);
    changelock (lockid, lockstatus, &resource);
  }
  return 1;
}

/* rereads the basic configuration data */
void
reread_conf_data ()
{
  free_lists ();		/* init the user and group lists */
  read_user_conf ();		/* users file */
  read_group_conf ();		/* groups file */
  read_auth_conf ();		/* authority file */
  read_database_conf ();	/* databases file */
  showlists ();
}

/* lists all connected servers (clients) to the client (including itself) */
int
list_connection_data (int sock, Clist *clist)
{
  Clist *cl = clist;
  int clc = 0, retval;

  while (cl) {
    clc++;
    cl = cl->next;
  }

  retval = writesockf (sock, "%c%s %d connections", KCP_OKC, KCP_OK, clc);
  if (retval < 0)
    goto errhd;
  if (retval == 0)
    goto sigpipehd;
  
  cl = clist;
  while (cl) {
    retval = writesockf (sock, "%c %d %s %d %s@%s:%d", KCP_NEXTC, 
			 cl->connid,
			 cl->connkey,
			 cl->uid,
			 getusrnme (cl->uid),
			 inet_ntoa (cl->client.sin_addr),
			 ntohs (cl->client.sin_port));
    if (retval == 0)
      goto sigpipehd;
    if (retval < 0)
      goto errhd;
    
    cl = cl->next;
  }

  retval = writesockf (sock, "%c ok.", KCP_DONEC);
  if (retval < 0)
    goto errhd;
  if (retval == 0)
    goto sigpipehd;
  
  return 1;
  
 sigpipehd:
  return 0;
  
 errhd:
  return -1;
}

/* ----------------------------------------------------------------------
   Lets check if our usr is authorized
   ---------------------------------------------------------------------- */
Uid
authorize_usr (int socket, char *logname, char *domain, char *keystr,
	       char **errmsg)
{
  Uid uid;
  int keyid;
  KdbKeyStruc key;
  
  *errmsg = NULL;

  getusrauth (logname, domain, &uid, &keyid);
  if (uid == UNKNOWNUSR) {
    *errmsg = _("unknown usr");
    return UNKNOWNUSR;
  }
  
  if (keyid > 0) {
    if (read_authorization_key (&key, keyid) < 0) {
      *errmsg = errstr;		/* errstr set by read_authorization_key */
      return UNKNOWNUSR;
    }
    
    if (strcmp (key.key, keystr) == 0) {
      if (!key.locked) {
	Date res;
	Date now = time (&res);
	
	/* ok, we've found the key, so we have to check for max_logins,
           etc.! */
	
	if (key.validity > 0) {
	  /* Check for a valid time, that is for a time span (validity
	     counts in days, that are 86400 seconds */
	  if (key.key_created + key.validity * 86400 < now) {
	    *errmsg = _("invalid key: key to old");
	    goto lock_key;
	  }
	}
	if (key.max_logins > 0) {
	  /* check for a maximum count of logins before the key will become
	     invalid */
	  if (key.login_count > key.max_logins) {
	    *errmsg = _("maximum number of logins reached");
	    goto lock_key;
	  }
	}

	key.login_count++;
	key.last_login = now;

	write_authorization_key (&key, keyid);
	
	return uid;
      }
      *errmsg = _("login is locked");
      return UNKNOWNUSR;
    }
    *errmsg = _("wrong key");
    return UNKNOWNUSR;
  }
  
  /* it seems the user mustn't have a key */
  return uid;

 lock_key:
  key.locked = true;
  write_authorization_key (&key, keyid);

  return UNKNOWNUSR;
}


/* ------------------------------------------------------------
   Zentrale Serverroutine
   ------------------------------------------------------------*/
void
log_request (char *req, Clist *cl, int connectid)
{
  tracef ("kdbd: %s request from ID %d (%s@%s:%d)\n",
	  req, connectid, getusrnme (cl->uid),
	  inet_ntoa (cl->client.sin_addr),
	  ntohs (cl->client.sin_port));
}

void
log_auth_request (char *req, Clist *cl, int connectid)
{
  tracef ("kdbd: unauthorized %s request from ID %d (%s@%s:%d)\n",
	  req, connectid, getusrnme (cl->uid),
	  inet_ntoa (cl->client.sin_addr),
	  ntohs (cl->client.sin_port));
}

int
listencycle (unsigned short port)
{
  struct sockaddr_in clientname;
  size_t size;
  int
    sock,			/* listening socket */
    testsock,			/* testing socket */
    newsock,			/* the accepted new socket */
    channel;
  fd_set
    active_fd_set, read_fd_set;	/* for select */
  int
    selectval, retval;		/* return values */
  struct timeval  
    timeout;			/* fr die timeoutroutine */

  int bufsize = 0 /* 1024 */;
  char *buffer = NULL; /* (char *) malloc (bufsize); */
  char *token;

#define getserverid(TOKEN)			\
({						\
  int _retval = 0;				\
  TOKEN = strtok (NULL, spacedelim);		\
  if (TOKEN) {					\
    char *_tail;				\
    _retval = strtol (TOKEN, &_tail, 0);	\
  }						\
  _retval;					\
})

#define checkserverkey(TOKEN, CONNID, CORRECT, CL)	\
({							\
  CORRECT = false;					\
  TOKEN = strtok (NULL, " \n\r");			\
  if (TOKEN) {						\
    CL = get_connection (clist, CONNID);		\
    if (CL) {						\
      if (checkconnkey (CL, CONNID, TOKEN))		\
	CORRECT = true;					\
    }							\
  }							\
})


  /* Create the socket and set it up to accept connections. */
  tracef ("kdbd: Connection established on port %d\n", port);

  sock = makeSocket (port);	/* jetzt socket einrichten */
  
  if (listen (sock, 1) < 0)
    GOERR (_("listen error"));
  
  FD_ZERO (&active_fd_set);	/* init activ-sockets for select */
  FD_SET (sock, &active_fd_set);
  
  while (!eodaemon) {		/* solange wie server activ ist */
    
    read_fd_set = active_fd_set;
    
    /* diese seltsame Schleife um Signale abzufangen */
  selectjump:
    if (lockreqlistcount > 0) {	/* locks angemeldet? */
      timeout.tv_sec = 0;	/* dann sofort berprfen */
      timeout.tv_usec = 500;	/* test 2 times a second */
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, &timeout);
    }
    else {			/* nein? check for eodaemon! */
      timeout.tv_sec = 5;	/*  */
      timeout.tv_usec = 0;
      selectval = select (FD_SETSIZE, &read_fd_set, NULL, NULL, &timeout);
    }
    if (selectval < 0) {	/* signal EINTR caught? */
      if (errno == EINTR) 
	goto selectjump;
      GOERR (_("select error"));
    }

    if (lockreqlistcount > 0) {	/* some locks activ? */
      check_for_complete_servers (); /* check for complete servers! */
    }
    
    for (testsock = 0; testsock < FD_SETSIZE; ++testsock) {
      if (FD_ISSET (testsock, &read_fd_set)) { /* activ socket? */

	/* ----- A NEW CONNECTION COMING IN ---------------------------- */
	if (testsock == sock) {	               /* request on serversocket */
	  
	  size = sizeof (clientname);
	  
	  if ((newsock = accept (sock, (struct sockaddr *) &clientname,
				 &size)) < 0)
	    GOERR (_("accept error"));
	  
	  tracef ("kdbd: connect from host %s, port %d.\n",
		  inet_ntoa (clientname.sin_addr),
		  ntohs (clientname.sin_port));
	  FD_SET (newsock, &active_fd_set); /* Verbindung steht */
	  
	}

	/* ----- DATA COMING FROM EXISTING CONNECTIONS ------------------- */
	else {			/* receiving data */
	  
	  retval = read_from_sock (testsock, &buffer, &bufsize);

	  if (retval < 0) {	/* Socket is brocken */
	    tracef ("kdbd: connection to client broken\n");
	    
	    close (testsock);
	    FD_CLR (testsock, &active_fd_set);
	    break;
	  }
	  else if (retval == 0) { /* end of file! Socket closed? */
	    tracef ("kdbd: client closed connection closed\n");
	    close (testsock);
	    FD_CLR (testsock, &active_fd_set);
	    break;
	  }
	  
	  token = strtok (buffer, spacedelim);

	  if (token) {
	    Searchkey skeys[] = {
	      {KCP_KILL, KILL_CODE},
	      {KCP_REREAD, REREAD_CODE},
	      {KCP_CLOSE, CLOSE_CODE},
	      {KCP_LOGIN, LOGIN_CODE},
	      {KCP_SSTARTED, SSTARTED_CODE},
	      {KCP_VERIFYDB, VERIFYDB_CODE},
	      {KCP_BEGINTRANS, BEGINTRANS_CODE},
	      {KCP_ADDTRANS, ADDTRANS_CODE},
	      {KCP_ENDTRANS, ENDTRANS_CODE},
	      {KCP_CHANGETRANS, CHANGETRANS_CODE},
	      {KCP_TRACE, TRACE_CODE},
	      {KCP_USRINGRP, USRINGRP_CODE},
	      {KCP_SUPERVISOR, SUPERVISOR_CODE},
	      {KCP_LISTCONN, LISTCONN_CODE},
	      {NULL, 0}
	    };
	    int i;
	    int connid, keyok;
	    Clist *cl;

	    for (i = 0; skeys[i].name; i++) {
	      if (strcasecmp (token, skeys[i].name) == 0)
		goto keyfound;
	    }
	    goto keynotfound;
	    
	  keyfound:
	    switch (skeys[i].code) {
	    case KILL_CODE:
	      /* ----- KILL ----------------------------------------------- */
	      /* The explicit kill request for the daemin */

	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		log_request ("kill", cl, connid);

		/* here we may ignore every error code since we will be
                   stopped a second later or so */
		writesockf (testsock, "%c%s", KCP_OKC, KCP_OK);
		stop_all_servers ();
		goto eodaemonhd;
	      }
	      else
		log_auth_request ("kill", cl, connid);
	      break;

	    case REREAD_CODE:
	      /* ----- REREAD --------------------------------------------- */
	      /* daemon: reread your configuration files */

	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		log_request ("reread", cl, connid);
		reread_conf_data ();
		
		if ((retval = writesockf (testsock, 
					  "%c%s", KCP_OKC, KCP_OK)) == 0) {
		  tracef ("kdbd: lost connection in reread\n");
		  close (testsock);
		  FD_CLR (testsock, &active_fd_set);
		}
		else if (retval < 0)
		  goto errhd;
	      }
	      else
		log_auth_request ("reread", cl, connid);
	      break;

	    case LISTCONN_CODE:
	      /* ----- LISTCONN ------------------------------------------- */
	      /* daemon: list all connected clients */

	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		log_request ("listconn", cl, connid);
		if (list_connection_data (testsock, clist) == 0) {
		  tracef ("kdbd: lost connection in listconn\n");
		  close (testsock);
		  FD_CLR (testsock, &active_fd_set);
		}
	      }
	      else
		log_auth_request ("listconn", cl, connid);
	      break;

	    case CLOSE_CODE:
	      /* ----- CLOSE ---------------------------------------------- */
	      /* The close request of a server. */
	      connid = getserverid (token); /* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		if (cl->status != C_OK) {
		  writesockf (cl->clientsock, "%c%s", KCP_ERRC, KCP_ERR);
		}
		log_request ("close", cl, connid);
		deleteallserverlocks (connid); /* transaction erst beenden */

		check_for_complete_servers (); /* grant others permissions */

		if (!cl->supervisor && (conn_count > 0))
		  conn_count--;
		
		free_connid (&clist, connid);

		close (testsock);
		FD_CLR (testsock, &active_fd_set);
	      }
	      else
		log_auth_request ("close", cl, connid);
	      break;

	    case LOGIN_CODE:
	    case SUPERVISOR_CODE:
	      /* ----- LOGIN Command --------------------------------------- */
	      /* This is the login commando for the client. */

	      getpeername (testsock, (struct sockaddr *) &clientname, &size);
	      
	      cl = get_new_conn ();
	      
	      if (cl) { /* still a connection free? */
		tracef ("kdbd: client login req\n");
		
		token = strtok (NULL, spacedelim); /* get a login name? */
		if (!token) {
		  tracef ("kdbd: anonymous login not accepted\n");
		  /* since testsock is close a moment later we may ignore
                     every error here */
		  writesockf (testsock,
			      _("%c%s anonymous login not accepted\n"),
			      KCP_ERRC, KCP_ERR);
		  close (testsock); /* close client socket */
		  FD_CLR (testsock, &active_fd_set); /* del from select */

		  free_conn (cl);
		}
		else {
		  char *logname = token;
		  char *auth_errstr;

		  token = strtok (NULL, "");

		  if (!token) {
		    tracef ("kdbd: no key in login message\n");
		    /* since testsock is closed a moment later we may
                       ignore every error here */
		    writesockf (testsock, 
				_("%c%s no login key\n"),
				KCP_ERRC, KCP_ERR);
		    close (testsock);
		    FD_CLR (testsock, &active_fd_set); /* del from select */

		    free_conn (cl);
		  }
		  else {
		    Uid uid = authorize_usr (testsock, 
					     logname, 
					     inet_ntoa(clientname.sin_addr),
					     token,
					     &auth_errstr);
		  
		    /* erweitere hier: Je nach Konfiguration knnen
		       unbekannte Usr zu Gsten erklrt werden! */
		    if (uid == UNKNOWNUSR) {
		      tracef ("kdbd: %s", auth_errstr);
		      /* since testsock is closed a moment later we may
                         ignore every error here */
		      writesockf (testsock, _("%c%s unknown user\n"), 
				  KCP_ERRC, KCP_ERR);
		      close (testsock); /* close client socket */
		      FD_CLR (testsock, &active_fd_set); /* del from select */
		      
		      free_conn (cl);
		    }
		    else {
		      Uentry *uentry;
		      
		      /* Set the connection list item data */
		      cl->status = C_OK;
		      cl->clientsock = testsock;
		      cl->serversock = 0; 
		      memcpy (&cl->client, &clientname, size); 
		      cl->uid = uid;
		      
		      switch (skeys[i].code) {
		      case LOGIN_CODE:
			if (conn_count < maxconn) {
			  cl->supervisor = false;

			  /* now insert data to the connection list */
			  insert_conn (&clist, cl);
			  /* 			showconns (clist); */
			  
			  /* now fork the server */
			  channel = startserver (testsock);
			
			  /* save the socket to the server */
			  cl->serversock = channel;
			  
			  uentry = getusrdata (uid);
			  
			  retval = 
			    writesockf (channel, /* talk init to backend */
					"%s %d %s %s %d %d\n", 
					KCP_HELO,
					cl->connid, 
					cl->connkey,
					uentry->name, /* the user name */
					uid, /* the user id */
					uentry->gid /* the default group id */
					);
			  if (retval == 0) {
			    /* shit, the channel faded away before we was
                               done, set a mark and serversock to 0, so we
                               may clean up clist afterwards! */
			    cl->status = C_ZOMBIE;
			    some_server_died = true;
			    close (testsock);
			    FD_CLR (testsock, &active_fd_set);
			  }
			  else if (retval > 0) {
			    /* close client socket in daemon, its reserved
                               for the backend */
			    close (testsock); 
			    FD_CLR (testsock, &active_fd_set);
			    FD_SET (channel, &active_fd_set); /* connect ok */
			  
			    conn_count++; /* inc connection counter */
			  }
			  else	/* retval < 0 */
			    goto errhd;
			}
			else {
			  free_conn (cl);
			  goto no_further_connections;
			}
			break;

		      case SUPERVISOR_CODE:
			if (uid == U_ROOT) {
			  cl->supervisor = true;
			  channel = cl->serversock = testsock;
			  
			  /* now insert data to the connection list */
			  insert_conn (&clist, cl);

			  uentry = getusrdata (uid);
		      
			  retval = 
			    writesockf (channel, /* talk init to backend */
					"%s %d %s %s %d %d\n", 
					KCP_HELO,
					cl->connid, 
					cl->connkey,
					uentry->name, /* the user name */
					uid, /* the user id */
					uentry->gid); /* default group id */
			  if (retval == 0) {
			    /* the fuck channel went away before we're
                               done! */
			    some_server_died = true;
			    cl->status = C_ZOMBIE;
			    FD_CLR (testsock, &active_fd_set);
			  }
			  else if (retval < 0)
			    goto errhd;
			}
			else {
			  tracef ("kdbd: client login refused -- no root\n");
			  /* since testsock is closed in a moment we may
                             ignore every error here */
			  writesockf (testsock, _("%c%s no permission\n"), 
				      KCP_ERRC, KCP_ERR);
			  close (testsock);
			  FD_CLR (testsock, &active_fd_set);
			  
			  free_conn (cl);
			}
			break;
		      }
		    }
		  }
		}
	      }
	      else {
     no_further_connections:
		tracef ("kdbd: client login refused - no free connections\n");
		
		/* since testsock is closed in a moment we may ignore every
                   error here */
		writesockf (testsock, _("%c%s no free connections\n"),
			    KCP_ERRC, KCP_ERR);
		close (testsock); /* close client socket */
		FD_CLR (testsock, &active_fd_set); /* del from select */
	      }
	      break;

	    case SSTARTED_CODE:
	      /* ----- SERVER command -------------------------------- */
	      /* with this commando the server starts his request to the
		 daemon. */

	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok)
		tracef ("kdbd: Server started on ID %d\n", connid);
	      else
		log_auth_request ("sstarted", cl, connid);
	      break;

	    case VERIFYDB_CODE:
	      /* ----- VERIFYDB command -------------------------------- */
	      /* with this commando the backend gets the accessrights for a
		 database. Uses two arguments: VERIFYDB id key UID DBNAME */

	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		char *p;
		Uid uid;
		Dbentry *dbentry;
		char *tail;
		
		log_request ("verifydb", cl, connid);
		
		p = strtok (NULL, spacedelim);
		if (p) {
		  uid = (Uid) strtol (p, &tail, 0);
		  
		  p = strtok (NULL, spacedelim);
		  if (p) {
		    dbentry = getdbdata_byname (p);
		    if (dbentry) {
		      if ((auth_usr_on_dbase (uid, dbentry->dbid))
			  || (uid == U_ROOT)) {
			retval = 
			  writesockf (cl->serversock,
				      "%c%s %d %s %s",
				      KCP_OKC, KCP_OK,
				      dbentry->dbid,
				      dbentry->path,
				      (uid == dbentry->owner) 
				      ? "owner" : "other"
				      );
		      }
		      else {
			retval = 
			  writesockf (cl->serversock,
				      _("%c%s permission denied"),
				      KCP_ERRC, KCP_ERR);
		      }
		      if (retval == 0) {
			some_server_died = true;
			cl->status = C_ZOMBIE;
		      }
		      else if (retval < 0) 
			goto errhd;
		    }
		    else {
		      retval = 
			writesockf (cl->serversock,
				    _("%c%s unknown database"), 
				    KCP_ERRC, KCP_ERR);
		      if (retval == 0) {
			some_server_died = true;
			cl->status = C_ZOMBIE;
		      }
		      else if (retval < 0)
			goto errhd;
		    }
		  }
		}
	      }
	      else
		log_auth_request ("verifydb", cl, connid);
	      break;

	    case BEGINTRANS_CODE:
	      /* ----- BEGINTRANS command -------------------------------- */
	      /* with this commando the server starts his lock requests to the
		 daemon.  The format of a lock request is restricted:
		 
		 BEGINTRANS <id> <key> 0 <lock><lock><lock>
		 
		 The 0 is an unused element -- it stands for the lockid,
		 which is of course, not known at this point.
		 
		 Each lock consists of 4 entries:

		 lockstatus  1 char: lock_LOCKED or lock_SHARED
		 locktype    1 char: lock_NEXTOID, lock_INDEX, lock_DATA or
		 lock_OPENLIST
		 lockdbase   2 char: a hex char value (database number!)
		 lockval     8 char: a hex (long) int value
		 
		 The lock-items follow without a space */

	      connid = getserverid (token);
	      checkserverkey (token, connid, keyok, cl);
	      
	      if (keyok) {
		char *p;
		int lockid;

		token = strtok (NULL, " "); /* take the zero lockid item */
		p = strtok (NULL, "\r");
		lockid = beginlock (connid); /* Lockeintragung beginnen! */
		add_lock_data (p, lockid); /* put new lock data */
		
 		if (lockchanged)
 		  showlocklist();
		
		retval =
		  writesockf (cl->serversock,
			      "%s %c%s %d", 
			      KCP_TRANS, KCP_WAITC, KCP_WAIT, /* protokoll */
			      lockid); /* die lock ID */
		if (retval == 0) {
		  some_server_died = true;
		  cl->status = C_ZOMBIE;
		}
		else if (retval < 0)
		  goto errhd;	/* to server: wait! */
		check_for_complete_servers (); /* grant others permissions */
	      }
	      else
		log_auth_request ("begintrans", cl, connid);
	      break;

	    case ADDTRANS_CODE:
	      /* ----- ADDTRANS command -------------------------------- */
	      /* with this commando the server adds additional resources to a
		 exisiting lock requests.  The format of a lock request is
		 the same as with the BEGINTRANS command:

		 ADDTRANS <id> <key> <lockid> <lock><lock><lock>
	       
		 Each lock consists of 4 entries:

		 lockstatus  1 char: lock_LOCKED or lock_SHARED
		 locktype    1 char: lock_NEXTOID, lock_INDEX, lock_DATA or
		 lock_OPENLIST
		 lockdbase   2 char: a hex char value (database number!)
		 lockval     8 char: a hex (long) int value
		 
		 The lock-items follow without a space */

	      connid = getserverid (token);
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		char *p, *tail;
		int lockid;
		
		token = strtok (NULL, " ");
		lockid = strtol (token, &tail, 0);

		p = strtok (NULL, "\r");
		add_lock_data (p, lockid); /* put new lock data */
		
 		if (lockchanged)
 		  showlocklist();
		
		retval =
		  writesockf (cl->serversock,
			      "%s %c%s %d", 
			      KCP_TRANS, KCP_WAITC, KCP_WAIT, /* protokoll */
			      lockid); /* die lock ID */
		if (retval == 0) {
		  some_server_died = true;
		  cl->status = C_ZOMBIE;
		}
		else if (retval < 0)
		  goto errhd;	/* to server: wait! */
		check_for_complete_servers (); /* grant others permissions */
	      }
	      else
		log_auth_request ("addtrans", cl, connid);
	      break;

	    case CHANGETRANS_CODE:
	      connid = getserverid (token);
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		char *p, *tail;
		int lockid;

		token = strtok (NULL, " ");
		lockid = strtol (token, &tail, 0);
		
		p = strtok (NULL, "\r");
		
		change_lock_data (p, lockid);

 		if (lockchanged)
		  showlocklist();
		
		retval = 
		  writesockf (cl->serversock,
			      "%s %c%s %d",
			      KCP_TRANS, KCP_WAITC, KCP_WAIT,
			      lockid);
		if (retval == 0) {
		  some_server_died = true;
		  cl->status = C_ZOMBIE;
		}
		else if (retval < 0)
		  goto errhd;	/* to server: wait! */
		check_for_complete_servers (); /* grant others permissions */
	      }
	      else
		log_auth_request ("changetrans", cl, connid);
	      break;

	    case ENDTRANS_CODE:
	      /* ----- ENDTRANS command ------------------------------------ */
	      /* ends a transaction status.  This command is as strict as the
		 begintrans, addtrans, changetrans commando, but it needs
		 only 1 Parameter:

		 ENDTRANS <id> <key> <lockid>

		 if lockid is 0, every lock for this server will be removed;
		 otherwise the specific lockrequest with the handle lockid
		 will be removed.  */

	      connid = getserverid (token);
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		char *tail;
		int lockid;
		
		token = strtok (NULL, " \r");
		lockid = strtol (token, &tail, 0);

		if (lockid == 0) { /* remove all locks */
		  deleteallserverlocks (connid); /* stop transaction */
		}
		else {
		  deletelock (lockid);
		}
 		if (lockchanged)
 		  showlocklist();

		retval = 
		  writesockf (cl->serversock,
			      "%s %c%s", KCP_ENDTRANS, KCP_OKC, KCP_OK);
		if (retval == 0) {
		  some_server_died = true;
		  cl->status = C_ZOMBIE;
		}
		else if (retval < 0)
		  goto errhd;
		check_for_complete_servers (); /* grant others permissions */
	      }
	      else
		log_auth_request ("endtrans", cl, connid);
	      break;

	    case TRACE_CODE:
	      /* ----- TRACE commando -------------------------------- */

	      connid = getserverid (token);
      	      checkserverkey (token, connid, keyok, cl);
	
	      if (keyok) {
		token = strtok (NULL, "\r");
		if (token)
		  tracef ("MESSAGE from ID %d: %s", connid, token);
		else
		  tracef ("Unspecified MESSAGE from ID %d", connid);
	      }
	      else
		log_auth_request ("trace", cl, connid);
	      break;

	    case USRINGRP_CODE:
	      /* ----- USRINGRP command -------------------------------- */
	      /* the daemon should look up a user in a specified group.  Two
		 arguments: USRINGRP id key UID GID */
	      
	      connid = getserverid (token);	/* get the server id */
	      checkserverkey (token, connid, keyok, cl);

	      if (keyok) {
		char *p;
		Uid uid;
		Gid gid;
		Gentry *gentry;
		char *tail;

		log_request ("usr_in_group", cl, connid);

		p = strtok (NULL, spacedelim);
		if (p) {
		  uid = (Uid) strtol (p, &tail, 0);

		  p = strtok (NULL, spacedelim);
		  if (p) {
		    gid = (Gid) strtol (p, &tail, 0);
		    
		    gentry = getgrpdata (gid);

		    if (gentry) {
		      if (on_ulist (uid, gentry->ulist))
			retval = writesockf (cl->serversock,
					     "%c%s", KCP_OKC, KCP_OK);
		      else
			retval = writesockf (cl->serversock,
					     "%c%s not in group",
					     KCP_ERRC, KCP_ERR);
		      if (retval == 0) {
			some_server_died = true;
			cl->status = C_ZOMBIE;
		      }
		      else if (retval < 0)
			goto errhd;	/* to server: msg! */
		    }
		    else {
		      retval = 
			writesockf (cl->serversock,
				    "%c%s unknown group", 
				    KCP_ERRC, KCP_ERR);
		      if (retval == 0) {
			some_server_died = true;
			cl->status = C_ZOMBIE;
		      }
		      else if (retval < 0)
			goto errhd;	/* to server: msg! */
		    }
		  }
		}
	      }
	      else
		log_auth_request ("usr_in_group", cl, connid);
	      break;

	    default:
	    keynotfound:
	      /* ----- unknown commands ------------------------------- */
	      tracef ("kdbd: unknown transfer protocol (%s)\n", token);
	    }
	  }
	  if (buffer && (bufsize > 1024)) {
	    free (buffer);
	    buffer = NULL;
	  }
        }
      } /* if (FD_ISSET ... */
    } /* for (testsock = 0; ... */

    /* if there somewhere died a server in der system, cleanup the
       connection list. */
    if (some_server_died) {
      cleanup_clist (&clist);
      some_server_died = false;
    }
  } /* while (!eodaemon) */

 eodaemonhd:  

  if (errstr)
    PRINTERR (GETERRDESC());
  stop_all_servers ();		/* auch bei Fehlern alle Server anhalten! */
  return 0;
  
 errhd:
  PRINTERR (GETERRDESC());
  stop_all_servers ();		/* auch bei Fehlern alle Server anhalten! */
  return -1;
}

