/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#define PROG_NAME "kdbd"

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <locale.h>

#include <getopt.h>

#include "kdbd.h"		/* autoincludes net.h */
#include "lock.h"

unsigned short port = 0;	/* the port */
char *host = NULL;		/* the host */
char daemon_user[128] = "kaenguru";

/* prototypes */
void showhelp (void);
void showversion (void);
int change_uid ();
volatile int eodaemon = false;	/* Daemon end? */
int verbose = false;

/* ----------------------------------------------------------------------
   SIGNAL-Handler
   ---------------------------------------------------------------------- */
void
sigpipe_handler (int sig)
{
  /* okay, we have to block the broken pipe signal, so we can handle the
     communication in a save way */
  signal (SIGPIPE, sigpipe_handler);
}

void
sigint_handler (int sig)
{
  /* we just ingnore the interrupt signal */
  signal (SIGINT, sigint_handler);
}

void
sigquit_handler (int sig)
{
  /* just ignore the quit signal */
  signal (SIGQUIT, sigquit_handler);
}

void
sigterm_handler (int sig)
{
  PRINTERR(_("receiving SIGTERM signal.  Quit."));
  eodaemon = true;		/* stop the daemon! */
  signal (SIGTERM, sigterm_handler);
}



/* ----------------------------------------------------------------------
   main, init and startup routines 
   ---------------------------------------------------------------------- */
#define HELP_OPT       0
#define PORT_OPT       1
#define TRACE_OPT      2
#define CONFFILE_OPT   3
#define VERSION_OPT    4
#define VERBOSE_OPT    5

int
main(int argc, char **argv)
{
  char
    *portstring = NULL,		/* port argument */
    *configfilename = NULL,
    *newtrace = NULL;
  char c;

  int retval;
  unsigned short portv = 0;

  /* set up signal-handlers */
  
  signal (SIGPIPE, sigpipe_handler); /* broken pipe */
  signal (SIGINT, sigint_handler); /* interrupt (close) */
  signal (SIGQUIT, sigquit_handler); /* interrupt (close) */
  signal (SIGTERM, sigterm_handler); /* interrupt (close) */

  setlocale (LC_MESSAGES, "");
  textdomain (PACKAGE);

  /* argumentpassing */
  while (1) {
    static struct option long_options[] = {
      {"help", 0, 0, HELP_OPT},
      {"port", 1, 0, PORT_OPT},
      {"trace", 1, 0, TRACE_OPT},
      {"conffile", 1, 0, CONFFILE_OPT},
      {"version", 0, 0, VERSION_OPT},
      {"verbose", 0, 0, VERBOSE_OPT},
      {0, 0, 0, 0}
    };
    int option_index = 0;
    
    c = getopt_long (argc, argv, "hp:t:f:", long_options, &option_index);
    
    if (c == -1)
      break;
    
    switch (c) {
    case HELP_OPT: 
    case 'h':			/* help */
      showhelp();
      exit (EXIT_SUCCESS);
    case PORT_OPT:
    case 'p':			/* port to establish to */
      portstring = optarg;
      portv = atoi (portstring);
      break;
    case CONFFILE_OPT:
    case 'f':			/* configfilename */
      configfilename = optarg;
      break;
    case TRACE_OPT:
    case 't':			/* tracefile */
      trace = true;
      newtrace = optarg;
      break;
    case VERSION_OPT:
      showversion ();
      exit (EXIT_SUCCESS);
    case VERBOSE_OPT:
      verbose = true;
      break;
    case '?':			/* wrong option */
      exit (EXIT_FAILURE);
      break;
    default:			/* other options */
      exit (EXIT_FAILURE);
    }
  }
  
  /* read global configuration file */
  read_configuration (configfilename);

  if (change_uid () < 0) {
    fprintf (stderr, _("Can't change UID\n"));
    exit (EXIT_FAILURE);
  }

  if (portv > 0) {
    port = portv;
  }
  if (port == 0)
    port = PORT;
  
  establishtracefile (newtrace); /* tracing ? */
  
  init_lists ();		/* init the user and group lists */
  read_user_conf ();		/* users file */
  read_group_conf ();		/* groups file */
  read_auth_conf ();		/* authority file */
  read_database_conf ();	/* databases file */
  
/*   showlists (); */

  initDaemon ();		/* init us as daemon */
  initreslist ();		/* init Lock lists */
  
  retval = listencycle (port);	/* start to listen for clients */
  
  if (retval < 0) {
    PRINTERR (GETERRDESC());
    exit (EXIT_FAILURE);
  }
  PRINTERR (GETERRDESC());
  exit (EXIT_SUCCESS);
}


int
change_uid ()
{
  struct passwd *pw;

  pw = getpwnam (daemon_user);
  if (!pw)
    return -1;

  return setuid (pw->pw_uid);
}


/*------------------------------------------------------------
  General routines
  ------------------------------------------------------------*/	
void
showhelp (void)
{
  fprintf(stdout, _(
"This is %s %s - The Kaenguru Database Daemon\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"
"Starts the Kaenguru Database Daemon\n"
"\n"
"General usage:\n"
"  %s [OPTION] &\n"
"\n"
"Options:\n"
"\n"
"  -h, --help             This help\n"
"  -p, --port=PORT        The port, to which the daemon will listen\n"
"  -t, --trace=FILE       Logfile, where errormessages, tracing output\n"
"                         etc. will go; if you specify 'stdout' (without\n"
"                         inverted commas) all output goes to stdout\n"
"  -f, --conffile=FILE    Don't use the standard configfile in\n"
"                         /etc/kaenguru.conf, but <file>\n"
"      --version          print the version\n"
"\n"), PROG_NAME, VERSION, COPYRIGHTYEAR, PROG_NAME);    
}

void
showversion (void)
{
  fprintf (stdout, "
This is Kaenguru %s - The Kaenguru Database System (%s)
Copyright (c) %s by Gregor Klinke

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.
  
This program ist distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public Lincense for more details.

", VERSION, PROG_NAME, COPYRIGHTYEAR);

}

