/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#if defined STDC_HEADERS || defined HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif

#include "kdbd.h"
#include "path.h"
#include "read_configuration.h"

char systempath[FILENAME_MAX] = SYSTEMPATH;
char confpath[FILENAME_MAX] = CONFIG_DEFAULT;

Uentry *users;
Gentry *groups;
Aentry *auths;
Dbentry *dbases;

char binpath[FILENAME_MAX] = BINPATH;
int maxconn = 5;

const char fielddelim[] = ":";


/* ----------------------------------------------------------------------
   Read the config lists etc.
   ---------------------------------------------------------------------- */
void
init_lists ()
{
  users = NULL;
  groups = NULL;
  auths = NULL;
  dbases = NULL;
}

void
free_lists ()
{
  if (users) {
    Uentry *ue = users, *nue;
    while (ue) {
      nue = ue->next;
      if (ue->name)
	free (ue->name);
      if (ue->comment)
	free (ue->comment);
      free (ue);
      ue = nue;
    }
    users = NULL;
  }
  if (groups) {
    Gentry *ge = groups, *nge;
    while (ge) {
      nge = ge->next;
      if (ge->name)
	free (ge->name);
      if (ge->comment)
	free (ge->comment);
      if (ge->ulist) {
	Ulist *ul = ge->ulist, *nul;
	while (ul) {
	  nul = ul->next;
	  free (ul);
	  ul = nul;
	}
      }
      free (ge);
      ge = nge;
    }
    groups = NULL;
  }
  if (auths) {
    Aentry *ae = auths, *nae;
    while (ae) {
      nae = ae->next;
      if (ae->login)
	free (ae->login);
      free (ae);
      ae = nae;
    }
    auths = NULL;
  }
  if (dbases) {
    Dbentry *dbe = dbases, *ndbe;
    while (dbe) {
      ndbe = dbe->next;
      if (dbe->name)
	free (dbe->name);
      if (dbe->path)
	free (dbe->path);
      if (dbe->ulist) {
	Ulist *ul = dbe->ulist, *nul;
	while (ul) {
	  nul = ul->next;
	  free (ul);
	  ul = nul;
	}
      }
      if (dbe->comment)
	free (dbe->comment);
      free (dbe);
      dbe = ndbe;
    }
    dbases = NULL;
  }
}

/* ----------------------------------------------------------------------
   Read the user configuration data
   ---------------------------------------------------------------------- */
#define ALLOC_UENTRY()					\
({							\
  Uentry *_nue = (Uentry *) malloc (sizeof (Uentry));	\
  _nue->name = NULL;					\
  _nue->uid = UNKNOWNUSR;				\
  _nue->gid = UNKNOWNGRP;				\
  _nue->comment = NULL;					\
  _nue->next = NULL;					\
  _nue;							\
})

#define USRCNFR_ERROR "user configuration file"
int
read_user_conf ()
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *tail, *token;
  Uentry *ueptr, *nue;
  
  sprintf (tmp, "%s/conf/%s", systempath, USRCONFFILE);
  stream = fopen (tmp, "r");

  if (!stream)
    GOERR (_(USRCNFR_ERROR));
  
  ueptr = users;
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_(USRCNFR_ERROR));
    }
    
    nue = ALLOC_UENTRY ();
    
    token = strtok (buffer, fielddelim);
    if (!token)
      GOERR (_("user configuration: bad user name"));
    nue->name = strdup (token);	/* get the user name */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("user configuration: bad user id"));
    nue->uid = strtol (token, &tail, 0); /* the user id */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("user configuration: bad default group id"));
    nue->gid = strtol (token, &tail, 0); /* his default group id */
    
    token = strtok (NULL, "\n");
    if (token)
      nue->comment = strdup (token); /* the comment */
    else
      nue->comment = NULL;
    
    if (ueptr) {
      ueptr->next = nue;
      ueptr = ueptr->next;
    }
    else
      users = ueptr = nue;
  }

eofhd:
  if (fclose (stream) != 0)
    GOERR (_(USRCNFR_ERROR));
  return 1;

errhd:
  fprintf (stderr, "kdbd: %s\n", GETERRDESC());
  exit (EXIT_FAILURE);
}


/* ----------------------------------------------------------------------
   Now read the groups config file
   ---------------------------------------------------------------------- */
#define ALLOC_GENTRY()					\
({							\
  Gentry *_nge = (Gentry *) malloc (sizeof (Gentry));	\
  _nge->name = NULL;					\
  _nge->gid = UNKNOWNGRP;				\
  _nge->ulist = NULL;					\
  _nge->comment = NULL;					\
  _nge->next = NULL;					\
  _nge;							\
})

#define ALLOC_ULIST()					\
({							\
  Ulist *_nul = (Ulist *) malloc (sizeof (Ulist));	\
  _nul->uid = UNKNOWNUSR;				\
  _nul->next = NULL;					\
  _nul;							\
})

#define GROUPCNFR_ERROR "group configuration file"
int
read_group_conf ()
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *tail, *token;
  Gentry *geptr, *nge;
  Ulist *ulptr, *nul;

  sprintf (tmp, "%s/conf/%s", systempath, GRPCONFFILE);
  stream = fopen (tmp, "r");

  if (!stream)
    GOERR (_(GROUPCNFR_ERROR));
  
  geptr = groups;
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_(GROUPCNFR_ERROR));
    }
    
    nge = ALLOC_GENTRY ();
    
    token = strtok (buffer, fielddelim); /* keywords are divided by : */
    if (!token)
      GOERR (_("group configuration: bad group name"));
    nge->name = strdup (token);	/* get the group name */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("group configuration: bad group id"));
    nge->gid = strtol (token, &tail, 0); /* the group id */
    
    token = strtok (NULL, fielddelim);
    if (token) {
      if (strlen (token) > 0) {
	nge->comment = strdup (token); /* the comment */
      }
      
      ulptr = NULL;		/* get the userlist */
      do {
	token = strtok (NULL, ", \n");
	if (token) {
	  Uid uid;
	  nul = ALLOC_ULIST ();
	  uid = getusrid (token);
	  if (uid != UNKNOWNUSR) {
	    nul->uid = uid;
	    if (ulptr) {
	      ulptr->next = nul;
	      ulptr = ulptr->next;
	    }
	    else
	      nge->ulist = ulptr = nul;
	  }
	  else {
	    PRINTERR (_("group configuration: bad userlist"));
	  }
	}
      } while (token);
    }

    if (geptr) {
      geptr->next = nge;
      geptr = geptr->next;
    }
    else
      groups = geptr = nge;
  }

eofhd:
  if (fclose (stream) != 0)
    GOERR (_(GROUPCNFR_ERROR));
  return 1;
  
 errhd:
  fprintf (stderr, "kdbd: %s\n", GETERRDESC());
  exit (EXIT_FAILURE);
}


/* ----------------------------------------------------------------------
   Read the authorisation configuration data
   ---------------------------------------------------------------------- */
#define ALLOC_AENTRY()					\
({							\
  Aentry *_nae = (Aentry *) malloc (sizeof (Aentry));	\
  _nae->login = NULL;					\
  _nae->uid = UNKNOWNUSR;				\
  _nae->key = 0;					\
  _nae->next = NULL;					\
  _nae;							\
})

#define AUTHCNFR_ERROR "authorization file" 
int
read_auth_conf ()
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *token, *tail;
  Aentry *aeptr, *nae;
  Uid uid;

  sprintf (tmp, "%s/conf/%s", systempath, AUTHFILE);
  stream = fopen (tmp, "r");

  if (!stream)
    GOERR (_(AUTHCNFR_ERROR));
  
  aeptr = auths;
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_(AUTHCNFR_ERROR));
    }
    
    nae = ALLOC_AENTRY ();
    
    token = strtok (buffer, fielddelim); /* keywords are divided by : */
    if (!token)
      GOERR (_("authorization file: bad login string"));
    nae->login = strdup (token); /* get the login */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("authorization file: bad authorization name"));
    uid = getusrid (token);
    if (uid == UNKNOWNUSR) {
      GOERR (_("authorization file: unknown user equivalance"));
    }
    else
      nae->uid = uid;		/* the user id */

    /* get here the dates, alterdata, key &c! */
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("authorization file: bad authorization key"));
    nae->key = strtol (token, &tail, 0);
    
    if (aeptr) {
      aeptr->next = nae;
      aeptr = aeptr->next;
    }
    else
      auths = aeptr = nae;
  }
  
eofhd:
  if (fclose (stream) != 0)
    GOERR (_(AUTHCNFR_ERROR));
  return 1;
  
 errhd:
  fprintf (stderr, "kdbd: %s\n", GETERRDESC());
  exit (EXIT_FAILURE);
}



/* ----------------------------------------------------------------------
   Read the database configuration data
   ---------------------------------------------------------------------- */
#define ALLOC_DENTRY()						\
({								\
  Dbentry *_nde = (Dbentry *) malloc (sizeof (Dbentry));	\
  _nde->name = NULL;						\
  _nde->dbid = UNKNOWNDB;					\
  _nde->owner = UNKNOWNUSR;					\
  _nde->path = NULL;						\
  _nde->ulist = NULL;						\
  _nde->comment = NULL;						\
  _nde->next = NULL;						\
  _nde;								\
})

#define DBCNFR_ERROR "database configuration"
int
read_database_conf ()
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *tail, *token;
  Dbentry *deptr, *nde;
  Ulist *ulptr, *nul;
  Uid uid;

  sprintf (tmp, "%s/conf/%s", systempath, DATABASEFILE);
    stream = fopen (tmp, "r");

  if (!stream)
    GOERR (_(DBCNFR_ERROR));
  
  deptr = dbases;
  
  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_(DBCNFR_ERROR));
    }
    
    nde = ALLOC_DENTRY ();
    
    token = strtok (buffer, fielddelim); /* keywords are divided by : */
    if (!token)
      GOERR (_("database configuration: bad database name"));
    nde->name = strdup (token);	/* get the database name */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("database configuration: bad database id"));
    nde->dbid = strtol (token, &tail, 0); /* the database id */
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("database configuration: bad database owner name"));
    uid = getusrid (token);	/* the owner name */
    if (uid != UNKNOWNUSR)
      nde->owner = uid;
    else
      PRINTERR (_("database configuration: bad db-owner equivelance id"));
    
    token = strtok (NULL, fielddelim);
    if (!token)
      GOERR (_("database configuration: bad database path"));
    nde->path = strdup (token);	/* the path */
    
    token = strtok (NULL, fielddelim);
    if (token) {
      if (strlen (token) > 0) {
	nde->comment = strdup (token); /* the comment */
      }
      
      ulptr = NULL;		/* get the userlist */
      do {
	token = strtok (NULL, ", \n");
	if ((token) &&
	    (strlen (token) > 0)) {
	  nul = ALLOC_ULIST ();

	  if (token[0] == '*')
	    uid = EVERYUSR;	/* a star means: every user has access */
	  else
	    uid = getusrid (token);
	  if (uid != UNKNOWNUSR) {
	    nul->uid = uid;
	    if (ulptr) {
	      ulptr->next = nul;
	      ulptr = ulptr->next;
	    }
	    else
	      nde->ulist = ulptr = nul;
	  }
	  else {
	    PRINTERR (_("database configuration: bad userlist"));
	  }
	}
      } while (token);
    }

    if (deptr) {
      deptr->next = nde;
      deptr = deptr->next;
    }
    else
      dbases = deptr = nde;
  }

 eofhd:
  if (fclose (stream) != 0)
    GOERR (_(DBCNFR_ERROR));
  return 1;
  
 errhd:
  fprintf (stderr, "kdbd: %s\n", GETERRDESC());
  exit (EXIT_FAILURE);
}



/* ----------------------------------------------------------------------
   Convenienve and search functions
   ---------------------------------------------------------------------- */
bool
on_ulist (Uid uid, Ulist *ulist)
{
  Ulist *ulptr = ulist;
  if (ulptr) {
    if (ulptr->uid == EVERYUSR)	/* if the first usr is EVERYUSR, every user */
      return true;		/* is granted ~untested~ */
    
    while (ulptr) {
      if (ulptr->uid == uid)
	return true;
      ulptr = ulptr->next;
    }
  }
  return false;
}

Uid
getusrid (char *usrname)
{
  Uentry *ueptr = users;
  while (ueptr) {
    if (strcmp (usrname, ueptr->name) == 0)
      return ueptr->uid;
    ueptr = ueptr->next;
  }
  return UNKNOWNUSR;
}

char *
getusrnme (Uid uid)
{
  Uentry *ueptr = users;
  while (ueptr) {
    if (uid == ueptr->uid)
      return ueptr->name;
    ueptr = ueptr->next;
  }
  return NULL;
}

Uentry *
getusrdata (Uid uid)
{
  Uentry *ueptr = users;
  while (ueptr) {
    if (uid == ueptr->uid)
      return ueptr;
    ueptr = ueptr->next;
  }
  return NULL;
}

Gid
getgrpid (char *grpname)
{
  Gentry *geptr = groups;
  while (geptr) {
    if (strcmp (grpname, geptr->name) == 0)
      return geptr->gid;
    geptr = geptr->next;
  }
  return UNKNOWNGRP;
}

char *
getgrpnme (Gid gid)
{
  Gentry *geptr = groups;
  while (geptr) {
    if (gid == geptr->gid)
      return geptr->name;
    geptr = geptr->next;
  }
  return NULL;
}

Gentry *
getgrpdata (Gid gid)
{
  Gentry *geptr = groups;
  while (geptr) {
    if (gid == geptr->gid)
      return geptr;
    geptr = geptr->next;
  }
  return NULL;
}

void
getusrauth (char *logname, char *domain, Uid *uid, int *key)
{
  Aentry *aeptr = auths;
  char tmp[127];
  
  sprintf (tmp, "%s@%s", logname, domain);
  
  while (aeptr) {
    if (strcmp (tmp, aeptr->login) == 0) {
      *uid = aeptr->uid;
      *key = aeptr->key;
      return;
    }
    aeptr = aeptr->next;
  }
  *uid = UNKNOWNUSR;
  *key = 0;
}

Dbid
getdbid (char *dbname)
{
  Dbentry *deptr = dbases;
  while (deptr) {
    if (strcmp (dbname, deptr->name) == 0)
      return deptr->dbid;
    deptr = deptr->next;
  }
  return UNKNOWNDB;
}

char *
getdbnme (Dbid dbid)
{
  Dbentry *deptr = dbases;
  while (deptr) {
    if (dbid == deptr->dbid)
      return deptr->name;
    deptr = deptr->next;
  }
  return NULL;
}

Dbentry *
getdbdata_byid (Dbid dbid)
{
  Dbentry *deptr = dbases;
  while (deptr) {
    if (dbid == deptr->dbid)
      return deptr;
    deptr = deptr->next;
  }
  return NULL;
}

Dbentry *
getdbdata_byname (char *dbname)
{
  Dbentry *deptr = dbases;
  while (deptr) {
    if (strcmp (dbname, deptr->name) == 0)
      return deptr;
    deptr = deptr->next;
  }
  return NULL;
}

bool
auth_usr_on_dbase (Uid uid, Dbid dbid)
{
  Dbentry *dbentry = getdbdata_byid (dbid);
  if (!dbentry)
    return false;
  
  return on_ulist (uid, dbentry->ulist);
}


/* ----------------------------------------------------------------------
   Testphase function: print the lists
   ---------------------------------------------------------------------- */
void
showlists ()
{
  Uentry *ueptr;
  Gentry *geptr;
  Aentry *aeptr;
  Dbentry *deptr;
  Ulist *ulptr;
  char *gname, *oname;

  printf ("************************************************************\n");
  printf (_("Known users: \n"));
  ueptr = users;
  while (ueptr) {
    gname = getgrpnme (ueptr->gid);
    if (gname)
      printf ("%s UID: %d Group: %s (%s)\n",
	      ueptr->name, ueptr->uid, gname, ueptr->comment);
    else
      printf ("%s UID: %d GID: %d (%s)\n",
	      ueptr->name, ueptr->uid, ueptr->gid, ueptr->comment);
    ueptr = ueptr->next;
  }

  printf (_("\nKnown groups: \n"));
  geptr = groups;
  while (geptr) {
    printf ("%s GID: %d (%s)\n",
	    geptr->name, geptr->gid, geptr->comment);
    ulptr = geptr->ulist;
    printf (_("    Members: "));
    while (ulptr) {
      printf ("%s ", getusrnme (ulptr->uid));
      ulptr = ulptr->next;
    }
    printf ("\n");
    geptr = geptr->next;
  }

  printf (_("\nAuthorization data: \n"));
  aeptr = auths;
  while (aeptr) {
    printf ("%s -> %s\n", aeptr->login, getusrnme (aeptr->uid));
    aeptr = aeptr->next;
  }

  printf (_("\nAvailable databases: \n"));
  deptr = dbases;
  while (deptr) {
    oname = getusrnme (deptr->owner);
    if (oname)
      printf (_("%s (DBID: %d), owned by: %s based at %s (%s)\n"),
	      deptr->name, deptr->dbid, oname, deptr->path, 
	      deptr->comment);
    else
      printf (_("%s (DBID: %d), owned by UID: %d based at %s (%s)\n"),
	      deptr->name, deptr->dbid, deptr->owner, deptr->path, 
	      deptr->comment);
    ulptr = deptr->ulist;
    printf (_("    Access to: "));
    while (ulptr) {
      printf ("%s ", getusrnme (ulptr->uid));
      ulptr = ulptr->next;
    }
    printf ("\n");
    deptr = deptr->next;
  }

  printf ("************************************************************\n");
}




/* ----------------------------------------------------------------------
   GENERAL CONFIG READ FUNCTION
   ---------------------------------------------------------------------- */
#define CNFR_ERROR "configuration file"
int
read_configuration (char *name)
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *token;
  int inscope = false;

  if (name) {
    sprintf (tmp, "%s", name);
  }
  else {
    sprintf (tmp, "%s", CONFIG_DEFAULT);
  }
  stream = fopen (tmp, "r");
  
  if (!stream)
    GOERR (_(CNFR_ERROR));

  strcpy (confpath, tmp);

  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_(CNFR_ERROR));
    }

    if (!inscope) {
      token = strtok (buffer, " (\n");
      if (token) {
	if (strcasecmp (token, "begin") == 0) {
	  token = strtok (NULL, " )\r\t\n");
	  if ((token) &&
	      ((strcasecmp (token, "daemon") == 0) ||
	       (strcasecmp (token, "global") == 0))) {
	    inscope = true;
	  }
	  /*	  else 
	    dm_seterr (DMEKEYWORD);*/
	}
      }
    }
    else {
      token = strtok (buffer, ": \n");
      if (token) {
	
	if (strcmp (token, "systemdir") == 0) {
	  GET_STRING (token, systempath);
	}
	else if (strcmp (token, "bindir") == 0) {
	  GET_STRING (token, binpath);
	}
	else if (strcmp (token, "port") == 0) {
	  port = GET_CONST (token, 1024, 65535);
	}
	else if (strcmp (token, "maxconn") == 0) {
	  maxconn = GET_CONST (token, 1, MAXCONNECTIONS);
	}
	else if (strcmp (token, "user") == 0) {
	  GET_STRING (token, daemon_user);
	}
	else if (strcmp (token, "log-file") == 0) {
	  token = strtok (NULL, " \r\t\n");
	  if (token) {
	    trace = true;
	    tracefilename = strdup (token);
	  }
	}
	else if (strcasecmp (token, "end") == 0) {
	  inscope = false;
	}
	else {
	  if (*token == '#') {}
	}
      }
    }
  }
  
eofhd:
  if (inscope) {
    fprintf (stderr, _("warning! unexpected eof in config file\n"));
  }
  if (fclose (stream) != 0)
    GOERR (_(CNFR_ERROR));
  return 1;
     
errhd:
  PRINTERR (GETERRDESC());
  exit (EXIT_FAILURE);
}

