/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997,98 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#define PROG_NAME "kdaddkey"

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
#endif
#include <errno.h>
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#if defined STDC_HEADERS || defined HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif
#include <sys/types.h>
#if defined HAVE_SYS_TIME_H
# include <sys/time.h>
#else
# include <time.h>
#endif
#include <locale.h>

#include <getopt.h>

#include "kdbd.h"		/* autoincludes net.h */
#include "path.h"
#include "read_configuration.h"

/* this vars are here only to reuse the config file! ignore them */
unsigned short port;		/* the port */
char *host;			/* the host */
char daemon_user[128];
bool trace;			/* tracing? */
char *tracefilename;		/* a tracefile? */

/* prototypes */
void showhelp (void);
void showversion (void);


int
write_authorization_file (char *filename, KdbKeyStruc *key)
{
  FILE *stream;

  stream = fopen (filename, "w+");
  if (!stream)
    return -1;

  fwrite (key->key, 1, strlen (key->key), stream);
  fclose (stream);

  return 1;
}

int
read_configuration_keytool (char *name)
{
  FILE *stream;
  char tmp[FILENAME_MAX], buffer[1024], *token;
  int inscope = false;

  if (name) {
    sprintf (tmp, "%s", name);
  }
  else {
    sprintf (tmp, "%s", CONFIG_DEFAULT);
  }
  stream = fopen (tmp, "r");
  
  if (!stream)
    GOERR (_("config file"));
  
  strcpy (confpath, tmp);

  while (!feof (stream)) {
    
    if (fgets (buffer, 1023, stream) == NULL) {
      if (feof (stream))
	goto eofhd;
      else
	GOERR (_("config file"));
    }

    if (!inscope) {
      token = strtok (buffer, " (\n");
      if (token) {
	if (strcasecmp (token, "begin") == 0) {
	  token = strtok (NULL, " )\r\t\n");
	  if ((token) &&
	      ((strcasecmp (token, "keytool") == 0) ||
	       (strcasecmp (token, "global") == 0))) {
	    inscope = true;
	  }
	}
      }
    }
    else {
      token = strtok (buffer, ": \n");
      if (token) {
	
	if (strcmp (token, "systemdir") == 0) {
	  GET_STRING (token, systempath);
	}
	else if (strcasecmp (token, "end") == 0) {
	  inscope = false;
	}
	else {
	  if (*token == '#') {}
	}
      }
    }
  }
  
 eofhd:
  if (inscope) {
    fprintf (stderr, _("warning! unexpected eof in config file\n"));
  }
  if (fclose (stream) != 0)
    GOERR (_("config file"));
  return 1;
  
 errhd:
  PRINTERR (GETERRDESC());
  exit (EXIT_FAILURE);
}

/* ----------------------------------------------------------------------
   main, init and startup routines 
   ---------------------------------------------------------------------- */
#define HELP_OPT       0
#define VERSION_OPT    1
#define MAX_LOGINS_OPT 2
#define LOCKED_OPT     3
#define NOT_LOCKED_OPT 4
#define ADD_OPT        5
#define SET_OPT        6
#define GET_OPT        7
#define KEYDB_OPT      8
#define CONFFILE_OPT   9
#define MAX_DAYS_OPT   10

int
main(int argc, char **argv)
{
  char c;
  int keyid = 0;
  Int16 maxlogins = 0;
  Int16 max_days = 0;
  enum {
    ADD_MODE,
    SET_MODE,
    GET_MODE,
    NONE_MODE
  } mode = NONE_MODE;
  Bool locked = false;
  KdbKeyStruc key;
  char *keydbfile = "kdbdauth";
  Date res;
  char *configfilename = NULL;

  setlocale (LC_MESSAGES, "");
  textdomain (PACKAGE);

  /* argumentpassing */
  while (1) {
    static struct option long_options[] = {
      {"help", 0, 0, HELP_OPT},
      {"version", 0, 0, VERSION_OPT},
      {"max-logins", 1, 0, MAX_LOGINS_OPT},
      {"locked", 0, 0, LOCKED_OPT},
      {"not-locked", 0, 0, NOT_LOCKED_OPT},
      {"add", 0, 0, ADD_OPT},
      {"set", 0, 0, SET_OPT},
      {"get", 0, 0, GET_OPT},
      {"file", 1, 0, KEYDB_OPT},
      {"conffile", 1, 0, CONFFILE_OPT},
      {"max-days", 1, 0, MAX_DAYS_OPT}, 
      {0, 0, 0, 0}
    };
    int option_index = 0;
    
    c = getopt_long (argc, argv, "hm:lasgo:f:t:", long_options, &option_index);
    
    if (c == -1)
      break;
    
    switch (c) {
    case HELP_OPT: 
    case 'h':			/* help */
      showhelp();
      exit (EXIT_SUCCESS);
    case MAX_LOGINS_OPT:
    case 'm':
      maxlogins = atoi (optarg);
      break;
    case LOCKED_OPT:
    case 'l':
      locked = true;
      break;
    case NOT_LOCKED_OPT:
      locked = false;
      break;
    case ADD_OPT:
    case 'a':
      mode = ADD_MODE;
      break;
    case SET_OPT:
    case 's':
      mode = SET_MODE;
      break;
    case GET_OPT:
    case 'g':
      mode = GET_MODE;
      break;
    case KEYDB_OPT:
    case 'o':
      keydbfile = optarg;
      break;
    case CONFFILE_OPT:
    case 'f':			/* configfilename */
      configfilename = optarg;
      break;
    case MAX_DAYS_OPT:
    case 't':
      max_days = atoi (optarg);
      break;
    case VERSION_OPT:
      showversion ();
      exit (EXIT_SUCCESS);
    case '?':			/* wrong option */
      exit (EXIT_FAILURE);
      break;
    default:			/* other options */
      exit (EXIT_FAILURE);
    }
  }

  if (optind >= argc) {
    fprintf (stderr, _("at least one key ID is missing\n"));
    exit (EXIT_FAILURE);
  }
  keyid = atoi (argv[optind]);
  
  read_configuration_keytool (configfilename);
  /*   read_auth_conf (); */
  
  switch (mode) {
  case ADD_MODE:
    new_authorization_key (&key, maxlogins, locked);

    write_authorization_key (&key, keyid);
    if (write_authorization_file (keydbfile, &key) < 0) {
      fprintf (stderr, _("Error writing authorization key file\n"));
      exit (EXIT_FAILURE);
    }
    printf (_("- Created new key for key id '%d'.\n"
	      "- Written to file '%s'\n"
	      "- Database updated\n"), keyid, keydbfile);
    break;
    
  case GET_MODE:
    if (read_authorization_key (&key, keyid) < 0) {
      fprintf (stderr, _("Error reading authorization data\n"));
      exit (EXIT_FAILURE);
    }

    printf (_("- Got key from key id '%d':\n"
	      "- Written to file '%s'\n"
	      "- Key data:\n"
	      "    max. count of logins: %d\n" 
	      "    Logincount: %d\n"
	      "    locked?: %s\n"
	      "    Date of creation: %d\n"
	      "    Validity (in days): %d\n"),
	    keyid, keydbfile,
	    key.max_logins,
	    key.login_count,
	    (key.locked ? "yes" : "no"),
	    key.key_created,
	    key.validity);

    if (write_authorization_file (keydbfile, &key) < 0) {
      fprintf (stderr, _("Error writing authorization key file\n"));
      exit (EXIT_FAILURE);
    }
    break;
    
  case SET_MODE:
    if (read_authorization_key (&key, keyid) < 0) {
      fprintf (stderr, _("Error reading authorization data\n"));
      exit (EXIT_FAILURE);
    }
    key.max_logins = maxlogins;
    key.locked = locked;
    key.key_created = time (&res);
    key.login_count = 0;
    key.validity = max_days;

    if (write_authorization_key (&key, keyid) < 0) {
      fprintf (stderr, _("Error writing authorization data\n"));
      exit (EXIT_FAILURE);
    }

    printf (_("- Keydata for key id '%d' set.\n"), keyid);
    break;
    
  case NONE_MODE:
    fprintf (stderr,
	     _("at least one of the options a, s, g must be specified\n"));
    exit (EXIT_FAILURE);
  }

  exit (EXIT_SUCCESS);
}


/*------------------------------------------------------------
  General routines
  ------------------------------------------------------------*/	
void
showhelp (void)
{
  fprintf(stdout, _(
"This is %s %s - The Kaenguru Database System Addkey Utility\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"
"%s is a utility for the Kaenguru Database System to set a\n"
"authorization key for a specified user.\n"
"\n"
"General usage:\n"
"  %s [a|s|g] [options] <KID>\n"
"\n"
"Function, one of the following:\n"
"\n"
"  -a, --add             Add a new key, write this to the database\n"
"  -s, --set             Set the data in an existing key id, without\n"
"                        creating a new key\n"
"  -g, --get             only read the key of an key id and write it to\n"
"                        the file specified by -o, --file=FILE\n"
"\n"
"Options:\n"
"\n"  
"  -h, --help            This help\n"
"  -l, --locked          Lock the specified key\n"
"      --not-locked      Open the specified key (default)\n"
"  -m, --max-logins=N    With this login are only N logins possible;\n"
"                        is N equal to 0 this feature is disabled\n"
"  -t, --max-days        Give a maximum time in day, before this logins\n"
"                        becomes invalid (locked)\n"
"  -o, --file=FILE       Write the created/read key to the file FILE\n"
"  -f, --conffile=FILE   Read the configurationfile FILE instead of\n"
"                        /etc/kaenguru.conf\n"
"      --version         print the version\n"
"\n"
"  KID                   The KeyId, responding to 3 option in the\n"
"                        `authority' file.\n"
"\n"), PROG_NAME, VERSION, COPYRIGHTYEAR, PROG_NAME, PROG_NAME);    
}

void
showversion (void)
{
  fprintf (stdout, 
"This is Kaenguru %s - The Kaenguru Database System (%s)\n"
"Copyright (c) %s by Gregor Klinke\n"
"\n"
"This program is free software; you can redistribute it and/or modify it\n"
"under the terms of the GNU General Public License as published by the\n"
"Free Software Foundation; either version 2 of the License, or (at your\n"
"option) any later version.\n"
"\n"
"This program ist distributed in the hope that it will be useful, but\n"
"WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n"
"General Public Lincense for more details.\n"
"\n", VERSION, PROG_NAME, COPYRIGHTYEAR);
}

