/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#if defined HAVE_UNISTD_H || defined _LIBC
# include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#if defined HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "kaenguru.h"

#define PORT            5555	/* dafaultport */
#define LOCALHOST       "localhost" /* defaulthost */
#define MAXMSG          512
#define MAXPACKET       1024

/* ------------------------------------------------------------
   Generelle Net-routinen
   ------------------------------------------------------------*/
/* erstellt einen socket. Als bergabewert den port.  Zurck gibt's den
  socket */
int
makeSocket (unsigned short port)
{
  int sock;
  Socketaddr name;
  int optval;

  sock = socket (PF_INET, SOCK_STREAM, 0); /* Create the socket. */
  if (sock < 0)
    KD_GOERR ("Socket failed");
  
  /* Give the socket a name. */
  name.sin_family = AF_INET;
  name.sin_port = htons (port);
  name.sin_addr.s_addr = htonl (INADDR_ANY);
  
  optval = 1;
  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (void *) &optval,
	     sizeof(SO_REUSEADDR));
  
  if (bind (sock, (struct sockaddr *) &name, sizeof (name)) < 0) {
    switch (errno) {
    case EADDRNOTAVAIL:
      KD_GOERR ("Address not available");
    case EADDRINUSE:
      KD_GOERR ("Address is in use");
    case EACCES:
      KD_GOERR ("no access");
    default:
      KD_GOERR ("bind failed");
    }
  }
  return sock;			/* sock-wert zurckgeben */
  
 errhd:
  return -1;
}


/* initialisiert und setzt eine sockaddr zusammen */
int
init_sockaddr (Socketaddr *name, const char *hostname,
	       unsigned short int port)
{
  struct hostent *hostinfo;
  name->sin_family = AF_INET;
  name->sin_port = htons (port);
  hostinfo = gethostbyname (hostname);
  if (hostinfo == NULL)
    KD_GOERR ("Unknown host");
  
  name->sin_addr = *(struct in_addr *) hostinfo->h_addr;
  
  return 1;
 errhd:
  return 0;
}



/* --------------------------------------------------
   I/O-Routinen
   -------------------------------------------------- */
/* liest Daten von filedes und gibt die Anzahl der gelesenen Bytes zurck. 
 * Tritt ein Fehler auf, gibt read_from_sock -1 zurck
 */
int
read_from_sock (int filedes, char **buffer, int *bufsize)
{
  int 
    nbytes, 
    count = 0;
  char c, *p;

  if (*buffer == NULL) {
    *bufsize = 128;
    *buffer = malloc (*bufsize);
  }

  p = *buffer;
  *p = '\0';
  
  do {
    nbytes = read (filedes, &c, 1);

    if (nbytes < 0) {
      KD_errstr = "Error reading socket";
      return -1;
    }
    if (nbytes == 0) {		/* end of file -> end of connection */
      return 0;
    }

    if (count >= *bufsize) {
      char *b = *buffer;
      
      *bufsize += 128;
      *buffer = (char*) realloc ((char*) * buffer, *bufsize);
      p = *buffer + (p - b);
    }
    *p = c;
    p++;
    count++;
  } while (c != KCP_EOL);
  
  *(p - 1) = '\0';
  
  return count;
}

/* schreibt den inhalt eines Buffers auf einen socket.  Rckgabewert ist
   die Anzahl der geschriebenen Zeichen, bzw. -1 fr einen Fehler.  */
int
write_to_sock (int filedes, char *buffer)
{
  int nbytes, len;
  char tmp[5];

  len = strlen (buffer);
  nbytes = write (filedes, buffer, len); 
  if (nbytes < 0)
    goto errhd;
  
  strcpy (tmp, "\r");
  nbytes = write (filedes, tmp, strlen(tmp));
  if (nbytes < 0)
    goto errhd;

  return nbytes;

 errhd:
  KD_errstr = "Error writing socket";
  return -1;
}


/* --------------------------------------------------
   Largebuffer I/O functions
   -------------------------------------------------- */
#define MAXKCPBUF      1024

/* bertrgt einen groen Buffer ber tcp/ip.  Die function geht davon aus,
   da es sich um Daten handelt; darauf wird der Buffer konsequenterweise
   nach nicht-sendbaren Zeichen durchscannt und dementsprechend umgesetzt.
   Die Daten knnen dann nur mit read_blob gelesen werden.

   Daten knnen sowohl text als auch binaer sein; deshalb mu die Laenge
   stets bergeben werden */

int
write_blob (int filedes, char *buffer, int len)
{
  register int pc, tc;		/* pc & tc are length counter */
  char tmp[MAXKCPBUF], *p, *t;

#define wo_initsendcharbuffer()			\
{						\
  t = tmp; 					\
  tc = 0;					\
}

#define wo_sendchar(CH, LIMIT)			\
{						\
  *t = CH; 					\
  t++; 						\
  tc++;						\
  if (tc >= LIMIT) {				\
    write (filedes, tmp, tc);			\
    wo_initsendcharbuffer ();			\
  }						\
}
    
  sprintf (tmp, "%d%c", len, KCP_EOL); /* send length of coming data */
  write (filedes, tmp, strlen (tmp));
  
  p = buffer;			/* p is pointer to the char */
  wo_initsendcharbuffer ();
  
  for (pc = 0; pc < len; pc++, p++) {
    if (*p == KCP_EOL) {
      wo_sendchar (KCP_ESC, MAXKCPBUF);
      wo_sendchar (KCP_SEC_EOL, MAXKCPBUF);
    }
    else if (*p == KCP_ESC) {
      wo_sendchar (KCP_ESC, MAXKCPBUF);
      wo_sendchar (KCP_SEC_ESC, MAXKCPBUF);
    }
    else {
      wo_sendchar (*p, MAXKCPBUF);
    }
  }
  wo_sendchar (KCP_EOL, 0); /* end of data */
  
  return len;
}

int
read_blob (int filedes, char **retbuf)
{
  register int pc;
  int nbytes, bufsize, tmpsize = 0;
  char *buffer = NULL, *p, *tmp = NULL, *token, *tail, c;
  
#define ro_nextchar()				\
({						\
  nbytes = read (filedes, &c, 1);		\
})

#define ro_storechar(C)				\
({						\
  *p = C;					\
  p++;       					\
  pc++;						\
  ro_nextchar ();				\
})

  *retbuf = NULL;


  read_from_sock (filedes, &tmp, &tmpsize);

  token = strtok (tmp, " \r");
  if (!token)
    return -1;
  
  bufsize = strtol (token, &tail, 0);
  free (tmp);

  buffer = (char *) malloc (bufsize);
  
  p = buffer; 
  pc = 0;
  nbytes = read (filedes, &c, 1);  

  do {
    if (c == KCP_ESC) {
      ro_nextchar ();
      
      if (c == KCP_SEC_EOL) {
	ro_storechar (KCP_EOL);
      }
      else if (c == KCP_SEC_ESC) {
	ro_storechar (KCP_ESC);
      }
    }
    else if (c == KCP_EOL) {
      goto donehd;
    }
    else {
      ro_storechar (c);
    }
  } while (5);
  
donehd:
  *retbuf = buffer;
  return pc;
}



/* ------------------------------------------------------------
   Setup Routinen
   ------------------------------------------------------------ */

unsigned short
establishport (unsigned short port)
{
  char *ps = NULL;
  
  if (port == 0) {
    ps = getenv ("KAENGURUPORT"); /* environment variable */
    if (ps) {
      char *tail;
      port = strtol (ps, &tail, 0);
    }
  }
  
  if (port <= 0)
    port = PORT;
  return port;
}


char *
establishhost (char *host)
{
  char * hs = NULL;
  
  if (host == NULL) {		/* get host */
    hs = getenv ("KAENGURUHOST"); /* environmentvariable einholen! */
    if (hs != NULL) {
      host = (char *) strdup (hs);
    }
    else
      host = (char *) strdup (LOCALHOST);
  }
  else
    host = (char *) strdup (host);
  
  return host;
}



