/*
  
  This file is part of the Kaenguru Database System
  Copyright (c) 1997 by Gregor Klinke
  
  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.
  
  This program ist distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public Lincense for more details.

  */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#if defined STDC_HEADERS || defined _LIBC
# include <stdlib.h>
# if defined HAVE_STRING_H
#  include <string.h>
# else
#  include <strings.h>
# endif
#endif
#include <sys/socket.h>

#include "kaenguru.h"

int kaenguru_errno = 0;
char kaenguru_server_errmsg[256] = "";
char *KD_errstr = "";

int
read_key_file (char *file, char **buf, int *bufsize)
{
  FILE *stream;
  char tmp[FILENAME_MAX];
  char *home;
  long lng, sizeread;

  if (file) {
    if (index(file, '/')) {
      strcpy (tmp, file);
    }
    else {
      home = getenv ("HOME");
      if (!home)
	home = "/";
      sprintf (tmp, "%s/%s", home, file);
    }
  }
  else {
    home = getenv ("HOME");
    if (!home)
      home = "/";
    sprintf (tmp, "%s/.kdbdauth", home);
  }

  stream = fopen (tmp, "r");
  if (!stream)
    return 0;
  
  fseek (stream, 0, SEEK_END);	/* get length of file */
  lng = ftell (stream);
  fseek (stream, 0, SEEK_SET);

  if (lng > *bufsize) {
    if (*bufsize == 0)
      *buf = malloc (lng + 1);
    else
      *buf = realloc (*buf, lng + 1);
    *bufsize = lng + 1;
  }
  
  sizeread = fread (*buf, 1, lng, stream);
  (*buf)[lng] = '\0';

  if (sizeread < lng) {
    fclose (stream);
    return -1;
  }

  fclose (stream);
  return 1;
}

KD_conn *
KD_connect (short port, char *host, char *usrname) 
{
  KD_conn *conn;
  int bufsize = 256;
  char *buf = (char *) malloc (bufsize);
  char tmp[1024];
  int retval;
  
  conn = (KD_conn *) malloc (sizeof (KD_conn));
  
  conn->port = establishport (port);	/* get correct port */
  conn->host = establishhost (host);	/* get correct host */
  
  /* Create a socket. */
  conn->sock = socket (PF_INET, SOCK_STREAM, 0);
  if (conn->sock < 0)
    KD_GOERR ("Socket failed");
  
  /* Connect to the daemon. */
  init_sockaddr (&conn->servername,
		 conn->host, conn->port);
  if (0 > connect (conn->sock, (struct sockaddr *) &conn->servername,
		   sizeof (conn->servername)))
    KD_GOERR ("Connection failed");
  
  retval = read_key_file (NULL, &buf, &bufsize);
  if (retval < 0)
    goto errhd;
  if (retval == 0)
    strcpy (buf, "<no key>");
  
  /* now login */
  if (usrname)
    sprintf (tmp, "%s %s %s", KCP_LOGIN, usrname, buf);
  else
    sprintf (tmp, "%s ", KCP_LOGIN);
  if (write_to_sock (conn->sock, tmp) < 0)
    goto errhd;
  
  if (read_from_sock (conn->sock, &buf, &bufsize) < 0)
    goto errhd;
  if (buf[0] != KCP_OKC) {
    free (buf);
    KD_GOERR ("Login failed");
  }
  
  free (buf);
  return conn;
  
errhd:
  free (buf);
  if (conn->host)
    free (conn->host);
  free (conn);
  return NULL;
}

int
KD_close (KD_conn *conn)
{
  char tmp[128];

  if (!conn)
    return -1;
  
  sprintf (tmp, "%s", KCP_QUIT);
  if (write_to_sock (conn->sock, tmp) < 0) {
    if (conn->host)
      free (conn->host);
    free (conn);
    return -1;
  }
  if (conn->host)
    free (conn->host);
  free (conn);
  return 0;
}

int
KD_send (KD_conn *conn, char *command)
{
  int len, tmpsize = 0;
  char *tmp = NULL;
  
  if (!conn)
    return -1;

  if (command) {
    len = strlen (command);

    if (len > 0) {
      tmpsize = len + 64;
      tmp = (char*) malloc (tmpsize);
      sprintf (tmp, "%s %s", KCP_LISP, command);

      if (write_to_sock (conn->sock, tmp) < 0)
	goto errhd;
      if (read_from_sock (conn->sock, &tmp, &tmpsize) < 0)
	goto errhd;
      
      if (tmp[0] != KCP_OKC) {
	char *token = strtok (tmp, " \r\n"); /* get message trail header */
	if (token) {
	  token = strtok (NULL, "\r\n"); /* get errormessage */
	  if (token) {
	    strcpy (kaenguru_server_errmsg, "Send error: ");
	    strcat (kaenguru_server_errmsg, token);
	  }
	  else
	    strcpy (kaenguru_server_errmsg,
		    "Send error: No message support.\n");
	}
	/* tmp is freed at errhd:, KD_SENDERR jumps there */
	KD_SENDERR();
      }
      free (tmp);
    }
  }
  return 0;

errhd:
  free (tmp);
  return -1;
}

#define KDR_ATOM       0
#define KDR_SEXP       1
#define KDR_STOP       2

#define SET_NOBJ_NUM(TOKEN, NOBJ, TYP)				\
({								\
  char *_tail;							\
  TOKEN = strtok (NULL, " \r");	/* this should be the value */	\
  if (TOKEN)							\
    NOBJ->data.num = strtol (TOKEN, &_tail, 0);			\
  else								\
    NOBJ->data.num = 0;						\
  NOBJ->typ = TYP;						\
})
#define SET_NOBJ_STR(CONN, TOKEN, NOBJ, TYP)		\
({							\
  char *_buf = NULL;					\
  int _readsize = read_blob (CONN->sock, &_buf);	\
  if (_readsize < 0) {					\
    free (_buf);					\
    KD_GOERR ("Error reading k4 value");		\
  }							\
  NOBJ->typ = TYP;					\
  NOBJ->data.str = _buf;				\
})
#define SET_NOBJ_UNUM(TOKEN, NOBJ, TYP)				\
({								\
  char *_tail;							\
  TOKEN = strtok (NULL, " \r");	/* this should be the value */	\
  if (TOKEN)							\
    NOBJ->data.unum = strtol (TOKEN, &_tail, 0);		\
  else								\
    NOBJ->data.unum = 0;					\
  NOBJ->typ = TYP;						\
})

KD_data *
KD_getval (KD_conn *conn)
{
  int tmpsize = 256;
  char *tmp = (char *) malloc (tmpsize);
  KD_data *dobj, *nobj, *rootobj;
  int readmode = KDR_ATOM;
  int ignoreobj;
  
  if (!conn)
    goto errhd;

  sprintf (tmp, "%s", KCP_GETVAL);
  if (write_to_sock (conn->sock, tmp) < 0)
    goto errhd;
  if (read_from_sock (conn->sock, &tmp, &tmpsize) < 0)
    goto errhd;


  dobj = rootobj = NULL;

  do {

    ignoreobj = 0;
    
    nobj = (KD_data *) malloc (sizeof (KD_data));
    nobj->next = NULL;
    nobj->typ = KD_NONE;
    
    if (tmp[0] == KCP_OKC) {
      char *token;
      
      token = strtok (tmp, " \r");
      if (token) {
	token = strtok (NULL, " \r"); /* this should be a INT, BLIST ... */
	
	if (token) {
	  if (strcmp (token, "INT") == 0) {
	    SET_NOBJ_NUM (token, nobj, KD_INT);
	  }
	  else if (strcmp (token, "UINT") == 0) {
	    SET_NOBJ_UNUM (token, nobj, KD_UINT);
	  }
	  else if (strcmp (token, "STR") == 0) {
	    SET_NOBJ_STR (conn, token, nobj, KD_STR);
	  }
	  else if (strcmp (token, "SYM") == 0) {
	    SET_NOBJ_STR (conn, token, nobj, KD_SYM);
	  }
	  if (strcmp (token, "CHAR") == 0) {
	    SET_NOBJ_NUM (token, nobj, KD_CHAR);
	  }
	  if (strcmp (token, "BOOL") == 0) {
	    SET_NOBJ_NUM (token, nobj, KD_BOOL);
	  }
	  else if (strcmp (token, "DFIELD") == 0) {
 	    nobj->typ = KD_DFIELD;
 	    nobj->data.dfield.size = 0;
 	    nobj->data.dfield.blobid = 0;
	    token = strtok (NULL, " \r"); /* this should be the size */
	    if (token) {
	      char *tail;
	      nobj->data.dfield.size = strtol (token, &tail, 0);
	      token = strtok (NULL, " \r"); /* this should be the id */
	      if (token) {
		nobj->data.dfield.blobid = strtol (token, &tail, 0);
	      }
	    }
	  }
	  else if (strcmp (token, "BLIST") == 0) {
	    readmode = KDR_SEXP;
	    ignoreobj = 1;
	  }
	  else if (strcmp (token, "ELIST") == 0) {
	    readmode = KDR_STOP;
	    ignoreobj = 1;
	  }
	  else if (strcmp (token, "VOID") == 0) {
	    ignoreobj = 1;
	  }

	  if (!ignoreobj) {
	    if (dobj) {
	      dobj->next = nobj;
	      dobj = nobj;
	    }
	    else {
	      rootobj = nobj;
	      dobj = nobj;
	    }
	  }
	  else {
	    if (nobj)
	      free (nobj);
	  }
	  
	} /* strtok: get INT, BLIST, ... */
      }	/* strtok: get +OK */
    } /* if tmp[0] = '+' ... */
    else {
      KD_GOERR ("Error reading k4 value");
    }
    
    if (readmode == KDR_SEXP) {
      if (read_from_sock (conn->sock, &tmp, &tmpsize) < 0) /* get next obj */
	goto errhd;
    }
    
  } while (readmode == KDR_SEXP);
  
  free (tmp);
  return rootobj;
  
errhd:

  free (tmp);  
  return NULL;
}

void
KD_free (KD_data *ptr)
{
  KD_data *nptr;
  while (ptr) {
    nptr = ptr->next;
    if ((ptr->typ == KD_STR) &&
	(ptr->data.str)) {
      free (ptr->data.str);
    }
    free (ptr);
    ptr = nptr;
  }
}

char *
KD_talklisp (KD_conn *conn)
{
  int readsize;
  int tmpsize = 256;
  char *tmp = (char*) malloc (tmpsize);
  char *buf = NULL;

  if (!conn)
    goto errhd;

  sprintf (tmp, "%s", KCP_TALKLISP);
  if (write_to_sock (conn->sock, tmp) < 0)
    goto errhd;
  if (read_from_sock (conn->sock, &tmp, &tmpsize) < 0)
    goto errhd;
  
  if (tmp[0] == KCP_OKC) {
    free (tmp);
    
    readsize = read_blob (conn->sock, &buf);
    if (readsize < 0) {
      free (buf);
      KD_GOERR ("Error reading talklisp value");
    }
    return buf;
  }
  else
    KD_GOERR ("Error reading talklisp value");
  
 errhd:
  free (tmp);
  return NULL;
}

char *
KD_read_blob (KD_conn *conn, int blobid, int *size)
{
  int tmpsize = 128;
  char *tmp = (char*) malloc (tmpsize);

  if (!conn)
    goto errhd;

  sprintf (tmp, "%s %d", KCP_READ_DFIELD, blobid);
  if (write_to_sock (conn->sock, tmp) < 0)
    goto errhd;
  if (read_from_sock (conn->sock, &tmp, &tmpsize) < 0)
    goto errhd;
  
  if (tmp[0] == KCP_OKC) {
    char *buf = NULL;

    free (tmp);

    *size = read_blob (conn->sock, &buf);
    if (*size < 0) {
      free (buf);
      KD_GOERR ("Error reading BLOB");
    }
    return buf;
  }
  
 errhd:
  free (tmp);
  *size = 0;
  return NULL;
}

int
KD_store_blob (KD_conn *conn, int blobid, char *buf, int size)
{
  char tmp[128];

  if (!conn)
    goto errhd;
  
  if ((size > 0) && buf) {
    sprintf (tmp, "%s %d", KCP_STORE_DFIELD, blobid);
    if (write_to_sock (conn->sock, tmp) < 0)
      goto errhd;
    
    write_blob (conn->sock, buf, size);
  }
  else {
    KD_GOERR ("Won't store empty BLOB");
  }
  return size;

 errhd:
  return 0;
}

char *
KD_geterrmsg ()
{
  if (kaenguru_errno == KD_SEND) {
    return kaenguru_server_errmsg;
  }
  else {
    return KD_errstr;
  }
  return NULL;
}

