/* -*- Mode: C -*- */
/* vm.cc - PageFile implementation (virtual memory management)
 * Created by Robert Heller on Sat Dec  7 00:19:52 1991
 * Updated for Version 2.0 on Sat Apr 26 15:30:17 1997
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Class library implementation code
 * ------------------------------------------------------------------
 * Modification History:
 * $Log: vm.cc,v $
 * Revision 2.3  1998/04/21 15:12:18  heller
 * Update copyright notice.
 *
 * Revision 2.2  1997/06/29 19:14:59  heller
 * Fixes for portablity with CW
 *
 * Revision 2.1  1997/04/26 21:59:06  heller
 * *** empty log message ***
 *
 * Revision 1.1  1997/04/26 21:57:43  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 *    Home Librarian Database -- a program for maintaining a database
 *                               for a home library
 *    Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 *			51 Locke Hill Road
 *			Wendell, MA 01379-9728
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */
#include <iostream.h>
#include <vm.h>
 
static char ID[] = "$Id: vm.cc,v 2.3 1998/04/21 15:12:18 heller Rel $";

#ifdef macintosh
#define NoErr 0
#endif

DiskPage::DiskPage (const DiskPage& a) {record_address = a.record_address;}
DiskPage::DiskPage (LongInt addr) {record_address = addr;}
Boolean operator == (const DiskPage a,const DiskPage b)
	{return(a.record_address == b.record_address);}
Boolean operator != (const DiskPage a,const DiskPage b)
	{return(a.record_address != b.record_address);}
DiskPage& DiskPage::operator = (const DiskPage a)
	{record_address = a.record_address;return *this;}
DiskPage& DiskPage::operator = (long a)
	{record_address = a;return *this;}
				
//const PageFile::NumPTEntries = 50;
PageFile::PageFile ()
{
	isopen = false;
	numpagesincore = 0;
	for (int i = 0;i<NumPTEntries;i++) {
		pagetable[i].isdirty = false;
		pagetable[i].corepage = 0;
	}
}

PageFile::~PageFile () 
{
	close();
}

Page * PageFile::operator [] (LongInt addr) 
{
	DiskPage page(addr);
	return((FindPage(page).corepage));
}

Page * PageFile::operator [] (DiskPage page) 
{
	return((FindPage(page).corepage));
}

PTEntry & PageFile::operator () (LongInt addr) 
{
	DiskPage page(addr);
	return(FindPage(page));
}

PTEntry & PageFile::operator () (DiskPage page) 
{
	return(FindPage(page));
}
	
// File open function.
// Open or create the page file
HLEnums::OpenStatus PageFile::open(const FileName filename,HLEnums::Direction direction,
#ifdef macintosh
			OSType creator,
			int flags,
#else
			int flags,
			int mode,
#endif
			Boolean create_if
					)
{
	HLEnums::OpenStatus result = HLEnums::failure;		// assume failure
#ifdef macintosh
	OSErr err = NoErr;
	err = FSpOpenDF(filename,flags,&fd);
	// failure?
	if (err != NoErr)
	{
		if (create_if && (direction == HLEnums::out || direction  == HLEnums::inout)) {
			// yes.  create a new file if posible
			err = FSpCreate(filename,creator,'BINA',smSystemScript);
			if (err != NoErr) return(HLEnums::failure);
			err = FSpOpenDF(filename,flags,&fd);
			if (err != NoErr) return(HLEnums::failure);
			result = HLEnums::opennew;
		} else return(HLEnums::failure);
	} else result = HLEnums::openold;       // old file open
#else
	fd = ::open(filename,flags);		// try to open the file
	// failure?
	if (fd < 0) {
		// yes.  create a new file if posible
		if (create_if && (direction == HLEnums::out || direction  == HLEnums::inout)) {
#ifdef OSK
			fd = create(filename,flags,mode); // OSK specific
#else
	 	  	fd = ::open(filename,flags|O_CREAT,mode);  // vanila UNIX
#endif
			// still a problem?  give up
			if (fd < 0) return(HLEnums::failure);
			// else note that it is a new file
			result = HLEnums::opennew;
		} else return(HLEnums::failure);
	} else result = HLEnums::openold;			// old file open
#endif
	PageFile::direction = direction;		// remember direction
	numpagesincore = 0;				// empty WS
	isopen = true;					// yes, it is open
	return(result);
}

// close a PageFile
void PageFile::close()
{
	// if the file is open...
 	if (isopen) {
		for (int i = 0; i < numpagesincore; i++) {
			// write out any dirty pages
			if (pagetable[i].isdirty) {
				if (!PageWrite(pagetable[i])) {
					int error = errno;
					cerr << "%%% PageWrite error. Page no. = " <<
						pagetable[i].diskpage.record_address <<
						"\n";
					exit(error);
				}
				pagetable[i].isdirty = false;
			}
			// free up and used core memory
			if (pagetable[i].corepage != 0) {
				delete pagetable[i].corepage;
				pagetable[i].corepage = 0;
			}
		}
		// really close the file
#ifdef macintosh
		(void) FSClose(fd);
#else
		::close(fd);
#endif
		// nope, not open anymore
		isopen = false;
		// no pages core resident
		numpagesincore = 0;
	}
}

void PageFile::FlushPages()
{
	// if the file is open...
 	if (isopen) {
		for (int i = 0; i < numpagesincore; i++) {
			// write out any dirty pages
			if (pagetable[i].isdirty) {
				if (!PageWrite(pagetable[i])) {
					int error = errno;
					cerr << "%%% PageWrite error. Page no. = " <<
						pagetable[i].diskpage.record_address <<
						"\n";
					exit(error);
				}
				pagetable[i].isdirty = false;
			}
		}
	}
}
	
// Read in a page.  ptentry says which page.
Boolean PageFile::PageRead(PTEntry& ptentry)
{
	//cerr << "*** PageFile::PageRead(): page's disk address is " << 
	//	ptentry.diskpage.record_address << "\n";
	// try to seek to page
#ifdef macintosh
	OSErr err = SetFPos(fd,fsFromStart,ptentry.diskpage.record_address);
	if (err != NoErr) return(false);
#else
	LongInt pos = lseek(fd,ptentry.diskpage.record_address,SEEK_SET);
	// can't seek? fail
	if (pos == -1L) return(false);
#endif
	// do we have memory available?
	if (ptentry.corepage == 0) {
		ptentry.corepage = new Page;	// allocate memory if needed
	}
	//cerr << "*** -: page's core address is 0x" <<
	//	form("%08x",ptentry.corepage) << "\n";
	// read in page
#ifdef macintosh
	LongInt len = sizeof(Page);
	err = FSRead(fd,&len,(void*)(ptentry.corepage));
	if (err != NoErr) return(false);
#else
	if (read(fd,(char *)(ptentry.corepage),sizeof(Page)) <
		sizeof(Page)) return(false);
#endif
	// page is ready and clean
	ptentry.isdirty = false;
	return(true);
}
// Write out a page
Boolean PageFile::PageWrite(PTEntry& ptentry)
{
	//cerr << "*** PageFile::PageWrite(): page's disk address is " << 
	//	ptentry.diskpage.record_address <<
	//	", and its core address is 0x" <<
	//	form("%08x",ptentry.corepage) << "\n";
	// seek to disk space
#ifdef macintosh
	OSErr err = SetFPos(fd,fsFromStart,ptentry.diskpage.record_address);
	if (err != NoErr) return(false);
#else
	LongInt pos = lseek(fd,ptentry.diskpage.record_address,SEEK_SET);
	// if seek failure, return failure
	if (pos == -1L) return(false);
#endif
	// write the page
#ifdef macintosh
	LongInt len = sizeof(Page);
	err = FSWrite(fd,&len,(void*)(ptentry.corepage));
	if (err != NoErr) return(false);
#else
	if (write(fd,(char *)(ptentry.corepage),sizeof(Page)) <
		sizeof(Page)) return(false);
#endif
	// no LongInter dirty...
	ptentry.isdirty = false;
	return(true);
}

// Locate a page.  Read the page in if necessary, dumping the least recently
// referenced page to make room
PTEntry & PageFile::FindPage(DiskPage diskpage)
{
	//cerr << "*** in PageFile::FindPage - looking for page " <<
	//	diskpage.record_address << "\n";
	// is page already resident?
	for (int i = 0; i < numpagesincore; i++) {
		if (pagetable[i].diskpage == diskpage) {
			// yes.  promote page to end of table, if it is not
			// already there
			//cerr << "*** -: Page is resident.\n";
			if (i < (numpagesincore-1)) {
				temp = pagetable[i];
				for (int j = i+1; j < numpagesincore; j++) {
					pagetable[j-1] = pagetable[j];
				}
				pagetable[numpagesincore-1] = temp;
			}
			//cerr << "*** -: core address is 0x" <<
			//	form("%08x",pagetable[numpagesincore-1].corepage) <<
			//	"\n";
			// return page table entry
			return(pagetable[numpagesincore-1]);
		}
	}
	//cerr << "*** -: Page is non-resident.\n";
	// page not in core.  Is WS (page table) full?
	if (numpagesincore < NumPTEntries) {
		//cerr << "*** -: WS grown.\n";
		// if not, use new entry at the end and increase WS
		pagetable[numpagesincore].diskpage = diskpage;
		if (!PageRead(pagetable[numpagesincore])) {
			int error = errno;
			cerr << "%%% PageRead error: page no. = " <<
				diskpage.record_address << "\n";
			exit(error);
		}
		numpagesincore++;
		//cerr << "*** -: core address is 0x" <<
		//	form("%08x",pagetable[numpagesincore-1].corepage) <<
		//	"\n";
		return(pagetable[numpagesincore-1]);
	} else {	// otherwise, bump a page
		// if old page is dirty, write it out
		if (pagetable[0].isdirty == true) {
			//cerr << "*** -: Page out " <<
			//	pagetable[0].diskpage.record_address <<
			//	"\n";
			if (!PageWrite(pagetable[0])) {
				int error = errno;
				cerr << "%%% PageWrite error: page no. = " <<
					pagetable[0].diskpage.record_address <<
					"\n";
				exit(error);
			}
		}
		// shift pages down
		temp = pagetable[0];
		for (int j = 1; j < numpagesincore; j++) {
			pagetable[j-1] = pagetable[j];
		}
		pagetable[numpagesincore-1] = temp;
		// setup page
		pagetable[numpagesincore-1].diskpage = diskpage;
		if(!PageRead(pagetable[numpagesincore-1])) {
			int error = errno;
			cerr << "%%% PageRead error: page no. = " <<
				diskpage.record_address << "\n";
			exit(error);
		}
		//cerr << "*** -: core address is 0x" <<
		//	form("%08x",pagetable[numpagesincore-1].corepage) <<
		//	"\n";
		return(pagetable[numpagesincore-1]);
	}
}

// allocate a new, fresh page
DiskPage PageFile::NewPage()
{
	DiskPage temppage;		// new page
	LongInt fill;			// filler to make page address fall
					// on a sector boundry
	static char buffer[SectorSize];	// filler
	static Boolean init = false;	// init flag
	if (init == false) {		// filler initialization
		memset(buffer,'\0',SectorSize);
		init = true;
	}
	//cerr << "*** in PageFile::NewPage()\n";
	// find eof
#ifdef macintosh
	LongInt eofPos;
	OSErr err = GetEOF(fd,&eofPos);
	temppage = eofPos;
	if (err != NoErr) return temppage;   // failure??
#else
	temppage = lseek(fd,0,SEEK_END);
	if (temppage == -1L) return temppage;	// failure??
#endif
	// compute filler size
	fill = SectorSize - (temppage.record_address & (SectorSize-1));
	// is filler even needed?
	if (fill == SectorSize) fill = 0;
	if (fill != 0) {
		// yep.  write filler and re-compute page address.
#ifdef macintosh
		LongInt len = fill;
		err = FSWrite(fd,&len,(void*)buffer);
		err = GetEOF(fd,&eofPos);
		temppage = eofPos;
#else
		write(fd,buffer,fill);
		temppage = lseek(fd,0,SEEK_END);
#endif
	}
	//cerr << "*** -: new page at " << temppage.record_address << "\n";
	// find a home for this page (code copied from FindPage, with
	// minor mode)
	if (numpagesincore < NumPTEntries) {
		//cerr << "*** -: WS grown.\n";
		pagetable[numpagesincore].diskpage = temppage;
		if (pagetable[numpagesincore].corepage == 0) {
			pagetable[numpagesincore].corepage = new Page;
		}
		//cerr << "*** -: core address is 0x" <<
		//	form("%08x",pagetable[numpagesincore].corepage) <<
		//	"\n";
		memset((char *) pagetable[numpagesincore].corepage,
			'\0',SectorSize);
		if (!PageWrite(pagetable[numpagesincore])) {
			int error = errno;
			cerr << "%%% PageWrite error: page no. = " <<
				pagetable[numpagesincore].diskpage.record_address <<
				"\n";
			exit(error);
		}
		pagetable[numpagesincore].isdirty = false;
		numpagesincore++;
		return(pagetable[numpagesincore-1].diskpage);
	} else {
		if (pagetable[0].isdirty == true) {
			//cerr << "*** -: Page out " <<
			//	pagetable[0].diskpage.record_address <<
			//	"\n";
			if (!PageWrite(pagetable[0])) {
				int error = errno;
				cerr << "%%% PageWrite error: page no. = " <<
					pagetable[0].diskpage.record_address <<
					"\n";
				exit(error);
			}
		}
		temp = pagetable[0];
		for (int j = 1; j < numpagesincore; j++) {
			pagetable[j-1] = pagetable[j];
		}
		pagetable[numpagesincore-1] = temp;
		pagetable[numpagesincore-1].diskpage = temppage;
		memset((char *) pagetable[numpagesincore-1].corepage,
			'\0',SectorSize);
		//cerr << "*** -: core address is 0x" <<
		//	form("%08x",pagetable[numpagesincore-1].corepage) <<
		//	"\n";
		if (!PageWrite(pagetable[numpagesincore-1])) {
			int error = errno;
			cerr << "%%% PageWrite error: page no. = " <<
				pagetable[numpagesincore-1].diskpage.record_address <<
				"\n";
			exit(error);
		}
		pagetable[numpagesincore-1].isdirty = false;
		return(pagetable[numpagesincore-1].diskpage);
	}
}

// Read in a data record
int PageFile::ReadRecord(DiskRecord& record, char *buffer, int buffsize)
{
#ifdef macintosh
	// seek to data offset
	OSErr err = SetFPos(fd,fsFromStart,record.record_address);
	if (err != NoErr) return(-1);
	// read in data (or at as much as will fit in record
	if (buffsize >= record.record_size) {
		LongInt len = record.record_size;
		err = FSRead(fd,&len,(void*)buffer);
		if (err != NoErr) return(-1);
		else return(len);
	} else
	{
		LongInt len = buffsize;
		err = FSRead(fd,&len,(void*)buffer);
		if (err != NoErr) return(-1);
		else return(len);
	}
#else
	// seek to data offset
	LongInt pos = lseek(fd,record.record_address,SEEK_SET);
	// seek failure?
	if (pos == -1) return(-1);
	// read in data (or at as much as will fit in record
	if (buffsize >= record.record_size) {
		return(read(fd,buffer,record.record_size));
	} else {
		return(read(fd,buffer,buffsize));
	}
#endif
}

// Write a redord out
DiskRecord PageFile::WriteRecord(char *buffer, int buffsize)
{
	DiskRecord newrec;	// new disk record
#ifdef macintosh
	// seek to EOF
	OSErr err = GetEOF(fd,&newrec.record_address);
	if (err != NoErr) return 0;
	LongInt len = buffsize;
	err = FSWrite(fd,&len,(void*)buffer);
	if (err != NoErr) return 0;
	newrec.record_size = len;
#else
	// seek to EOF
	newrec.record_address = lseek(fd,0,SEEK_END);
	// seek failure??
	if (newrec.record_address == -1L) {
		return 0;
	}
	//cerr.form("*** in WriteRecord(): newrec.record_address = 0x%08x\n",
	//	 newrec.record_address);
	// write out record
	newrec.record_size = write(fd,buffer,buffsize);
	if (newrec.record_size < 0) return 0;
	//cerr.form("*** in WriteRecord(): buffsize = %d, newrec.record_size = %d\n",
	//	buffsize,newrec.record_size);
#endif
	return(newrec);
}

// update old record or write new one
DiskRecord PageFile::ReWriteRecord(DiskRecord& record, char *buffer, int buffsize)
{

	//cerr << "*** PageFile::ReWriteRecord(): record = (:SIZE " << 
	//	record.record_size << " :ADDR " << record.record_address << 
	//	"), buffsize = " << buffsize << "\n";
	// will new record fill in old place?
	if (buffsize > record.record_size) {
		// no, create new record
		return(PageFile::WriteRecord(buffer,buffsize));
	} else {
		// yes, rewrite old record
#ifdef macintosh
		OSErr err = SetFPos(fd,fsFromStart,record.record_address);
		if (err != NoErr) return(0);
		LongInt len = buffsize;
		err = FSWrite(fd,&len,(void*)buffer);
		if (err != NoErr) return 0;
		record.record_size = len;
#else
		LongInt pos;
		pos = lseek(fd,record.record_address,SEEK_SET);
		if (pos == -1) return(0);
		record.record_size = write(fd,buffer,buffsize);	
		if (record.record_size < 0) return 0;
#endif
		return(record);
	}
	
}

