/* -*- Mode: C -*- */
/* vBTree.h - virtual BTrees - a modification of the BTree class
 *	      described in "C++, a guide for C programmers", by
 *	      Sharam Hekmatpour, (c) 1990 by Prentice Hall of
 *	      Australia Pty Ltd.
 *
 * Created by Robert Heller on Fri Dec  6 20:55:06 1991
 * Updated for Version 2.0 on Sat Apr 26 18:01:51 1997
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Header Files
 * ------------------------------------------------------------------
 * Modification History:
 * $Log: vBTree.h,v $
 * Revision 2.5  1998/05/16 23:43:28  heller
 * Add indexing to the documentation.
 *
 * Revision 2.4  1998/04/21 15:12:18  heller
 * Update copyright notice.
 *
 * Revision 2.3  1997/07/27 04:10:07  heller
 * Fix totally silly boundary condition
 *
 * Revision 2.2  1997/07/15 19:20:06  heller
 * Add documentation
 *
 * Revision 2.1  1997/07/06 21:44:54  heller
 * Misc. changes
 *
 * Revision 2.0  1997/07/01 01:42:52  heller
 * *** empty log message ***
 *
 * Revision 1.1  1997/06/29 19:14:59  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 *    Home Librarian Database -- a program for maintaining a database
 *                               for a home library
 *    Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 *			51 Locke Hill Road
 *			Wendell, MA 01379-9728
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */
#ifndef _VBTREE_
#define _VBTREE_
#include <common.h>
#include <vm.h>
#include <Card.h>

//@ChapterName:Virtual B-Tree Class

/*@ManDoc:
 \typeout{Generated from: $Id: vBTree.h,v 2.5 1998/05/16 23:43:28 heller Rel $} 
 \index{vBTree!C++ Interface|(}
 \index{vBTree!C++ Interface!vBTree class|bold}
 This class implements a Library Card Catalog as a group a four
 (virtual) BTrees of Order 10.
 The nodes of the BTrees are handled using the virtual memory management
 of the PageFile class.
 */
class vBTree : public PageFile {
public:
	//@ManDoc: Callback pointer type for user callback functions.
	//@Doc: \index{vBTree!C++ Interface!UserData}
	typedef const void * UserData;
	//@Man: TravFunc
	//@Type: int (*)(CoreItem*, int, UserData)
	//@Doc: Callback function type for tree traversal.
	//@Doc: \index{vBTree!C++ Interface!TravFunc}
	typedef int  ( * TravFunc )( CoreItem*, int, UserData );
	//@Man: ErrFun
	//@Type: void (*)(HLEnums::ErrKind, const char*, UserData)
	//@Doc: Callback function for error handling.
	//@Doc: \index{vBTree!C++ Interface!ErrFun}
	typedef void ( * ErrFun )( HLEnums::ErrKind, const char*, UserData);
private:
	//@ManMemo: open status
	HLEnums::OpenStatus openstat;
	//@ManMemo: home (header) block
	HomeBlock homeblock;
	//@ManMemo: has the home block been modified?
	Boolean homedirty;
	//@ManMemo: scratch buffer
	Item* buf;
	/*@ManMemo: Last page searched.  Set by SearchAux,
	  used and updated by SearchAgain. 
	 */
	DiskPage LastSearchPage, CurrentSearchPage;
	//@ManMemo: Last search key.
	Key LastSearchKey;
	//@ManMemo: Last search index
	int LastSearchIdx;
	//@ManMemo: Last search type
	HLEnums::LSType LastSearchType;
	//@ManMemo: Error hander function
	ErrFun errFun;
	//@ManMemo: Callback data for user error handler.
	UserData errData;
	//@ManMemo: library file version;
	int LibrVersion;
	//@ManMemo: Can we write???
	Boolean CanWriteP;
	//@ManMemo: CardType to card type name mapping vectors.
	struct CardTypes {
		Card::CardType t;
		const char* n;
	} cardtypenames[Card::NumCardTypes];
	static const CardTypes BaseCardTypeNames[Card::NumCardTypes];
	//@ManMemo: LocationType to location type name mapping vectors.
	struct LocationTypes {
		Card::LocationType t;
		const char* n;
	} locationtypenames[Card::NumLocationTypes];
	static const LocationTypes BaseLocationTypeNames[Card::NumLocationTypes];
	//@ManMemo: Category code mapping vectors.
	struct CategoryCodes {
		unsigned char c;
		const char *n;
	} *categorycodes;
	//@ManMemo: number of category codes in use, number allocated.
	int numcatcodes, allocatcodes;
	//@ManMemo: Category allocation size.
	enum {CCAllocSize = 64};
	//@ManMemo: head of user defined code blocks.
	UserDefinedCodesInCore *udefcodes;
	//@ManMemo: internal function to set a card type name.
	void SetCardType(Card::CardType tt,const char *nn);
	//@ManMemo: internal function to set a location type name
	void SetLocationType(Card::LocationType tt, const char *nn);
	//@ManMemo: internal function to set a category name.
	void SetCategory(unsigned char cc,const char *nn);
	//@ManMemo: User Defined Code string allocator.
	char *AllocateUDC(int len);
	//@ManMemo: User Defined Code string dealloccator.
	void FreeUDC(char *ptr);
	//@ManMemo: new page allocator.
	DiskPage NewPage	();
	//@ManMemo: Free up a page.
	void FreePage		(const DiskPage page);
	//@ManMemo: Searching functions
	Boolean SearchAux	(const DiskPage dtree,const Key key,CoreItem* item);
	Boolean SearchAgain	(CoreItem* item,Boolean DontGoDown = false);
	Boolean BinarySearch	(const Page* node,const Key key,int* idx);
	//@ManMemo: Insertion function
	Item* InsertAux		(Item* item,const DiskPage dnode);
	//@ManMemo: Deletion functions
	void DeleteAux1		(const Key key,const DiskPage dnode,Boolean* underflow);
	void DeleteAux2		(const DiskPage dparent,const DiskPage dnode,int idx,
				 Boolean* underflow);
	void Underflow		(const DiskPage dnode,const DiskPage dchild,int idx,
				 Boolean* underflow);
	//@ManMemo: Parentage adjustment
	void AdjustParent	(const DiskPage dparent);
	//@ManMemo: Item vector hacking
	int CopyItems		(const Item* src,Item* dest,int count);
	int InsertItem		(const Item* item,Item* items,int idx,int size);
	int DeleteItem		(Item* items,int idx,int size);
	//@ManMemo: Raw tree print function
	void PrintAux		(const DiskPage node,int margin);
	//@ManMemo: Page counter
	int CountPagesAux	(const DiskPage node);
	//@ManMemo: Key compare function
	int CompareKeys		(const Key keypat, const Key key);
	//@ManMemo: Tree traversal function
	int TravAux		(const DiskPage node,TravFunc tfun,int level,UserData ud);
public:
	//@ManDoc: Writable flag function.
	//@Doc: \index{vBTree!C++ Interface!Writable}
	Boolean Writable () const {return CanWriteP;}
	//@ManDoc: Library Version function.
	//@Doc: \index{vBTree!C++ Interface!LibraryVersion}
	int LibraryVersion () const {return LibrVersion;}
	//@ManDoc: Open status function.
	//@Doc: \index{vBTree!C++ Interface!OpenStat}
	HLEnums::OpenStatus OpenStat() const {return openstat;}
	//@ManDoc: Error handler access.
	//@Doc: \index{vBTree!C++ Interface!ErrorFun}
	const ErrFun ErrorFun () const {return errFun;}
	//@ManDoc: Error handler data access.
	//@Doc: \index{vBTree!C++ Interface!ErrorData}
	const UserData ErrorData () const {return errData;}
	//@ManDoc: Set the error handler.
	//@Doc: \index{vBTree!C++ Interface!SetErrorFun}
	void SetErrorFun (ErrFun efun,const UserData edata);
	//@ManDoc:  Error handler.
	//@Doc: \index{vBTree!C++ Interface!Error}
	void Error	(HLEnums::ErrKind err,const char* msg) const;
	//@ManDoc: Open file function.
	//@Doc: \index{vBTree!C++ Interface!open}
	HLEnums::OpenStatus open (const FileName filename,HLEnums::OpenMode mode = HLEnums::ReadOnly,
				 int nfree = MaxNumFree / 10);
	//@ManDoc: Default Constructor.
	//@Doc: \index{vBTree!C++ Interface!Default Constructor}
	     vBTree		();
	//@ManDoc: Constructor that opens the file.
	//@Doc: \index{vBTree!C++ Interface!Constructor that opens the file.}
	     vBTree		(const FileName filename,HLEnums::OpenMode mode = HLEnums::ReadOnly,
				 int nfree = MaxNumFree / 10);
	//@ManDoc: Destructor.
	//@Doc: \index{vBTree!C++ Interface!Destructor}
	    ~vBTree		();
	//@ManDoc: Search Id tree.
	//@Doc: \index{vBTree!C++ Interface!SearchId}
	Boolean SearchId	(const Key key,CoreItem* item)
					{LastSearchType = HLEnums::id;
					 LastSearchIdx = -1;
					 return(SearchAux(homeblock.IdRoot,key,
							  item));}
	//@ManDoc: Search Id tree for more matches.
	//@Doc: \index{vBTree!C++ Interface!SearchIdAgain}
	Boolean SearchIdAgain	(CoreItem* item)
					{return(LastSearchType == HLEnums::id && 
						SearchAgain(item));}
	//@ManDoc: Search Title tree.
	//@Doc: \index{vBTree!C++ Interface!SearchTitle}
	Boolean SearchTitle	(const Key key,CoreItem* item)
					{LastSearchType = HLEnums::title;
					 LastSearchIdx = -1;
					 return(SearchAux(homeblock.TitleRoot,
							  key,item));}
	//@ManDoc: Search Title tree for more matches.
	//@Doc: \index{vBTree!C++ Interface!SearchTitleAgain}
	Boolean SearchTitleAgain (CoreItem* item)
					{return(LastSearchType == HLEnums::title &&
						SearchAgain(item));}
	//@ManDoc: Search Author tree.
	//@Doc: \index{vBTree!C++ Interface!SearchAuthor}
	Boolean SearchAuthor	(const Key key,CoreItem* item)
					{LastSearchType = HLEnums::author;
					 LastSearchIdx = -1;
					 return(SearchAux(homeblock.AuthorRoot,
							  key,item));}
	//@ManDoc: Search Author tree for more matches.
	//@Doc: \index{vBTree!C++ Interface!SearchAuthorAgain}
	Boolean SearchAuthorAgain (CoreItem* item)
					{return(LastSearchType == HLEnums::author &&
						SearchAgain(item));}
	//@ManDoc: Search Subject tree.
	//@Doc: \index{vBTree!C++ Interface!SearchSubj}
	Boolean SearchSubj	(const Key key,CoreItem* item)
					{LastSearchType = HLEnums::subj;
					 LastSearchIdx = -1;
					 return(SearchAux(homeblock.SubjRoot,
							  key,item));}
	//@ManDoc: Search Subject tree for more matches.
	//@Doc: \index{vBTree!C++ Interface!SearchSubjAgain}
	Boolean SearchSubjAgain	(CoreItem* item)
					{return(LastSearchType == HLEnums::subj &&
						SearchAgain(item));}
	//@ManDoc: Insert into Id tree.
	//@Doc: \index{vBTree!C++ Interface!InsertId}
	DiskRecord InsertId	(const Key key,const Record* newdata);
	//@ManDoc: Insert into Title tree.
	//@Doc: \index{vBTree!C++ Interface!InsertTitle}
	DiskRecord InsertTitle	(const Key key,const Record* newdata);
	//@ManDoc: Insert into Author tree.
	//@Doc: \index{vBTree!C++ Interface!InsertAuthor}
	DiskRecord InsertAuthor	(const Key key,const Record* newdata);
	//@ManDoc: Insert into Subject tree.
	//@Doc: \index{vBTree!C++ Interface!InsertSubj}
	DiskRecord InsertSubj	(const Key key,const Record* newdata);
	//@ManDoc: Delete from Id tree.
	//@Doc: \index{vBTree!C++ Interface!DeleteId}
	void DeleteId		(const Key key);
	//@ManDoc: Delete from Title tree.
	//@Doc: \index{vBTree!C++ Interface!DeleteTitle}
	void DeleteTitle	(const Key key);
	//@ManDoc: Delete from Author tree.
	//@Doc: \index{vBTree!C++ Interface!DeleteAuthor}
	void DeleteAuthor	(const Key key);
	//@ManDoc: Delete from Subject tree.
	//@Doc: \index{vBTree!C++ Interface!DeleteSubj}
	void DeleteSubj		(const Key key);
	//@ManDoc: Raw printing.
	//@Doc: \index{vBTree!C++ Interface!Print}
	void Print		(int margin=0)
				{
					PrintAux(homeblock.IdRoot,margin);
					PrintAux(homeblock.TitleRoot,margin);
					PrintAux(homeblock.AuthorRoot,margin);
					PrintAux(homeblock.SubjRoot,margin);
				}
	//@ManDoc: Page counter.
	//@Doc: \index{vBTree!C++ Interface!CountPages}
	int CountPages		()
				{
					return (CountPagesAux(homeblock.IdRoot) +
						CountPagesAux(homeblock.TitleRoot) +
						CountPagesAux(homeblock.AuthorRoot) +
						CountPagesAux(homeblock.SubjRoot));
				}
	//@ManDoc: Tree traversal (Id Tree).
	//@Doc: \index{vBTree!C++ Interface!TraverseId}
	int TraverseId		(TravFunc tfun,UserData ud)
					{return TravAux(homeblock.IdRoot,tfun,0,ud);}
	//@ManDoc: Tree traversal (Author Tree).
	//@Doc: \index{vBTree!C++ Interface!TraverseAuthor}
	int TraverseAuthor	(TravFunc tfun,UserData ud)
					{return TravAux(homeblock.AuthorRoot,tfun,0,ud);}
	//@ManDoc: Tree traversal (Title Tree).
	//@Doc: \index{vBTree!C++ Interface!TraverseTitle}
	int TraverseTitle	(TravFunc tfun,UserData ud)
					{return TravAux(homeblock.TitleRoot,tfun,0,ud);}
	//@ManDoc: Tree traversal (Subject Tree).
	//@Doc: \index{vBTree!C++ Interface!TraverseSubj}
	int TraverseSubj	(TravFunc tfun,UserData ud)
					{return TravAux(homeblock.SubjRoot,tfun,0,ud);}
	//@ManDoc: Type name given a type code.
	//@Doc: \index{vBTree!C++ Interface!CardTypeName}
	const char *CardTypeName(Card::CardType type) const;
	//@ManDoc: Location name given a location code.
	//@Doc: \index{vBTree!C++ Interface!LocationTypeName}
	const char *LocationTypeName(Card::LocationType type) const;
	//@ManDoc: Category name given a category code. 
	//@Doc: \index{vBTree!C++ Interface!Category}
	const char *Category(unsigned char catcode) const;
	//@ManDoc: CardType given a type name.
	//@Doc: \index{vBTree!C++ Interface!CardTypeFromName}
	Card::CardType CardTypeFromName(const char *name) const;
	//@ManDoc: LocationType given a location code.
	//@Doc: \index{vBTree!C++ Interface!LocationTypeFromName}
	Card::LocationType LocationTypeFromName(const char *name) const;
	//@ManDoc: Catcode given a category.
	//@Doc: \index{vBTree!C++ Interface!CatCodeFromName}
	unsigned char CatCodeFromName(const char *name) const;
	//@ManDoc: Is this a Type name?
	//@Doc: \index{vBTree!C++ Interface!CardTypeNameP}
	Boolean CardTypeNameP(const char *name) const;
	//@ManDoc: Is this a Location name?
	//@Doc: \index{vBTree!C++ Interface!LocationTypeNameP}
	Boolean LocationTypeNameP(const char *name) const;
	//@ManDoc: Is this a Category?
	//@Doc: \index{vBTree!C++ Interface!CatCodeNameP}
	Boolean CatCodeNameP(const char *name) const;
	//@ManDoc: Add a new type name.
	//@Doc: \index{vBTree!C++ Interface!AddCardTypeName}
	void  AddCardTypeName(Card::CardType type,const char *name);
	//@ManDoc: Add a new location type name.
	//@Doc: \index{vBTree!C++ Interface!AddLocationTypeName}
	void  AddLocationTypeName(Card::LocationType type,const char *name);
	//@ManDoc: Add a new category.
	//@Doc: \index{vBTree!C++ Interface!AddCategory}
	//@Doc: \index{vBTree!C++ Interface|)}
	void  AddCategory(unsigned char catcode,const char *name);
};

#endif

