/* -*- Mode: C -*- */
/* vBTree.cc - (virtual) BTree implementation
 *		This code was heavily inspired by the BTree class
 *		described in chapter 10 of "C++, a guide for C programmers",
 *		by Sharam Hekmatpour, (c) 1990 by Prentice Hall of
 *		Australia Pty Ltd.
 * Created by Robert Heller on Sat Dec  7 20:59:18 1991
 * Updated for Version 2.0 on Sat Apr 26 18:20:48 1997
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Class library implementation code
 * ------------------------------------------------------------------
 * Modification History:
 * $Log: vBTree.cc,v $
 * Revision 2.7  1998/04/21 15:12:18  heller
 * Update copyright notice.
 *
 * Revision 2.6  1997/09/20 03:01:12  heller
 * Bulletproof "df" variable (get rid of uninitized warning).
 *
 * Revision 2.5  1997/08/05 01:16:58  heller
 * Fix vBTree::LocationTypeName to be consistent with vBTree::CardTypeName
 * and vBTree::Category.
 *
 * Revision 2.4  1997/07/27 04:10:07  heller
 * Fix totally silly boundary condition
 *
 * Revision 2.3  1997/07/23 23:48:28  heller
 * Fix spelling errors.
 *
 * Revision 2.2  1997/07/08 14:10:58  heller
 * fix small problem in dumphome (debug code)
 *
 * Revision 2.1  1997/07/06 21:44:54  heller
 * Misc. changes
 *
 * Revision 2.0  1997/07/01 01:42:52  heller
 * *** empty log message ***
 *
 * Revision 1.1  1997/06/29 19:14:59  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 *    Home Librarian Database -- a program for maintaining a database
 *                               for a home library
 *    Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 *			51 Locke Hill Road
 *			Wendell, MA 01379-9728
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */
#include <vBTree.h>
#include <ctype.h>
#include <iostream.h>
#include <stdio.h>

#ifdef macintosh
#define NoErr 0
#endif

#ifdef __MWERKS__
#include <ListRecord.h>
#endif

static char ID[] = "$Id: vBTree.cc,v 2.7 1998/04/21 15:12:18 heller Rel $";

const char  * const HomeBlock::OldMagic = "LIBRV000";
const char  * const HomeBlock::Magic = "LIBRV001";

const vBTree::CardTypes vBTree::BaseCardTypeNames[Card::NumCardTypes] = {
	{Card::Book,"Book"}, {Card::Magazine,"Magazine"}, {Card::CD,"CD"},
	  {Card::AudioCassette,"AudioCassette"}, {Card::Album,"Album"}, 
	  {Card::LaserDisk,"LaserDisk"}, {Card::VHSVideo,"VHSVideo"}, 
	  {Card::BetaVideo,"BetaVideo"}, {Card::EightMM,"EightMM"},
	  {Card::EightTrack,"EightTrack"}, {Card::DAT,"DAT"}, 
	  {Card::Other,"Other"}, {Card::UC1, NULL}, {Card::UC2, NULL}, 
	  {Card::UC3, NULL}, {Card::UC4, NULL}, {Card::UC5, NULL},
	  {Card::UC6, NULL}, {Card::UC7, NULL}, {Card::UC8, NULL},
	  {Card::UC9, NULL}, {Card::UC10, NULL}
};

const vBTree::LocationTypes vBTree::BaseLocationTypeNames[Card::NumLocationTypes] = 
	{ {Card::OnShelf, "On Shelf"}, {Card::OnLoan, "On Loan"}, 
	  {Card::OnOrder, "On Order"}, {Card::Destroyed, "Destroyed"},
	  {Card::InStorage, "In Storage"}, {Card::Unknown, "Unknown"},
	  {Card::UL1, NULL}, {Card::UL2, NULL}, {Card::UL3, NULL}, 
	  {Card::UL4, NULL}, {Card::UL5, NULL}, {Card::UL6, NULL}, 
	  {Card::UL7, NULL}, {Card::UL8, NULL}, {Card::UL9, NULL}, 
	  {Card::UL10, NULL}
	};

UserDefinedCodesInCore::UserDefinedCodesInCore(LongInt addr)
{
	diskBlock = DiskRecord(1024,addr);
	memset(&thisBlock,0,sizeof(thisBlock));
	nextBlock = NULL;
}

// This function is for debugging.  It formats a page for inspection.
// (Needed since g++ for OSK does not generate any usefull debugging
// info for Microware's srcdbg.
static void DumpPage (Page* page)
{
	static char buffer[2048];

	sprintf(buffer,"*** Page* at 0x%08lx:",(unsigned long int)page); cout << buffer << endl;
	sprintf(buffer,"    ->size = %ld",page->size); cout << buffer << endl;
	sprintf(buffer,"    ->left.record_address = %ld",
			page->left.record_address); cout << buffer << endl;
	sprintf(buffer,"    ->parent.record_address = %ld",
			page->parent.record_address); cout << buffer << endl;
	sprintf(buffer,"    ->parentidx = %ld",page->parentidx); cout << buffer << endl;
	for (int i = 0; i < page->size; i++) {
		sprintf(buffer,"    ->items[%d].key = %s",i,page->items[i].key); cout << buffer << endl;
		sprintf(buffer,"    ->items[%d].data = [%ld:%ld]",i,
				page->items[i].data.record_size,
				page->items[i].data.record_address); cout << buffer << endl;
		sprintf(buffer,"    ->items[%d].right.record_address = %ld",i,
				page->items[i].right.record_address); cout << buffer << endl;
	}
}

static void dumphome(const HomeBlock* block,const char* message)
{
	static char buffer[2048];

	cerr << "\nHomeBlock (" << message << ") [hex dump]:\n\n";
	unsigned char* p = (unsigned char*) block;
	cerr <<
	"  Addr     0 1  2 3  4 5  6 7  8 9  A B  C D  E F 0 2 4 6 8 A C E\n" <<
	"--------  ---- ---- ---- ---- ---- ---- ---- ---- ----------------\n";
	unsigned int i;
	for (i = 0; i < sizeof(HomeBlock); i += 16, p += 16) {
		sprintf(buffer,"%08x ",i); cerr << buffer;
		int j;
		for (j = 0; j < 16; j += 2) {
			sprintf(buffer," %02x%02x",p[j] & 0x0FF,
						     p[j+1] & 0x0FF);
			cerr << buffer;
		}
		cerr << " ";
		for (j = 0; j < 16; j++) {
			char c = p[j];
			if (c >= ' ' && c <= '~') cerr << c;
			else cerr << ".";
		}
		cerr << endl;
	}
	cerr << "\nHomeBlock (" << message << ") [structured dump]:\n\n";
	cerr << "magic = '";
	for (p = (unsigned char*)block->magic,i = 0; i < 8; p++, i++) cerr << (char)*p;
	cerr << "'" << endl;
	cerr << "IdRoot @ ";
	sprintf(buffer,"%08lx ",block->IdRoot.record_address);
	cerr << buffer << ", TitleRoot @ ";
	sprintf(buffer,"%08lx ",block->TitleRoot.record_address);
	cerr << buffer << ", AuthorRoot @ ";
	sprintf(buffer,"%08lx ",block->AuthorRoot.record_address);
	cerr << buffer << ", SubjRoot @ ";
	sprintf(buffer,"%08lx ",block->SubjRoot.record_address);
	cerr << buffer << endl;
	cerr << "numfree = " << block->numfree << endl;
	for (i = 0; i < block->numfree; i++) {
		cerr << "freepages[" << i << "] @ ";
		sprintf(buffer,"%08lx ",block->freepages[i].record_address);
		cerr << buffer << endl;
	}
}

// Basic constructor.  Does not actually open the file, just sets things up.
vBTree::vBTree() {
	LastSearchType = HLEnums::none;
	homedirty = false;
	buf = new Item[2*Order + 2];
	numcatcodes = 0;
	udefcodes = NULL;
	int i;
	for (i = 0; i < Card::NumCardTypes; i++)
	{
		cardtypenames[i] = BaseCardTypeNames[i];
	}
	for (i = 0; i < Card::NumLocationTypes; i++)
	{
		locationtypenames[i] = BaseLocationTypeNames[i];
	}
	numcatcodes = 0;
	allocatcodes = 0;
	categorycodes = NULL;
	udefcodes = NULL;
	LibrVersion = 2;
	errData = NULL;
	errFun = NULL;
	CanWriteP = false;
}

// This constructor also opens the file.
vBTree::vBTree(const FileName filename,HLEnums::OpenMode mode,int nfree) 
{
	LastSearchType = HLEnums::none;
	homedirty = false;
	buf = new Item[2*Order + 2];
	numcatcodes = 0;
	allocatcodes = 0;
	categorycodes = NULL;
	udefcodes = NULL;
	int i;
	for (i = 0; i < Card::NumCardTypes; i++)
	{
		cardtypenames[i] = BaseCardTypeNames[i];
	}
	for (i = 0; i < Card::NumLocationTypes; i++)
	{
		locationtypenames[i] = BaseLocationTypeNames[i];
	}
	udefcodes = NULL;
	LibrVersion = 2;
	errData = NULL;
	errFun = NULL;
	CanWriteP = false;
	open(filename,mode,nfree);
}

// Open a vBTree file (library card catalog file)
HLEnums::OpenStatus vBTree::open(const FileName filename, HLEnums::OpenMode mode,int nfree)
{
	// Figure type of access and whether we should try to create a
	// new file, if the file does not exist.

#ifdef macintosh
	{
		OSErr err;
		FInfo finfo;
		err = FSpGetFInfo(filename,&finfo);
		if (err == NoErr)
		{
			if (finfo.fdType != 'BINA' || 
				(finfo.fdCreator != 'HL1X' &&
				 finfo.fdCreator != 'HL2X'))
			{
				errno = EDOM;
				return(HLEnums::failure);
			}
		}
	}
#endif


	if ((mode & HLEnums::ModeMask) == HLEnums::ReadWrite) {
		openstat = PageFile::open(filename, HLEnums::inout,
					  ReadWriteFlags,
#ifdef macintosh
					  'HL2X',
#else
					  ReadWriteMode,
#endif
					  ((mode & HLEnums::Create) != 0) ?  true :
									false);
	} else {
		openstat = PageFile::open(filename, HLEnums::in, ReadFlags,
#ifdef macintosh
					  'HL2X',
#else
					  ReadMode,
#endif
					  false);
	}
	// File might have been opened.  Fan out based on how PageFile::open
	// fared.
	switch (openstat) 
	{
	  case HLEnums::openold :	// An existing file was opened.
					// Read in the home block and
					// set things up.
	  {
	    DiskRecord homerecord(sizeof(HomeBlock),0L);
	    if (PageFile::ReadRecord(homerecord,
				     (char *) &homeblock,
				     sizeof(HomeBlock)) <
		(int) sizeof(HomeBlock)) 
	    {
	      PageFile::close();
	      return(openstat = HLEnums::failure);
	    }
	    // Is it really a library file??
	    if (strncmp(homeblock.magic,homeblock.OldMagic,8) == 0) 
	    {
	      LibrVersion = 1;
	    } else if (strncmp(homeblock.magic,homeblock.Magic,8) != 0) 
	    {
	      PageFile::close();
	      return(openstat = HLEnums::failure);
	    } else LibrVersion = 2;
	    // Home block is fresh and clean
	    homedirty = false;
	    if (LibrVersion == 2)
	    {
	      // load in user defined codes
	      udefcodes = new UserDefinedCodesInCore(1024L);
	      if (PageFile::ReadRecord(udefcodes->diskBlock,
				       (char *) &udefcodes->thisBlock,
				       sizeof(UserDefinedCodesInFile)) <
		  (int) sizeof(UserDefinedCodesInFile))
	      {
		PageFile::close();
		return(openstat = HLEnums::failure);
	      }
	      udefcodes->isdirty = false;
	      UserDefinedCodesInCore *n = udefcodes,
				     **nn = &udefcodes->nextBlock;
	      while (n->thisBlock.NextBlock != 0)
	      {
		*nn =  new UserDefinedCodesInCore(n->thisBlock.NextBlock);
		if (PageFile::ReadRecord((*nn)->diskBlock,
					 (char *) &(((*nn)->thisBlock)),
					 sizeof(UserDefinedCodesInFile)) <
		    (int) sizeof(UserDefinedCodesInFile))
		{
		  PageFile::close();
		  return(openstat = HLEnums::failure);
		}
		n = *nn;
		n->isdirty = false;
		nn = &(n->nextBlock);
	      }
	      for (n = udefcodes;n != NULL;n = n->nextBlock)
	      {
		char *p = n->thisBlock.buffer;
		while (p < (&n->thisBlock.buffer[1024-sizeof(LongInt)]))
		{
		  char uctype = *p++;
		  switch (uctype)
		  {
		    case '\0': break;
		    case 'D':
		    {
		      short int len = 0;
		      if (*p >= '0' && *p <= '9')
		      {
			len  = (short int)((*p) - '0') << 8;
		      } else
		      {
			len  = (short int)((*p) - 'A' + 10) << 8;
		      }
		      p++;
		      if (*p >= '0' && *p <= '9')
		      {
			len |= (short int)((*p) - '0') << 4;
		      } else
		      {
			len |= (short int)((*p) - 'A' + 10) << 4;
		      }
		      p++;
		      if (*p >= '0' && *p <= '9')
		      {
			len |= (short int)(*p) - '0';
		      } else
		      {
			len |= (short int)(*p) - 'A' + 10;
		      }
		      p++;
		      p += len;
		    }
		    case 'C':
		    {
		      char CT = *p++;
		      SetCardType((Card::CardType)CT,p);
		      p += strlen(p)+1;
		      break;
		    }
		    case 'L':
		    {
		      char LT = *p++;
		      SetLocationType((Card::LocationType)LT,p);
		      p += strlen(p)+1;
		      break;
		    }
		    case 'c':
		    {
		      unsigned char CC = 0;
		      if (*p >= '0' && *p <= '9')
		      {
			CC  = ((*p) - '0') << 4;
		      } else
		      {
			CC  = ((*p) - 'A' + 10) << 4;
		      }
		      p++;
		      if (*p >= '0' && *p <= '9')
		      {
			CC |= (*p) - '0';
		      } else
		      {
			CC |= (*p) - 'A' + 10;
		      }
		      p++;
		      SetCategory(CC,p);
		      p += strlen(p)+1;
		      break;
		    }
		  }
		}
	      }
	    }
	    CanWriteP = ((mode & HLEnums::ModeMask) == HLEnums::ReadWrite);
	    // All set.
	    return(openstat);
	  }
	  case HLEnums::opennew :		// new file.  Initialize the
						// file with a specified
						// number of free pages
						// and a properly intialized
						// home block.
#ifdef macintosh
	  {
	    OSErr err;
	    FSpCreateResFile(filename,'HL2X','BINA',smSystemScript);
	    err = ResError();
	    if (err == NoErr || err == dupFNErr)
	    {
	      short int CurRes = CurResFile();
	      short int refNum = FSpOpenResFile(filename,fsRdWrPerm);
	      err = ResError();
	      if (err == NoErr)
	      {
	      	static  char missAppvalue[] = "Librarian";
		Handle missApp = NewHandle(strlen(missAppvalue)+1);
		char hs = HGetState(missApp);
		HLockHi(missApp);
		unsigned char *ma = (unsigned char *) *missApp;
		*ma++ = strlen(missAppvalue);
		memcpy(ma,missAppvalue,strlen(missAppvalue));
		HSetState(missApp,hs);
		UseResFile(refNum);
		AddResource(missApp,'STR ',-16396,"\p");
		if (ResError() == NoErr)
		{
		  UpdateResFile(refNum);
		}
		UseResFile(CurRes);
		CloseResFile(refNum);
	      }
	    }
	  }
#endif
	  if (nfree > MaxNumFree) nfree = MaxNumFree;
	  else if (nfree < 0) nfree = 0;
	  strncpy(homeblock.magic,homeblock.Magic,8);
	  homeblock.IdRoot = 0L;
	  homeblock.TitleRoot = 0L;
	  homeblock.AuthorRoot = 0L;
	  homeblock.SubjRoot = 0L;
	  homeblock.numfree = 0;
	  // pre-write home block (tie up first "sector"
	  // so it won't get used as a page).
	  PageFile::WriteRecord((char *) &homeblock,
			        sizeof(HomeBlock));
	  // Initialize and write a base UserDefinedCodesInCore block
	  udefcodes = new UserDefinedCodesInCore(1024L);
	  memset(&udefcodes->thisBlock,0,sizeof(udefcodes->thisBlock));
	  udefcodes->thisBlock.NextBlock = 0;
	  {
	    char *p = udefcodes->thisBlock.buffer;
	    sprintf(p,"D%03X",1024-sizeof(LongInt)-4);
	  }
	  PageFile::WriteRecord((char *) &udefcodes->thisBlock,
				sizeof(UserDefinedCodesInFile));
	  udefcodes->isdirty = false;
	  // get a bunch of free pages
	  for (int i = nfree-1; i >= 0; i--) 
	  {
	    DiskPage newp = PageFile::NewPage();
	    //cerr << "*** Allocated a new page at " << newp.record_address << endl;
	    homeblock.freepages[i] = newp;
	  }
	  // note the number of available pages
	  homeblock.numfree = nfree;
	  // the home block is already dirty...
	  homedirty = true;
	  //dumphome(&homeblock,"open (new file)");
	  // all set.
	  {
	    DiskRecord homerecord(sizeof(HomeBlock),0L);
	    PageFile::ReWriteRecord(homerecord,(char *) &homeblock,
				    sizeof(HomeBlock));
	  }
	  homedirty = false;
	  CanWriteP = true;
	  //dumphome(&homeblock,"New block");
	  return(openstat);
	  default: 
	  case HLEnums::failure: return(openstat);	// PageFile::open() failed.
	}
}

// Destructor.  Close file and generally clean up
vBTree::~vBTree()
{
	if (isopen) {		// if file is open...
		if (homedirty == true) {	// home block needs writing??
			//dumphome(&homeblock,"Closing, updating homeblock");
			// yep.  rewrite it.
			DiskRecord homerecord(sizeof(HomeBlock),0L);
			PageFile::ReWriteRecord(homerecord,
						(char *) &homeblock,
						sizeof(HomeBlock));
		}
		UserDefinedCodesInCore *n = udefcodes, *nn;
		while (n != NULL)
		{
		  if (n->isdirty)
		  {
		    PageFile::ReWriteRecord(n->diskBlock,
					    (char *)&(n->thisBlock),
					    sizeof(UserDefinedCodesInFile));
		  }
		  nn = n->nextBlock;
		  delete n;
		  n = nn;
		}
		// Close file.  (dirty pages will get written out)
		PageFile::close();
#ifdef __DMALLOC_H__
	_dmalloc_file = __FILE__; _dmalloc_line = __LINE__;
#endif
		delete [] buf;
	}
}

// Free up a disk page.
void vBTree::FreePage (const DiskPage dpage)
{
	if (dpage != 0) {
		if (homeblock.numfree == MaxNumFree) return;
		for (int i = 0; i < homeblock.numfree; i++)
			if (homeblock.freepages[i] == dpage) return;
		homeblock.freepages[homeblock.numfree++] = dpage;
		homedirty = true;
		//dumphome(&homeblock,"FreePage");
	}
}

// allocate a new disk page.
DiskPage vBTree::NewPage()
{
	if (homeblock.numfree > 0) {	// any reuseable pages??
		homeblock.numfree--;
		homedirty = true;
		//dumphome(&homeblock,"NewPage");
		return (homeblock.freepages[homeblock.numfree]);
	} else return (PageFile::NewPage());	// nope, get a new one
}

// Main searching function.  I've modified things to remember where we found
// match. My key compare function matches proper prefixes and I've added a
// search again function - allows the user to only give a prefix and find
// all matches.
Boolean vBTree::SearchAux(const DiskPage dtree, const Key key, register CoreItem* item)
{
	int idx;
	Page *tree;

	if (dtree == 0) {			// fell through on exact match
						// maybe a prefix matched...
		if (LastSearchIdx >= 0) {
			// yep.  save state and return match.
			strncpy(LastSearchKey,key,KeySize);
			tree = (*this)[LastSearchPage];	// page in page
			// copy actual key
			strcpy(item->key,tree->items[LastSearchIdx].key);
			// allocate a buffer and read in the data record
			item->data.NewBuffer(tree->items[LastSearchIdx].data.record_size);
			int bytesread =
			PageFile::ReadRecord(tree->items[LastSearchIdx].data,
				     item->data.buffer,item->data.size);
			// I/O error?  report error and return empty buffer
			if (bytesread < 0) {
				Error(HLEnums::sysErr,"PageFile::ReadRecord");
				item->data.NewBuffer(0);
			}
			// success
			return true;
		} else {	// nope.  search over.  
			LastSearchType = HLEnums::none;
			return false;
		}
	}
	CurrentSearchPage = dtree;		// remember where er are
	tree = (*this)[dtree];		// page in page
	if (BinarySearch(tree,key,&idx)) {	// is key on this page??
		// yep.  return key and data
		strcpy(item->key,tree->items[idx].key);
		item->data.NewBuffer(tree->items[idx].data.record_size);
		int bytesread =
		PageFile::ReadRecord(tree->items[idx].data,
			     item->data.buffer,item->data.size);
		if (bytesread < 0) {
			Error(HLEnums::sysErr,"PageFile::ReadRecord");
			item->data.NewBuffer(0);
		}
		// save state
		strncpy(LastSearchKey,key,KeySize);
		LastSearchIdx = idx;
		LastSearchPage = CurrentSearchPage;
		// success
		return true;
	}
	// not here.  descend down the tree.
	return SearchAux((idx < 0 ? tree->left
				  : tree->items[idx].right),key,item);
}

// Binary search function
Boolean vBTree::BinarySearch (const Page *node, const Key key, int *idx)
{
	int low = 0;
	int up = node->size - 1;
	register int mid, comp;
	register Boolean found, exact;
	do {					// binary chop
		mid = (low + up) / 2;
		comp = CompareKeys(key,node->items[mid].key);
		exact = comp==0 && strlen(key) == strlen(node->items[mid].key);
		if (comp==0) {			 // state preservation
			LastSearchIdx = mid;
			LastSearchPage = CurrentSearchPage;
		}
		if (comp==0 && !exact) comp = -1; // make sure we get
						  // the leftmost match
		if (comp <= 0) up = mid - 1;	// restrict to lower half
		if (comp >= 0) low = mid + 1;	// restrict to upper half
	} while (low <= up);
	*idx = (found = low - up > 1) ? mid : up;
	return (found);
}

// Key comparison function.  Will return 0 if keypat is a proper prefix
// of key.  Otherwise it behaves simularly to strcmp(), except it does
// a case insensitive comparison.
int vBTree::CompareKeys(const Key keypat,const Key key)
{
	const char *p = keypat, *k = key;
	int ch, comp;

	if (*p == '\0') return(0);
	do {
		ch = *p++;
		if (isalpha(ch) && islower(ch)) ch = toupper(ch);
		comp = ch - *k++;
	} while (comp == 0 && *p != '\0');
	return (comp);
}

// This function continues the search (prefix string matching)
Boolean vBTree::SearchAgain(register CoreItem* item,Boolean DontGoDown)
{
	Page *tree;
	int comp;
	int idx, tidx;
	DiskPage tpage;
	HLEnums::LSType ttype;
	
	// remember place
	//cerr << "*** Entering SearchAgain: LastSearchPage = " <<
	//	LastSearchPage.record_address <<
	//	" LastSearchIdx = " << LastSearchIdx << "\n";
	tpage = LastSearchPage;
	tidx = LastSearchIdx;
	ttype = LastSearchType;
	tree = (*this)[LastSearchPage];		// page in page
	if (!DontGoDown && LastSearchIdx >= 0 &&	// a real index?
	    tree->items[LastSearchIdx].right != 0) {	// a child??
		// decend t  he tree to the right
		LastSearchPage = tree->items[LastSearchIdx].right;
		LastSearchIdx  = -1;		// left most node
		// recurse...
		//cerr << "*** -: Down and to the left..." << endl;
		if (SearchAgain(item,false)) return(true);
		// failed.  reset
		LastSearchPage = tpage;
		LastSearchIdx = tidx;
		LastSearchType = ttype;
	}
	// next node to the right
	idx = ++LastSearchIdx;
	// nodes remaining??
	//cerr << "*** -: idx = " << idx << ", tree->size = " << tree->size << endl;
	if (idx < tree->size) {
		// yep.  does this node match??
		comp = CompareKeys(LastSearchKey,tree->items[idx].key);
		if (comp < 0) {
			// nope.  we are done...
			LastSearchType = HLEnums::none;
			return (false);
		}
		// yep.  return it.
		//cerr << "*** -: tree->items[LastSearchIdx].key = " << tree->items[LastSearchIdx].key << endl;
		strcpy(item->key,tree->items[LastSearchIdx].key);
		item->data.NewBuffer(tree->items[LastSearchIdx].data.record_size);
		int bytesread =
		PageFile::ReadRecord(tree->items[LastSearchIdx].data,
			     item->data.buffer,item->data.size);
		if (bytesread < 0) {
			Error(HLEnums::sysErr,"PageFile::ReadRecord");
			item->data.NewBuffer(0);
		}
		// we are all set for next time
		return (true);
	} else {	// no more items here.  pop up and try next item
			// in parent
		if (tree->parent == 0) {	// if any...
			// no parent.  end of the line..
			LastSearchType = HLEnums::none;
			return(false);
		} else {
			// up and to the right...
			LastSearchPage = tree->parent;
			LastSearchIdx  = tree->parentidx;
			//cerr << "*** -: up and to the right..." << endl;
			return (SearchAgain(item,true));
		}
	}
}

// helper function for InsertXXX
// copy a key, converting to uppercase and truncating to fit.
static void strucase(Key dest, const Key source)
{
	char ch, *a, *b;
	int i;

	for (a=(char*)source,b=dest,i=1; *a != '\0' && i < KeySize; a++,b++,i++) {
		ch = *a;
		if (isalpha(ch) && islower(ch)) ch = toupper(ch);
		*b = ch;
	}
	*b = '\0';
}	

// Insertion function for the Id tree
DiskRecord vBTree::InsertId (const Key key,const Record* newdata)
{
	Item *receive, item;
	Page *page, *root;
	DiskPage dpage;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::InsertId)");
		return(item.data);
	}
	// copy the key
	strucase(item.key,key);
	// and write the data out
	item.data = PageFile::WriteRecord(newdata->buffer,newdata->size);
	item.right = 0;
	if (homeblock.IdRoot == 0) {		// empty tree
		homeblock.IdRoot = NewPage();
		homedirty = true;
		root = (*this)[homeblock.IdRoot];
		root->left = 0;
		root->parent = 0;
		root->parentidx = -1;
		root->items[0] = item;
		root->size = 1;
		(*this)(homeblock.IdRoot).isdirty = true;
		//cout << "InsertId (new tree)" << endl;
		//DumpPage(root);
		//dumphome(&homeblock,"InsertId (new tree)");
	} else if ((receive = InsertAux(&item,homeblock.IdRoot)) != 0) {
		dpage = NewPage();		// new root
		page = (*this)[dpage];
		page->size = 1;
		page->left = homeblock.IdRoot;
		page->parent = 0;
		page->parentidx = -1;
		page->items[0] = *receive;
		AdjustParent(dpage);		// fixup this node's offspring
						// to point back.
		(*this)(dpage).isdirty = true;
		root = (*this)[homeblock.IdRoot];
		root->parent = dpage;
		root->parentidx = -1;
		AdjustParent(homeblock.IdRoot);
		(*this)(homeblock.IdRoot).isdirty = true;
		homeblock.IdRoot = dpage;
		//cout << "InsertId (new root [new root])" << endl;
		//DumpPage(page);
		//cout << "InsertId (new root [old root])" << endl;
		//DumpPage(root);
		homedirty = true;
		//dumphome(&homeblock,"InsertId (new root)");
	}
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
	return(item.data);
}

// Insertion for the Title tree
DiskRecord vBTree::InsertTitle (const Key key,const Record* newdata)
{
	Item *receive, item;
	Page *page, *root;
	DiskPage dpage;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::InsertTitle)");
		return(item.data);
	}
	strucase(item.key,key);
	item.data = PageFile::WriteRecord(newdata->buffer,newdata->size);
	item.right = 0;
	if (homeblock.TitleRoot == 0) {
		homeblock.TitleRoot = NewPage();
		homedirty = true;
		root = (*this)[homeblock.TitleRoot];
		root->left = 0;
		root->parent = 0;
		root->parentidx = -1;
		root->items[0] = item;
		root->size = 1;
		(*this)(homeblock.TitleRoot).isdirty = true;
		//dumphome(&homeblock,"InsertTitle (new tree)");
	} else if ((receive = InsertAux(&item,homeblock.TitleRoot)) != 0) {
		dpage = NewPage();
		page = (*this)[dpage];
		page->size = 1;
		page->left = homeblock.TitleRoot;
		page->parent = 0;
		page->parentidx = -1;
		page->items[0] = *receive;
		AdjustParent(dpage);
		(*this)(dpage).isdirty = true;
		root = (*this)[homeblock.TitleRoot];
		root->parent = dpage;
		root->parentidx = -1;
		AdjustParent(homeblock.TitleRoot);
		(*this)(homeblock.TitleRoot).isdirty = true;
		homeblock.TitleRoot = dpage;
		homedirty = true;
		//dumphome(&homeblock,"InsertTitle (new root)");
	}
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
	return(item.data);
}

// Insertion for the Author tree
DiskRecord vBTree::InsertAuthor (const Key key,const Record* newdata)
{
	Item *receive, item;
	Page *page, *root;
	DiskPage dpage;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::InsertAuthor)");
		return(item.data);
	}
	strucase(item.key,key);
	item.data = PageFile::WriteRecord(newdata->buffer,newdata->size);
	item.right = 0;
	if (homeblock.AuthorRoot == 0) {
		homeblock.AuthorRoot = NewPage();
		homedirty = true;
		root = (*this)[homeblock.AuthorRoot];
		root->left = 0;
		root->parent = 0;
		root->parentidx = -1;
		root->items[0] = item;
		root->size = 1;
		(*this)(homeblock.AuthorRoot).isdirty = true;
		//dumphome(&homeblock,"InsertAuthor (new tree)");
	} else if ((receive = InsertAux(&item,homeblock.AuthorRoot)) != 0) {
		dpage = NewPage();
		page = (*this)[dpage];
		page->size = 1;
		page->left = homeblock.AuthorRoot;
		page->parent = 0;
		page->parentidx = -1;
		page->items[0] = *receive;
		AdjustParent(dpage);
		(*this)(dpage).isdirty = true;
		root = (*this)[homeblock.AuthorRoot];
		root->parent = dpage;
		root->parentidx = -1;
		AdjustParent(homeblock.AuthorRoot);
		(*this)(homeblock.AuthorRoot).isdirty = true;
		homeblock.AuthorRoot = dpage;
		homedirty = true;
		//dumphome(&homeblock,"InsertAuthor (new root)");
	}
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
	return(item.data);
}

// Insertion for the Subject tree
DiskRecord vBTree::InsertSubj (const Key key,const Record* newdata)
{
	Item *receive, item;
	Page *page, *root;
	DiskPage dpage;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::InsertSubj)");
		return(item.data);
	}
	strucase(item.key,key);
	item.data = PageFile::WriteRecord(newdata->buffer,newdata->size);
	item.right = 0;
	if (homeblock.SubjRoot == 0) {
		homeblock.SubjRoot = NewPage();
		homedirty = true;
		root = (*this)[homeblock.SubjRoot];
		root->left = 0;
		root->parent = 0;
		root->parentidx = -1;
		root->items[0] = item;
		root->size = 1;
		(*this)(homeblock.SubjRoot).isdirty = true;
		//dumphome(&homeblock,"InsertSubj (new tree)");
	} else if ((receive = InsertAux(&item,homeblock.SubjRoot)) != 0) {
		dpage = NewPage();
		page = (*this)[dpage];
		page->size = 1;
		page->left = homeblock.SubjRoot;
		page->parent = 0;
		page->parentidx = -1;
		page->items[0] = *receive;
		AdjustParent(dpage);
		(*this)(dpage).isdirty = true;
		root = (*this)[homeblock.SubjRoot];
		root->parent = dpage;
		root->parentidx = -1;
		AdjustParent(homeblock.SubjRoot);
		(*this)(homeblock.SubjRoot).isdirty = true;
		homeblock.SubjRoot = dpage;
		homedirty = true;
		//dumphome(&homeblock,"InsertSubj (new root)");
	}
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
	return(item.data);
}

// Common helper function for Inserting
Item* vBTree::InsertAux (Item* item, const DiskPage dnode)
{
	register Page* node = (*this)[dnode];
	int idx, size, half;

	if (BinarySearch(node,item->key,&idx)) {
		// Modified to allow replacement
		node->items[idx].data = item->data;
		(*this)(dnode).isdirty = true;
		return(0);		// already in tree
	}
	DiskPage dchild = (idx < 0 ? node->left : node->items[idx].right);
	if (dchild != 0) item = InsertAux(item,dchild);	// child is not e leaf
	// node is a leaf or passed up
	if (item != 0) {
		node = (*this)[dnode];
		if  (node->size < 2*Order) {	// insert in the node
			node->size = InsertItem(item,node->items,idx+1,
						node->size);
			//cout << "InsertAux (item inserted)" << endl;
			//DumpPage(node);
			AdjustParent(dnode);
			(*this)(dnode).isdirty = true;
		} else {		// node is full, split
			DiskPage dpage =  NewPage();
			register Page* page;
			node = (*this)[dnode];
			size = CopyItems(node->items, buf, node->size);
			size = InsertItem(item,buf,idx+1,size);
			node->size  = CopyItems(buf,node->items,half=size/2);
			(*this)(dnode).isdirty = true;
			//cout << "InsertAux (left half)" << endl;
			//DumpPage(node);
			page = (*this)[dpage];
			page->size  = CopyItems(buf+half+1,page->items,size-half-1);
			page->left =  buf[half].right;
			(*this)(dpage).isdirty = true;
			//cout << "InsertAux (right half)" << endl;
			//DumpPage(page);
			*item = buf[half];	// the mid item
			item->right = dpage;
			return(item);
		}
	}
	return(0);
}

// Deletion for Id tree...
void vBTree::DeleteId (const Key key)
{
	Boolean underflow;
	DiskPage dtemp;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::DeleteId)");
		return;
	}
	if (homeblock.IdRoot == 0) return;
	DeleteAux1(key,homeblock.IdRoot,&underflow);
	Page *root = (*this)[homeblock.IdRoot];
	if (underflow && root->size == 0)  {	// dispose root
		dtemp = homeblock.IdRoot;
		homeblock.IdRoot = root->left;
		homedirty = true;
		FreePage(dtemp);
		//dumphome(&homeblock,"DeleteId (new root)");
		if (homeblock.IdRoot == 0) goto Finish;
		root = (*this)[homeblock.IdRoot];
		root->parent = 0;
		root->parentidx = 0;
		(*this)(homeblock.IdRoot).isdirty = true;
	}
Finish:
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
}

// Deletion for Title tree
void vBTree::DeleteTitle (const Key key)
{
	Boolean underflow;
	DiskPage dtemp;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::DeleteTitle)");
		return;
	}
	if (homeblock.TitleRoot == 0) return;
	DeleteAux1(key,homeblock.TitleRoot,&underflow);
	Page *root = (*this)[homeblock.TitleRoot];
	if (underflow && root->size == 0)  {		// dispose root
		dtemp = homeblock.TitleRoot;
		homeblock.TitleRoot = root->left;
		homedirty = true;
		FreePage(dtemp);
		//dumphome(&homeblock,"DeleteTitle (new root)");
		if (homeblock.TitleRoot == 0) goto Finish;
		root = (*this)[homeblock.TitleRoot];
		root->parent = 0;
		root->parentidx = 0;
		(*this)(homeblock.TitleRoot).isdirty = true;
	}
Finish:
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
}

// Deletion for Author tree...
void vBTree::DeleteAuthor (const Key key)
{
	Boolean underflow;
	DiskPage dtemp;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::DeleteAuthor)");
		return;
	}
	if (homeblock.AuthorRoot == 0) return;
	DeleteAux1(key,homeblock.AuthorRoot,&underflow);
	Page *root = (*this)[homeblock.AuthorRoot];
	if (underflow && root->size == 0)  {		// dispose root
		dtemp = homeblock.AuthorRoot;
		homeblock.AuthorRoot = root->left;
		homedirty = true;
		FreePage(dtemp);
		//dumphome(&homeblock,"DeleteAuthor (new root)");
		if (homeblock.AuthorRoot == 0) goto Finish;
		root = (*this)[homeblock.AuthorRoot];
		root->parent = 0;
		root->parentidx = 0;
		(*this)(homeblock.AuthorRoot).isdirty = true;
	}
Finish:
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
}

// Deletion for Subject tree
void vBTree::DeleteSubj (const Key key)
{
	Boolean underflow;
	DiskPage dtemp;
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::DeleteSubj)");
		return;
	}
	if (homeblock.SubjRoot == 0) return;
	DeleteAux1(key,homeblock.SubjRoot,&underflow);
	Page *root = (*this)[homeblock.SubjRoot];
	if (underflow && root->size == 0)  {		// dispose root
		dtemp = homeblock.SubjRoot;
		homeblock.SubjRoot = root->left;
		homedirty = true;
		FreePage(dtemp);
		//dumphome(&homeblock,"DeleteSubj (new root)");
		if (homeblock.SubjRoot == 0) goto Finish;
		root = (*this)[homeblock.SubjRoot];
		root->parent = 0;
		root->parentidx = 0;
		(*this)(homeblock.SubjRoot).isdirty = true;
	}
Finish:
	FlushPages();
	if (homedirty == true) {
		DiskRecord homerecord(sizeof(HomeBlock),0L);
		PageFile::ReWriteRecord(homerecord,
					(char *) &homeblock,
					sizeof(HomeBlock));
		homedirty = false;
	}
}

// First helper function for deleting
void vBTree::DeleteAux1(const Key key, const DiskPage dnode, Boolean* underflow)
{
	Page *node;
	DiskPage dchild;
	int idx;

	*underflow = false;
	if (dnode == 0) return;
	node = (*this)[dnode];
	if (BinarySearch(node,key,&idx)) {
		dchild = (idx - 1 < 0 ? node->left
				      : node->items[idx-1].right);
		if (dchild == 0) {
			// node is a leaf
			node->size = DeleteItem(node->items,idx,node->size);
			(*this)(dnode).isdirty = true;
			*underflow = node->size < Order;
		} else {	// node is a subtree
			// delete from subtree
			DeleteAux2(dnode,dchild,idx,underflow);
			if (*underflow)
			    Underflow(dnode,dchild,idx-1,underflow);
		}
	} else {		// is not in this node
		dchild = (idx < 0 ? node->left : node->items[idx].right);
		DeleteAux1(key,dchild,underflow);	// should be in child
		if (*underflow)
		    Underflow(dnode,dchild,idx,underflow);
	}
}

// Second helper function, subtree deletion
void vBTree::DeleteAux2 (const DiskPage dparent, const DiskPage dnode, int idx, Boolean* underflow)
{
	Page* node = (*this)[dnode];
	DiskPage dchild;
	Page* child;

	dchild = node->items[node->size-1].right;
	if (dchild != 0) {	// node is not is leaf
		child = (*this)[dchild];
		// go another level down
		DeleteAux2(dparent,dchild,idx,underflow);
		node = (*this)[dnode];
		if (*underflow)
		    Underflow(dnode,dchild,node->size-1,underflow);
	} else {		// node is a leaf
		DiskPage dright;
		Page* parent = (*this)[dparent];
		node = (*this)[dnode];
		dright = parent->items[idx].right;
		parent->items[idx] = node->items[node->size-1];
		parent->items[idx].right  = dright;
		(*this)(dparent).isdirty = true;
		node->size = DeleteItem(node->items,node->size-1,node->size);
		(*this)(dnode).isdirty = true;
		*underflow = node->size < Order;
	}
}

// Underflow handler...
void vBTree::Underflow (const DiskPage dnode, const DiskPage dchild,int idx,Boolean* underflow)
{
	register Page* node = (*this)[dnode];
	DiskPage dleft;
	if (idx < ((node->size)-1))
	    dleft = dchild;
	else {
		if (idx == 0) dleft = node->left;
		else {
			int prev = idx-1;
			//cout << "Taking prev = " << prev << "\n";
			dleft = node->items[prev].right;
		}
	}
	register Page* left;
	DiskPage dright;
	if (dleft == dchild) {
		idx++;
		node = (*this)[dnode];
		dright = node->items[idx].right;
	} else dright = dchild;
	register Page* right;
	register int size, half;
	// copy contents of the left, parent, and right into buf
	left = (*this)[dleft];
	size = CopyItems(left->items,buf,left->size);
	node = (*this)[dnode];
	buf[size] =  node->items[idx];
	right = (*this)[dright];
	buf[size++].right = right->left;
	size += CopyItems(right->items,buf+size,right->size);
	if (size > 2*Order) {	// distribute buf between the left and right
		left = (*this)[dleft];
		left->size = CopyItems(buf,left->items,half = size/2);
		AdjustParent(dleft);
		(*this)(dleft).isdirty = true;
		right = (*this)[dright];
		right->size = CopyItems(buf+half+1,right->items,size-half-1);
		right->left = buf[half].right;
		AdjustParent(dright);
		right = (*this)[dright];
		right->parent = dnode;
		right->parentidx = idx;
		(*this)(dright).isdirty = true;
		node = (*this)[dnode];
		node->items[idx] = buf[half];
		node->items[idx].right = dright;
		(*this)(dnode).isdirty = true;
		*underflow = false;
	} else {	// merge, and free the right page.
		left = (*this)[dleft];
		left->size = CopyItems(buf,left->items,size);
		AdjustParent(dleft);
		(*this)(dleft).isdirty = true;
		node = (*this)[dnode];
		node->size = DeleteItem(node->items,idx,node->size);
		(*this)(dnode).isdirty = true;
		FreePage(dright);
		*underflow = node->size < Order;
	}
}

// Parentage adjuster.  Makes sure the parent pointers are correct.
void vBTree::AdjustParent (const DiskPage dparent)
{
	int idx;
	DiskPage dnode;
	Page *node, *parent = (*this)[dparent];
	int psize = parent->size;

	parent = (*this)[dparent];
	dnode = parent->left;
	if (dnode != 0) {
		node = (*this)[dnode];
		if (node->parent != dparent ||
		    node->parentidx != -1) {
		    	node->parent = dparent;
		    	node->parentidx = -1;
		    	(*this)(dnode).isdirty = true;
		}
	}
	for (idx=0; idx < psize; idx++) {
		parent = (*this)[dparent];
		dnode = parent->items[idx].right;
		if (dnode != 0) {
			node = (*this)[dnode];
			if (node->parent !=
				dparent ||
			    node->parentidx != idx) {
				node->parent = dparent;
				node->parentidx = idx;
				(*this)(dnode).isdirty = true;
			}
		}
	}
}

// Item hacking code - copy items, insert an item and delete an item
int vBTree::CopyItems (const Item* src,Item* dest,int count)
{
	for (int i = 0; i < count;  ++i)	// straight copy
	    dest[i] = src[i];
	return count;
}

int vBTree::InsertItem (const Item* item,Item* items,int idx,int size)
{
	for (int i = size; i > idx; --i)	// shift right
	    items[i] = items[i-1];
	items[idx] = *item;			// insert
	return size + 1;
}

int vBTree::DeleteItem (Item* items,int idx,int size)
{
	for (int i = idx; i < size; ++i)	// shift left
	    items[i] = items[i+1];
	return size - 1;
}

// Raw printing function.  Much like in the book. The data is printed as
// the size and offset of the disk record.
void vBTree::PrintAux(const DiskPage dnode, int margin)
{
	static char  margBuf[128], buffer[2048];
	Page* node;

	if (dnode != 0) {
		node = (*this)[dnode];
		int i;
		for (i = 0; i < margin; ++i) margBuf[i] = ' ';
		margBuf[i] = 0;
		int nsize = node->size;
		PrintAux(node->left,margin+8);
		for (i = 0; i < nsize; ++i) {
			node = (*this)[dnode];
			sprintf(buffer,"%s(%s [%ld:%ld])",
				     margBuf,node->items[i].key,
				     node->items[i].data.record_size,
				     node->items[i].data.record_address);
			cout << buffer << endl;
			PrintAux(node->items[i].right,margin+8);
		}
	}
}

// page counting function.  Allows applications that need this info
// do things smart (preallocating the pages for a compact output file)
int vBTree::CountPagesAux(const DiskPage dnode)
{
	int count = 0;
	Page* node;

	if (dnode != 0) {
		count++;
		node = (*this)[dnode];
		//cout << "*** dnode = " << dnode.record_address << "\n"; DumpPage(node);
		count += CountPagesAux(node->left);
		node = (*this)[dnode];
		int nsize = node->size;
		for (int i = 0; i < nsize; ++i) {
			count += CountPagesAux(node->items[i].right);
			node = (*this)[dnode];
		}
	}
	return(count);
}

// Tree traversal function.  Allows "sequential" access to a tree
int vBTree::TravAux(const DiskPage dnode,TravFunc tfun,int level,UserData ud)
{
	CoreItem item;
	Page* node;
	int status;

	if (dnode != 0) {
		node = (*this)[dnode];
		status = TravAux(node->left,tfun,level+1,ud);
		if (status != 0) return status;
		node = (*this)[dnode];
		int nsize = node->size;
		for (int i = 0; i < nsize; ++i) {
			strcpy(item.key,node->items[i].key);
			item.data.NewBuffer(node->items[i].data.record_size);
			int bytesread = 
			PageFile::ReadRecord(node->items[i].data,
					     item.data.buffer,
					     item.data.size);
			if (bytesread < 0) {
				Error(HLEnums::sysErr,"PageFile::ReadRecord");
				item.data.NewBuffer(0);
			}
			status = (*tfun)(&item,level,ud);
			if (status != 0) return status;
			node = (*this)[dnode];
			status = TravAux(node->items[i].right,tfun,level+1,ud);
			if (status != 0) return status;
		}
	}
	return 0;
}

void vBTree::SetErrorFun (ErrFun efun,UserData edata)
{
	errFun = efun;
	errData = edata;
}

// Generic error handler
void vBTree::Error (HLEnums::ErrKind err,const char* msg) const
{
	static char buffer[2048];

	if (errFun != 0) (*errFun)(err,msg,errData);
	else {
		if (err == HLEnums::sysErr) {
			int error = errno;
			sprintf(buffer,"Error: %s %s",strerror(error),msg);
			cerr << buffer << endl;
			exit(error);
		} else {
			sprintf(buffer,"Error: Memory: %s",msg);
			cerr << buffer << endl;
			exit(1);
		}
	}
}


void vBTree::SetCardType(Card::CardType tt,const char *nn)
{
	if (LibrVersion < 2) return;
	if (tt < Card::UC1 || tt > Card::UC10) return;
	int index = ((int)Card::IndexUC1) + (((int)tt) - ((int)Card::UC1));
	if (cardtypenames[index].n != NULL)
		FreeUDC((char*)cardtypenames[index].n);
	cardtypenames[index].n = nn;
}

void vBTree::SetLocationType(Card::LocationType tt, const char *nn)
{
	if (LibrVersion < 2) return;
	if (tt < Card::UL1 || tt > Card::UL10) return;
	int index = ((int)Card::IndexUL1) + (((int)tt) - ((int)Card::UL1));
	if (locationtypenames[index].n != NULL)
		FreeUDC((char*)locationtypenames[index].n);
	locationtypenames[index].n = nn;
}

void vBTree::SetCategory(unsigned char cc,const char *nn)
{
	if (LibrVersion < 2) return;
	int i;
	for (i = 0; i < numcatcodes; i++)
	{
		if (categorycodes[i].c == cc)
		{
			if (categorycodes[i].n != NULL)
				FreeUDC((char*)categorycodes[i].n);
			categorycodes[i].n = nn;
			return;
		}
	}
	if (i < allocatcodes)
	{
		categorycodes[numcatcodes].c = cc;
		categorycodes[numcatcodes].n = nn;
		numcatcodes++;
	} else
	{
		CategoryCodes *newcategorycodes =
			new CategoryCodes[allocatcodes+CCAllocSize];
		for (i = 0; i < allocatcodes; i++)
			newcategorycodes[i] = categorycodes[i];
		delete [] categorycodes;
		categorycodes = newcategorycodes;
		allocatcodes += CCAllocSize;
		for (i = numcatcodes; i < allocatcodes; i++)
		{
			categorycodes[numcatcodes].c = 0;
			categorycodes[numcatcodes].n = NULL;
		}
		categorycodes[numcatcodes].c = cc;
		categorycodes[numcatcodes].n = nn;
		numcatcodes++;
	}
}

char *vBTree::AllocateUDC(int len)
{
	if (LibrVersion < 2) return NULL;
	//cerr << "*** Entering AllocateUDC: len = " << len << ", (1024-sizeof(LongInt)) = " << (1024-sizeof(LongInt)) << endl;
	if (len > (1024-sizeof(LongInt))) return NULL;
	UserDefinedCodesInCore **tail = &udefcodes;
	static Boolean Bdummy;
	Boolean *df = &Bdummy;
	LongInt *prev = &(udefcodes->thisBlock.NextBlock);
	for (UserDefinedCodesInCore *n = udefcodes; n != NULL; n = n->nextBlock)
	{
	  //cerr << "*** -: n = ";
	  //cerr.form("%08X",(unsigned long int)n);
	  //cerr << endl;
	  tail = &n->nextBlock;
	  df = &n->isdirty;
	  prev = &n->thisBlock.NextBlock;
	  char *p = n->thisBlock.buffer;
	  while (p < (&n->thisBlock.buffer[1024-sizeof(LongInt)]))
	  {
	    char *pp = p;
	    char uctype = *p++;
	    //cerr << "*** -: uctype = " << uctype << " (" << ((unsigned short)uctype) << ")" << endl;
	    switch (uctype)
	    {
	      case '\0': break;
	      case 'D':
	      {
	        short int blen = 0;
	        if (*p >= '0' && *p <= '9')
	        {
		  blen  = (short int)((*p) - '0') << 8;
	        } else
	        {
		  blen  = (short int)((*p) - 'A' + 10) << 8;
	        }
	        p++;
	        if (*p >= '0' && *p <= '9')
	        {
	          blen |= (short int)((*p) - '0') << 4;
	        } else
	        {
	          blen |= (short int)((*p) - 'A' + 10) << 4;
	        }
	        p++;
	        if (*p >= '0' && *p <= '9')
	        {
	          blen |= (short int)(*p) - '0';
	        } else
	        {
	          blen |= (short int)(*p) - 'A' + 10;
	        }
	        p++;
	        //cerr << "*** :- deleted block, blen = " << blen << endl;
	        if ((blen+4) > len)
	        {
	          p = pp + len;
	          int remainder = blen - len;
	          if (remainder >= 0)
	          {
	            sprintf(p,"D%03X",remainder);
		  }
		  n->isdirty = true;
		  //cerr << "*** :- returns ";
		  //cerr.form("%08X",pp);
		  //cerr << endl;
		  return pp;
	        } else p += blen;
	      }
	      case 'C':
	      case 'L':
	      case 'c': p += strlen(p)+1;
	    }
	  }
	}
	UserDefinedCodesInCore *newBl = new UserDefinedCodesInCore();
	*tail = newBl;
	*df = true;
	newBl->nextBlock = NULL;
	memset(&newBl->thisBlock,0,1024);
	newBl->isdirty = false;
	{
	  char *p = newBl->thisBlock.buffer;
	  sprintf(p,"D%03X",1024-sizeof(LongInt)-4);
	}
	newBl->diskBlock =
		PageFile::WriteRecord((char *) &newBl->thisBlock,
				      sizeof(UserDefinedCodesInFile));
	*prev = newBl->diskBlock.record_address;
	char *pp = newBl->thisBlock.buffer, *p = pp+len;
	int remainder = (1024-sizeof(LongInt)-4) - len;
	if (remainder >= 0)
	{
	  sprintf(p,"D%03X",remainder);
	}
	newBl->isdirty = true;
	//cerr << "*** :- returns ";
	//cerr.form("%08X",pp);
	//cerr << endl;
	return pp;
}

void vBTree::FreeUDC(char *ptr)
{
	if (LibrVersion < 2) return;
	for (UserDefinedCodesInCore *n = udefcodes; n != NULL; n = n->nextBlock)
	{
	  char *prevHole = NULL;
	  char *p = n->thisBlock.buffer;
	  while (p < (&n->thisBlock.buffer[1024-sizeof(LongInt)]))
	  {
	    if (p != ptr)
	    {
	      if (*p == 'C' || *p == 'L' || *p == 'c') prevHole = NULL;
	      else if (prevHole == NULL) prevHole = p;
	      if (*p == '\0') p++;
	      else if (*p == 'D')
	      {
	      	p++;
	      	short int blen = 0;
		if (*p >= '0' && *p <= '9')
		{
		  blen  = (short int)((*p) - '0') << 8;
	        } else
	        {
		  blen  = (short int)((*p) - 'A' + 10) << 8;
	        }
	        p++;
	        if (*p >= '0' && *p <= '9')
	        {
	          blen |= (short int)((*p) - '0') << 4;
	        } else
	        {
	          blen |= (short int)((*p) - 'A' + 10) << 4;
	        }
	        p++;
	        if (*p >= '0' && *p <= '9')
	        {
	          blen |= (short int)(*p) - '0';
	        } else
	        {
	          blen |= (short int)(*p) - 'A' + 10;
	        }
	        p++;
	        p += blen;
	      } else p += strlen(p)+1;
	    } else {
	      if (prevHole == NULL) prevHole = p;
	      p += strlen(p)+1;
	      while (p < (&n->thisBlock.buffer[1024-sizeof(LongInt)]))
	      {
	        if (*p == '\0') p++;
	        else if (*p == 'D')
	        {
		  p++;
		  short int blen = 0;
		  if (*p >= '0' && *p <= '9')
		  {
		    blen  = (short int)((*p) - '0') << 8;
	          } else
	          {
		    blen  = (short int)((*p) - 'A' + 10) << 8;
	          }
	          p++;
	          if (*p >= '0' && *p <= '9')
	          {
	            blen |= (short int)((*p) - '0') << 4;
	          } else
	          {
	            blen |= (short int)((*p) - 'A' + 10) << 4;
	          }
	          p++;
	          if (*p >= '0' && *p <= '9')
	          {
	            blen |= (short int)(*p) - '0';
	          } else
	          {
	            blen |= (short int)(*p) - 'A' + 10;
	          }
	          p++;
	          p += blen;
	        } else break;
	      }
	      n->isdirty = true;
	      int bsize = (p - prevHole) - 4;
	      sprintf(prevHole,"D%03X",bsize);
	      return;
	    }
	  }
	}
}

const char *vBTree::CardTypeName(Card::CardType type) const
{
	for (int i = 0; i < Card::NumCardTypes; i++)
	{
		if (type == cardtypenames[i].t) return(cardtypenames[i].n);
	}
	static char* dummy = "Other";
	return dummy;
}

const char *vBTree::LocationTypeName(Card::LocationType type) const
{
	for (int i = 0; i < Card::NumLocationTypes; i++)
	{
		if (type == locationtypenames[i].t) return(locationtypenames[i].n);
	}
	static char* dummy = "Unknown";
	return dummy;
}

const char *vBTree::Category(unsigned char catcode) const
{
	for (int i = 0; i < numcatcodes; i++)
	{
		if (catcode == categorycodes[i].c) return(categorycodes[i].n);
	}
	static char* dummy = "";
	return dummy;
}

Card::CardType vBTree::CardTypeFromName(const char *name) const
{
	for (int i = 0; i < Card::NumCardTypes; i++)
	{
	  if (cardtypenames[i].n != NULL &&
	      strcasecmp(name,cardtypenames[i].n) == 0)
		return(cardtypenames[i].t);
	}
	return Card::Other;
}

Card::LocationType vBTree::LocationTypeFromName(const char *name) const
{
	for (int i = 0; i < Card::NumLocationTypes; i++)
	{
	  if (locationtypenames[i].n != NULL &&
	      strcasecmp(name,locationtypenames[i].n) == 0)
		return(locationtypenames[i].t);
	}
	return Card::Unknown;
}

unsigned char vBTree::CatCodeFromName(const char *name) const
{
	for (int i = 0; i < numcatcodes; i++)
	{
	  if (categorycodes[i].n != NULL &&
	      strcasecmp(name,categorycodes[i].n) == 0)
	  	return(categorycodes[i].c);
	}
	return 0;
}

Boolean vBTree::CardTypeNameP(const char *name) const
{
	for (int i = 0; i < Card::NumCardTypes; i++)
	{
	  if (cardtypenames[i].n != NULL &&
	      strcasecmp(name,cardtypenames[i].n) == 0) return true;
	}
	return false;
}

Boolean vBTree::LocationTypeNameP(const char *name) const
{
	for (int i = 0; i < Card::NumLocationTypes; i++)
	{
	  if (locationtypenames[i].n != NULL &&
	      strcasecmp(name,locationtypenames[i].n) == 0) return true;
	}
	return false;
}

Boolean vBTree::CatCodeNameP(const char *name) const
{
	for (int i = 0; i < numcatcodes; i++)
	{
	  if (categorycodes[i].n != NULL &&
	      strcasecmp(name,categorycodes[i].n) == 0) return true;
	}
	return false;
}


void  vBTree::AddCardTypeName(Card::CardType type,const char *name)
{
	if (LibrVersion < 2) 
	{
		Error(HLEnums::hlErr,"V1 file (vBTree::AddCardTypeName)");
		return;
	}
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::AddCardTypeName)");
		return;
	}
	int len = strlen(name);
	if ((len+3) > (1024-sizeof(LongInt)))
		len = (1024-sizeof(LongInt))-3;
	char *p = AllocateUDC(len+3);
	*p++ = 'C';
	*p++ = (char)type;
	strncpy(p,name,len);
	SetCardType(type,p);		
}

void  vBTree::AddLocationTypeName(Card::LocationType type,const char *name)
{
	if (LibrVersion < 2) 
	{
		Error(HLEnums::hlErr,"V1 file (vBTree::AddLocationTypeName)");
		return;
	}
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::AddLocationTypeName)");
		return;
	}
	int len = strlen(name);
	if ((len+3) > (1024-sizeof(LongInt)))
		len = (1024-sizeof(LongInt))-3;
	char *p = AllocateUDC(len+3);
	*p++ = 'L';
	*p++ = (char)type;
	strncpy(p,name,len);
	SetLocationType(type,p);
}

void  vBTree::AddCategory(unsigned char catcode,const char *name)
{
	if (LibrVersion < 2) 
	{
		Error(HLEnums::hlErr,"V1 file (vBTree::AddCategory)");
		return;
	}
	if (!CanWriteP)
	{
		Error(HLEnums::hlErr,"ReadOnly file (vBTree::AddCategory)");
		return;
	}
	int len = strlen(name);
	if ((len+4) > (1024-sizeof(LongInt)))
		len = (1024-sizeof(LongInt))-4;
	char *p = AllocateUDC(len+4);
	sprintf(p,"c%02X",catcode);
	p += 3;
	strncpy(p,name,len);
	SetCategory(catcode,p);
}



