/* -*- Mode: C -*- */
/* common.h - common include file - all other include files (and
 *	      programs) include this file
 * Created by Robert Heller on Fri Dec  6 19:42:59 1991
 * Updated for Version 2.0 on Sat Apr 26 12:59:25 1997
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Header Files
 * ------------------------------------------------------------------
 * Modification History:
 * $Log: common.h,v $
 * Revision 2.16  1999/04/07 23:16:14  heller
 * Update for RedHat 5.2 (glibc / egcs)
 *
 * Revision 2.15  1998/05/20 21:57:33  heller
 * Further fix the small documentation error.
 *
 * Revision 2.14  1998/05/20 21:53:10  heller
 * Fix small documentation error
 *
 * Revision 2.13  1998/05/16 23:43:28  heller
 * Add indexing to the documentation.
 *
 * Revision 2.12  1998/04/21 15:42:41  heller
 * Fix documentation...
 *
 * Revision 2.11  1998/04/21 15:12:18  heller
 * Update copyright notice.
 * Small doc. changes.
 *
 * Revision 2.10  1997/07/20 21:13:11  heller
 * Fun with \ref
 *
 * Revision 2.9  1997/07/20 20:34:53  heller
 * Squigle fun...
 *
 * Revision 2.8  1997/07/15 12:51:06  heller
 * Small documentation changes.
 *
 * Revision 2.7  1997/07/14 22:23:51  heller
 * Add in automagical hooks for doc++ (chapter heading)
 *
 * Revision 2.6  1997/07/13 23:45:43  heller
 * Fix spelling errors
 *
 * Revision 2.5  1997/07/13 23:19:52  heller
 * Fix some comment problems.
 *
 * Revision 2.4  1997/07/13 23:17:54  heller
 * Add in DOC++ documentation.
 *
 * Revision 2.3  1997/07/06 21:44:54  heller
 * Misc. changes
 *
 * Revision 2.2  1997/06/29 19:14:59  heller
 * Fixes for portablity with CW
 *
 * Revision 2.1  1997/04/26 21:57:00  heller
 * Small changes
 *
 * Revision 2.0  1997/04/26 19:06:49  heller
 * *** empty log message ***
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *
 *    Home Librarian Database -- a program for maintaining a database
 *                               for a home library
 *    Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 *			51 Locke Hill Road
 *			Wendell, MA 01379-9728
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


//@ChapterName:Common Declarations

//@Man: Common Declarations
//@Memo: Common Declarations
/*@Doc:
  \typeout{Generated from: $Id: common.h,v 2.16 1999/04/07 23:16:14 heller Exp $}
  This chapter describes the common (not class specific) declarations used
  by the Home Librarian package.

  This file is included by all of the other source files.
  */


#ifndef _COMMON_
#define _COMMON_
#include <stdlib.h>		// standard C library functions
#include <string>
#include <string.h>
#include <errno.h>

//#include <iostream.h>

/*
 * O/S dependent stuff:
 *    we need file modes (passed as args to open and creat (create under
 *	 OSK) (FSpCreate/FSpOpenDF under MacOS)
 */
#ifdef OSK
#  include <modes.h>		// file modes
// generic symbols (hide O/S dependent names)
#  define ReadMode S_IREAD
#  define ReadWriteMode S_IREAD|S_IWRITE
#  define ReadFlags S_IREAD
#  define ReadWriteFlags S_IREAD|S_IWRITE
   extern "C" char* strerror(int);	// missing from the OSK library...
#else
#  ifdef macintosh
#    include <Files.h>
#    define ReadMode 0
#    define ReadWriteMode 0warning: comparison between signed and unsigned
#    define ReadFlags fsRdPerm
#    define ReadWriteFlags fsRdWrPerm
#  else
#    include <sys/stat.h>
#    include <fcntl.h>
//   generic symbols (hide O/S dependent names)
#    define ReadMode S_IREAD
#    define ReadWriteMode S_IREAD|S_IWRITE
#    include <unistd.h>
#    ifdef unix
#      define ReadFlags O_RDONLY
#      define ReadWriteFlags O_RDWR
#    else
#      define ReadFlags O_RDONLY|O_BINARY
#      define ReadWriteFlags O_RDWR|O_BINARY
#    endif
#  endif
#endif

   
//@Man: Scalar objects
//@Doc: \index{Scalar objects|(}
//@{
//@Man: LongInt
/*@Doc: \index{Scalar objects!LongInt}
  LongInt is a typedef for a 32-bit integer */
// support for 64-bit boxes
#ifdef _LONGLONG
typedef int LongInt;
#else
typedef long int LongInt;
#endif
//@Doc: \index{Scalar objects|)}
//@}
//@Man: Global constants:
//@Doc: \index{Global constants|(}
//@{
//@ManDoc: \index{Global constants!KeySize} KeySize is how long key strings can be.  This includes the nul byte
const KeySize = 36;
//@ManDoc:  \index{Global constants!Order}  Tree order.  This is the ``Order'' of the tree.
const Order   =  10;
//@ManDoc: Key type. Keys are strings of not more then 35 characters.
typedef char Key[KeySize];
/*@ManDoc:
 \index{Global constants!SectorSize}
 With an order 10 tree and a 32 byte key string + standard overhead
 (pointers, size info, data record pointers), we will get a tree node
 (page) of just under 1024 bytes.  With a small filler, pages will
 be 1024 bytes in size.  This is a reasonable size for most O/Ss.
 \index{Global constants|)}
 */
const SectorSize = 1024;
//@}
// Your basic true/false type:
#ifdef __GNUC__
//#include <bool.h>
typedef bool Boolean;
#endif

/*@ManDoc: \index{Global enumerated types|bold}
  \index{Global enumerated types|(}
  Global enumerated types
  All of the global enumerated types are embedded in this class.
 */
class HLEnums {
public:
	/*@ManDoc:
	  \index{Global enumerated types!Direction}
	  File input/output directions.
	 */
	enum Direction { 
	  //@ManDoc: Input only.
	  in, 
	  //@ManDoc: Output only.
	  out, 
	  //@ManDoc: Both Input and Output.
	  inout 
	};
	/*@ManDoc:
	  \index{Global enumerated types!OpenStatus}
	  File open status codes.
	 */
	enum OpenStatus {
		//@ManDoc: Open failure.
		failure, 
		//@ManDoc: Opened old file.
		openold, 
		//@ManDoc: Opened (created) new file.
		opennew 
	};
	/*@ManDoc:
	  \index{Global enumerated types!LSType}
	  Last search type.  Used by SearchXX and SearchXXAgain.
	 */
	enum LSType {
		//@ManDoc: No previous search.
		none, 
		//@ManDoc: Previous search was on the Id tree.
		id, 
		//@ManDoc: Previous search was on the Title tree.
		title, 
		//@ManDoc: Previous search was on the Author tree.
		author, 
		//@ManDoc: Previous search was on the Subject tree.
		subj 
	};
	/*@ManDoc:
	  \index{Global enumerated types!OpenMode}
	  Open modes.  How a file is opened.
	 */
	enum OpenMode {
		//@ManDoc: Open the file for reading only.
		ReadOnly = 0x001, 
		//@ManDoc: Open the file for reading and writing.
		ReadWrite = 0x003,
		//@ManDoc: Mask for the Read/Write bits.
		ModeMask = 0x0FF,
		//@ManDoc: Create a new file if the file does not exist.
		Create = 0x100 
	};
	/*@ManDoc:
	   \index{Global enumerated types!ErrKind}
	   Types of errors.
	 */
	enum ErrKind {
		//@ManDoc: Memory error.
		memErr,  
		//@ManDoc: Operating System error.
		sysErr, 
		//@ManDoc: \index{Global enumerated types|)} Internal Error.
		hlErr
	};
};


//@Man: Basic structure classes
/*@Doc: \index{Basic structure classes|(}
  \index{Basic structure classes|bold}
  These low-level structures represent the low-level file structure. */
//@{

/*@ManDoc:  \index{Basic structure classes!DiskRecord}
 \index{DiskRecord|bold}
 A disk record contains a size and a file offset.
 */
struct DiskRecord {
	//@ManDoc: This is the size of the record.
	LongInt record_size;
	//@ManDoc: This is the file offset of the record.
	LongInt record_address;
	/*@ManDoc:
	  The constructor makes sure we also have a properly
	  initialized object.
	 */
	DiskRecord (LongInt size = 0, LongInt addr = 0L) 
		   {record_size = size; record_address = addr;}
};

/*@ManDoc:
 \index{Basic structure classes!DiskPage}
 \index{DiskPage|bold}
 A disk page is similar, except the size is fixed.
 there are some special operator and constructor methods
 used to make things convenient, since these are used like
 pointers
 */
struct DiskPage {
	//@Man: record_size
	//@Type const
	//@Doc: Hard wired in size
	enum {record_size = SectorSize};
	//@ManDoc: This is the file offset of the Page
	LongInt  record_address;
	//@ManDoc: Construct a disk page from another disk page.
	DiskPage (const DiskPage& a);
	//@ManDoc: Construct a disk page from from a file offset.
	DiskPage (const LongInt addr = 0L);
	//@ManDoc: Are the pages the same page?
	friend Boolean operator == (const DiskPage a,const DiskPage b);
	//@ManDoc: Are the pages different pages?
	friend Boolean operator != (const DiskPage a,const DiskPage b);
	//@ManDoc: assign one disk page to another.
	DiskPage& operator = (const DiskPage a);
	//@ManDoc: assign a new disk address to a disk page.
	DiskPage& operator = (LongInt a);

};

/*@ManDoc:
  \index{Basic structure classes!Item}
  \index{Item|bold}
  A data item - has a key, a data record ``pointer'', and a child page
  ``pointer''.
 */
struct Item {
	//@ManDoc: The item's ``key''.
	Key key;
	//@ManDoc: The item's ``data''.
	DiskRecord data;
	//@ManDoc: Child page to the ``right''.
	DiskPage right;
};

/*@ManDoc:
   \index{Basic structure classes!Page}
   \index{Page|bold}
   A page is a tree node.
   It has a size (number of used items), pointers to a child and parent,
   a vector of items, and some filler (used to make sure the page is
   exactly 1024 bytes in size).
 */
struct Page {
	//@ManDoc: The number of items currently in the page
	LongInt size;
	//@ManDoc: The ``left'' child.
	DiskPage left;
	//@ManDoc: The parent (used in the SearchAgain code).
	DiskPage parent;
	//@ManDoc: The parent index (used in the SearchAgain code).
	LongInt parentidx;
	//@Man: items[]
	//@Type: Item
	//@Doc: The items in this node (page).
	Item items[Order*2];
	//@Man: filler[]
	//@Type: char
	//@Doc: filler is to round out the size to 1024 bytes.
	char filler[SectorSize-(sizeof(LongInt) +
				(2*sizeof(DiskPage)) +
				sizeof(LongInt) +
				(sizeof(Item)*Order*2))];
};

/*@ManDoc:
  \index{Basic structure classes!PTEntry}
  \index{PTEntry|bold}
   This is a ``Page Table Entry''.  This is used to keep track of the
   ``virtual'' memory used by the B-Trees
 */
struct PTEntry {
	/*@ManDoc: dirty flag -- if set, the page needs to be written out
	  to disk.
	 */
	Boolean isdirty;
	//@ManDoc: where the page is on disk.
	DiskPage diskpage;
	//@ManDoc: a core resident copy of the page.
	Page     *corepage;
};

const int MaxNumFree = (SectorSize - ( 8 + (4 * sizeof(DiskPage)) + 
					   sizeof(LongInt))) /
		       sizeof(DiskPage);

/*@ManDoc:
  \index{Basic structure classes!HomeBlock}
  \index{HomeBlock|bold}
  This is a 1024 byte record at the start of the file.
  it contains a 8 byte magic header, pointers to the roots of the
  four trees and a vector of available pre-allocated pages.
 */
struct HomeBlock {
	//@ManDoc: Version 1.x magic header string value
	static const char  * const OldMagic/* = "LIBRV000"*/;
	//@ManDoc: Version 2.x magic header string value
	static const char  * const Magic/* = "LIBRV001"*/;
	//@ManDoc: magic header string
	char magic[8];
	//@ManDoc: Id tree's root
	DiskPage IdRoot;
	//@ManDoc: Title tree's root
	DiskPage TitleRoot;
	//@ManDoc: Author tree's root
	DiskPage AuthorRoot;
	//@ManDoc: Subject tree's root
	DiskPage SubjRoot;
	//@ManDoc: number of free pages available.
	LongInt numfree;
	//@ManDoc: free page vector.
	DiskPage freepages[MaxNumFree];
};

/*@ManDoc:
  \index{Basic structure classes!UserDefinedCodesInFile}
  \index{UserDefinedCodesInFile|bold}
  These records are used to store the User Defined text strings for Card types,
  Location types, and Categories.
 */
struct UserDefinedCodesInFile {
	//@ManDoc: File offset to next block
	LongInt NextBlock;
	//@Man: buffer[]
	//@Type: char
	/*@Doc: Block of text strings (see Chapter\space\ref{vBTree}
	  for information of the format of these strings).
	 */
	char buffer[1024-sizeof(LongInt)];
};

/*@ManDoc:
  \index{Basic structure classes!UserDefinedCodesInCore}
  \index{UserDefinedCodesInCore|bold}
  This is the memory resident version of the User Defined text string blocks.
 */
struct UserDefinedCodesInCore {
	//@ManDoc: This block's file offset.
	DiskRecord diskBlock;
	//@ManDoc: The image of the disk record.
	UserDefinedCodesInFile thisBlock;
	//@ManDoc: Link to the next block.
	UserDefinedCodesInCore *nextBlock;
	//@ManDoc: Dirty flag.
	Boolean isdirty;
	/*@ManDoc: constructor -- make sure these structures are always
          properly initialized.
         */
	UserDefinedCodesInCore(LongInt addr = 0);
};

/*@ManDoc:
  \index{Basic structure classes!Record}
  \index{Record|bold}
  Core resident record - it has a size and a chunk of memory (the
  record itself).
 */
// Lots of fun.  I wish C++ had a garbage collector...
struct Record {
	//@ManDoc: Size of the record.
	LongInt size;
	//@ManDoc: Buffer for the data.
	char *buffer;
	//@ManDoc: Base constructor: empty buffer
	     Record () {size = 0; buffer = 0;}		// empty buffer
	//@ManDoc: Constructor: preallocated buffer.
	     Record (LongInt size) {Record::size = size; // preallocated buffer
				     buffer = new char[size];
				     }
	//@ManDoc: Constructor: record from another record
	     Record (const Record& rec)	{
			size = rec.size;
			if (size > 0) {
				buffer = new char[size];
				memcpy(buffer,rec.buffer,size);
			}
		}
	//@ManDoc: Destructor: free up allocated memory.
	    ~Record () {
		if (size > 0) {
			delete buffer;
		}
		buffer = 0;
	     } // Oh for a GC!
	//@ManDoc: assignment operator (copy a record) -- allocate fresh memory.
	Record& operator = (const Record& rec)	{
			if (size < rec.size) {
				if (buffer !=0) delete buffer;
				buffer = new char[rec.size];
			} else if (rec.size <= 0 && size > 0) {
			    delete buffer; buffer = 0; size = 0;
			}
			size = rec.size;
			if (rec.size > 0)
			    memcpy(buffer,rec.buffer,rec.size);
			return *this;
		}
	//@ManDoc: Buffer (re-)allocator function.
	void NewBuffer(LongInt size) {
			if (Record::size > 0) {
				delete buffer; buffer = 0; Record::size = 0;
			}
			Record::size = size;
			if (size > 0) {
				buffer = new char[size];
			}
		}			
};

/*@ManDoc:
  \index{Basic structure classes!CoreItem}
  \index{CoreItem|bold}
   In core item.  Code outside of vBTree.cc never does actual disk I/O.
   Instead it passes in-core records and the insertion, searching, and
    traversal code reads and writes the data from and to the disk file.
 */
struct CoreItem {
	//@ManDoc: Item key.
	Key key;
	//@ManDoc: Item data.
	//@Doc: \index{Basic structure classes|)}
	Record data;
};
//@}
#endif

