/* 
 * ------------------------------------------------------------------
 * Home Libarian 1.1 by Deepwoods Software
 * ------------------------------------------------------------------
 * TkvBTree_Import.cc - Import code for TkvBTree class
 * Created by Robert Heller on Sat Sep 20 19:16:21 1997
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: TkvBTree_Import.cc,v $
 * Revision 1.4  1999/04/07 23:16:14  heller
 * Update for RedHat 5.2 (glibc / egcs)
 *
 * Revision 1.3  1998/05/16 23:43:28  heller
 * Add indexing to the documentation.
 *
 * Revision 1.2  1998/02/15 18:39:09  heller
 * Update for additional inport (delimited) and export (formatted)
 *
 * Revision 1.1  1998/01/29 00:09:04  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Home Librarian Database -- a program for maintaining a database
 *                                for a home library
 *     Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

//@Man: ASCII Import functions.
/*@Doc:
  \typeout{Generated from: $Id: TkvBTree_Import.cc,v 1.4 1999/04/07 23:16:14 heller Rel $}
  This section describes the ASCII import constructors for vBTrees. These Tcl
  commands import a formatted ASCII file into a newly created Home Librarian
  file.  They all return a vBTree handle to the newly created file.
 */

//@{


static char ID[] = "$Id: TkvBTree_Import.cc,v 1.4 1999/04/07 23:16:14 heller Rel $";

#include <TkvBTree.h>
#include <TkCardRecord.h>
#include <ListRecord.h>
#include <stdio.h>
#include <errno.h>
#include <iostream.h>
#include <fstream.h>
#include <string>	// ANSI C++ string library

const initbuffsize = 2048;
const buffergrowth = 2048;

static char* ReadQuotedString(istream& in)
{
	char* outbuffer = 0;
	char* newbuffer = 0;
	char* p;
	char  ch;
	int   buffsize, charsleft, chindex;

	//cerr << "*** Entering ReadQuotedString\n";
	do {
		if (!in) return NULL;
		in.get(ch);
		//cerr << "*** Skiping '" << ch << "'\n";
	} while (ch != '"');
	outbuffer = new char[initbuffsize];
	//cerr << "*** ReadQuotedString(): outbuffer = ";cerr.form("0x%08x",outbuffer);cerr << endl;
	buffsize = initbuffsize;
	charsleft = initbuffsize;
	chindex = 0;
	p = outbuffer;
	while (true) {
		if (!in) return NULL;
		in.get(ch);
		if (ch == '"') break;
		else if (ch == '\\') {
			in >> ch;
			//cerr << "*** Inserting '" << ch << "' (was escaped)\n";
			*p++ = ch;
			charsleft--;
			chindex++;
		} else {
			//cerr << "*** Inserting '" << ch << "'\n";
			*p++ = ch;
			charsleft--;
			chindex++;
		}
		if (charsleft < 1) {
			newbuffer = new char[buffsize+buffergrowth];
	//cerr << "*** : newbuffer = ";cerr.form("0x%08x",newbuffer);cerr << endl;
			strcpy(newbuffer,outbuffer);
			delete outbuffer;
			outbuffer = newbuffer;
			buffsize += buffergrowth;
			p = outbuffer+chindex;
			charsleft = buffergrowth;
		}
	}
	*p = 0;
	return outbuffer;
}

//@Man: ImportV1ASCIIvBTree
//@Args: infilename treefilename ?minfree?
/*@Doc:
  \index{ImportV1ASCIIvBTree}
  This Tcl command imports a V1 ASCII file (as created by Ascii2Libr (HLV1) or
  exportv1ascii vBTree handle command (HLV2)), by creating a vBTree instance 
  and reading the V1 ASCII file into it.  The first argument (required) is the 
  name of the input (ASCII) file, the second argument (required) is the name 
  of the Home Librarian card catalog file to create.  The third (optional) 
  argument is the minimum number of pages to pre-allocate.  The number of pages 
  pre-allocated is the maximum of this number and the number supplied by the 
  V1 ASCII file.

  This function returns a string which is a ``handle'' to a vBTree instance.
  This handle is defined as a Tcl command that will access the vBTree instance.
  */

int TkvBTree::tkvBTreeImportV1ASCII(ClientData ,Tcl_Interp *interp,int argc, char *argv[])
{
	int numpages,numkeys,minpages;
	char* p;
	Key key;
	char ch;
	static char wordbuffer[80], errorBuffer[256];
	static char* vector[2048];
	int ielt;
	char *infile, *treefile;
	TkvBTree *tkvbtree;

	if (argc < 3 || argc > 4)
	{
		Tcl_AppendResult(interp, "Wrong # args: should be \"",
				 argv[0]," infilename treefile ?minpages?\"",
				 (char*) NULL);
		return TCL_ERROR;
	}

	
	infile = argv[1];
	treefile = argv[2];
	if (argc > 3) {
		if (Tcl_GetInt(interp,argv[3],&minpages) != TCL_OK) return TCL_ERROR;
	}

	ifstream infp(infile,ios::in);
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, argv[0],": Error opening ",infile,"!",
				 (char*) NULL);
		return TCL_ERROR;
	}
	infp >> numpages;
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, "Read error (numpages) on ",infile,
				 (char*) NULL);
		return TCL_ERROR;
	}
	if (argc > 3 && minpages > numpages)
	    numpages = minpages;
	tkvbtree = new TkvBTree(treefile,(HLEnums::OpenMode)(HLEnums::ReadWrite|HLEnums::Create),numpages);
	if (tkvbtree->WasError)
	{
		tkvbtree->FormatError(interp);
		delete tkvbtree;
		return TCL_ERROR;
	}
	if (tkvbtree->Tree.OpenStat() == HLEnums::failure)
	{
		int err = errno;
		Tcl_AppendResult(interp, "Open Failure on file: ",argv[1],
				 ": ",strerror(err),(char*) NULL);
		delete tkvbtree;
		return TCL_ERROR;
	} else if (tkvbtree->Tree.OpenStat() == HLEnums::openold)
	{
		Tcl_AppendResult(interp, treefile, " already exists!",
				 (char*) NULL);
		delete tkvbtree;
		return TCL_ERROR;
	}
				      
	infp >> numkeys;
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, "Read error (numkeys / cards) on ",
				 infile,(char*) NULL);
		delete tkvbtree;
		return TCL_ERROR;
	}
	int i;
	Card *tempcard = new Card(Card::Other,0,0,0,0,0,0,0,Card::Unknown,"",0);
	for (i = 0;i < numkeys;i++) {
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		p = ReadQuotedString(infp);
		if (p == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		strncpy(key,p,36);
		key[35] = 0;
		//cerr << "*** main(): key = \"" << key << "\"" << endl;
		//cerr << "*** main: freeing(p): ";cerr.form("0x%08x",(long int)p); cerr << endl;
		delete p; p = 0;
		//cerr << "*** Key = |" << key << "|\n";
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			infp >> ch;
		} while (ch != '#');
		infp >> ch;
		if (ch != 'C') {
			sprintf(errorBuffer,
				"Syntax error reading Card %d: expected a C, saw a %c\n",
				i,ch);
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp, errorBuffer, (char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		infp >> ch;
		if (ch != '(') {
			sprintf(errorBuffer,
				"Syntax error reading Card %d: expected a (, saw a %c\n",
				i,ch);
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp, errorBuffer, (char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		while (infp) {
			while (infp) {
				infp >> ch;
				//cerr << "*** ch = '" << ch << "'\n";
				if (ch == ')') break;
				else if (ch > ' ') {
					//cerr << "*** putting back '" << ch << "'\n";
					infp.putback(ch);
					break;
				}
			}
			if (ch == ')') break;
			infp >> wordbuffer;
			//cerr << "*** wordbuffer = |" << wordbuffer << "|\n";
			for (p = wordbuffer;*p != 0;p++) {
				ch = *p;
				if (islower(ch)) ch = toupper(ch);
				*p = ch;
			}
			if (strcmp(wordbuffer,":TYPE") == 0) {
				infp >> wordbuffer;
				tempcard->type = tkvbtree->Tree.CardTypeFromName(wordbuffer);
			} else if (strcmp(wordbuffer,":AUTHOR") == 0) {
				tempcard->author = ReadQuotedString(infp);
			} else if (strcmp(wordbuffer,":TITLE") == 0) {
				tempcard->title = ReadQuotedString(infp);
			} else if (strcmp(wordbuffer,":PUBLISHER") == 0) {
				tempcard->publisher = ReadQuotedString(infp);
			} else if (strcmp(wordbuffer,":CITY") == 0) {
				tempcard->city = ReadQuotedString(infp);
			} else if (strcmp(wordbuffer,":DESCRIPTION") == 0) {
				tempcard->description = ReadQuotedString(infp);
			} else if (strcmp(wordbuffer,":VOLUME") == 0) {
				infp >> tempcard->vol;
			} else if (strcmp(wordbuffer,":YEAR") == 0) {
				infp >> tempcard->year;
			} else {
				Tcl_ResetResult(interp);
				sprintf(errorBuffer,
					"Bad key in Card #%d: %s\n",i,wordbuffer);
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,errorBuffer,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
		}
		if (infp.eof() || infp.bad())
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		//cerr << "*** main(): tempcard(";cerr.form("0x%08x",(long int)tempcard);cerr << "):" << endl;
		//cerr << "\ttype: " << tempcard->type << endl;
		//cerr << "\tauthor: ";cerr.form("0x%08x",(long int)(tempcard->author)); cerr << "\"" << tempcard->author << "\"" << endl;
		//cerr << "\ttitle: ";cerr.form("0x%08x",(long int)(tempcard->title)); cerr << "\"" << tempcard->title << "\"" << endl;
		//cerr << "\tpublisher: ";cerr.form("0x%08x",(long int)(tempcard->publisher)); cerr << "\"" << tempcard->publisher << "\"" << endl;
		//cerr << "\tcity: ";cerr.form("0x%08x",(long int)(tempcard->city)); cerr << "\"" << tempcard->city << "\"" << endl;
		//cerr << "\tdescription: ";cerr.form("0x%08x",(long int)(tempcard->description)); cerr << "\"" << tempcard->description << "\"" << endl;
		//cerr << "\tvol " << tempcard->vol << endl;
		//cerr << "\tyear " << tempcard->year << endl;
		CardRecord crec(tempcard);
		Record rawrec = crec;
		tkvbtree->Tree.InsertId(key,&rawrec);
		if (tempcard->author != 0) {
		//cerr << "*** main: freeing(tempcard->author): ";cerr.form("0x%08x",(long int)(tempcard->author)); cerr << endl;
			delete tempcard->author;
			tempcard->author = 0;
		}
		if (tempcard->title != 0) {
		//cerr << "*** main: freeing(tempcard->title): ";cerr.form("0x%08x",(long int)(tempcard->title)); cerr << endl;
			delete tempcard->title;
			tempcard->title = 0;
		}
		if (tempcard->publisher != 0) {
		//cerr << "*** main: freeing(tempcard->publisher): ";cerr.form("0x%08x",(long int)(tempcard->publisher)); cerr << endl;
			delete tempcard->publisher;
			tempcard->publisher = 0;
		}
		if (tempcard->city != 0) {
		//cerr << "*** main: freeing(tempcard->city): ";cerr.form("0x%08x",(long int)(tempcard->city)); cerr << endl;
			delete tempcard->city;
			tempcard->city = 0;
		}
		if (tempcard->description != 0) {
		//cerr << "*** main: freeing(tempcard->description): ";cerr.form("0x%08x",(long int)(tempcard->description)); cerr << endl;
			delete tempcard->description;
			tempcard->description = 0;
		}
		//cerr << "*** main: freeing(crec): ";cerr.form("0x%08x",(long int)crec); cerr << endl;
		//delete crec;
	}
	//cerr << "*** main: freeing(tempcard): ";cerr.form("0x%08x",(long int)tempcard); cerr << endl;
	delete tempcard;
	tempcard = 0;
	infp >> numkeys;
	for (i = 0;i < numkeys;i++) {
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		p = ReadQuotedString(infp);
		if (p == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		strncpy(key,p,36);
		key[35] = 0;
		//cerr << "*** main(): key = \"" << key << "\"" << endl;
		//cerr << "*** main: freeing(p): ";cerr.form("0x%08x",(long int)p); cerr << endl;
		delete p;
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			infp >> ch;
		} while (ch != '#');
		infp >> ch;
		if (ch != '(') {
			sprintf(errorBuffer,
				"Syntax error reading Title %d: expected a (, saw a %c\n",
				i,ch);
			Tcl_AppendResult(interp, errorBuffer, (char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		for (ielt = 0;ielt < 2048;ielt++) {
			while (infp) {
				ch = infp.peek();
				if (ch == ')') {ch = infp.get();break;}
				else if (ch > ' ') break;
			}
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (ch == ')') break;
			vector[ielt] = ReadQuotedString(infp);
		}
		ListRecord lrec(ielt,vector);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertTitle(key,&rawrec);
		for (;ielt > 0;ielt--)
		{
			//cerr << "*** main: freeing(vector[" << ielt-1 << "]): ";cerr.form("0x%08x",(long int)(vector[ielt-1])); cerr << endl;
			delete vector[ielt-1];
		}
	}
	infp >> numkeys;
	for (i = 0;i < numkeys;i++) {
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		p = ReadQuotedString(infp);
		if (p == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		strncpy(key,p,36);
		key[35] = 0;
		//cerr << "*** main(): key = \"" << key << "\"" << endl;
		//cerr << "*** main: freeing(p): ";cerr.form("0x%08x",(long int)p); cerr << endl;
		delete p;
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			infp >> ch;
		} while (ch != '#');
		infp >> ch;
		if (ch != '(') {
			sprintf(errorBuffer,
				"Syntax error reading Author %d: expected a (, saw a %c\n",
				i,ch);
			Tcl_AppendResult(interp, errorBuffer, (char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		for (ielt = 0;ielt < 2048;ielt++) {
			while (infp) {
				ch = infp.peek();
				if (ch == ')') {ch = infp.get();break;}
				else if (ch > ' ') break;
			}
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (ch == ')') break;
			vector[ielt] = ReadQuotedString(infp);
		}
		ListRecord lrec(ielt,vector);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertAuthor(key,&rawrec);
		for (;ielt > 0;ielt--)
		{
			//cerr << "*** main: freeing(vector[" << ielt-1 << "]): ";cerr.form("0x%08x",(long int)(vector[ielt-1])); cerr << endl;
			delete vector[ielt-1];
		}
	}
	infp >> numkeys;
	for (i = 0;i < numkeys;i++) {
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		p = ReadQuotedString(infp);
		if (p == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		strncpy(key,p,36);
		key[35] = 0;
		//cerr << "*** main(): key = \"" << key << "\"" << endl;
		//cerr << "*** main: freeing(p): ";cerr.form("0x%08x",(long int)p); cerr << endl;
		delete p;
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			infp >> ch;
		} while (ch != '#');
		infp >> ch;
		if (ch != '(') {
			sprintf(errorBuffer,
				"Syntax error reading Subject list %d: expected a (, saw a %c\n",
				i,ch);
			Tcl_AppendResult(interp, errorBuffer, (char*) NULL);
			delete tkvbtree;
			return TCL_ERROR;
		}
		for (ielt = 0;ielt < 2048;ielt++) {
			while (infp) {
				ch = infp.peek();
				if (ch == ')') {ch = infp.get();break;}
				else if (ch > ' ') break;
			}
			if (infp.eof() || infp.bad())
			{
				Tcl_ResetResult(interp);
				Tcl_AppendResult(interp,"Premature EOF on file ",infile,(char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (ch == ')') break;
			vector[ielt] = ReadQuotedString(infp);
		}
		ListRecord lrec(ielt,vector);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertSubj(key,&rawrec);
		for (;ielt > 0;ielt--)
		{
			//cerr << "*** main: freeing(vector[" << ielt-1 << "]): ";cerr.form("0x%08x",(long int)(vector[ielt-1])); cerr << endl;
			delete vector[ielt-1];
		}
	}
	tkvbtree->newHandle();
	Tcl_CreateCommand(interp,tkvbtree->myHandle,
			  (Tcl_CmdProc*)TkCppWraper::TclCommand,
			  (ClientData)tkvbtree,
			  (Tcl_CmdDeleteProc*)TkCppWraper::deleteTkCppWraper);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp,tkvbtree->myHandle,(char *) NULL);
	return TCL_OK;
}

static char * StripBraces(const char *p)
{
	//cerr << "*** StripBraces: " << p << endl;
	char *result = new char[strlen(p)+1];
	if (*p != '{') strcpy(result,p);
	else
	{
		strcpy(result,p+1);
		char *closebrace = strrchr(result,'}');
		if (closebrace != NULL) *closebrace = '\0';
	}
	return result;
}

//@Man: ImportV2ASCIIvBTree
//@Args: infilename treefilename ?minfree?
/*@Doc:
  \index{ImportV2ASCIIvBTree}
  This Tcl command imports a V2 ASCII file (as created by exportv2ascii vBTree 
  handle command), by creating a vBTree instance and reading the V2 ASCII file 
  into it.  The first argument (required) is the name of the input (ASCII) 
  file, the second argument (required) is the name of the Home Librarian card 
  catalog file to create.  The third (optional) argument is the minimum number 
  of pages to pre-allocate.  The number of pages pre-allocated is the maximum 
  of this number and the number supplied by the V2 ASCII file.

  This function returns a string which is a ``handle'' to a vBTree instance.
  This handle is defined as a Tcl command that will access the vBTree instance.
  */

int TkvBTree::tkvBTreeImportV2ASCII(ClientData ,Tcl_Interp *interp,int argc, char *argv[])
{
	int numpages,numkeys,minpages, ikey;
	char* p;
	Key key;
	/*char ch;*/
	static char wordbuffer[80], errorBuffer[256];
	static char lineBuffer[4096];
	char **listElts = 0, **subList = 0;
	int  listEltsC, subListC;
	int ielt;
	char *infile, *treefile;
	TkvBTree *tkvbtree;
	Tcl_DString record;

	if (argc < 3 || argc > 4)
	{
		Tcl_AppendResult(interp, "Wrong # args: should be \"",
				 argv[0]," infilename treefile ?minpages?\"",
				 (char*) NULL);
		return TCL_ERROR;
	}
	infile = argv[1];
	treefile = argv[2];
	if (argc > 3) {
		if (Tcl_GetInt(interp,argv[3],&minpages) != TCL_OK) return TCL_ERROR;
	}

	ifstream infp(infile,ios::in);
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, argv[0],": Error opening ",infile,"!",
				 (char*) NULL);
		return TCL_ERROR;
	}
	infp.getline(wordbuffer,80);
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, "Read error (header line) on ",
				 infile,(char*) NULL);
		return TCL_ERROR;
	}
	if (strcmp(wordbuffer,"Library-V2-Ascii") != 0)
	{
		Tcl_AppendResult(interp, "Read mismatch on header in ",
				 infile,(char*) NULL);
		return TCL_ERROR;
	}
	infp >> numpages;
	if (infp.eof() || infp.bad())
	{
		Tcl_AppendResult(interp, "Read error (numpages) on ",infile,
				 (char*) NULL);
		return TCL_ERROR;
	}
	if (infp.peek() == '\n') infp.get();
	if (argc > 3 && minpages > numpages)
	    numpages = minpages;
	tkvbtree = new TkvBTree(treefile,(HLEnums::OpenMode)(HLEnums::ReadWrite|HLEnums::Create),numpages);
	if (tkvbtree->WasError)
	{
		tkvbtree->FormatError(interp);
		delete tkvbtree;
		return TCL_ERROR;
	}
	if (tkvbtree->Tree.OpenStat() == HLEnums::failure)
	{
		int err = errno;
		Tcl_AppendResult(interp, "Open Failure on file: ",argv[1],
				 ": ",strerror(err),(char*) NULL);
		delete tkvbtree;
		return TCL_ERROR;
	} else if (tkvbtree->Tree.OpenStat() == HLEnums::openold)
	{
		Tcl_AppendResult(interp, treefile, " already exists!",
				 (char*) NULL);
		delete tkvbtree;
		return TCL_ERROR;
	}
	Tcl_DStringInit(&record);
	for (ikey = 0; ikey < (Card::NumCardTypes - Card::IndexUC1); ikey++)
	{
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (CardTypes) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC != 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (CardTypes) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		const char *tn = tkvbtree->Tree.CardTypeName((Card::CardType)(listElts[0][0]));
		char *name = listElts[1];
		if (!(tn == NULL && *name == '\0') && (strcmp(tn,name) != 0))
		{
			tkvbtree->Tree.AddCardTypeName((Card::CardType)(listElts[0][0]),name);
		}
		Tcl_Free((char*)listElts);
		Tcl_DStringFree(&record);
	}
	for (ikey = 0; ikey < (Card::NumLocationTypes - Card::IndexUL1); ikey++)
	{
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (LocationTypes) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC != 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (LocationTypes) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		const char *tn = tkvbtree->Tree.LocationTypeName((Card::LocationType)(listElts[0][0]));
		char *name = listElts[1];
		if (!(tn == NULL && *name == '\0') && (strcmp(tn,name) != 0))
		{
			tkvbtree->Tree.AddLocationTypeName((Card::LocationType)(listElts[0][0]),name);
		}
		Tcl_Free((char*)listElts);
		Tcl_DStringFree(&record);
	}
	infp >> numkeys;
	if (infp.peek() == '\n') infp.get();
	for (ikey = 0; ikey < numkeys; ikey++)
	{
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (Categories) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC != 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (Categories) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		int ccode = atoi(listElts[0]);
		const char *tn = tkvbtree->Tree.Category(ccode);
		char *name = listElts[1];
		if (!(tn == NULL && *name == '\0') && (strcmp(tn,name) != 0))
		{
			tkvbtree->Tree.AddCategory(ccode,name);
		}
		Tcl_Free((char*)listElts);
		Tcl_DStringFree(&record);
	}
	infp >> numkeys;
	if (infp.peek() == '\n') infp.get();
	Card *tempcard = new Card(Card::Other,0,0,0,0,0,0,0,Card::Unknown,0,0);
	for (ikey = 0; ikey < numkeys; ikey++)
	{
		//cerr << "*** ikey = " << ikey << endl;
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		Tcl_ResetResult(interp);
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
 			{
				Tcl_AppendResult(interp,
						 "Read error (Cards) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		p = StripBraces(p);
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC < 3)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (Cards) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		strncpy(key,listElts[0],36);
		key[35] = 0;
		if (strcmp(listElts[1],"CardRecord") != 0)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,
					 "} (Cards: missing header) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		for (ielt = 2; ielt < listEltsC; ielt++)
		{
			cerr << "*** listElts[" << ielt << "] = " << listElts[ielt] << endl;
			if (Tcl_SplitList(interp,listElts[ielt],&subListC,&subList) != TCL_OK)
			{
				delete tkvbtree;
				Tcl_Free((char*)listElts);
				return TCL_ERROR;
			}
			if (subListC != 2)
			{
				Tcl_AppendResult(interp,
						 "Format error {",
						 listElts[ielt],
						 "} (Cards, item element) on ",
						 infile,(char*) NULL);
				delete tkvbtree;
				Tcl_Free((char*)subList);
				Tcl_Free((char*)listElts);
				return TCL_ERROR;
			}
			cerr << "*** subList[0] = '" << subList[0] << "'" << endl;
			if (strcmp(subList[0],"cardtype") == 0)
			{
				tempcard->type = (Card::CardType) subList[1][0];
			} else if (strcmp(subList[0],"author") == 0)
			{
				if (tempcard->author != 0)
				{
					delete tempcard->author;
					tempcard->author = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->author = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"title") == 0)
			{
				if (tempcard->title != 0)
				{
					delete tempcard->title;
					tempcard->title = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->title = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"publisher") == 0)
			{
				if (tempcard->publisher != 0)
				{
					delete tempcard->publisher;
					tempcard->publisher = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->publisher = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"city") == 0)
			{
				if (tempcard->city != 0)
				{
					delete tempcard->city;
					tempcard->city = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->city = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"description") == 0)
			{
				if (tempcard->description != 0)
				{
					delete tempcard->description;
					tempcard->description = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->description = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"vol") == 0)
			{
				int val;
				if (Tcl_GetInt(interp,subList[1],&val) != TCL_OK)
				{
					delete tkvbtree;
					Tcl_Free((char*)listElts);
					Tcl_Free((char*)subList);
					return TCL_ERROR;
				}
				tempcard->vol = val;
			} else if (strcmp(subList[0],"year") == 0)
			{
				int val;
				if (Tcl_GetInt(interp,subList[1],&val) != TCL_OK)
				{
					delete tkvbtree;
					Tcl_Free((char*)listElts);
					Tcl_Free((char*)subList);
					return TCL_ERROR;
				}
				tempcard->year = val;
			} else if (strcmp(subList[0],"locationtype") == 0)
			{
				tempcard->ltype = (Card::LocationType) subList[1][0];
			} else if (strcmp(subList[0],"locationdetail") == 0)
			{
				if (tempcard->locdetail != 0)
				{
					delete tempcard->locdetail;
					tempcard->locdetail = 0;
				}
				//char *word = StripBraces(subList[1]);
				//char *str = new char[strlen(word+1)];
				//strcpy(str,word);
				tempcard->locdetail = StripBraces(subList[1]);
			} else if (strcmp(subList[0],"categorycode") == 0)
			{
				int val;
				if (Tcl_GetInt(interp,subList[1],&val) != TCL_OK)
				{
					delete tkvbtree;
					Tcl_Free((char*)listElts);
					Tcl_Free((char*)subList);
					return TCL_ERROR;
				}
				tempcard->catcode = val;
			} else
			{
				sprintf(errorBuffer,
					"Bad key in Card #%d: %s\n",ikey,subList[0]);
				Tcl_AppendResult(interp,errorBuffer,(char*) NULL);
				delete tkvbtree;
				Tcl_Free((char*)listElts);
				Tcl_Free((char*)subList);
				return TCL_ERROR;
				
			}
			Tcl_Free((char*)subList);
		}
		//cerr << "*** -: tempcard(";cerr.form("0x%08x",(long int)tempcard);cerr << "):" << endl;
		//cerr << "\ttype: " << (char)(tempcard->type) << endl;
		//cerr << "\tauthor: ";cerr.form("0x%08x",(long int)(tempcard->author)); cerr << "\"" << tempcard->author << "\"" << endl;
		//cerr << "\ttitle: ";cerr.form("0x%08x",(long int)(tempcard->title)); cerr << "\"" << tempcard->title << "\"" << endl;
		//cerr << "\tpublisher: ";cerr.form("0x%08x",(long int)(tempcard->publisher)); cerr << "\"" << tempcard->publisher << "\"" << endl;
		//cerr << "\tcity: ";cerr.form("0x%08x",(long int)(tempcard->city)); cerr << "\"" << tempcard->city << "\"" << endl;
		//cerr << "\tdescription: ";cerr.form("0x%08x",(long int)(tempcard->description)); cerr << "\"" << tempcard->description << "\"" << endl;
		//cerr << "\tvol " << tempcard->vol << endl;
		//cerr << "\tyear " << tempcard->year << endl;
		//cerr << "\tltype " << (char)(tempcard->ltype) << endl;
		//cerr << "\tlocdetail "; cerr.form("0x%08x",(long int)(tempcard->locdetail)); cerr << "\"" << tempcard->locdetail << "\"" << endl;
		//cerr << "\tcatcode " << (unsigned int)(tempcard->catcode) << endl;
		CardRecord crec(tempcard);
		//cerr << "*** crec built" << endl;
		Record rawrec = crec;
		//cerr << "*** Inserting " << tempcard->title << " at " << key << endl;
		tkvbtree->Tree.InsertId(key,&rawrec);
		//cerr << "*** inserted..." << endl;
		//cerr << "*** listElts = ";cerr.form("0x%08x",(long int)(listElts)); cerr << endl;
		Tcl_Free((char*)listElts);
		listElts = 0;
		//cerr << "*** listElts freed" << endl;
		Tcl_DStringFree(&record);
		//cerr << "*** record freed" << endl;
		delete p;
	}
	if (tempcard->author != 0) delete tempcard->author;
	if (tempcard->title != 0) delete tempcard->title;
	if (tempcard->publisher != 0) delete tempcard->publisher;
	if (tempcard->city != 0) delete tempcard->city;
	if (tempcard->description != 0) delete tempcard->description;
	if (tempcard->locdetail != 0) delete tempcard->locdetail;
	delete tempcard;
	infp >> numkeys;
	if (infp.peek() == '\n') infp.get();
	for (ikey = 0; ikey < numkeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		Tcl_ResetResult(interp);
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (Titles) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		//cerr << "*** titlelist: p = " << p << endl;
		p = StripBraces(p);
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC < 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (Titles) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		strncpy(key,listElts[0],36);
		key[35] = 0;
		ListRecord lrec(listEltsC-1,listElts+1);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertTitle(key,&rawrec);
		Tcl_Free((char*)listElts);
		delete p;
		Tcl_DStringFree(&record);
	}
	infp >> numkeys;
	if (infp.peek() == '\n') infp.get();
	for (ikey = 0; ikey < numkeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		Tcl_ResetResult(interp);
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (Authors) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		p = StripBraces(p);
		//cerr << "*** authorlist: p = " << p << endl;
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC < 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (Authors) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		strncpy(key,listElts[0],36);
		key[35] = 0;
		ListRecord lrec(listEltsC-1,listElts+1);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertAuthor(key,&rawrec);
		Tcl_Free((char*)listElts);
		delete p;
		Tcl_DStringFree(&record);
	}
	infp >> numkeys;
	if (infp.peek() == '\n') infp.get();
	for (ikey = 0; ikey < numkeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		Tcl_ResetResult(interp);
		if (infp.peek() == '\n') infp.get();
		do {
			if (infp.eof() || infp.bad())
			{
				Tcl_AppendResult(interp,
						 "Read error (Subjects) on ",
						 infile,", buffer = ",
						 Tcl_DStringValue(&record),
						 (char*) NULL);
				delete tkvbtree;
				return TCL_ERROR;
			}
			if (infp.peek() == '\n')
			{
				infp.get();
				Tcl_DStringAppend(&record,"\n",-1);
			}
			infp.get(lineBuffer,4096);
			Tcl_DStringAppend(&record,lineBuffer,-1);
			p = Tcl_DStringValue(&record);
		} while (!Tcl_CommandComplete(p));
		p = StripBraces(p);
		//cerr << "*** subjectlist: p = " << p << endl;
		if (Tcl_SplitList(interp,p,&listEltsC,&listElts) != TCL_OK)
		{
			delete tkvbtree;
			return TCL_ERROR;
		}
		if (listEltsC < 2)
		{
			Tcl_AppendResult(interp,
					 "Format error {",p,"} (Subjects) on ",
					 infile,(char*) NULL);
			delete tkvbtree;
			Tcl_Free((char*)listElts);
			return TCL_ERROR;
		}
		strncpy(key,listElts[0],36);
		key[35] = 0;
		ListRecord lrec(listEltsC-1,listElts+1);
		Record rawrec = lrec;
		tkvbtree->Tree.InsertSubj(key,&rawrec);
		Tcl_Free((char*)listElts);
		delete p;
		Tcl_DStringFree(&record);
	}
	tkvbtree->newHandle();
	Tcl_CreateCommand(interp,tkvbtree->myHandle,
			  (Tcl_CmdProc*)TkCppWraper::TclCommand,
			  (ClientData)tkvbtree,
			  (Tcl_CmdDeleteProc*)TkCppWraper::deleteTkCppWraper);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp,tkvbtree->myHandle,(char *) NULL);
	return TCL_OK;
}
	
//@Man: ImportDelimitedvBTree
//@Args: infilename treefilename ?minfree?
/*@Doc:
  \index{ImportDelimitedvBTree}
  This Tcl command imports a delimited ASCII fileset, in the same format as
  exported by the exportdelimited vBTree handle command.  A vBTree instance
  is created and the elimited ASCII fileset is read into it. The first argument (required) is the name of the 
  input (ASCII) file, the second argument (required) is the name of the Home 
  Librarian card catalog file to create.  The third (optional) argument is the 
  minimum number of pages to pre-allocate.

  This function returns a string which is a ``handle'' to a vBTree instance.
  This handle is defined as a Tcl command that will access the vBTree instance.
*/


int TkvBTree::tkvBTreeImportDelimited(ClientData ,Tcl_Interp *interp,int argc, char *argv[])
{
	const char DelimiterIs[] = "^# delimiter is (.)$",
		   QuotedWith[]  = "^# quoted with (.)$",
		   EscapedWith[] = "^# escaped with (.)$",
		   CardRecs[]    = "^# (.*_cards):([0-9]+) CardRecords: <cardtype>%c<author>%c<title>%c<publisher>%c<city>%c<description>%c<volume>%c<year>%c<locationtype>%c<locationdetail>%c<catcode>$",
		   TitleRecs[]   = "^# (.*_titles):([0-9]+) titles: <title>%c<ids>$",
		   AuthorRecs[]  = "^# (.*_authors):([0-9]+) authors: <author>%c<ids>$",
		   SubjRecs[]    = "^# (.*_subjects):([0-9]+) subjects: <subject>%c<ids>$";
	char PatternBuffer[512];
	Tcl_RegExp regexpToken;
	int numpages,numkeys, dlen;
	char quoesc, del;
	bool qe;
	static char lineBuffer[4096];
	char *infile, *treefile;
	char *dfilename, *extension, *dfilename2;
	ifstream infp2;
	TkvBTree *tkvbtree;

	if (argc < 3 || argc > 4)
	{
		Tcl_AppendResult(interp, "Wrong # args: should be \"",
				 argv[0]," infilename treefile ?numpages?\"",
				 (char*) NULL);
		return TCL_ERROR;
	}
	infile = argv[1];
	treefile = argv[2];
	if (argc > 3) {
		if (Tcl_GetInt(interp,argv[3],&numpages) != TCL_OK) return TCL_ERROR;
	}
	ifstream infp1(infile,ios::in);
	if (!infp1)
	{
		int err = errno;
		Tcl_AppendResult(interp, "Could not open main input file ",
				 infile,": ",strerror(err),(char*) NULL);
		return TCL_ERROR;
	}
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	regexpToken = Tcl_RegExpCompile(interp,DelimiterIs);
	if (regexpToken == NULL) return TCL_ERROR;
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
		    Tcl_AppendResult(interp,"Format error reading ",infile,
				     ", read '",lineBuffer,
				     "' expected '# delimiter is '",
				     (char*) NULL);
		    infp1.close();
		    return TCL_ERROR;
		case -1: infp1.close(); return TCL_ERROR;
		case 1: del = lineBuffer[15]; break;
	}
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	regexpToken = Tcl_RegExpCompile(interp,QuotedWith);
	if (regexpToken == NULL)
	{
		infp1.close();
		return TCL_ERROR;
	}
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
			regexpToken = Tcl_RegExpCompile(interp,EscapedWith);
			if (regexpToken == NULL)
			{
				infp1.close();
				return TCL_ERROR;
			}
			switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
			{
				case 0:
				    Tcl_AppendResult(interp,
						     "Format error reading ",
						     infile,", read '",
						     lineBuffer,
						     "' expected '# quoted with ' or '# escaped with '",
						     (char*) NULL);
				    infp1.close();
				    return TCL_ERROR;
				case -1: infp1.close(); return TCL_ERROR;
				case 1:
				    qe = false;
				    quoesc = lineBuffer[15];
				    break;
			}
			break;
		case -1: infp1.close(); return TCL_ERROR;
		case 1:
		    qe = true;
		    quoesc = lineBuffer[14];
		    break;
	}
	if (argc > 3) {
		tkvbtree = new TkvBTree(treefile,
					(HLEnums::OpenMode)(HLEnums::ReadWrite|HLEnums::Create),
					numpages);
	} else {
		tkvbtree = new TkvBTree(treefile,
					(HLEnums::OpenMode)(HLEnums::ReadWrite|HLEnums::Create));
	}
	if (tkvbtree->WasError)
	{
		tkvbtree->FormatError(interp);
		delete tkvbtree;
		infp1.close();
		return TCL_ERROR;
	}
	if (tkvbtree->Tree.OpenStat() == HLEnums::failure)
	{
		int err = errno;
		Tcl_AppendResult(interp, "Open Failure on file: ",argv[1],
				 ": ",strerror(err),(char*) NULL);
		delete tkvbtree;
		infp1.close();
		return TCL_ERROR;
	} else if (tkvbtree->Tree.OpenStat() == HLEnums::openold)
	{
		Tcl_AppendResult(interp, treefile, " already exists!",
				 (char*) NULL);
		delete tkvbtree;
		infp1.close();
		return TCL_ERROR;
	}
	tkvbtree->quoesc = quoesc;
	tkvbtree->del    = del;
	tkvbtree->qe     = qe;
	dfilename = new char[strlen(infile)+10];
	strcpy(dfilename,infile);
	extension = dfilename + strlen(dfilename);
	strcpy(extension,"_cards");
	dlen = strlen(dfilename);
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	sprintf(PatternBuffer,CardRecs,del,del,del,del,del,del,del,del,del,del);
	regexpToken = Tcl_RegExpCompile(interp,PatternBuffer);
	if (regexpToken == NULL)
	{
		infp1.close();
		delete tkvbtree;
		return TCL_ERROR;
	}
	
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
			Tcl_AppendResult(interp,"Format error reading ",
					 infile,", read '",lineBuffer,
					 "' expected '",PatternBuffer,"'",
					 (char*) NULL);
			infp1.close();
			delete tkvbtree;
			return TCL_ERROR;
		case -1: infp1.close(); delete tkvbtree; return TCL_ERROR;
		case 1:
		{
			char *start,*end,temp[32];
			int len;
			Tcl_RegExpRange(regexpToken,2,&start,&end);
			len = end - start;
			strncpy(temp,start,len);
			temp[len] = '\0';
			numkeys = atoi(temp);
			Tcl_RegExpRange(regexpToken,1,&start,&end);
			len = end - start;
			dfilename2 = new char[len+1];
			strncpy(dfilename2,start,len);
			dfilename2[len] = '\0';
			break;
		}
	}
	if (!(tkvbtree->tkvBTreeImportDelimitedCardRecs(interp,numkeys,dfilename,dfilename2)))
	{
		infp1.close();
		delete tkvbtree;
		delete dfilename2;
		return TCL_ERROR;
	}
	delete dfilename2;
	strcpy(extension,"_titles");
	dlen = strlen(dfilename);
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	sprintf(PatternBuffer,TitleRecs,del);
	regexpToken = Tcl_RegExpCompile(interp,PatternBuffer);
	if (regexpToken == NULL)
	{
		infp1.close();
		delete tkvbtree;
		return TCL_ERROR;
	}
	
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
			Tcl_AppendResult(interp,"Format error reading ",
					 infile,", read '",lineBuffer,
					 "' expected '",PatternBuffer,"'",
					 (char*) NULL);
			infp1.close();
			delete tkvbtree;
			return TCL_ERROR;
		case -1: infp1.close(); delete tkvbtree; return TCL_ERROR;
		case 1:
		{
			char *start,*end,temp[32];
			int len;
			Tcl_RegExpRange(regexpToken,2,&start,&end);
			len = end - start;
			strncpy(temp,start,len);
			temp[len] = '\0';
			numkeys = atoi(temp);
			Tcl_RegExpRange(regexpToken,1,&start,&end);
			len = end - start;
			dfilename2 = new char[len+1];
			strncpy(dfilename2,start,len);
			dfilename2[len] = '\0';
			break;
		}
	}
	if (!(tkvbtree->tkvBTreeImportDelimitedTitleRecs(interp,numkeys,dfilename,dfilename2)))
	{
		infp1.close();
		delete tkvbtree;
		delete dfilename2;
		return TCL_ERROR;
	}
	delete dfilename2;
	strcpy(extension,"_authors");
	dlen = strlen(dfilename);
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	sprintf(PatternBuffer,AuthorRecs,del);
	regexpToken = Tcl_RegExpCompile(interp,PatternBuffer);
	if (regexpToken == NULL)
	{
		infp1.close();
		delete tkvbtree;
		return TCL_ERROR;
	}
	
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
			Tcl_AppendResult(interp,"Format error reading ",
					 infile,", read '",lineBuffer,
					 "' expected '",PatternBuffer,"'",
					 (char*) NULL);
			infp1.close();
			delete tkvbtree;
			return TCL_ERROR;
		case -1: infp1.close(); delete tkvbtree; return TCL_ERROR;
		case 1:
		{
			char *start,*end,temp[32];
			int len;
			Tcl_RegExpRange(regexpToken,2,&start,&end);
			len = end - start;
			strncpy(temp,start,len);
			temp[len] = '\0';
			numkeys = atoi(temp);
			Tcl_RegExpRange(regexpToken,1,&start,&end);
			len = end - start;
			dfilename2 = new char[len+1];
			strncpy(dfilename2,start,len);
			dfilename2[len] = '\0';
			break;
		}
	}
	if (!(tkvbtree->tkvBTreeImportDelimitedAuthorRecs(interp,numkeys,dfilename,dfilename2)))
	{
		infp1.close();
		delete tkvbtree;
		delete dfilename2;
		return TCL_ERROR;
	}
	delete dfilename2;
	strcpy(extension,"_subjects");
	dlen = strlen(dfilename);
	infp1.getline(lineBuffer,sizeof(lineBuffer));
	sprintf(PatternBuffer,SubjRecs,del);
	regexpToken = Tcl_RegExpCompile(interp,PatternBuffer);
	if (regexpToken == NULL)
	{
		infp1.close();
		delete tkvbtree;
		return TCL_ERROR;
	}
	
	switch (Tcl_RegExpExec(interp,regexpToken,lineBuffer,lineBuffer))
	{
		case 0:
			Tcl_AppendResult(interp,"Format error reading ",
					 infile,", read '",lineBuffer,
					 "' expected '",PatternBuffer,"'",
					 (char*) NULL);
			infp1.close();
			delete tkvbtree;
			return TCL_ERROR;
		case -1: infp1.close(); delete tkvbtree; return TCL_ERROR;
		case 1:
		{
			char *start,*end,temp[32];
			int len;
			Tcl_RegExpRange(regexpToken,2,&start,&end);
			len = end - start;
			strncpy(temp,start,len);
			temp[len] = '\0';
			numkeys = atoi(temp);
			Tcl_RegExpRange(regexpToken,1,&start,&end);
			len = end - start;
			dfilename2 = new char[len+1];
			strncpy(dfilename2,start,len);
			dfilename2[len] = '\0';
			break;
		}
	}
	if (!(tkvbtree->tkvBTreeImportDelimitedSubjRecs(interp,numkeys,dfilename,dfilename2)))
	{
		infp1.close();
		delete tkvbtree;
		delete dfilename2;
		return TCL_ERROR;
	}
	delete dfilename2;
	infp1.close();
	tkvbtree->newHandle();
	Tcl_CreateCommand(interp,tkvbtree->myHandle,
			  (Tcl_CmdProc*)TkCppWraper::TclCommand,
			  (ClientData)tkvbtree,
			  (Tcl_CmdDeleteProc*)TkCppWraper::deleteTkCppWraper);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp,tkvbtree->myHandle,(char *) NULL);
	return TCL_OK;
}


bool TkvBTree::tkvBTreeImportDelimitedCardRecs(Tcl_Interp *interp,int numKeys,const char *inf1,const char *inf2)
{
	int ikey;
	const char *item;
	Key key;
	const char *fn = inf1;
	ifstream infp2(inf1,ios::in);
	if (!infp2)
	{
		infp2.open(inf2,ios::in);
		fn = inf2;
		if (!infp2)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Could not open ",inf1," or ",inf2,(char *)NULL);
			return false;
		}
	}
	for (ikey = 0; ikey < numKeys; ikey++)
	{
		Card tempcard;
	
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card key]",(char *)NULL);
			infp2.close();
			return false;
		}
		strncpy(key,item,sizeof(key)-1);
		key[sizeof(key)-1] = 0;
		delete item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card type]",(char *)NULL);
			infp2.close();
			return false;
		}
		if (Tree.CardTypeNameP(item))
		{
			tempcard.type = Tree.CardTypeFromName(item);
		} else
		{
			static Card::CardType UCtypes[] = {
				Card::UC1, Card::UC2, Card::UC3, Card::UC4, 
				Card::UC5, Card::UC6, Card::UC7, Card::UC8, 
				Card::UC9, Card::UC10};
			static int nUCtypes = sizeof(UCtypes) / sizeof(UCtypes[0]);
			int i;
			for (i = 0; i < nUCtypes; i++)
			{
				if (strcmp(Tree.CardTypeName(UCtypes[i]),"Other") == 0) break;
			}
			if (i >= nUCtypes)
			{
			}
			Tree.AddCardTypeName(UCtypes[i],item);
			tempcard.type = UCtypes[i];
		}
		delete item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card author]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.author = item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card title]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.title = item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card publisher]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.publisher = item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card city]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.city = item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card description]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.description = item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card volume]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.vol = atoi(item);
		delete item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card year]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.year = atoi(item);
		delete item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card ltype]",(char *)NULL);
			infp2.close();
			return false;
		}
		if (Tree.LocationTypeNameP(item))
		{
			tempcard.ltype = Tree.LocationTypeFromName(item);
		} else
		{
			static Card::LocationType ULtypes[] = {
				Card::UL1, Card::UL2, Card::UL3, Card::UL4,
				Card::UL5, Card::UL6, Card::UL7, Card::UL8,
				Card::UL9, Card::UL10};
			static int nULtypes = sizeof(ULtypes) / sizeof(ULtypes[0]);
			int i;
			for (i = 0; i < nULtypes; i++)
			{
				if (strcmp(Tree.LocationTypeName(ULtypes[i]),"Unknown") == 0) break;
			}
			if (i >= nULtypes)
			{
			}
			Tree.AddLocationTypeName(ULtypes[i],item);
			tempcard.ltype = ULtypes[i];
		}
		delete item;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card location detail]",(char *)NULL);
			infp2.close();
			return false;
		}
		tempcard.locdetail = item;
		item = GetNextDelimitedToken(infp2,true);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card category]",(char *)NULL);
			infp2.close();
			return false;
		}
		if (Tree.CatCodeNameP(item))
		{
			tempcard.catcode = Tree.CatCodeFromName(item);
		} else
		{
			unsigned short i;
			for (i = 0; i < 256; i++)
			{
				if (strlen(Tree.Category(i)) == 0) break;
			}
			if (i >= 256)
			{
			}
			Tree.AddCategory(i,item);
			tempcard.catcode = i;
		}
		delete item;
		CardRecord crec(&tempcard);
		Record rawrec = crec;
		Tree.InsertId(key,&rawrec);
		delete tempcard.author;
		delete tempcard.title;
		delete tempcard.publisher;
		delete tempcard.city;
		delete tempcard.description;
		delete tempcard.locdetail;
	}
	infp2.close();
	return true;
}

bool TkvBTree::tkvBTreeImportDelimitedTitleRecs(Tcl_Interp *interp,int numKeys,const char *inf1,const char *inf2)
{
	int ikey;
	const char *item;
	Key key;
	const char *fn = inf1;
	ifstream infp2(inf1,ios::in);
	if (!infp2)
	{
		infp2.open(inf2,ios::in);
		fn = inf2;
		if (!infp2)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Could not open ",inf1," or ",inf2,(char *)NULL);
			return false;
		}
	}
	for (ikey = 0; ikey < numKeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card key]",(char *)NULL);
			infp2.close();
			return false;
		}
		strncpy(key,item,sizeof(key)-1);
		key[sizeof(key)-1] = 0;
		delete item;
		item = GetNextDelimitedToken(infp2,true);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		const ListRecord *lrec = PeelDelimitedTokenList(item);
		if (lrec == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [syntax error in id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		delete item;
		Record rawrec = *lrec;
		delete lrec;
		Tree.InsertTitle(key,&rawrec);
	}
	infp2.close();
	return true;
}

bool TkvBTree::tkvBTreeImportDelimitedAuthorRecs(Tcl_Interp *interp,int numKeys,const char *inf1,const char *inf2)
{
	int ikey;
	const char *item;
	Key key;
	const char *fn = inf1;
	ifstream infp2(inf1,ios::in);
	if (!infp2)
	{
		infp2.open(inf2,ios::in);
		fn = inf2;
		if (!infp2)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Could not open ",inf1," or ",inf2,(char *)NULL);
			return false;
		}
	}
	for (ikey = 0; ikey < numKeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card key]",(char *)NULL);
			infp2.close();
			return false;
		}
		strncpy(key,item,sizeof(key)-1);
		key[sizeof(key)-1] = 0;
		delete item;
		item = GetNextDelimitedToken(infp2,true);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		const ListRecord *lrec = PeelDelimitedTokenList(item);
		if (lrec == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [syntax error in id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		delete item;
		Record rawrec = *lrec;
		delete lrec;
		Tree.InsertAuthor(key,&rawrec);
	}
	infp2.close();
	return true;
}

bool TkvBTree::tkvBTreeImportDelimitedSubjRecs(Tcl_Interp *interp,int numKeys,const char *inf1,const char *inf2)
{
	int ikey;
	const char *item;
	Key key;
	const char *fn = inf1;
	ifstream infp2(inf1,ios::in);
	if (!infp2)
	{
		infp2.open(inf2,ios::in);
		fn = inf2;
		if (!infp2)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Could not open ",inf1," or ",inf2,(char *)NULL);
			return false;
		}
	}
	for (ikey = 0; ikey < numKeys; ikey++)
	{
		while (Tcl_DoOneEvent(TCL_DONT_WAIT)) ;
		item = GetNextDelimitedToken(infp2,false);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing Card key]",(char *)NULL);
			infp2.close();
			return false;
		}
		strncpy(key,item,sizeof(key)-1);
		key[sizeof(key)-1] = 0;
		delete item;
		item = GetNextDelimitedToken(infp2,true);
		if (item == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [missing id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		const ListRecord *lrec = PeelDelimitedTokenList(item);
		if (lrec == NULL)
		{
			Tcl_ResetResult(interp);
			Tcl_AppendResult(interp,"Format error on file ",fn," [syntax error in id list]",(char *)NULL);
			infp2.close();
			return false;
		}
		delete item;
		Record rawrec = *lrec;
		delete lrec;
		Tree.InsertSubj(key,&rawrec);
	}
	infp2.close();
	return true;
}


const char *TkvBTree::GetNextDelimitedToken(istream &stream,bool eolIsDelimeter) const
{
	string result;
	char theDel = del;
	if (eolIsDelimeter) theDel = '\n';

	if (qe)	// might be quoted
	{
		if (stream.peek() == quoesc)
		{
			// is quoted.
			stream.get();	// gobble quote mark
			while (stream)
			{
				if (stream.peek() == quoesc) // close quote?
				{
					stream.get();	// gobble quote mark
					if (!stream) return NULL;
					if (stream.peek() == quoesc) // quote-quote?
					{
						result += (char)stream.get(); // yep
					} else
					{
						break;
					}
				} else
				{
					result += (char)stream.get();
				}
			}
			if (!stream) return NULL;
			if (stream.peek() != theDel) return NULL;
		}			
	} else	// might be escaped
	{
		while (stream)
		{
			if (stream.peek() == theDel) break;
			else if (stream.peek() == quoesc) // escaped??
			{
				// yep.  Gobble escape character and include
				// next character.
				stream.get();
				if (!stream) return NULL;
				result += (char)stream.get();
			} else
			{
				result += (char)stream.get();
			}
		}
		if (!stream) return NULL;
	}
	stream.get(); // gobble delimiter.
	char *toreturn = new char[result.size()+1];
	strcpy(toreturn,result.c_str());
	return toreturn;
}

const char *TkvBTree::GetNextDelimitedToken(const char **str,bool eosIsDelimeter) const
{
	string result;
	char theDel = del;
	if (eosIsDelimeter) theDel = '\0';
	const char *p_str = *str;
	
	if (qe)	// might be quoted
	{
		if (*p_str == quoesc)
		{
			// is quoted.
			p_str++;	// gobble quote mark
			while (*p_str != '\0')
			{
				if (*p_str == quoesc) // close quote?
				{
					p_str++;	// gobble quote mark
					if (*p_str == quoesc) // quote-quote?
					{
						result += *p_str++; // yep
					} else
					{
						break;
					}
				} else
				{
					result += *p_str++;
				}
			}
			if (p_str[-1] != quoesc) return NULL;
			if (*p_str != theDel) return NULL;
		}				
	} else	// might be escaped
	{
		while (*p_str != '\0')
		{
			if (*p_str == theDel) break;
			else if (*p_str == quoesc) // escaped??
			{
				// yep.  Gobble escape character and include
				// next character.
				p_str++;
				if (*p_str == '\0') return NULL;
				result += *p_str++;
			} else
			{
				result += *p_str++;
			}
		}
	}
	if (*p_str == theDel)
	{
		if (*p_str != '\0') p_str++; // gobble delimiter.
		*str = p_str;		     // update string position;
		char *toreturn = new char[result.size()+1];
		strcpy(toreturn,result.c_str());
		return toreturn;
	} else
	{
		return NULL;
	}
}

const ListRecord *TkvBTree::PeelDelimitedTokenList(const char *idlist) const
{
	ListRecord *result = new ListRecord(0,NULL);
	const char *ptr = idlist,*iptr;
	while ((iptr = GetNextDelimitedToken(&ptr,false)) != NULL)
	{
		result->AddElement(iptr);
		delete iptr;
	}
	iptr = GetNextDelimitedToken(&ptr,true);
	if (iptr == NULL)
	{
		delete result;
		result =  NULL;
	} else
	{
		result->AddElement(iptr);
		delete iptr;
	}		
	return result;
}

//@}



