/* -*- Mode: C -*- */
/* Card.h - Structure for representing catalog cards
 * Created by Robert Heller on Fri Dec  6 20:11:05 1991
 * Updated for Version 2.0 on Sat Apr 26 12:59:50 1997
 *
 * ------------------------------------------------------------------
 * Home Libarian by Deepwoods Software
 * Common Header Files
 * ------------------------------------------------------------------
 * Modification History:
 * $Log: Card.h,v $
 * Revision 2.15  1998/05/16 23:43:28  heller
 * Add indexing to the documentation.
 *
 * Revision 2.14  1998/04/21 15:12:18  heller
 * Update copyright notice.
 *
 * Revision 2.13  1997/09/06 23:25:08  heller
 * Change chapter name.
 *
 * Revision 2.12  1997/08/10 16:41:29  heller
 * Change default Card constructor to not fill a default card with the string
 * "Unknown" all over the place. Instead, the empty string ("") is used.
 *
 * Revision 2.11  1997/07/23 23:20:32  heller
 * Fix unterminated comment
 *
 * Revision 2.10  1997/07/20 21:13:11  heller
 * Fun with \ref
 *
 * Revision 2.9  1997/07/20 20:34:53  heller
 * Squigle fun...
 *
 * Revision 2.8  1997/07/20 19:50:20  heller
 * Spelling errors
 *
 * Revision 2.7  1997/07/15 13:56:46  heller
 * Added in class documentation.
 *
 * Revision 2.6  1997/07/06 21:44:54  heller
 * Misc. changes
 *
 * Revision 2.5  1997/07/01 16:40:04  heller
 * Remove "const" from static check functions
 *
 * Revision 2.4  1997/07/01 16:28:14  heller
 * Make the check functions static
 *
 * Revision 2.3  1997/07/01 15:58:57  heller
 * Add in check functions for CardType and LocationType
 *
 * Revision 2.2  1997/06/29 19:14:59  heller
 * Fixes for portablity with CW
 *
 * Revision 2.1  1997/04/26 22:11:49  heller
 * Add in Card::NumCardTypes and Card::NumLocationTypes
 *
 * Revision 2.0  1997/04/26 19:06:49  heller
 * *** empty log message ***
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 * 
 *    Home Librarian Database -- a program for maintaining a database
 *                               for a home library
 *    Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
 *			51 Locke Hill Road
 *			Wendell, MA 01379-9728
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

//@ChapterName:Catalog Card Class

#ifndef _CARD_
#define _CARD_
#include <common.h>		// common defs

/*@ManDoc:
  \typeout{Generated from: $Id: Card.h,v 2.15 1998/05/16 23:43:28 heller Rel $}
  \index{Card!C++ Interface|(}
  \index{Card!C++ Interface!Card class|bold}
  Card class.  This class is really just a structure pointing to
  various pieces of data describing an item in the library.

  This class is used to represent an image of a library catalog card.
  The various public data members contain much the same information that is
  typical of the cards used by libraries to catalog their collections.
 */
class Card {
public:
	/*@ManDoc: \index{Card!C++ Interface!Types of cards|(}
	  Types of cards.  This enumerated type defines the 22 types
	  of items.  12 of these types are ``standard'' and 10 are user
	  definable. */
	enum CardType {
		//@ManDoc: Your basic Book.
		Book = 'B', 
		//@ManDoc: Your basic magazine or journal.
		Magazine = 'M', 
		//@ManDoc: Audio CDs
		CD = 'D',
		//@ManDoc: Audio Cassettes
		AudioCassette = 'C',
		//@ManDoc: Vinyl Disks (remember?).
		Album = 'A',
		//@ManDoc: Laser (video) Disks
		LaserDisk = 'L', 
		//@ManDoc: Your ``standard'' VHS Video tapes.
		VHSVideo = 'V',
		//@ManDoc: Old Sony Beta Videos
		BetaVideo = 'S',
		//@ManDoc: 8mm Video Tapes
		EightMM = '8',
		//@ManDoc: Old 8-track Cartridges
		EightTrack = 'E',
		//@ManDoc: 4mm Digital Audio Tapes
		DAT = '4',
		//@ManDoc: Otherwise not classified
		Other = 'O' ,
		//@ManDoc: User Defined Type 1
		UC1 = 'a',
		//@ManDoc: User Defined Type 2
		UC2 = 'b',
		//@ManDoc: User Defined Type 3
		UC3 = 'c',
		//@ManDoc: User Defined Type 4
		UC4 = 'd',
		//@ManDoc: User Defined Type 5
		UC5 = 'e',
		//@ManDoc: User Defined Type 6
		UC6 = 'f', 
		//@ManDoc: User Defined Type 7
		UC7 = 'g',
		//@ManDoc: User Defined Type 8
		UC8 = 'h',
		//@ManDoc: User Defined Type 9
		UC9 = 'i',
		//@ManDoc: User Defined Type 10 \index{Card!C++ Interface!Types of cards|)}
		UC10 = 'j' 
	};
	/*@ManDoc: \index{Card!C++ Interface!Types of Locations|(}
	  Types of Locations.  This enumerated type defines the 16
	  types of ``locations''.  6 of these types are ``standard'' and 10
	  are user definable. */
	enum LocationType {
		//@ManDoc: The item is on a shelf.
		OnShelf = 'S', 
		//@ManDoc: The item is on loan.
		OnLoan = 'L', 
		//@ManDoc: The item is on order.
		OnOrder = 'O',
		//@ManDoc: The item was destroyed.
		Destroyed = 'D', 
		//@ManDoc: The item is in storage.
		InStorage = 's', 
		//@ManDoc: The item's location is unknown.
		Unknown = 'U',
		//@ManDoc: User Defined Location 1
		UL1 = 'a', 
		//@ManDoc: User Defined Location 2
		UL2 = 'b', 
		//@ManDoc: User Defined Location 3
		UL3 = 'c', 
		//@ManDoc: User Defined Location 4
		UL4 = 'd', 
		//@ManDoc: User Defined Location 5
		UL5 = 'e', 
		//@ManDoc: User Defined Location 6
		UL6 = 'f', 
		//@ManDoc: User Defined Location 7
		UL7 = 'g', 
		//@ManDoc: User Defined Location 8
		UL8 = 'h', 
		//@ManDoc: User Defined Location 9
		UL9 = 'i', 
		//@ManDoc: User Defined Location 10  \index{Card!C++ Interface!Types of Locations|)}
		UL10 = 'j' 
	};
	//@Man: General constants.
	/*@Doc: \index{Card!C++ Interface!General constants|(}
 	  These 4 constants are used by the code in vBTree class
	  (see Chapter\space\ref{vBTree}) for the vectors that hold the textual
	  names for the item types and locations and for indexing into these
	  vectors. 
	 */
	//@{
	enum {
		//@ManDoc: The number of Card Types.
		//@See: CardType
		NumCardTypes = 22, 
		//@ManDoc: The index of the first user defined type.
		//@See: UC1
		IndexUC1 = 12,
		//@ManDoc: The number of Location types.
		//@See: LocationType
		NumLocationTypes = 16, 
		//@ManDoc: The index of the first user defined location type.
		//@See: UL1 \index{Card!C++ Interface!General constants|)}
		IndexUL1 = 6
	};
	//@}
	/*@ManDoc: \index{Card!C++ Interface!type}
	  Type of item (book, magazine, etc.)
	 */
	//@See: CardType
	CardType type;
	/*@ManDoc: \index{Card!C++ Interface!author}
	  Who wrote it.  Could also be the performer for recordings.
	  Or the editor if it is an edited work.
	 */
	const char* author;
	//@ManDoc: \index{Card!C++ Interface!title} What it is called.
	const char* title;
	//@ManDoc: \index{Card!C++ Interface!publisher} Who published it.
	const char* publisher;
	//@ManDoc: \index{Card!C++ Interface!city} Where it was published.
	const char* city;
	/*@ManDoc: \index{Card!C++ Interface!description}
	  Detailed description of the item.  This is an unstructured
	  field.  Any text can be placed here.
	 */
	const char* description;
	//@ManDoc: \index{Card!C++ Interface!vol} The volume number, if any.
	int vol;
	//@ManDoc: \index{Card!C++ Interface!year} The Year it was published.
	int year;
	//@ManDoc: \index{Card!C++ Interface!ltype} The Location type (On a shelf, out on loan, etc.).
	LocationType ltype;
	//@ManDoc: \index{Card!C++ Interface!locdetail} Location detail.  Which shelf, where it is stored, Who borrowed it, etc.
	//@See: LocationType
	const char* locdetail;
	/*@ManDoc: \index{Card!C++ Interface!catcode}
	  A Category code.  This is a code from 0 to 255 that is the
	  index into an array of user defined categories defined in the
	  vBTree class (see Chapter\space\ref{vBTree}).
	 */
	unsigned char catcode;
	/*@ManDoc:  \index{Card!C++ Interface!Constructor (base)}
	  The constructor.  All of the arguments have default values.*/
	      Card(
	           //@ManDoc: type type for the new item.
	           //@See: CardType
		   CardType type = Other,
		   //@ManDoc: The author for the new item.
		   const char* author = "",
		   //@ManDoc: The title of the new item.
		   const char* title = "",
		   //@ManDoc: The publisher of the new item.
		   const char* publisher = "",
		   //@ManDoc: Where the new item was published.
	           const char* city  = "",
		   //@ManDoc: The description of the new item.
		   const char* description = "",
		   //@ManDoc: The volume number of the new item.
		   int vol = 0,
		   //@ManDoc: The publication year of the new item.
		   int year = 0,
		   //@ManDoc: The location type of the new item.
		   //@See: LocationType
		   LocationType ltype = Unknown, 
		   //@ManDoc: The location detail of the new item.
		   const char* locdetail = "",
		   //@ManDoc: The category code of the new item.
		   unsigned char catcode = 0
		   )
			{
				Card::type = type;
				Card::author = author;
				Card::title = title;
				Card::publisher = publisher;
				Card::city = city;
				Card::description = description;
				Card::vol = vol;
				Card::year = year;
				Card::ltype = ltype;
				Card::locdetail = locdetail;
				Card::catcode = catcode;
			}
	     /*@ManDoc: \index{Card!C++ Interface!Constructor (from another card reference)}
	       Construct a card from a reference to an existing card.
	      */
	     Card(
	     	  //@ManDoc: The source card.
		  const Card& card
		 )
		{
			type = card.type; author = card.author; 
			title = card.title; publisher = card.publisher;
			city = card.city; description = card.description;
			vol = card.vol; year = card.year;
			ltype = card.ltype; locdetail = card.locdetail;
			catcode = card.catcode;			
		}
 	     /*@ManDoc: \index{Card!C++ Interface!Constructor (from another card pointer)}
	       Construct a card from a pointer to an existing card.
	      */
	     Card(
	        //@ManDoc: The source card.
		const Card* card
		)
		{
			type = card->type; author = card->author; 
			title = card->title; publisher = card->publisher;
			city = card->city; description = card->description;
			vol = card->vol; year = card->year;
			ltype = card->ltype; locdetail = card->locdetail;
			catcode = card->catcode;
			
		}
	/*@ManDoc: \index{Card!C++ Interface!"=@"= operator}
	  Card assignment operator.
	 */
	Card& operator = (
		//@ManDoc: The source card.
		const Card& card
		)
		{
			type = card.type; author = card.author; 
			title = card.title; publisher = card.publisher;
			city = card.city; description = card.description;
			vol = card.vol; year = card.year;
			ltype = card.ltype; locdetail = card.locdetail;
			catcode = card.catcode;
			return *this;
		}
	/*@ManDoc: \index{Card!C++ Interface!IsCardType}
	  Predicate to test a character to see if it is an allowed
	  Card type.
	*/
	static Boolean IsCardType (char tc);
	/*@ManDoc: \index{Card!C++ Interface!IsLocationType}
	   Predicate to test a character to see if it is an allowed 
	   Location type.\index{Card!C++ Interface|)}
	 */
	static Boolean IsLocationType (char tc);
};

#endif
