#* 
#* ------------------------------------------------------------------
#* Home Libarian 2.0 by Deepwoods Software
#* ------------------------------------------------------------------
#* UnixPrinter.tcl - Unix Printer (PostScript) device
#* Created by Robert Heller on Sat Jan 17 15:25:40 1998
#* ------------------------------------------------------------------
#* Modification History: 
#* $Log: UnixPrinter.tcl,v $
#* Revision 1.4  1998/05/17 21:52:04  heller
#* Add indexing
#*
#* Revision 1.3  1998/04/14 18:01:06  heller
#* Put in code to reset the page number.
#*
#* Revision 1.2  1998/02/08 19:16:23  heller
#* Complete the package
#*
#* Revision 1.1  1998/01/29 00:07:38  heller
#* Initial revision
#*
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Home Librarian Database -- a program for maintaining a database
#*                                for a home library
#*     Copyright (C) 1991-1997  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 


#@Chapter: UnixPrinter.tcl - Unix Printer (PostScript) device
#@Label: UnixPrinter.tcl
#$Id: UnixPrinter.tcl,v 1.4 1998/05/17 21:52:04 heller Rel $

# This is the UNIX ``printer'' implementation.  Under UNIX, this is 
# implemented by generating PostScript code that is feed either to a file or
# to a pipe to a printer command, typically ``lpr''.
#

global Printer
# This variable contains all of the ``printer'''s state info.
# It is an array containing the elements:
#
# <FileOrPrinter>	Contains either the string ``Printer'' or ``File''.
# <PrinterCommand>	Contains the piped command to send data to the
#			printer spool.  Typically ``|lpr -''.
# <PrintFileName>	Contains the name of the file to write to.
# <fp>			An open file handle or the null string.
# <Copies>		The number of copies to print.
# <FirstPage>		The first page to print.
# <LastPage>		The last page to print.
# <CurrentPage>		The page currently being printed.
# <DocOpened>		A flag to indicate if the document is opened.
# <PageOpened>		A flag to indicate if the page is opened.
# <TmpFileIndex>	An every increasing index for use as part of a temp
#			file name.
# [index] Printer!global variable

set Printer(FileOrPrinter) Printer
set Printer(PrinterCommand) {|lpr -}
set Printer(PrintFileName) {dummy.ps}
set Printer(fp) {}
set Printer(Copies) 1
set Printer(FirstPage) 1
set Printer(LastPage) 99999
set Printer(CurrentPage) 0
set Printer(DocOpened) 0
set Printer(PageOpened) 0
set Printer(TmpFileIndex) 0

global PrID
# This variable contains identification info for the Postscript file
# [index] PrID!global variable

set PrID {$Id: UnixPrinter.tcl,v 1.4 1998/05/17 21:52:04 heller Rel $}

proc printer {function args} {
# Unix Printer Dispatching Function.  This function dispatches to the internal
# functions to that implement the UNIX printer functionality.
# <in> function -- this is the function to be performed.
# <in> args     -- any additional parameters that are used.
#
# Functions supported:
# <open> -- open a connection to the printer device.
#	    Takes three arguments: fOrp pCmd pFile: fOrp should be ``File'' or
#           ``Printer'', pCmd should be the pipe command to send to the print
#	    spool, and pFile should be the name of a file to write to.
#           If pFile is ``File'' only pFile needs to be legitimate file name,
#	    and if pFile is ``Printer'' only pCmd needs to be legitimate pipe 
#	    command.
# <close> -- closes the connection to the printer device.
# <styledialog> -- pops up the printer style dialog (a noop under UNIX).
# <jobdialog> -- pops up the printer job dialog (a noop under UNIX).
# <opendoc> -- opens the document for printing.
# <closedoc> -- closes the print document.
# <openpage> -- opens a new page to print.
# <closepage> -- closes the currently printing page.
# <getattr> -- gets the current print attributes.  Takes one argument, which
#              should be one of the strings ``copies'', ``firstpage'', or 
#	       ``lastpage''.  The corresponding values from the printer
#	       state variable are returned.
# <getunitsperinch> -- returns the number of printer device units per inch.
#		For PostScript this is 72.0 (points).
# <getpagebbox> -- returns the printer's page bounding box as a list 
#		{top left bottom right}.  For PostScript this is {792 0 0 612}.
#		PostScript's graphic origin is at the lower left and these
#		numbers describe a page that is 11 inches high and 8.5 inches
#		wide.
# <test> -- print a test page.
# <setfont> -- set the current font.  Takes two arguments, the name of the
#              font and the size of the font (in printer units).
# <setcolor> -- set the current color.  Takes three arguments, the red, green,
#		and blue values as floating point numbers between 0.0 and 1.0,
#		with 0.0 being no color and 1.0 being maximum color brightness.
#		Thus setcolor with the argument list {0.0 0.0 0.0} would
#		print in black and with the argument list {1.0 1.0 1.0} would
#		display in white.
# <moveto> -- sets the current position.  Takes two arguments, x and y, in 
#		printer units.
# <drawtextat> -- draws text at a specified location. Takes three arguments,
#		x, y, and text.  x and y are in printer units and text is a
#		string.  The currently set font and color are used.
# <drawtext> -- draws text at the current position. Takes one argument, text.
#		text is a string.  The currently set font and color are used.
# <drawtextinbbox> -- draws text is a specified bounding box.  The text is
#		word wrapped to fit and clipped to the bounding box.  Takes
#		5 arguments: x1, y1, x2, y2, and text.  x1, y1, x2, y2 define 
#		the bounding box and text is the text string to print. The 
#		currently set font and color are used.
# <drawline> -- draws a line in the current color. Takes 4 or 5 arguments:
#		x1, y1, x2, y2, and optionally linewidth.  linewidth defaults 
#		to 1.
# <drawrectangle> -- draws an open rectangle in the current color.  Takes 4 or 
#		5 arguments: x1, y1, x2, y2, and optionally thickness. thickness
#		defaults to 1.
# <drawfilledrectangle> -- draws a filled rectangle in the current color.
#		Takes 4 arguments: x1, y1, x2, and y2.
# <drawoval> -- draws an open oval in the current color. Takes 4 or 5 
#		arguments: x1, y1, x2, y2, and optionally thickness. thickness
#		defaults to 1.
# <drawfilledoval> -- draws a filled oval in the current color. Takes 4 
#		arguments: x1, y1, x2, and y2.
# <drawimage> -- draws a bitmap image in the current color.  Takes 5 
#		arguments: x1, y1, x2, y2, and bitmap.
# <maketempfilename> -- create a unique file name for use as a temporary file.
# <spoolplaintextfile> -- spool a plain text file to the printer.  Takes one
#		argument, the name of the file.
#
# [index] printer!procedure

  switch -exact -- "$function" {
    open {eval PrOpen $args}
    close {eval PrClose $args}
    styledialog {eval PrStyleDialog $args}
    jobdialog {eval PrJobDialog $args}
    opendoc {eval PrOpenDoc $args}
    closedoc {eval PrCloseDoc $args}
    openpage {eval PrOpenPage $args}
    closepage {eval PrClosePage $args}
    getattr {eval PrGetAttributes $args}
    getunitsperinch {eval PrGetUnitsPerInch $args}
    getpagebbox {eval PrGetPageBBox $args}
    test {eval PrTest $args}
    setfont {eval PrSetFont $args}
    setcolor {eval PrSetColor $args}
    moveto {eval PrMoveTo $args}
    drawtextat {eval PrDrawTextAt $args}
    drawtext {eval PrDrawText $args}
    drawtextinbbox {eval PrDrawTextInBBox $args}
    drawline {eval PrDrawLine $args}
    drawrectangle {eval PrDrawRectangle $args}
    drawfilledrectangle {eval PrDrawFilledRectangle $args}
    drawoval {eval PrDrawOval $args}
    drawfilledoval {eval PrDrawFilledOval $args}
    drawimage {eval PrDrawImage $args}
    maketempfilename {eval PrMakeTempFileName $args}
    spoolplaintextfile {eval PrSpoolPlainTextFile $args}
    default {
	error "printer: unknown function $function"
    }
  }
}

proc PSQuote {string} {
# Quotify a string for PostScript
# [index] PSQuote!procedure

  if {[regsub -all {[\\()%]} "$string" {\\&} newstring] > 0} {
    return "$newstring"
  } else {
    return "$string"
  }
}


proc PrOpen {fOrp pCmd pFile} {
# Open a connection to the ``printer''
# Arguments:
# <in> fOrp - print destination.  Should be wither ``Printer'' or ``File''.
# <in> pCmd - printer command pipeline -- only used if fOrp is ``Printer''.
# <in> pFile - printer filename -- only used if fOrp is ``File''.
# [index] PrOpen!procedure

  global Printer
  if {[string length "$Printer(fp)"] > 0} {
    error "Printer already open: $Printer(FileOrPrinter): \{$Printer(PrinterCommand)\} : \{$Printer(PrintFileName)\}"
  }
  set Printer(FileOrPrinter) "$fOrp"
  set Printer(PrinterCommand) "$pCmd"
  set Printer(PrintFileName) "$pFile"
  switch -exact -- "$fOrp" {
    Printer {
	if {[catch [list open "|$Printer(PrinterCommand)" w] fp]} {
	  error "printer cannot open printer command: $Printer(PrinterCommand): $fp"
	}
	set Printer(fp) "$fp"
    }
    File {
	if {[catch [list open "$Printer(PrintFileName)" w] fp]} {
	  error "printer cannot file: $Printer(PrintFileName): $fp"
	}
	set Printer(fp) "$fp"
    }
    default {
	error "printer: undefined print destination: $Printer(FileOrPrinter)"
    }
  }
  set Printer(DocOpened) 0
  set Printer(PageOpened) 0
}

proc PrClose {} {
# Close connection with the ``printer''.
# [index] PrClose!procedure

  global Printer
  if {[string length "$Printer(fp)"] == 0} {
    error "Printer not open"
  }
  if {$Printer(DocOpened)} {
    PrCloseDoc
  }
  if {[catch [list close $Printer(fp)] mess]} {
    set Printer(fp) {}
    error "printer: error closing printer: $mess"
  }
  set Printer(fp) {}
}

proc PrStyleDialog {} {
# Popup ``Style'' dialog -- a noop for UNIX.
# [index] PrStyleDialog!procedure

  global Printer
  if {[string length "$Printer(fp)"] == 0} {
    error "Printer not open"
  }
}

proc PrJobDialog {} {
# Popup ``Job'' dialog -- a noop for UNIX.
# [index] PrJobDialog!procedure

  global Printer
  if {[string length "$Printer(fp)"] == 0} {
    error "Printer not open"
  }
}

proc PrOpenDoc {} {
# Open a print document.  Document level initialization.
# [index] PrOpenDoc!procedure

  global Printer
  if {[string length "$Printer(fp)"] == 0} {
    error "Printer not open"
  }
  if {$Printer(DocOpened)} {return}
  global PrID
  puts $Printer(fp) "%!PS-Adobe-2.0"
  puts $Printer(fp) "%%Title: Home Librarian 2.0 document"
  puts $Printer(fp) "%%Creator: [PSQuote $PrID]"
  puts $Printer(fp) "%%EndComments"
  puts $Printer(fp) "%%BeginProlog"
  puts $Printer(fp) "/UnixPrinterDict 100 dict def"
  puts $Printer(fp) "/UnixPrinterLocals 100 dict def"
  puts $Printer(fp) "UnixPrinterDict begin"
  puts $Printer(fp) "/inch {72 mul} def"
  puts $Printer(fp) "/TopOfPage 8 inch def"
  puts $Printer(fp) "/WidthOfPage 6 inch def"
  puts $Printer(fp) "/wordbreak ( ) def"
  puts $Printer(fp) "/nl (\\n) def"
  puts $Printer(fp) "/BreakIntoLines"
  puts $Printer(fp) " { UnixPrinterLocals begin"
  puts $Printer(fp) "   /proc exch def"
  puts $Printer(fp) "   /linewidth exch def"
  puts $Printer(fp) "   /textstring exch def"
  puts $Printer(fp) "   /breakwidth wordbreak stringwidth pop def"
  puts $Printer(fp) "   /curwidth 0 def"
  puts $Printer(fp) "   /lastwordbreak 0 def"
  puts $Printer(fp) "   /startchar 0 def"
  puts $Printer(fp) "   /restoftext textstring def"
  puts $Printer(fp) "   { restoftext wordbreak search"
  puts $Printer(fp) "     {/nextword exch def pop"
  puts $Printer(fp) "      /restoftext exch def"
  puts $Printer(fp) "      /wordwidth nextword stringwidth pop def"
  puts $Printer(fp) "      curwidth wordwidth add linewidth gt"
  puts $Printer(fp) "       { textstring startchar"
  puts $Printer(fp) "           lastwordbreak startchar sub "
  puts $Printer(fp) "           getinterval proc"
  puts $Printer(fp) "         /startchar lastwordbreak def"
  puts $Printer(fp) "         /curwidth wordwidth breakwidth add def }"
  puts $Printer(fp) "       { /curwidth curwidth wordwidth add"
  puts $Printer(fp) "           breakwidth add def"
  puts $Printer(fp) "       } ifelse"
  puts $Printer(fp) "     /lastwordbreak lastwordbreak"
  puts $Printer(fp) "      nextword length add 1 add def"
  puts $Printer(fp) "     }"
  puts $Printer(fp) "     { pop exit }"
  puts $Printer(fp) "     ifelse"
  puts $Printer(fp) "   } loop"
  puts $Printer(fp) "   /lastchar textstring length def"
  puts $Printer(fp) "   textstring startchar lastchar startchar  sub"
  puts $Printer(fp) "     getinterval proc"
  puts $Printer(fp) "   end"
  puts $Printer(fp) "   } def"
  puts $Printer(fp) "/TextInBBox"
  puts $Printer(fp) " { UnixPrinterLocals begin"
  puts $Printer(fp) "   /textblock exch def"
  puts $Printer(fp) "   /y2 exch def"
  puts $Printer(fp) "   /x2 exch def"
  puts $Printer(fp) "   /ypos exch def"
  puts $Printer(fp) "   /x1 exch def"
  puts $Printer(fp) "   /linewidth x2 x1 sub def"
  puts $Printer(fp) "   gsave newpath 0 0 moveto textblock false charpath"
  puts $Printer(fp) "         flattenpath pathbbox exch pop exch sub exch pop"
  puts $Printer(fp) "   grestore 1.1 mul /yheight exch def"
  puts $Printer(fp) "   gsave"
  puts $Printer(fp) "     newpath x1 ypos moveto x1 y2 lineto"
  puts $Printer(fp) "    x2 y2 lineto x2 ypos lineto closepath clip"
  puts $Printer(fp) "   /ypos ypos yheight sub def"
  puts $Printer(fp) "   /lastlinebreak 0 def"
  puts $Printer(fp) "   /startofline 0 def"
  puts $Printer(fp) "   /restofblock textblock def"
  puts $Printer(fp) "     { restofblock nl search"
  puts $Printer(fp) "       {/nextline exch def pop"
  puts $Printer(fp) "        /restofblock exch def"
  puts $Printer(fp) "        nextline linewidth"
  puts $Printer(fp) "          {x1 ypos moveto show /ypos ypos yheight sub def}"
  puts $Printer(fp) "          BreakIntoLines}"
  puts $Printer(fp) "       {pop exit} ifelse"
  puts $Printer(fp) "     } loop"
  puts $Printer(fp) "     restofblock linewidth"
  puts $Printer(fp) "     {x1 ypos moveto show /ypos ypos yheight sub def}"
  puts $Printer(fp) "     BreakIntoLines"
  puts $Printer(fp) "    grestore"
  puts $Printer(fp) "   end"
  puts $Printer(fp) "   } def"
  puts $Printer(fp) " /Oval { UnixPrinterLocals begin"
  puts $Printer(fp) "   /thickness exch def"
  puts $Printer(fp) "   /y2 exch def"
  puts $Printer(fp) "   /x2 exch def"
  puts $Printer(fp) "   /y1 exch def"
  puts $Printer(fp) "   /x1 exch def"
  puts $Printer(fp) "   /xsize x2 x1 sub abs def /ysize y2 y1 sub abs def"
  puts $Printer(fp) "   /scalethick xsize ysize gt {thickness xsize div} {thickness ysize div} ifelse def"
  puts $Printer(fp) "   gsave"
  puts $Printer(fp) "   x2 x1 add 2 div y2 y1 add 2 div"
  puts $Printer(fp) "   translate xsize ysize scale"
  puts $Printer(fp) "   scalethick pstack setlinewidth"
  puts $Printer(fp) "   0 0 .5 0 360 arc stroke"
  puts $Printer(fp) "   grestore"
  puts $Printer(fp) "   end } def"
  puts $Printer(fp) " /FillOval { UnixPrinterLocals begin"
  puts $Printer(fp) "   /y2 exch def"
  puts $Printer(fp) "   /x2 exch def"
  puts $Printer(fp) "   /y1 exch def"
  puts $Printer(fp) "   /x1 exch def"
  puts $Printer(fp) "   /xsize x2 x1 sub abs def /ysize y2 y1 sub abs def"
  puts $Printer(fp) "   gsave"
  puts $Printer(fp) "   x2 x1 add 2 div y2 y1 add 2 div"
  puts $Printer(fp) "   translate xsize ysize scale"
  puts $Printer(fp) "   0 0 .5 0 360 arc fill"
  puts $Printer(fp) "   grestore"
  puts $Printer(fp) "   end } def"
  puts $Printer(fp) "%%EndProlog"
  set Printer(PageOpened) 0
  set Printer(DocOpened) 1
  set Printer(CurrentPage) 0
}

proc PrCloseDoc {} {
# Document close function -- perform document level run-down.
# [index] PrCloseDoc!procedure

  global Printer
  if {[string length "$Printer(fp)"] == 0} {
    error "Printer not open"
  }
  if {$Printer(DocOpened)} {
    if {$Printer(PageOpened)} {
      PrClosePage
    }
    puts $Printer(fp) "%%Trailer"
    puts $Printer(fp) "end"
    puts $Printer(fp) "%%EOF"    
    set Printer(DocOpened) 0
  }
}

proc PrOpenPage {} {
# Page open function -- open a new page and do page level initialization.
# [index] PrOpenPage!procedure

  global Printer
  if {$Printer(PageOpened)} {
    return
  } else {
    set Printer(PageOpened) 1
  }
  incr Printer(CurrentPage)
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "%%Page: $Printer(CurrentPage)"
}

proc PrClosePage {} {
# Page close function -- close the current page and do page level run-down.
# [index] PrClosePage!procedure

  global Printer
  if {$Printer(PageOpened)} {
    set Printer(PageOpened) 0
  } else {
    return
  }
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "showpage"
  flush $Printer(fp)
}

proc PrGetAttributes {attr} {
# Fetch printer attributes.
# [index] PrGetAttributes!procedure

  global Printer
  switch -exact -- "$attr" {
    copies {return $Printer(Copies)}
    firstpage {return $Printer(FirstPage)}
    lastpage {return $Printer(LastPage)}
    default {error "printer: bad attribute: $attr"}
  }
}

proc PrTest {} {
# Print test page.
# [index] PrTest!procedure

  PrDrawTextAt 100 100 {Test page}
}

proc PrGetUnitsPerInch {} {
# Return printer units.
# [index] PrGetUnitsPerInch!procedure

  return 72
}

# 72*11 = 792
# 72*8.5 = 612

proc PrGetPageBBox {} {
# Return printer page bounding box.
# [index] PrGetPageBBox!procedure

  return {792 0 0 612}
}

proc PrSetColor {r g b} {
# Set the current color.
# [index] PrSetColor!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "$r $g $b  setrgbcolor"
}

proc PrMoveTo {x y} {
# Set the current location.
# [index] PrMoveTo!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "$x1 $y1 moveto"
}


proc PrDrawRectangle {x1 y1 x2 y2 {thickness 1}} {
# Draw a rectangle.
# [index] PrDrawRectangle!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  set dx [expr $x2 - $x1]
  set dy [expr $y2 - $y1]
  puts $Printer(fp) "newpath"
  puts $Printer(fp) "$x1 $y1 moveto $dx 0 rlineto 0 $dy rlineto [expr -$dx] 0 rlineto"
  puts $Printer(fp) "closepath"
  puts $Printer(fp) "$thickness setlinewidth"
  puts $Printer(fp) "stroke"
}

proc PrDrawFilledRectangle {x1 y1 x2 y2} {
# Draw a filled rectangle.
# [index] PrDrawFilledRectangle!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  set dx [expr $x2 - $x1]
  set dy [expr $y2 - $y1]
  puts $Printer(fp) "newpath"
  puts $Printer(fp) "$x1 $y1 moveto $dx 0 rlineto 0 $dy rlineto [expr -$dx] 0 rlineto"
  puts $Printer(fp) "closepath"
  puts $Printer(fp) "fill"
}

proc PrSetFont {name size} {
# Set the current font.
# [index] PrSetFont!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "/$name findfont $size scalefont setfont"
}

proc PrDrawTextAt {x y text} {
# Draw text at a selected location.
# [index] PrDrawTextAt!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "$x $y moveto ([PSQuote $text]) show"
}

proc PrDrawText {text} {
# Draw text at the current location.
# [index] PrDrawText!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "([PSQuote $text]) show"
}

proc PrDrawTextInBBox {x1 y1 x2 y2 text} {
# Draw text in a bounding box.
# [index] PrDrawTextInBBox!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "$x1 $y1 $x2 $y2 ([PSQuote $text]) TextInBBox"  
}

proc PrDrawLine {x1 y1 x2 y2 {linewidth 1}} {
# Draw a line.
# [index] PrDrawLine!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "newpath $x1 $y1 moveto $x2 $y2 lineto $linewidth setlinewidth stroke"
}

proc PrDrawOval {x1 y1 x2 y2 {thickness 1}} {
# Draw an oval.
# [index] PrDrawOval!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "newpath $x1 $y1 $x2 $y2 $thickness Oval"
}

proc PrDrawFilledOval {x1 y1 x2 y2} {
# Draw a filled oval.
# [index] PrDrawFilledOval!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}
  puts $Printer(fp) "newpath $x1 $y1 $x2 $y2 FillOval"
}

proc PrDrawImage {x1 y1 x2 y2 bitmap} {
# Draw an image (bitmap).
# [index] PrDrawImage!procedure

  global Printer
  if {!$Printer(PageOpened)} {return}
  if {$Printer(CurrentPage) < $Printer(FirstPage) ||
      $Printer(CurrentPage) > $Printer(LastPage)} {return}

# Missing Tcl/Tk functionallity: It would be nice to be able to get the
# bits from a bitmap without having to go all through these fun & games
# with a canvas widget.

  catch "destroy .prdrawimage"
  canvas .prdrawimage
  set tag [.prdrawimage create bitmap 0 0 -anchor nw -bitmap $bitmap]
  set bbox [.prdrawimage bbox $tag]
  set bwidth [lindex $bbox 2]
  set bheight [lindex $bbox 3]
  set ps "[.prdrawimage postscript -x 0 -y 0 -width $bwidth -height $bheight]"
  catch "destroy .prdrawimage"
  if {[regexp {true matrix {(.*)} imagemask} "$ps" whole bits] <= 0} {return}
  puts $Printer(fp) \
	"gsave $x1 $y1 translate [expr $x2 - $x1] [expr $y2 - $y1] scale"
  puts $Printer(fp) \
	"$bwidth $bheight true \[$bwidth 0 0 $bheight 0 0\] \{$bits\} imagemask"
  puts $Printer(fp) "grestore"
}



proc PrMakeTempFileName {} {
# Create a new temporary filename.
# [index] PrMakeTempFileName!procedure

  global Printer
  global env
  if {[catch "set env(TMPDIR)" tmpdir]} {
    set tmpdir "/tmp"
  }
  while {1} {
    incr Printer(TmpFileIndex)
    set tname [file join $tmpdir \
			 [format {UPTMP%08x%08x} [pid] $Printer(TmpFileIndex)]]
    if {![file exists $tname]} {return $tname}
  }
}

proc PrSpoolPlainTextFile {file} {
# Spool file to printer.
# [index] PrSpoolPlainTextFile!procedure

  catch [list exec lpr "$file"]
}
  

package provide printer 1.0
