//-< TSTRTREE.CXX >--------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jun-97    K.A. Knizhnik  * / [] \ *
//                          Last update: 17-Oct-97    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Test program for R-tree class 
//-------------------------------------------------------------------*--------*

#include "goods.h"
#include "dbscls.h"
#include "rtree.h"

class spatial_object : public object { 
  public:
    rectangle rect;
    int4      id;

    void print() const { 
	console::output("Object '%s' (%d,%d,%d,%d)\n", name, 
			rect.boundary[0], rect.boundary[1], 
			rect.boundary[2], rect.boundary[3]);
    }
    static ref<spatial_object> create(int x0, int y0, int x1, int y1,
				      const char* name, int id) 
    {
	int len = strlen(name)+1; 
	return new (self_class,len) spatial_object(x0, y0, x1, y1, 
						   id, name, len);
    }
    boolean is(const char* name) const {return strcmp(this->name, name) == 0;}

    METACLASS_DECLARATIONS(spatial_object, object);
  protected:
    char      name[1];

    spatial_object(int x0, int y0, int x1, int y1,
		   int ident, const char* name, int len)
    : object(self_class, len), id(ident) {
	rect.boundary[0] = x0;
	rect.boundary[1] = y0;
	rect.boundary[2] = x1;
	rect.boundary[3] = y1;
	memcpy(this->name, name, len);
    }
};

field_descriptor& spatial_object::describe_components()
{
    return FIELD(rect), FIELD(id), VARYING(name);
}

REGISTER(spatial_object, object, pessimistic_repeatable_read_scheme);


class spatial_base : public R_tree { 
  protected:
    ref<H_tree> hash;

    class spatial_object_callback : public callback { 
	virtual void apply(ref<object> obj) { 
	    ref<spatial_object> sobj = obj;
	    sobj->print();
	}
    };
    static boolean print_object(const char*, ref<object> obj) { 
	ref<spatial_object> sobj = obj;
	sobj->print();
	return False;
    }
	
  public:
    void find(int x0, int y0, int x1, int y1) const {
	rectangle rect;
	spatial_object_callback cb;
	rect.boundary[0] = x0;
	rect.boundary[1] = y0;
	rect.boundary[2] = x1;
	rect.boundary[3] = y1;
	console::output("Found %d objects\n", search(rect, cb));
    }
    void find(const char* name) const { 
	ref<spatial_object> obj = hash->get(name);
	if (obj.is_nil()) { 
	    console::output("No object found\n");
	} else { 
	    obj->print();
	}
    }
    void print_all() const { 
	console::output("Total %d objects\n", hash->get_numer_of_elements());
	hash->apply(print_object);
    }
    void insert(int x0, int y0, int x1, int y1, const char* name, int id = 0) {
	ref<spatial_object> obj = spatial_object::create(x0, y0, x1, y1, 
							 name, id);
	R_tree::insert(obj->rect, obj);
	modify(hash)->put(name, obj);
    }
    boolean remove(const char* name) { 
	ref<spatial_object> obj = hash->get(name);
	if (!obj.is_nil()) { 
	    modify(hash)->del(name);
	    R_tree::remove(obj->rect, obj);
	    return True;
	}
	return False;
    }
    void initialize() const { 
	if (is_abstract_root()) { 
	    ref<spatial_base> root = this;
	    modify(root)->become(new spatial_base);
	} 
    }
    void clear() { 
	prune();
	modify(hash)->reset();
    }
    spatial_base() : R_tree(self_class) { 
	hash = new H_tree(525313);
    }
    METACLASS_DECLARATIONS(spatial_base, R_tree);
};

field_descriptor& spatial_base::describe_components()
{
    return FIELD(hash);
}

REGISTER(spatial_base, R_tree, pessimistic_repeatable_read_scheme);

static void input(char* prompt, char* buf, size_t buf_size)
{
    char* p;
    do { 
	console::output(prompt);
	*buf = '\0';
	console::input(buf, buf_size);
	p = buf + strlen(buf);
    } while (p <= buf+1); 
    
    if (*(p-1) == '\n') {
	*--p = '\0';
    }
}


struct test_record { 
    union { 
	test_record* next_free;
	struct { 
	    int2 x0;
	    int2 y0;
	    int2 x1;
	    int2 y1;
	} coord;
    };
    nat4 data[2];
};

inline unsigned random(unsigned mod) { return rand() % mod; }

static void random_test(ref<spatial_base> base, long n_iterations) { 
    const int max_records = 1024*1024;
    const int max_x = 32*1024;
    const int max_y = 32*1024;
    const int max_width = 1024;
    const int max_height = 1024;
    
    test_record* record = new test_record[max_records];
    test_record* free_chain;
    int i;
    int n_inserts = 0, n_searches = 0, n_removes = 0;
    test_record* rp;
    int x0, y0, x1, y1;
    rectangle rect;
    search_buffer sb;
    char buf[64];

    srand(time(NULL));
    for (i = 0; i < max_records; i++) { 
	record[i].data[0] = 0;
	record[i].next_free = &record[i+1];
    }
    record[max_records-1].next_free = NULL;
    free_chain = record;
    modify(base)->clear();
    time_t start_time = time(NULL);

    while (--n_iterations >= 0) { 
	switch (random(8)) { 
	  default: // insert
	    if (free_chain != NULL) { 
		rp = free_chain;
		free_chain = free_chain->next_free;
		rp->coord.x0 = x0 = random(max_x*2) - max_x; 
		rp->coord.x1 = x1 = x0 + random(max_width); 
		rp->coord.y0 = y0 = random(max_y*2) - max_y; 
		rp->coord.y1 = y1 = y0 + random(max_height); 
		rp->data[0] = rand();
		if (rp->data[0] == 0) { 
		    rp->data[0] += 1;
		}
		rp->data[1] = rand();
		i = rp - record;
		sprintf(buf, "%x%x.%x", rp->data[0], rp->data[1], i);
		modify(base)->insert(x0, y0, x1, y1, buf, i);
		n_inserts += 1;
	    }
	    break;
	  case 0: // remove
	    i = random(max_records);
	    rp = &record[i];
	    if (rp->data[0]) { 
		sprintf(buf, "%x%x.%x", rp->data[0], rp->data[1], i);
		boolean removed = modify(base)->remove(buf);
		assert(removed);
		rp->data[0] = 0;
		rp->next_free = free_chain;
		free_chain = rp;
		n_removes += 1;
	    } 
	    break;
	  case 1: // search
	    x0 = random(max_x*2) - max_x; 
	    x1 = x0 + random(max_width); 
	    y0 = random(max_y*2) - max_y; 
	    y1 = y0 + random(max_height); 
	    rect.boundary[0] = x0;
	    rect.boundary[1] = y0;
	    rect.boundary[2] = x1;
	    rect.boundary[3] = y1;
	    sb.clean();
	    int found = base->search(rect, sb);
	    rp = record;
	    for (i = 0; i < max_records; i++, rp++) { 
		if (rp->data[0]) { 
		    if (x0 <= rp->coord.x1 && rp->coord.x0 <= x1
			&& y0 <= rp->coord.y1 && rp->coord.y0 <= y1) 
		    {
			sb.reset();
			while (True) { 
			    assert(!sb.is_empty());
			    ref<spatial_object> obj = sb.get();
			    if (obj->id == i) { 
				sprintf(buf, "%x%x.%x", 
					rp->data[0], rp->data[1], i);
				assert(obj->rect.boundary[0] == rp->coord.x0 &&
				       obj->rect.boundary[1] == rp->coord.y0 &&
				       obj->rect.boundary[2] == rp->coord.x1 &&
				       obj->rect.boundary[3] == rp->coord.y1 &&
				       obj->is(buf));
				found -= 1;
				break;
			    }
			}
		    }
		}
	    }
	    assert (found == 0); 
	    n_searches += 1;
	}
	console::output("Proceed %d inserts, %d removes, %d searches\r", 
			n_inserts, n_removes, n_searches);
    }
    console::output("\nElapsed time: %ld\n", time(NULL) - start_time);
    delete[] record;
}

main() { 
    database db;
    task::initialize(task::normal_stack);
    basic_metaobject::set_cache_limit(1024*1024,4*1024*1024);
    if (db.open("rtree.cfg")) { 
	ref<spatial_base> root;
	db.get_root(root);
        root->initialize();
	
	while (True) { 
	    char buf[256], name[256];
	    int  action = 0;
	    int  x0, y0, x1, y1;
	    int n_iterations;

	    console::output("\nSpatial database menu:\n"
			    "\t1. Insert object\n"
			    "\t2. Remove object\n"
			    "\t3. Find object by name\n"
			    "\t4. Find object by coorinates\n"
			    "\t5. List of all objects\n"
			    "\t6. Random test\n"
			    "\t7. Exit\n");
	    input(">>", buf, sizeof buf);
	    sscanf(buf, "%d", &action);
	    switch (action) { 
	      case 1:
		input("Please specify object name: ", name, sizeof name);
		while (True) { 
		    input("Please specify object coordinates x0 y0 x1 y1: ",
			  buf,  sizeof buf);
		    if (sscanf(buf, "%d %d %d %d", &x0, &y0, &x1, &y1) == 4) {
			modify(root)->insert(x0, y0, x1, y1, name);
			break;
		    }
		}
		continue;
	      case 2:
		input("Please specify removed object name: ", 
		      name, sizeof name);
		if (!modify(root)->remove(name)) { 
		    console::output("Object is not in database\n");
		}
		continue;
	      case 3:
		input("Please specify object name: ", name, sizeof name);
		root->find(name);
		continue;
	      case 4:
		while (True) { 
		    input("Please specify object coordinates x0 y0 x1 y1: ",
			  buf,  sizeof buf);
		    if (sscanf(buf, "%d %d %d %d", &x0, &y0, &x1, &y1) == 4) {
			root->find(x0, y0, x1, y1);
			break;
		    }
		}
		continue;
	      case 5:
		root->print_all();
		continue;
	      case 6:
		while (True) {
		    input("Please sepcify number of iterations: ", 
			  buf, sizeof buf);
		    if (sscanf(buf, "%d", &n_iterations) == 1) {
			random_test(root, n_iterations);
			break;
		    }
		}
		continue;
	      case 7:
		db.close();
		console::output("Session is closed\n");
		return EXIT_SUCCESS;
	      default:
		console::output("Please chose action 1..7\n");
	    }
	}
    } else { 
	console::output("Failed to open database\n");
	return EXIT_FAILURE;
    }
}
