//-< STORAGE.H >-----------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jan-97    K.A. Knizhnik  * / [] \ *
//                          Last update: 23-Apr-97    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Abstract database storage interface.  
//-------------------------------------------------------------------*--------*

#ifndef __STORAGE_H__
#define __STORAGE_H__

#include "protocol.h"

//
// This class is used to provide interface with application to storage. 
// Since storage is implemented in application independent way it
// requires an agent to handle servers requests 
//
class dbs_application { 
  public: 
    //
    // Function which is called when server informs client
    // that object instance was changed. 
    //
    virtual void invalidate(sid_t sid, opid_t opid) = 0;
    //
    // Function is called when server is diconnected from client
    //
    virtual void disconnected(sid_t sid) = 0;

    //
    // Authorization procedure fails at server sid 
    //
    virtual void login_refused(sid_t sid) = 0;
};  

   
//
// Interface to database object storage. All methods waiting answer 
// from server should be called synchronously 
// (concurrent requests are not allowed)
//

class dbs_storage { 
  protected:
    //
    // Identifier of storage within database. 
    // 
    const sid_t    id;
    //
    // Interface with application
    //
    dbs_application* application;

  public: 
    virtual opid_t  allocate(cpid_t cpid, size_t size, boolean aligned) = 0;
    virtual void    deallocate(opid_t opid) = 0;

    virtual boolean lock(opid_t opid, lck_t lck, int attr) = 0;
    virtual void    unlock(opid_t opid, lck_t lck) = 0;
    
    //
    // Get class descriptor by class identifier.
    // Class descriptor is placed in the buffer supplied by application.
    // If there is no such class at server buf.size() is set to 0.
    //
    virtual void    get_class(cpid_t cpid, dnm_buffer& buf) = 0;
    //
    // Store class descriptor at server receive it's identifier
    //
    virtual cpid_t  put_class(dbs_class_descriptor* desc) = 0;
    //
    // Change existed class descriptor
    //
    virtual void    change_class(cpid_t cpid, dbs_class_descriptor* desc) = 0;
					    
    //
    // Load object from server into the buffer supplied by application.
    // Before each object dbs_object_header structure is placed.
    // If there is no such object at server then "cpid" field of 
    // dbs_object_header is set to 0.
    //
    virtual void    load(opid_t* opid, int n_objects, 
			 int flags, dnm_buffer& buf) = 0;

    virtual void    load(opid_t opid, int flags, dnm_buffer& buf) = 0;


    //
    // Inform server that client no more has reference to specified object
    // 
    virtual void    forget_object(opid_t opid) = 0;
    //
    // Inform server that client no more has instance of specified object
    //
    virtual void    throw_object(opid_t opid) = 0; 

    //
    // Initiate transaction at server. Allocate place for transaction header
    // in buffer. All objects involved in transaction should be appended 
    // to this buffer. 
    //
    virtual void    begin_transaction(dnm_buffer& buf) = 0; 

    //
    // Commit local transaction or part of global transaction 
    // at coordinator server. In latter case coordinator will return 
    // transaction indentifier which should be 
    // passed to all other servers participated in transaction.
    // If transaction is aborted by server "False" is returned. 
    //
    virtual boolean commit_coordinator_transaction(int n_trans_servers, 
						   sid_t* trans_servers,
						   dnm_buffer& buf, 
						   tid_t& tid) = 0;
    //
    // Commit local part of global transaction at server. 
    //
    virtual void    commit_transaction(sid_t coordinator, 
				       int n_trans_servers,
				       sid_t* trans_servers,
				       dnm_buffer& buf, 
				       tid_t tid) = 0;
    //
    // Wait completion of global transaction (request to coordinator)
    // If transaction is aborted by server "False" is returned. 
    //
    virtual boolean wait_global_transaction_completion() = 0;

    //
    // Establish connection with server, returns 'False' if connection failed
    //
    virtual boolean open(const char* server_connection_name) = 0;
    virtual void    close() = 0;

    dbs_storage(sid_t sid, dbs_application* app) 
    : id(sid), application(app) {}
};

#endif

