//-< SERVER.H >-----------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET   *     ?  *
// (Generic Object Oriented Database System)                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jan-97    K.A. Knizhnik * / [] \ *
//                          Last update: 14-Sep-97    K.A. Knizhnik * GARRET *
//------------------------------------------------------------------*--------*
// Database storage server
//------------------------------------------------------------------*--------*

#ifndef __SERVER_H__
#define __SERVER_H__

#include "stdinc.h"
#include "protocol.h"
#include "osfile.h"
#include "mmapfile.h"
#include "sockio.h"

#include "poolmgr.h"
#include "objmgr.h"
#include "memmgr.h"
#include "transmgr.h"
#include "classmgr.h"

//
// Storage server absrtaction
//

class server_agent;
class client_agent;

class dbs_server { 
  public:
    const sid_t            id;

    boolean                opened;         // server is openned
    boolean                backup_started; // set by 'backup_start', 
                                           // cleared by 'backup_stop'
    time_t                 backup_start_delay;
    fsize_t                backup_start_log_size;
    typedef void (*backup_finish_callback)(dbs_server& server, 
					   file& backup_file, 
					   boolean status);

    object_access_manager* obj_mgr;
    pool_manager*          pool_mgr;
    class_manager*         class_mgr; 
    memory_manager*        mem_mgr; 
    transaction_manager*   trans_mgr; 

    //
    // Clients are assigned successive identifiers. This method
    // returns identifier of client which was connected longest time
    // ago, i.e. client having smallest identifier.
    //
    virtual unsigned get_oldest_client_id() = 0;

    //
    // Set limitation on object cluster size which can be sent to client
    //
    virtual void set_object_cluster_size_limit(size_t cluster_size) = 0;

    virtual unsigned get_number_of_servers() const = 0;
    virtual char const* get_name() const = 0;

    virtual void    remote_server_connected(sid_t sid) = 0;

    virtual server_agent* create_server_agent(int id) = 0; 
    virtual client_agent* create_client_agent() = 0;
    virtual void    remove_client_agent(client_agent* agent) = 0; 
    virtual void    remove_server_agent(server_agent* agent) = 0; 

    //
    // Send request message to remote server. Message consists of fixed part
    // (class dbs_request) and 'body_len' bytes of message body.
    // Header of message is automatically in place converted to universal 
    // representation.  
    //
    virtual void    send(sid_t sid, dbs_request* req,
			 size_t body_len = 0) = 0; 

    //
    // Read message body. This function should be called from
    // 'gc_sync' or 'tm_sync' methods to read body part of message.
    //
    virtual void    read_msg_body(sid_t sid, void* buf, size_t size) = 0;

    //
    // Dump current server state. 'What' paramter specifies list of properties
    // which should be dumped. Properties can be separated by any separators. 
    //
    virtual void    dump(char* what) = 0;

    //
    // Initiate online backup of storage files . 
    //
    virtual void    start_backup(file&   backup_file, 
				 time_t  backup_start_delay = 0,
				 fsize_t backup_start_log_size = 0,
				 backup_finish_callback callback = NULL) = 0;
    virtual void    stop_backup() = 0;
    //
    // Offline storage restore. Storage should be closed before
    // calling this method. If storage is succefully restored, 
    // method "open" is called to open the server.
    // 
    virtual boolean restore(file& backup_file, 
			    const char* database_configuration_file) = 0; 


    //
    // Notify clients about modified objects
    //
    virtual void    notify_clients() = 0;

    //
    // Method performs login authorization. This method should return
    // 'True' value for valid clients (or servers) and 'False'
    // otherwise.
    //
    virtual boolean authorize(dbs_request const& req, char* name) = 0;

    virtual boolean open(char const* database_configuration_file) = 0;
    virtual void    close() = 0;

    dbs_server(sid_t sid) : id(sid) { 
	opened = False; 
	backup_started = False;
    }
};     

//
// Communication protocol 
// 

class communication { 
  protected: 
    char      name[MAX_LOGIN_NAME]; // connection name 
    boolean   connected; 
    socket_t* sock;
    mutex     cs;   // mutex for synchronization of concurrent writes
                    // and connect/disconnect operations
    task*     receiver;  
    eventex   shutdown_event;
    boolean   reconnect_flag;

    virtual boolean  handle_communication_error();

    static void task_proc receive(void* arg);

  public: 
    virtual void  write(void const* buf, size_t size);
    virtual void  read(void* buf, size_t size);

    virtual void  poll() = 0;

    virtual void  remove() = 0;

    virtual void  connect(socket_t* connection, char* name, boolean fork=False);
    virtual void  disconnect();

    static task*  create_receiver(task::fptr f, void* arg) { 
	return task::create(f, arg, task::pri_normal, task::small_stack); 
    }

    communication() : shutdown_event(cs) { 
	connected = False; 
	*name = '\0';
	receiver = NULL; 
	reconnect_flag = False;
	sock = NULL;
    } 
    virtual~communication();
};

//
// Client agent 
// 

class client_agent : public client_process, public communication 
{
  public: 
    //
    // Insert invalidation signal in notification queue
    //
    virtual void invalidate(opid_t opid);
 
    //
    // Send notification requests to client
    //
    virtual void notify();

    //
    // Poll and serve client requests
    //
    virtual void poll();

    //
    // Disconnect client from server
    //
    virtual void disconnect();


    virtual void remove();

    //
    // Send object (or bulk of objects) to client
    //
    virtual void send_object(dbs_request const& req);

    virtual char* get_name();
    
    client_agent(dbs_server* server, int client_id, size_t max_cluster_size) 
    : client_process(server, client_id),
      cluster_size(max_cluster_size) {}

  protected:
    void handle_notifications(int n_requests);
    virtual void write(void const* buf, size_t size);
    
    size_t     cluster_size; 
    dnm_buffer buf; 

    mutex      notify_cs;
    dnm_array<dbs_request> notify_buf;    
}; 

//
// Remote server agent
//

class server_agent : public communication { 
  protected:
    const sid_t  id;
    dbs_server*  server;  
    mutex        connect_cs;

  public:
    boolean      is_online() { return connected; }
    char*        get_name() { return name; }

    //
    // Make connection with another server by sending self indentifing 
    // information to it and receive authorization result. 
    // This methods is called before connect(), so ommunication class fields 
    // are not initialized.
    //
    virtual boolean handshake(socket_t* sock, sid_t sid, char const* my_name); 
    //
    // Poll and serve requests of remote server
    //
    virtual void poll();
    //
    // Write to remote server
    //
    virtual void write(void const* buf, size_t size);
    //
    // Remove server
    //
    virtual void remove();
    //
    // Send request to remote server
    //
    virtual void send(dbs_request* req, size_t body_len = 0); 
    //
    //
    //
    virtual void connect(socket_t* connection, char* name, boolean fork=False);

    //
    // Read body of message send by remote server
    //
    virtual void read_msg_body(void* buf, size_t size);
    
    server_agent(dbs_server* server, sid_t sid) : id(sid) {
	this->server = server; 
    }
};

//
// Local storage server
//

#define DEFAULT_CLUSTER_SIZE 512

class storage_server : public dbs_server  
{ 
  protected:
    l2elem         clients;   // list if client agents
    server_agent** servers;   // array of server agents
    int            n_servers; // total number of servers in database
    mutex          cs;        // synchronize access to class components

    socket_t*      local_gateway;    // listening sockets
    socket_t*      global_gateway;    

    int            n_online_remote_servers; 
    int            n_opened_gateways; 
    eventex        term_event; // event notifing about thread termination
    size_t         object_cluster_size_limit; 

    int            client_id; // generator of sequence numbers 
                              // used to identify clients 

    char           name[MAX_LOGIN_NAME]; // address of server

    file*          backup_file;   
    backup_finish_callback backup_callback;

    eventex        backup_finished_event;

    class login_data : public l2elem {
      public: 
	socket_t*       sock;
	storage_server* server;
	
	login_data(storage_server* server, socket_t* sock) { 
	    this->server = server;
	    this->sock = sock;
	}
	~login_data() { 
	    unlink(); 
	    delete sock;
	}
    };
    l2elem handshake_list; // list of accepted but not established connections

    static void task_proc start_local_gatekeeper(void* arg);
    static void task_proc start_global_gatekeeper(void* arg);
    static void task_proc start_backup_process(void* arg);
    static void task_proc start_handshake(void* arg);

    virtual void          accept(socket_t* gateway); // accept connections
    virtual void          handshake(login_data* login); 

    virtual void          remote_server_connected(sid_t sid);

    virtual server_agent* create_server_agent(int id); 
    virtual client_agent* create_client_agent();

    virtual boolean       backup();
    
  public: 
    virtual void        notify_clients();

    virtual void        send(sid_t sid, dbs_request* req, size_t body_len=0); 

    virtual void        read_msg_body(sid_t sid, void* buf, size_t size);

    virtual unsigned    get_number_of_servers() const;

    virtual char const* get_name() const;

    virtual unsigned    get_oldest_client_id();

    virtual void        remove_client_agent(client_agent* agent); 

    virtual void        set_object_cluster_size_limit(size_t cluster_size);

    virtual void        remove_server_agent(server_agent* agent); 

    virtual void        dump(char* what);

    virtual void        start_backup(file&   backup_file, 
				     time_t  backup_start_delay = 0,
				     fsize_t backup_start_log_size = 0,
				     backup_finish_callback callback = NULL);
    virtual void        stop_backup();
    virtual boolean     restore(file& backup_file, 
			     const char* database_configuration_file); 

    virtual boolean     authorize(dbs_request const& req, char* name);

    virtual boolean     open(char const* database_configuration_file);
    virtual void        close();

    storage_server(sid_t sid, 
		   object_access_manager&, 
		   pool_manager&,
		   class_manager&,
		   memory_manager&,
		   transaction_manager&,
		   size_t object_cluster_size = DEFAULT_CLUSTER_SIZE);
};

#endif
