//-< SERVER.CXX >---------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET   *     ?  *
// (Generic Object Oriented Database System)                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jan-97    K.A. Knizhnik * / [] \ *
//                          Last update: 14-Sep-97    K.A. Knizhnik * GARRET *
//------------------------------------------------------------------*--------*
// Database storage server
//------------------------------------------------------------------*--------*

#include "server.h"

//
// Communication
//

boolean communication::handle_communication_error()
{
    cs.enter();
    if (connected) { 
	msg_buf buf; 
	sock->get_error_text(buf, sizeof buf); 
	console::message(msg_error|msg_time, 
			 "Connection with '%s' is lost: %s\n", name, buf);
	connected = False;
    }
    cs.leave();
    return False; 
}

void communication::read(void* buf, size_t size)
{
    if (!sock->read(buf, size) &&
	!handle_communication_error()) 
    { 
	disconnect();
    }
}

void communication::write(void const* buf, size_t size)
{
    ((dbs_request*)buf)->pack();
    cs.enter();
    if (connected) { 
        if (!sock->write(buf, size) &&
	    !handle_communication_error()) 
	{ 
	    disconnect();
        }
    }
    cs.leave();
}

void task_proc communication::receive(void* arg)
{
    ((communication*)arg)->poll();
}

void communication::connect(socket_t* connection, char* connection_name, 
			    boolean fork)
{
    sock = connection; 
    strcpy(name, connection_name); 
    connected = True;
    receiver = fork ? create_receiver(receive, this) : task::current();
} 

void communication::disconnect()
{
    cs.enter();
    if (connected) { 
	console::message(msg_login|msg_time, "Disconnect '%s'\n", name);
	if (sock != NULL) { 
	    dbs_request req; 
	    req.cmd = dbs_request::cmd_bye; 
	    sock->write(&req, sizeof req);
	}
	connected = False;
    }
    if (sock != NULL) { 
	sock->shutdown();
    }

    if (task::current() == receiver) { 
	TRACE_MSG((msg_important, "Server agent '%s' terminated\n", name));
	delete sock;
	sock = NULL;
	if (reconnect_flag) { 
	    shutdown_event.signal();
        }
	receiver = NULL;
	cs.leave();
	remove();
	task::exit();
    } else { 
	cs.leave();
    }
}

communication::~communication()
{
    delete sock;
} 

//
// Client agent
//

void client_agent::invalidate(opid_t opid)
{
    dbs_request req;
    req.object.cmd = dbs_request::cmd_invalidate;
    req.object.opid = opid;
    req.object.extra = 0;
    TRACE_MSG((msg_object, 
	       "Push invalidation signal for object %x to client '%s'\n",
	       opid, name));
    notify_cs.enter();
    notify_buf.push(req);
    notify_cs.leave();
}

void client_agent::notify()
{
    notify_cs.enter();
    int n = notify_buf.size();
    if (n > 0) { 
	dbs_request* req = &notify_buf;
	TRACE_MSG((msg_object, 
		   "Send invalidation signals for to client '%s': %x", 
		   name, req->object.opid));
	req->object.extra = n - 1;
	while (--n > 0) {
	    req += 1;
	    TRACE_MSG((msg_object, ",%x", req->object.opid));
	    req->pack();
	}	
	communication::write(&notify_buf, 
			     notify_buf.size()*sizeof(dbs_request));
	notify_buf.change_size(0);
	TRACE_MSG((msg_object, "\n"));
    }
    notify_cs.leave();
}

void client_agent::write(void const* buf, size_t size)
{
    //
    // Make sure that all invalidations messages generated during 
    // handling of client request will be send to the client before 
    // response to the request
    //
    notify();
    communication::write(buf, size);
}

#define MAX_FAILED_ATTEMPTS 1

void client_agent::send_object(dbs_request const& req)
{ 
    dbs_handle hnd; 
    opid_t opid;
    int n, n_objects = req.object.extra + 1;
    dbs_request *reqp; 
    dbs_object_header *hps, *hpd;
    dnm_array<dbs_request> req_buf;

    buf.put(cluster_size); // reserve enough space in buffer
    buf.put(sizeof(dbs_request));

    if (n_objects > 1) { 
	req_buf.change_size(n_objects);
	reqp = &req_buf;
	read(reqp+1, (n_objects-1)*sizeof(dbs_request));
	for (n = 1; n < n_objects; n++) { 
	    reqp[n].unpack();
	    if (reqp[n].cmd != dbs_request::cmd_load) { 
		console::message(msg_error|msg_time, 
				 "Invalid load request %d from client '%s'\n",
		                 reqp[n].cmd, name);			     
		n_objects = n;
            }
        }
	*reqp = req;
    } else { 
	reqp = (dbs_request*)&req;
    }

    for (n = 0; n < n_objects; n++) { 
    	opid = reqp[n].object.opid;

  	server->obj_mgr->load_object(opid, reqp[n].object.flags, this); 
	server->mem_mgr->get_handle(opid, hnd);
	size_t size = hnd.get_size();
	cpid_t cpid = hnd.get_cpid();
        
	if (cpid == 0) { 
	    console::message(msg_error|msg_time, 
			     "Client '%s' request unexisted object %x\n",
			     name, opid);
	} 
        hpd = (dbs_object_header*)buf.append(sizeof(dbs_object_header)+size);
        hpd->set_opid(opid);
        hpd->set_size(size);
        hpd->set_cpid(cpid);

        if (size != 0) {
	    server->pool_mgr->read(hnd.get_pos(), hpd->body(), size);
        } 
	server->obj_mgr->release_object(opid); 
    }    
    size_t msg_size = buf.size();
    hps = (dbs_object_header*)(&buf + sizeof(dbs_request));
    hpd = (dbs_object_header*)(&buf + msg_size);

    int n_failed_attempts = 0;

    for (n = 0; n < n_objects && msg_size < cluster_size; n++) { 
	if (reqp[n].object.flags & lof_cluster) { 
	    cpid_t cpid = hps->get_cpid();
	    size_t size = hps->get_size();
	    if (size == 0) { 
		hps = (dbs_object_header*)
		      ((char*)hps + sizeof(dbs_object_header));
		continue;
	    } 
	    int n_refs = 
		server->class_mgr->get_number_of_references(cpid, size);
	    internal_assert(n_refs >= 0);
	    char* p = hps->body(); 

	    while (--n_refs >= 0) { 
		opid_t co_opid; 
		sid_t  co_sid; 
		p = unpackref(co_sid, co_opid, p);
		if (co_opid != 0 && co_sid == server->id &&
		    server->obj_mgr->get_object_state(co_opid, this)==cis_none)
		{
		    server->obj_mgr->load_object(co_opid, lof_none, this); 
		    server->mem_mgr->get_handle(co_opid, hnd);
		    size_t co_size = hnd.get_size();
		    if (msg_size + co_size + sizeof(dbs_object_header)
			<= cluster_size) 
		    { 
			hpd->set_opid(co_opid);
			hpd->set_size(co_size);
			hpd->set_cpid(hnd.get_cpid());
			server->pool_mgr->read(hnd.get_pos(), hpd->body(),
					       co_size);
			hpd = (dbs_object_header*)(hpd->body() + co_size);
			msg_size += co_size + sizeof(dbs_object_header); 
			server->obj_mgr->release_object(co_opid); 
		    } else { 
			server->obj_mgr->forget_object(co_opid, this); 
			//
			// Continue construction of closure only if we have 
			// enough space in the buffer
			//
			if (++n_failed_attempts >= MAX_FAILED_ATTEMPTS) { 
			    n = n_objects;
			    break;
			} 
		    }
		} 
	    }
	}
    }	
    dbs_request* reply = (dbs_request*)&buf;
    reply->result.cmd = dbs_request::cmd_object; 
    reply->result.size = buf.size() - sizeof(dbs_request); 
    write(reply, msg_size);
} 

void client_agent::disconnect() 
{
    terminated = True;
    lock_sem.signal();
    communication::disconnect();
}

void client_agent::remove()
{
    server->obj_mgr->disconnect_client(this);
    server->remove_client_agent(this); 
}

void client_agent::handle_notifications(int n_requests)
{
    if (n_requests > 0) { 
	dnm_array<dbs_request> buf;
	buf.change_size(n_requests);
	dbs_request* rp = &buf;
	read(rp, n_requests*sizeof(dbs_request));
	while (--n_requests >= 0) { 
	    rp->unpack();
	    switch (rp->cmd) { 
	      case dbs_request::cmd_forget:
		server->obj_mgr->forget_object(rp->object.opid, this);  
		break; 
		
	      case dbs_request::cmd_throw:
		server->obj_mgr->throw_object(rp->object.opid, this);  
		break; 
		
	      default: 
		console::message(msg_error|msg_time, 
				 "Invalid notification %d from client '%s'\n",
				 rp->cmd, name);			     
	    }
	    rp += 1;
	}
    }
}

void client_agent::poll()
{
    dbs_transaction_header* trans; 
    dbs_class_descriptor* desc; 
    dbs_request req, *reqp;
    lck_t       lck;

    while(connected) { 
	read(&req, sizeof req);
	req.unpack();
	TRACE_MSG((msg_request, "Receive request %d from client '%s'\n", 
		   req.cmd, name));
	req_count += 1;
	switch (req.cmd) { 
	  case dbs_request::cmd_load: 
	    send_object(req); 
	    break;
	    
	  case dbs_request::cmd_getclass: 
	    desc = server->class_mgr->get_and_lock_class(req.clsdesc.cpid, 
	                                                 this);
	    if (desc != NULL) { 
		size_t len = desc->get_size();
		reqp = (dbs_request*)buf.put(sizeof(dbs_request) + len);
		reqp->result.cmd = dbs_request::cmd_classdesc; 
		reqp->result.size = len; 
		dbs_class_descriptor* dst_desc = 
		    (dbs_class_descriptor*)(reqp+1);
		memcpy(dst_desc, desc, len);
		server->class_mgr->unlock_class(req.clsdesc.cpid);
		TRACE_MSG((msg_important, "Send class '%s' to client\n", 
			   dst_desc->name()));
		dst_desc->pack();
		write(reqp, sizeof(dbs_request)+len);
	    } else { 
		req.result.cmd = dbs_request::cmd_classdesc; 
		req.result.size = 0;
		console::message(msg_error|msg_time, 
				 "Class with descriptor %x is not found\n", 
				  req.clsdesc.cpid);
		write(&req, sizeof req);
	    }
	    break;

	  case dbs_request::cmd_putclass: 
	    desc = (dbs_class_descriptor*)buf.put(req.clsdesc.size);
	    read(desc, req.clsdesc.size);
	    desc->unpack();
	    req.clsdesc.cmd = dbs_request::cmd_classid;
	    req.clsdesc.cpid = server->class_mgr->put_class(desc, this);
	    write(&req, sizeof req);
	    break;
	    
	  case dbs_request::cmd_modclass: 
	    desc = (dbs_class_descriptor*)buf.put(req.clsdesc.size);
	    desc->unpack();
	    read(desc, req.clsdesc.size);
	    server->class_mgr->modify_class(req.clsdesc.cpid, desc, this);
	    break;

	  case dbs_request::cmd_lock: 
	    req.result.status = 
		server->obj_mgr->lock_object(req.lock.opid, 
					     lck_t(req.lock.type), 
	                                     req.lock.attr, 
					     lck, this);
	    req.result.cmd = dbs_request::cmd_lockresult; 
	    write(&req, sizeof req);
	    break;

	  case dbs_request::cmd_unlock: 
	    server->obj_mgr->unlock_object(req.lock.opid, 
	                                   lck_t(req.lock.type), this);
	    break;

	  case dbs_request::cmd_transaction: 
	    trans = (dbs_transaction_header*)
		buf.put(sizeof(dbs_transaction_header) + req.trans.size);
	    read(trans->body(), req.trans.size);
	    trans->coordinator = server->id;
	    trans->size = req.trans.size; 
	    req.cmd = dbs_request::cmd_transresult;
	    req.result.status = 
		server->trans_mgr->do_transaction(req.trans.n_servers,
						  trans, this);
	    write(&req, sizeof req);
	    break; 

	  case dbs_request::cmd_subtransact: 
	    trans = (dbs_transaction_header*)
		buf.put(sizeof(dbs_transaction_header) + req.trans.size); 
	    read(trans->body(), req.trans.size);
	    trans->coordinator = req.trans.coordinator;
	    trans->size = req.trans.size; 
	    trans->tid = req.trans.tid; 
	    server->trans_mgr->do_transaction(req.trans.n_servers,trans,this); 
	    break;
	    
	  case dbs_request::cmd_alloc:
	    req.object.opid = server->mem_mgr->alloc(req.alloc.cpid, 
						     req.alloc.size,
						     req.alloc.align,
						     this);
	    if (req.object.opid == 0) {
		TRACE_MSG((msg_object|msg_important|msg_warning, 
			   "Failed to allocate object of size %u\n",  
			   req.alloc.size));
		disconnect();
	    }
	    TRACE_MSG((msg_object, "Allocate object %x\n", req.object.opid));
	    req.cmd = dbs_request::cmd_location; 
	    write(&req, sizeof req);
	    break;

	  case dbs_request::cmd_free: 
	    server->mem_mgr->free(req.object.opid);
	    server->obj_mgr->forget_object(req.object.opid, this);  
	    break; 
	    
	  case dbs_request::cmd_forget:
	    server->obj_mgr->forget_object(req.object.opid, this);  
	    handle_notifications(req.object.extra);
	    break; 

	  case dbs_request::cmd_throw:
	    server->obj_mgr->throw_object(req.object.opid, this);  
	    handle_notifications(req.object.extra);
	    break; 
	    
	  case dbs_request::cmd_logout: 
	    console::message(msg_login|msg_time, 
			     "Client '%s' send logout request\n", name);
	    disconnect();
	    break;

	  default: 
	    console::message(msg_error|msg_time, 
			     "Invalid request %d from client '%s'\n",
			     req.cmd, name);			     
	}
    }
    disconnect();
}

char* client_agent::get_name() { return name; }

//
// Remote server agent
//

void server_agent::connect(socket_t* connection, char* name, boolean fork)
{
    //
    // connect() is called with fork == False from handshake() method 
    // and with fork == True from open() and write() methods.
    // In write() mutex was already locked and it is not neccessary to lock
    // mutex when open() is executed. Nested lock should be avoided otherwise
    // shutdown_event.wait() will not unlock mutex.
    //
    if (!fork) cs.enter(); 
    TRACE_MSG((msg_important, 
	       "Connect '%s', fork = %d, connected = %d, receiver = %p\n", 
	       name, fork, connected, receiver)); 
    if (receiver != NULL) { 
	reconnect_flag = True;
	shutdown_event.reset();
	sock->shutdown();
	shutdown_event.wait();
	reconnect_flag = False;
    }
    server->remote_server_connected(id);
    communication::connect(connection, name, fork);
    if (!fork) cs.leave();
}

void server_agent::write(void const* buf, size_t size)
{
    cs.enter();
    if (!connected) { 
        if (id < server->id) { 
	    if (reconnect_flag) { 
		while (reconnect_flag) { 
		    shutdown_event.wait();
		}
		if (connected) { 
		    communication::write(buf, size);
		}
	    } else { 
		socket_t* s =
		    socket_t::connect(name,socket_t::sock_any_domain, 1, 0); 
		if (s->is_ok()) { 
		    console::message(msg_notify|msg_login|msg_time, 
				     "Reestablish connection with server "
				     "'%s'\n", name);
		    if (handshake(s, server->id, server->get_name())) { 
			connect(s, name, True);
			communication::write(buf, size);
		    } else { 
			delete s;
		    }
		} else { 
		    delete s;
		}
	    }
        }
    } else { 
	communication::write(buf, size);
    }
    cs.leave();
}

void server_agent::remove()
{
    server->remove_server_agent(this);
}
    
void server_agent::poll() 
{
    dbs_request req; 

    while (connected) { 
	read(&req, sizeof req);
	req.unpack();
	TRACE_MSG((msg_request, "Receive request %d from server '%s'\n", 
		   req.cmd, name));
	switch (req.cmd) { 
	  case dbs_request::cmd_tmsync: 
	    server->trans_mgr->tm_sync(id, req);
	    break;

	  case dbs_request::cmd_gcsync: 
	    server->mem_mgr->gc_sync(id, req);
	    break;
	    
	  case dbs_request::cmd_bye:
	    console::message(msg_login|msg_notify|msg_time, 
			     "Server '%s' send logout request\n", name);
	    connected = False;
	    disconnect();
	    break;

	  default: 
	    console::message(msg_error|msg_time, 
			     "Illegal request %d from server '%s'\n", 
			     req.cmd, name); 
	}
    }
    disconnect();
}

void server_agent::send(dbs_request* req, size_t body_len)
{
    write(req, sizeof(dbs_request) + body_len);
}

void server_agent::read_msg_body(void* buf, size_t size) 
{
    read(buf, size);
}     

boolean server_agent::handshake(socket_t* s, sid_t sid, char const* my_name)
{
    int len = strlen(my_name);
    assert(len < MAX_LOGIN_NAME); 
    struct {
	dbs_request hdr;
	char        name[MAX_LOGIN_NAME];
    } snd_req;
    dbs_request rcv_req;

    snd_req.hdr.cmd = dbs_request::cmd_connect; 
    snd_req.hdr.login.sid = sid;
    snd_req.hdr.login.name_len = len;
    memcpy(snd_req.name, my_name, len); 
    snd_req.hdr.pack();
    if (s->write(&snd_req, sizeof(dbs_request) + len) &&
	s->read(&rcv_req, sizeof rcv_req)) 
    { 
	if (rcv_req.cmd == dbs_request::cmd_ok) { 
	    return True; 
	}	
	console::message(msg_login|msg_error|msg_time, 
			 "Login is refused by server %d\n", id);
    } else { 
	msg_buf buf; 
	s->get_error_text(buf, sizeof buf); 
	console::message(msg_login|msg_error|msg_time, 
			 "Handshake with server %d failed: %s\n", id, buf);
    }	
    return False;
}

//
// Local storage server
//


void task_proc storage_server::start_handshake(void* arg)
{
    login_data* login = (login_data*)arg;
    login->server->handshake(login);
}


void storage_server::handshake(login_data* login)
{
    dbs_request req, reply;

    if (login->sock->read(&req, sizeof req)) { 
	req.unpack();
	if (req.cmd != dbs_request::cmd_login 
	    && req.cmd != dbs_request::cmd_connect) 
	{ 
	    console::message(msg_error|msg_time, 
			     "Bad request received while handshake: %d\n", 
			     req.cmd); 
	    reply.cmd = dbs_request::cmd_bye;
	    login->sock->write(&reply, sizeof reply);
	} else if (req.login.name_len >= MAX_LOGIN_NAME) {
	    console::message(msg_error|msg_time, "Login name too long: %d",
			     req.login.name_len); 
	    reply.cmd = dbs_request::cmd_bye;
	    login->sock->write(&reply, sizeof reply);
	} else { 		    
	    char name[MAX_LOGIN_NAME];
	    if (login->sock->read(name, req.login.name_len)) { 
		name[req.login.name_len] = '\0';	
		if (!authorize(req, name)) { 
		    console::message(msg_login|msg_error|msg_time,
				     "Authorization of %s '%s' failed\n", 
				     (req.cmd == dbs_request::cmd_connect)
				     ? "server" : "client", name);
		    reply.cmd = dbs_request::cmd_refused;
		    login->sock->write(&reply, sizeof reply);
		} else { 
		    reply.cmd = dbs_request::cmd_ok;
		    login->sock->write(&reply, sizeof reply);
		    if (req.cmd == dbs_request::cmd_connect) {
			if (req.login.sid == id) { 
			    console::message(msg_error|msg_time, 
					     "Attempt to start duplicated "
					     "server %d\n", id);
			} else if (req.login.sid >= n_servers) { 
			    console::message(msg_error|msg_time, 
					     "Server %d is not in "
					     "configuration file\n", 
					     req.login.sid);
			} else { 
			    console::message(msg_login|msg_time, 
					     "Establish connection with"
					     " server %d: \"%s\"\n", 
					     req.login.sid, name);
			    //
			    // Server agent "connect" method will wait until
			    // previous connection is terminated. So only
			    // one task per server_agent exists at each moment
			    // of time.
			    //
			    servers[req.login.sid]->connect(login->sock, name);
			    cs.enter();
			    login->sock = NULL; // do not delete
			    delete login;
			    if (!opened && handshake_list.empty()) { 
				term_event.signal();
			    }
			    cs.leave();
			    servers[req.login.sid]->poll();
			    return; 
			}
		    } else { 
			client_agent* client = create_client_agent();
			console::message(msg_login|msg_time, 
					 "Open session for client '%s'\n",
					 name);
			cs.enter();
			client->link_after(&clients); 
			client->connect(login->sock, name);
			login->sock = NULL; // do not delete
			delete login;
			if (!opened && handshake_list.empty()) { 
			    term_event.signal();
			}
			cs.leave();
			client->poll();
			return; 
		    }
		}
	    }
	}
    } else { 
	if (opened) {
	    msg_buf buf; 
	    login->sock->get_error_text(buf, sizeof buf);
	    console::message(msg_error|msg_time, 
			     "Handshake failed: %s\n", buf);
	}
    }
    cs.enter();
    delete login;
    if (!opened && handshake_list.empty()) { 
	term_event.signal();
    }
    cs.leave();
}

void storage_server::accept(socket_t* gateway)
{ 
    msg_buf buf; 
    while (opened) { 
	socket_t* new_sock = gateway->accept();
	if (new_sock != NULL) { 
	    if (!opened) { 
		delete new_sock;
		break;
	    }
	    login_data* login = new login_data(this, new_sock);
	    cs.enter();
	    login->link_after(&handshake_list);
	    cs.leave();
	    communication::create_receiver(start_handshake, login);
	} else { 	
	    if (opened) { 
		gateway->get_error_text(buf, sizeof buf);
		console::message(msg_error|msg_time, 
				 "Failed to accept socket: %s\n", buf);
	    }
	}
    }
    cs.enter();
    if (--n_opened_gateways == 0) { 
	term_event.signal();
    }
    cs.leave();
}

void storage_server::remove_client_agent(client_agent* agent)
{
    cs.enter();
    delete agent;
    if (!opened && clients.empty()) { 
	term_event.signal();
    }
    cs.leave();
}

void storage_server::remove_server_agent(server_agent* agent)
{
    cs.enter();
    TRACE_MSG((msg_important, "Server '%s' is now offline\n", 
	       agent->get_name()));
    n_online_remote_servers -= 1;
    if (!opened && n_online_remote_servers == 0) { 
	term_event.signal();
    }
    cs.leave();
}

server_agent* storage_server::create_server_agent(int id)
{ 
    return new server_agent(this, id);
}					

client_agent* storage_server::create_client_agent()
{ 
    return new client_agent(this, ++client_id, object_cluster_size_limit);
}					

void storage_server::send(sid_t sid, dbs_request* req, size_t body_len)
{
    servers[sid]->send(req, body_len);
}

void storage_server::read_msg_body(sid_t sid, void* buf, size_t size)
{ 
    servers[sid]->read_msg_body(buf, size);
}

void task_proc storage_server::start_local_gatekeeper(void* arg)
{
    storage_server* server = (storage_server*)arg; 
    server->accept(server->local_gateway);
}

void task_proc storage_server::start_global_gatekeeper(void* arg)
{
    storage_server* server = (storage_server*)arg; 
    server->accept(server->global_gateway);
}

boolean storage_server::open(char const* database_configuration_file)
{
    int  i; 
    socket_t* sock;
    char buf[MAX_CFG_FILE_LINE_SIZE];
    msg_buf err;

    FILE* cfg = fopen(database_configuration_file, "r");

    if (cfg == NULL) { 
	console::message(msg_error, 
			 "Failed to open database configuration file '%s'\n", 
			 database_configuration_file);
	return False;
    }
    if (fgets(buf,sizeof buf,cfg) == NULL || sscanf(buf,"%d",&n_servers) != 1)
    {
	console::message(msg_error, "Bad format of configuration file '%s'\n",
			 database_configuration_file);
	fclose(cfg);
	return False;
    }
    if (id >= n_servers) { 
	console::message(msg_error, "Can't open server %d since configuration"
			 "file contains only %d entries\n",
			 id, n_servers);
	fclose(cfg);
	return False;
    }
	
    servers = new server_agent*[n_servers];

    for (i = 0; i < n_servers; i++) { 
	if (i != id) { 
	    servers[i] = create_server_agent(i); 
	} else { 
	    servers[i] = NULL;
	}
    }
    *name = '\0';
    n_opened_gateways = 0;
    n_online_remote_servers = 0;
    backup_started = False;	
    global_gateway = NULL;
    local_gateway = NULL;
    clients.prune();	
    handshake_list.prune();	

    while (fgets(buf, sizeof buf, cfg)) { 
	char hostname[MAX_CFG_FILE_LINE_SIZE];
	int j;
	if (sscanf(buf, "%d:%s", &j, hostname) == 2 && j == id) {
	    strcpy(name, hostname);
	    break;
	}
    }
    if (*name == '\0') { 
	console::message(msg_error, 
			 "Local server %d name is not defined in configuration"
			 " file '%s'\n", id, database_configuration_file);
	fclose(cfg);
	return False; 
    }
    fseek(cfg, 0, 0); // seek to the beginning of the file 

    while (fgets(buf, sizeof buf, cfg)) { 
	char hostname[MAX_CFG_FILE_LINE_SIZE];
	int j;
	if (sscanf(buf, "%d:%s", &j, hostname) == 2) { 
	    if (strlen(hostname) >= MAX_LOGIN_NAME) { 
		console::message(msg_error, 
				 "Server name too long: '%s'\n", hostname);
	    } else { 
		if (j < id) { 
		    sock = socket_t::connect(hostname); 
		    if (sock->is_ok()) { 
			if (servers[j]->handshake(sock, id, name)) { 
			    servers[j]->connect(sock, hostname, True);
			    continue;
			}
		    } else {  
			sock->get_error_text(err, sizeof err);
			console::message(msg_error, 
					 "Failed to connect server '%s': %s\n",
					 hostname, err);
		    }
		    delete sock;
		}
	    }
	}
    }
    fclose(cfg);

    if (!pool_mgr->open(this)  || 
	!mem_mgr->open(this)   ||
	!obj_mgr->open(this)   ||
	!trans_mgr->open(this) ||
	!class_mgr->open(this))
    {
	close();
	return False;
    }
    opened = True;

    pool_mgr->initialize();
    mem_mgr->initialize(); 
    obj_mgr->initialize(); 
    trans_mgr->initialize();
    class_mgr->initialize();

    client_id = 0;
    global_gateway = socket_t::create_global(name); 
    if (global_gateway) { 
	if (global_gateway->is_ok()) { 
	    n_opened_gateways = 1;
	    task::create(start_global_gatekeeper, this);
	} else { 
	    global_gateway->get_error_text(err, sizeof err);
	    console::message(msg_error, 
			     "Failed to open global accept socket: %s\n",err);
	    delete global_gateway;
	    global_gateway = NULL; 
	}
    }
    local_gateway = socket_t::create_local(name); 
    if (local_gateway) { 
	if (local_gateway->is_ok()) { 
	    cs.enter();
	    n_opened_gateways += 1;
	    cs.leave();
	    task::create(start_local_gatekeeper, this);
	} else { 
	    local_gateway->get_error_text(err, sizeof err);
	    console::message(msg_error, 
			     "Failed to open local accept socket: %s\n", err);
	    delete local_gateway;
	    local_gateway = NULL; 
	}
    }
    if (n_opened_gateways == 0) { 
	console::message(msg_error, "Failed to create gateways\n");
	close();
	return False;
    }
    return True;    
}

void storage_server::close()
{
    cs.enter();
    opened = False;

    if (local_gateway != NULL) { 
	local_gateway->cancel_accept();
    }
    if (global_gateway != NULL) {
	global_gateway->cancel_accept();
    }
    while (n_opened_gateways != 0) {
	TRACE_MSG((msg_important, 
		   "Close server: number of opened gateways: %d\n",
		   n_opened_gateways));
	term_event.reset();
	term_event.wait();
    }
	
    login_data* login = (login_data*)handshake_list.next;
    while (login != &handshake_list) { 
	login_data* next = (login_data*)login->next;
	login->sock->shutdown();
	login = next;
    }
    while (!handshake_list.empty()) { 
	TRACE_MSG((msg_important, 
		   "Close server: waiting pending handshakes\n"));
	term_event.reset();
	term_event.wait();
    }

    client_agent* client = (client_agent*)clients.next; 
    while (client != &clients) { 
	client_agent* next = (client_agent*)client->next;
	client->disconnect();
	client = next;
    }

    cs.leave();
    pool_mgr->shutdown();
    mem_mgr->shutdown();
    obj_mgr->shutdown();
    trans_mgr->shutdown();
    class_mgr->shutdown();
    cs.enter();

    while (!clients.empty()) {  
	TRACE_MSG((msg_important, 
		   "Close server: waiting until all clients will "
		   "be disconnected\n"));
	term_event.reset();
	term_event.wait();
    }

    if (backup_started) { 
	console::message(msg_notify, 
			 "Waiting for termination of current backup\n"); 
	backup_finished_event.reset();
	backup_finished_event.wait();
	console::message(msg_notify, "Backup terminated\n"); 
    }

    if (servers != NULL) { 
	for (int i = 0; i < n_servers; i++) { 
	    if (i != id) { 
		servers[i]->disconnect();
	    }
	}
    }	     

    while (n_online_remote_servers != 0)
    {
	TRACE_MSG((msg_important, "Close server: disconnect remote servers, "
		   "n_online_remote_servers=%d\n", n_online_remote_servers));
	term_event.reset();
	term_event.wait();
    }
    cs.leave();
    //
    // At this moment all server threads are terminated and no new
    // one can be created since gateways are closed
    //
    TRACE_MSG(( msg_important, "Close managers...\n"));
    delete local_gateway;
    delete global_gateway;
    obj_mgr->close(); 
    class_mgr->close();
    mem_mgr->close(); 
    pool_mgr->close();  
    trans_mgr->close(); 

    if (servers != NULL) {
	for (int i = 0; i < n_servers; i++) { 
	    if (i != id) { 
		delete servers[i]; 
	    }
	} 
	delete[] servers; 
	servers = NULL; 
    } 	    
    TRACE_MSG((msg_important, "Database shutdown completed...\n"));
}

unsigned storage_server::get_number_of_servers() const 
{
    return n_servers; 
}

const char* storage_server::get_name() const 
{
    return name; 
}

void storage_server::remote_server_connected(sid_t sid) 
{
    //
    // Restart garbage collection each time some server is rebooted 
    // and connected to coordinator or when server reestablish connection with 
    // coordinator
    //
    if (id == GC_COORDINATOR || sid == GC_COORDINATOR) { 
	mem_mgr->gc_abort(False);
    }
    cs.enter();
    n_online_remote_servers += 1;
    cs.leave();
}

unsigned storage_server::get_oldest_client_id()
{
    cs.enter();
    unsigned id = clients.empty() ? 0 : ((client_agent*)clients.prev)->id; 
    cs.leave();
    return id;
}

void storage_server::dump(char* what) 
{
    cs.enter();
    if (opened) { 
	if (strstr(what, "server")) { 
	    for (int i = 0; i < n_servers; i++) { 
		if (i != id) { 
		    if (servers[i] != NULL) { 
			console::output("Remote server %d: '%s' - %s\n", 
					 i, servers[i]->get_name(), 
					 servers[i]->is_online() 
					 ? "online" : "offline");
		    }
		} else { 
		    console::output("Local server %d: '%s'\n", i, name);
		}
	    }    
	    if (backup_started) { 
		console::output("Active backup process to file '%s'\n",
				backup_file->get_name());
	    }
	}
	if (strstr(what, "client")) { 
	    console::output("Attached clients:\n");
	    for (client_agent* agent = (client_agent*)clients.next;
		 agent != &clients;
		 agent = (client_agent*)agent->next)
            {
		obj_mgr->dump(agent, what);
	    }
	}
	if (strstr(what, "transaction")) { 
	    trans_mgr->dump(what);
	} 
	if (strstr(what, "memory")) { 
	    mem_mgr->dump(what);
	    obj_mgr->dump(what);
	    pool_mgr->dump(what);
	}
	if (strstr(what, "class")) { 
	    class_mgr->dump(what);
	}
    } else { 
	console::output("Server not opened");
    }
    cs.leave();
}

void task_proc storage_server::start_backup_process(void* arg)
{
    ((storage_server*)arg)->backup();
}


boolean storage_server::backup()
{
    boolean result = trans_mgr->backup(*backup_file, 
				       backup_start_delay, 
				       backup_start_log_size);
    cs.enter();
    backup_started = False;
    backup_finished_event.signal();
    cs.leave();
    if (backup_callback) { 
	(*backup_callback)(*this, *backup_file, result);
    } 
    return result;
}

void storage_server::start_backup(file&   backup_file, 
				  time_t  backup_start_delay,
				  fsize_t backup_start_log_size,
				  backup_finish_callback backup_callback)
{
    cs.enter();
    if (backup_started || !opened) { 
	cs.leave();
	return;
    }
    this->backup_file = &backup_file;
    this->backup_start_delay = backup_start_delay;
    this->backup_start_log_size = backup_start_log_size;
    this->backup_callback = backup_callback;
    backup_started = True;

    task::create(start_backup_process, this, task::pri_background); 
    cs.leave();
}

void storage_server::stop_backup()
{
    cs.enter();
    assert(opened && backup_started);
    backup_started = False;
    trans_mgr->stop_backup();
    cs.leave();
}

boolean storage_server::restore(file& backup_file,
				const char* database_config_file)
{
    assert(!opened);
    return trans_mgr->restore(backup_file)
	&& open(database_config_file);
}

void storage_server::notify_clients()
{
    cs.enter();
    for (l2elem* agent = clients.next; agent != &clients; agent = agent->next)
    {
	((client_agent*)agent)->notify();
    }
    cs.leave();
}

void storage_server::set_object_cluster_size_limit(size_t cluster_size)
{
    object_cluster_size_limit = cluster_size;	 
}

boolean storage_server::authorize(dbs_request const& /*req*/, char* /*name*/) 
{
    return True; // everybody are welcome
}

storage_server::storage_server(sid_t                  sid,
			       object_access_manager& omgr, 
			       pool_manager&          pmgr,
			       class_manager&         cmgr,
			       memory_manager&        mmgr,
			       transaction_manager&   tmgr,
 			       size_t                 object_cluster_size)
: dbs_server(sid), term_event(cs), backup_finished_event(cs)
{ 
    servers = NULL; 
    n_servers = 0;

    obj_mgr   = &omgr;
    mem_mgr   = &mmgr; 
    pool_mgr  = &pmgr;
    class_mgr = &cmgr; 
    trans_mgr = &tmgr; 

    opened = False;
    local_gateway = global_gateway = NULL; 
    object_cluster_size_limit = object_cluster_size; 
}     

