//-< MOP.H >---------------------------------------------------------*--------*
// GOODS                     Version 1.0         (c) 1997  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
//                                                                   *  /  \  *
//                          Created:      7-Jan-97    K.A. Knizhnik  * / [] \ *
//                          Last update: 17-Nov-97    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Metaobjects are used to specify synchronization policy for GOODS objects
//-------------------------------------------------------------------*--------*

#ifndef __MOP_H__
#define __MOP_H__

//
// Methods of metaobject are called with global mutex locked so 
// thier execution is not concurrent 
//

class metaobject { 
  public: 
    //
    // Make transient object persistent as a result of 
    // referencing to it from persistent object
    //
    virtual void make_persistent(hnd_t hnd, hnd_t parent_hnd) = 0; 

    //
    // This method is called by object destructor and performs cleanup of
    // deleted object. 
    //
    virtual void destroy(hnd_t hnd) = 0;

    //
    // Intertasking object locking
    //
    virtual void lock(hnd_t hnd) = 0;
    virtual void unlock(hnd_t hnd) = 0;
    
    //
    // Intertasking synchronization primitives
    // 
    virtual void wait(hnd_t hnd) = 0;
    virtual boolean wait(hnd_t hnd, time_t timeout) = 0;
    virtual void notify(hnd_t hnd) = 0;

    //
    // Access to object
    //
    virtual void begin_read(hnd_t hnd) = 0;
    virtual void end_read(hnd_t hnd) = 0;

    virtual void begin_write(hnd_t hnd) = 0;
    virtual void end_write(hnd_t hnd) = 0;

    //
    // Instance of object is deteriorated
    //
    virtual void invalidate(hnd_t hnd) = 0;

    //
    // Signal event when persistent object is modified by another client
    //
    virtual void signal_on_modification(hnd_t hnd, event& e) = 0;

    //
    // Insert persistent object to object cache. Replacement policy 
    // for objects should be implemented by this method
    //
    virtual void put_in_cache(hnd_t hnd) = 0;
    //
    // Prevent object from replacement algorithm activity.
    //
    virtual void get_from_cache(hnd_t hnd) = 0;

    //
    // Specify boundaries of nested transaction. Parent transaction will 
    // not be commited until all nested transactions are finished. 
    //
    virtual void begin_transaction() = 0;
    virtual void end_transaction() = 0;

    //
    // Transaction management 
    //
    enum abort_reason { aborted_by_server, aborted_by_user };

    virtual void commit_transaction() = 0;
    virtual void abort_transaction(const database* dbs, 
				   abort_reason reason = aborted_by_user) = 0;

    //
    // This methods are called after transaction completion 
    // (normal or abnormal) for each object involved in transaction
    //
    virtual void commit_object_changes(hnd_t hnd) = 0;
    virtual void abort_object_changes(hnd_t hnd, abort_reason reason) = 0;

    //
    // This method is called for all object from transaction list 
    // when no changes were made within transaction
    // 
    virtual void release_transaction_object(hnd_t hnd) = 0;

    //
    // Specify flags for transaction object
    //
    virtual int get_transaction_object_flags(hnd_t hnd) = 0;

  protected: 
    inline object_header* get_header(hnd_t hnd) { 
	return (object_header*)hnd->obj;
    }

    //
    // Insert object in database transaction list 
    //
    virtual void insert_in_transaction_list(hnd_t hnd, 
					    hnd_t parent_hnd = 0) = 0;
    //
    // Remove object from database transaction list 
    //
    virtual void remove_from_transaction_list(hnd_t hnd) = 0;
}; 

//
// Loaded persistent objects are linked either in l2-list of cached object
// or in l2-list of objects participated in transaction
//
class object_l2_list { 
  public: 
    hnd_t head; 
    hnd_t tail; 

    void put_at_head(object_header* obj) { 
	obj->next = head; 
	obj->prev = 0;
	if (head != 0) { 
	    head->obj->prev = obj->hnd; 
	} else { 
	    tail = obj->hnd; 
	}
	head = obj->hnd; 
    }

    void put_at_tail(object_header* obj) { 
	obj->next = 0; 
	obj->prev = tail;
	if (tail != 0) { 
	    tail->obj->next = obj->hnd; 
	} else { 
	    head = obj->hnd; 
	}
	tail = obj->hnd; 
    }

    void put_after(object_header* after, object_header* obj) { 
	obj->next = after->next;
	obj->prev = after->hnd;
	after->next = obj->hnd;
	if (obj->next != 0) { 
	    obj->next->obj->prev = obj->hnd;
	} else { 
	    tail = obj->hnd;
	}
    }

    void put_before(object_header* before, object_header* obj) { 
	obj->next = before->hnd;
	obj->prev = before->prev;
	before->prev = obj->hnd;
	if (obj->prev != 0) { 
	    obj->prev->obj->next = obj->hnd;
	} else { 
	    head = obj->hnd;
	}
    }

    void unlink(object_header* obj) { 
	if (obj->next) { 
	    obj->next->obj->prev = obj->prev; 
	} else { 
	    tail = obj->prev; 
	}
	if (obj->prev) { 
	    obj->prev->obj->next = obj->next; 
	} else { 
	    head = obj->next; 
	}
	obj->next = obj->prev = 0;
    }

    boolean empty() { return head == 0; }

    void truncate() { head = tail = 0; }

    object_l2_list() { truncate(); }
};

//
// Class providing mutual exclusion for access to object. 
// Since size of 'monitor' object is not very small,
// allocating monitor for each object will be space expansive.
// So "turnstile" of monitor objects is used for more space-effective monitor
// allocation.
//

class object_monitor { 
    friend  class monitor_turnstile;
  protected: 
    static mutex* global_cs; 

    mutex*      cs;
    semaphorex* sem;
    hnd_t       hnd;
    int         busy;
    int         signaled;
    int         n_nested_locks;
    task*       owner;

  public:
    //
    // Synchronize accesses to object system
    //
    inline static void lock_global()   { global_cs->enter(); }
    inline static void unlock_global() { global_cs->leave(); }

    void lock();
    void unlock(); 

    void wait();
    boolean wait(time_t timeout);
    void notify();

    void attach(hnd_t new_owner);
    void detach();

    void init();
    void destroy();
}; 
    
#define INIT_TURNSTILE_SIZE 1024

class monitor_turnstile { 
  private: 
    int             pos;
    int             size; 
    object_monitor* turnstile; 
  public: 
    int  attach_monitor(hnd_t hnd); 
    void lock(int monitor);
    void unlock(int monitor);
    void wait(int monitor);
    boolean wait(int monitor, time_t timeout);
    void notify(int monitor);
    void detach_monitor(int monitor);

    monitor_turnstile(int init_size = 1024);
    ~monitor_turnstile();
}; 

//
// Metaobject supporting basic transaction and cache management functionality
//

class basic_metaobject : public metaobject { 
  protected: 
    //
    // Lists of cached objects sorted in LRU order
    // Two separate list are used: 
    //  one for objects accessed one or zero times
    //  and one for objects accessed more than once(with 'useful' bit in state)
    // Objects from first queue are replaced first. 
    static object_l2_list lru[2]; 
    //
    // List of objects accessed within current transaction
    //
    static object_l2_list trans_objects; 
    //
    // Total size of loaded objects. 
    // Sizes are calculated separately for objects accessed 0 or 1 times and 
    // for objects acccessed more than once. 
    //
    static size_t         used_cache_size[2];
    //
    // Limit for total size of objects in cache
    // Limits are specified separately for objects accessed 0 or 1 times and 
    // for objects acccessed more than once. 
    //
    static size_t         cache_size_limit[2];

    typedef void (*abort_hook_t)(metaobject*); 
    abort_hook_t abort_hook;

    static event          trans_commit_event;
    static boolean        trans_commit_wait_flag;
    static boolean        trans_commit_in_progress;   

    static monitor_turnstile turnstile;

    struct notify_item { 
        notify_item* chain;
        hnd_t       hnd;
        event&       e;
	notify_item(hnd_t id, event& evt, notify_item* list)
	: chain(list), hnd(id), e(evt) {}
    };
    static notify_item* notification_hash[];

    //
    // This virtual methods can be used for metaobject specific 
    // transaction object lists. Method 'commit_transaction' 
    // tries to commit transactions in all databases.
    // Methos "abort_transaction" abort transaction in concrete database.
    //
    virtual void commit_transaction(object_l2_list& trans_objects);
    virtual void abort_transaction(const database* dbs,
				   object_l2_list& trans_objects,
				   abort_reason reason);

    virtual void insert_in_transaction_list(hnd_t hnd, 
					    hnd_t parent_hnd = 0);
    virtual void remove_from_transaction_list(hnd_t hnd);

    virtual void reload_object(hnd_t hnd);
    virtual void update_object(hnd_t hnd);
    
  public: 
    virtual void destroy(hnd_t hnd);

    virtual void lock(hnd_t hnd);
    virtual void unlock(hnd_t hnd);
    
    virtual void wait(hnd_t hnd);
    virtual boolean wait(hnd_t hnd, time_t timeout);
    virtual void notify(hnd_t hnd);

    virtual void make_persistent(hnd_t hnd, hnd_t parent_hnd); 

    virtual void invalidate(hnd_t hnd);

    virtual void signal_on_modification(hnd_t hnd, event& e);

    virtual void put_in_cache(hnd_t hnd);
    virtual void get_from_cache(hnd_t hnd);

    virtual void commit_transaction();
    virtual void abort_transaction(const database* dbs, abort_reason reason);

    //
    // Primary cache size (limit0) should be greater then maximal cluster size 
    // otherwise sluster can never be loaded
    //
    static void  set_cache_limit(size_t limit0, size_t limit1) { 
	cache_size_limit[0] = limit0; 
	cache_size_limit[1] = limit1; 
    }    
    //
    // Set hook which will be called if transaction will be aborted by server.
    // Hook is called with global mutex locked and should not release this lock
    //
    void set_abort_transaction_hook(abort_hook_t hook) { abort_hook = hook; }

    basic_metaobject();
};
    

//
// Optimistic approach: objects are not locked before access, instead
// of this modified object instances are checked at server while transaction 
// commit
//
class optimistic_metaobject : public basic_metaobject { 
  protected: 
    static unsigned n_nested_transactions; 

  public: 
    virtual void begin_read(hnd_t hnd);
    virtual void end_read(hnd_t hnd);

    virtual void begin_write(hnd_t hnd);
    virtual void end_write(hnd_t hnd);

    virtual void begin_transaction();
    virtual void end_transaction();
    
  protected: 
    virtual int  get_transaction_object_flags(hnd_t hnd);

    virtual void commit_object_changes(hnd_t hnd);
    virtual void abort_object_changes(hnd_t hnd, abort_reason reason); 

    virtual void release_transaction_object(hnd_t hnd);
};
    
extern optimistic_metaobject optimistic_scheme; 

//
// Optimistic approach providing repatable read: 
// While transaction commit server checks uptodate status not only of 
// modified objects but of all objects accessed within transaction
//
class optimistic_repeatable_read_metaobject : public optimistic_metaobject { 
  public: 
    virtual void begin_read(hnd_t hnd);
}; 

extern optimistic_repeatable_read_metaobject 
    optimistic_repeatable_read_scheme;


//
// Metaobjects providing modification correctness (changes can't be lost)
// Conflicts are also possible in that scheme since readonly accessed
// objects are not locked. Conflict can happen if invalidated object 
// is accessed for modification and it is impossible to reload this object 
// because there are some active invokations of methods for this object. 
//
class pessimistic_metaobject : public optimistic_metaobject { 
  protected: 
    int lck_attr; // attributes of locking (lck_nowait,...)

    virtual boolean storage_lock(hnd_t hnd, lck_t lck); 

    //
    // This function is called when modification method was called from 
    // readonly method and object found to be deteriorated after locking. 
    // So it is impossible to update object instance (as there are active
    // invokations for this object). If this method returns 'True'
    // then version of object in databsae will be overwritten by
    // local object instance after commite of transaction. 
    // If method returns 'False' transaction will be aborted. 
    //
    virtual boolean solve_write_conflict(hnd_t hnd);

    //
    // This method is called when lock request is failed
    //
    virtual boolean solve_lock_conflict(hnd_t hnd, lck_t lck);

    virtual int     get_transaction_object_flags(hnd_t hnd);

    virtual void    release_transaction_object(hnd_t hnd);
     
    virtual void    commit_object_changes(hnd_t hnd);
    virtual void    abort_object_changes(hnd_t hnd, abort_reason reason); 

  public: 
    virtual void    begin_write(hnd_t hnd);
    
    pessimistic_metaobject(int attr = 0) { lck_attr = attr; }
};

extern pessimistic_metaobject pessimistic_scheme;

extern pessimistic_metaobject nowait_pessimistic_scheme;


//
// Lazy pessimistic metaobject. This metaobject combines
// characterstics of both optimistic and pessimistic protocols. 
// Lock is set after object has been modified.
//

class lazy_pessimistic_metaobject : public pessimistic_metaobject { 
  protected: 
    virtual boolean solve_write_conflict(hnd_t hnd);

  public: 
    virtual void begin_write(hnd_t hnd);
    virtual void end_write(hnd_t hnd);

    lazy_pessimistic_metaobject(int attr = 0) 
    : pessimistic_metaobject(attr) {}
};

extern lazy_pessimistic_metaobject lazy_pessimistic_scheme;

extern lazy_pessimistic_metaobject lazy_nowait_pessimistic_scheme;


//
// Pessimistic metaobject enforing repeatable read policy 
//

class pessimistic_repeatable_read_metaobject : public pessimistic_metaobject { 
  protected:
    lck_t read_lock;
  public: 
    virtual void begin_read(hnd_t hnd);

    pessimistic_repeatable_read_metaobject(int attr = 0, 
					   lck_t read_lock_mode = lck_shared) 
    : pessimistic_metaobject(attr) { this->read_lock = read_lock_mode; }
}; 

extern pessimistic_repeatable_read_metaobject
    pessimistic_repeatable_read_scheme;

extern pessimistic_repeatable_read_metaobject 
    pessimistic_exclusive_scheme;

extern pessimistic_repeatable_read_metaobject 
    nowait_pessimistic_repeatable_read_scheme;

extern pessimistic_repeatable_read_metaobject 
    nowait_pessimistic_exclusive_scheme;

//
// Pessimistic metaobject for child objects accessed always through parent 
// object. So locking parent object effectivly syncronize access to child
// and no explicit child locking is required (parent and child
// should be located in the same storage, otherwise we can access
// deteriorated instance of child object).
//
class hierarchical_access_metaobject : public optimistic_metaobject { 
  public:
    int  get_transaction_object_flags(hnd_t);
};

extern hierarchical_access_metaobject hierarchical_access_scheme;

#endif
