//-< Query.java >----------------------------------------------------*--------*
// GOODS                      Version 2.04       (c) 1998  GARRET    *     ?  *
// (Generic Object Oriented Database System)                         *   /\|  *
// GOODS Persistent Class Library                                    *  /  \  *
//                          Created:      5-Mar-99    K.A. Knizhnik  * / [] \ *
//                          Last update:  6-Mar-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Java embedded SQL
//-------------------------------------------------------------------*--------*

package goodslib;

import  goodsjpi.*;
import  java.lang.reflect.*;
import  java.util.Hashtable;

class Node { 
    int type;
    int tag;

    final static int tpBool          = 0;
    final static int tpInt           = 1;
    final static int tpReal          = 2;
    final static int tpFreeVar       = 3;
    final static int tpList          = 4;
    final static int tpObj           = 5;
    final static int tpStr           = 6;
    final static int tpArrayBool     = 7;
    final static int tpArrayChar     = 8;
    final static int tpArrayInt1     = 9;
    final static int tpArrayInt2     = 10;
    final static int tpArrayInt4     = 11;
    final static int tpArrayInt8     = 12;
    final static int tpArrayReal4    = 13;
    final static int tpArrayReal8    = 14;
    final static int tpArrayStr      = 15;
    final static int tpArrayObj      = 16;
    final static int tpDnmArrayBool  = 17;
    final static int tpDnmArrayChar  = 18;
    final static int tpDnmArrayInt1  = 19;
    final static int tpDnmArrayInt2  = 20;
    final static int tpDnmArrayInt4  = 21;
    final static int tpDnmArrayInt8  = 22;
    final static int tpDnmArrayReal4 = 23;
    final static int tpDnmArrayReal8 = 24;
    final static int tpDnmArrayStr   = 25;
    final static int tpDnmArrayObj   = 26;



    final static int opNop     = 0;
    final static int opIntAdd  = 1;
    final static int opIntSub  = 2;
    final static int opIntMul  = 3;
    final static int opIntDiv  = 4;
    final static int opIntAnd  = 5;
    final static int opIntOr   = 6;
    final static int opIntNeg  = 7;
    final static int opIntNot  = 8;
    final static int opIntAbs  = 9;
    final static int opIntPow  = 10;
    final static int opIntEq   = 11;
    final static int opIntNe   = 12;
    final static int opIntGt   = 13;
    final static int opIntGe   = 14;
    final static int opIntLt   = 15;
    final static int opIntLe   = 16;
    final static int opIntBetween = 17;

    final static int opRealEq   = 18;
    final static int opRealNe   = 19;
    final static int opRealGt   = 20;
    final static int opRealGe   = 21;
    final static int opRealLt   = 22;
    final static int opRealLe   = 23;
    final static int opRealBetween = 24;

    final static int opStrEq   = 25;
    final static int opStrNe   = 26;
    final static int opStrGt   = 27;
    final static int opStrGe   = 28;
    final static int opStrLt   = 29;
    final static int opStrLe   = 30;
    final static int opStrBetween = 31;
    final static int opStrLike    = 32;
    final static int opStrLikeEsc = 33;

    final static int opBoolEq   = 34;
    final static int opBoolNe   = 35;
    
    final static int opObjEq   = 36;
    final static int opObjNe   = 37;
    
    final static int opRealAdd = 38;
    final static int opRealSub = 39;
    final static int opRealMul = 40;
    final static int opRealDiv = 41;
    final static int opRealNeg = 42;
    final static int opRealAbs  = 43;
    final static int opRealPow  = 44;

    final static int opIntToReal = 45;
    final static int opRealToInt = 46;
    final static int opIntToStr = 47;
    final static int opRealToStr = 48;

    final static int opIsNull = 49;

    final static int opStrGetAt  = 50;
    final static int opGetAtBool = 51;
    final static int opGetAtChar = 52;
    final static int opGetAtInt1 = 53;
    final static int opGetAtInt2 = 54;
    final static int opGetAtInt4 = 55;
    final static int opGetAtInt8 = 56;
    final static int opGetAtReal4 = 57;
    final static int opGetAtReal8 = 58;
    final static int opGetAtStr = 59;
    final static int opGetAtObj = 60;

    final static int opDnmGetAtBool = 61;
    final static int opDnmGetAtChar = 62;
    final static int opDnmGetAtInt1 = 63;
    final static int opDnmGetAtInt2 = 64;
    final static int opDnmGetAtInt4 = 65;
    final static int opDnmGetAtInt8 = 66;
    final static int opDnmGetAtReal4 = 67;
    final static int opDnmGetAtReal8 = 68;
    final static int opDnmGetAtObj = 69;

    final static int opStrLength = 70;
    final static int opLength = 71;
    final static int opDnmLength = 72;

    final static int opExists = 73;
    final static int opIndexVar = 74;

    final static int opFalse   = 75;
    final static int opTrue    = 76;
    final static int opNull    = 77;
    final static int opCurrent = 78;

    final static int opIntConst = 79;
    final static int opRealConst = 80;
    final static int opStrConst = 81;

    final static int opInvoke = 82;

    final static int opScanArrayBool = 83;
    final static int opScanArrayChar = 84;
    final static int opScanArrayInt1 = 85;
    final static int opScanArrayInt2 = 86;
    final static int opScanArrayInt4 = 87;
    final static int opScanArrayInt8 = 88;
    final static int opScanArrayReal4 = 89;
    final static int opScanArrayReal8 = 90;
    final static int opScanArrayStr = 91;
    final static int opScanArrayObj = 92;
    final static int opInString = 93;

    final static int opScanDnmArrayBool = 94;
    final static int opScanDnmArrayChar = 95;
    final static int opScanDnmArrayInt1 = 96;
    final static int opScanDnmArrayInt2 = 97;
    final static int opScanDnmArrayInt4 = 98;
    final static int opScanDnmArrayInt8 = 99;
    final static int opScanDnmArrayReal4 = 100;
    final static int opScanDnmArrayReal8 = 101;
    final static int opScanDnmArrayObj = 102;

    final static int opRealSin = 103;
    final static int opRealCos = 104;
    final static int opRealTan = 105;
    final static int opRealAsin = 106;
    final static int opRealAcos = 107;
    final static int opRealAtan = 108;
    final static int opRealSqrt = 109;
    final static int opRealExp = 110;
    final static int opRealLog = 111;
    final static int opRealCeil = 112;
    final static int opRealFloor = 113;

    final static int opBoolAnd = 114;
    final static int opBoolOr  = 115;
    final static int opBoolNot = 116;

    final static int opStrLower = 117;
    final static int opStrUpper = 118;
    final static int opStrConcat = 119;

    final static int opLoadBool = 120;
    final static int opLoadChar = 121;
    final static int opLoadInt1 = 122;
    final static int opLoadInt2 = 123;
    final static int opLoadInt4 = 124;
    final static int opLoadInt8 = 125;
    final static int opLoadReal4 = 126;
    final static int opLoadReal8 = 127;
    final static int opLoadStr = 128;
    final static int opLoadObj = 129;


    boolean evaluateBool(SearchThread t) { 
	throw new AbstractMethodError();
    }
    long    evaluateInt(SearchThread t) {
	throw new AbstractMethodError();
    }
    double  evaluateReal(SearchThread t) {
	throw new AbstractMethodError();
    }
    String  evaluateStr(SearchThread t) {
	throw new AbstractMethodError();
    }
    Object  evaluateObj(SearchThread t) {
	return evaluateStr(t);
    }
    Class   getType() { 
	return null;
    }
    
    public Node(int type, int tag) {
	this.type = type;
	this.tag = tag;
    }
}

class EmptyNode extends Node { 
    boolean evaluateBool(SearchThread t) { 
	return true;
    }

    EmptyNode() { 
	super(tpBool, opTrue);
    }
}


class IntLiteralNode extends Node { 
    long value;
    
    long evaluateInt(SearchThread t) {
	return value;
    }
    
    public int hashCode() {
	return (int)(value ^ (value >> 32));
    }

    IntLiteralNode(long value) { 
	super(tpInt, opIntConst);
	this.value = value;
    }
}

class RealLiteralNode extends Node { 
    double value;
    
    double evaluateReal(SearchThread t) {
	return value;
    }
    
    RealLiteralNode(double value) { 
	super(tpReal, opRealConst);
	this.value = value;
    }
}

class StrLiteralNode extends Node { 
    String value;
    
    String evaluateStr(SearchThread t) {
	return value;
    }
    
    StrLiteralNode(String value) { 
	super(tpStr, opStrConst);
	this.value = value;
    }
}

class ConstantNode extends Node { 
    Object evaluateObj(SearchThread t) {
	if (tag == opCurrent) { 
	    return t.currObj;
	}
	return null;
    }
    
    boolean evaluateBool(SearchThread t) { 
	return tag != opFalse;
    }

    ConstantNode(int type, int tag) { 
	super(type, tag);
    }
}

class IndexOutOfRangeError extends Error {
    int loopId;

    IndexOutOfRangeError(int loop) {
	loopId = loop;
    }
}

class ExistsNode extends Node {
    Node expr;
    int  loopId;
    
    boolean evaluateBool(SearchThread t) { 
	t.indexVar[loopId] = 0;
	try { 
	    while (!expr.evaluateBool(t)) { 
		t.indexVar[loopId] += 1;
	    }
	    return true;
	} catch (IndexOutOfRangeError x) { 
	    if (x.loopId != loopId) { 
		throw x;
	    }
	    return false;
	}
    }
    
    ExistsNode(Node expr, int loopId) { 
	super(tpBool, opExists);
	this.expr = expr;
	this.loopId = loopId;
    }
}


class IndexNode extends Node {
    int loopId;
    
    long evaluateInt(SearchThread t) { 
	return t.indexVar[loopId];
    }
    
    IndexNode(int loop) { 
	super(tpInt, opIndexVar);
	loopId = loop;
    }
}

class GetAtNode extends Node { 
    Node left;
    Node right;
    
    long evaluateInt(SearchThread t) { 
	Object arr = left.evaluateObj(t);
	int index = (int)right.evaluateInt(t);
	try {
	    switch (tag) { 
	      case opGetAtInt1:
		return ((byte[])arr)[index];
	      case opGetAtInt2:
		return ((short[])arr)[index];
	      case opGetAtInt4:
		return ((int[])arr)[index];
	      case opGetAtInt8:
		return ((long[])arr)[index];
	      case opGetAtChar:
		return ((char[])arr)[index];
	      case opStrGetAt:
		return ((String)arr).charAt(index);
		
	      case opDnmGetAtInt1:
		return ((ArrayOfByte)arr).getAt(index);
	      case opDnmGetAtInt2:
		return ((ArrayOfShort)arr).getAt(index);
	      case opDnmGetAtInt4:
		return ((ArrayOfInt)arr).getAt(index);
	      case opDnmGetAtInt8:
		return ((ArrayOfLong)arr).getAt(index);
	      case opDnmGetAtChar:
		return ((ArrayOfChar)arr).getAt(index);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch(ArrayIndexOutOfBoundsException x) { 
	    if (right.tag == opIndexVar) { 
		throw new IndexOutOfRangeError(((IndexNode)right).loopId);
	    }
	    throw x;
	}
    }	    

    double evaluateReal(SearchThread t) { 
	Object arr = left.evaluateObj(t);
	int index = (int)right.evaluateInt(t);
	try { 
	    switch (tag) { 
	      case opGetAtReal4:
		return ((float[])arr)[index];
	      case opGetAtReal8:
		return ((double[])arr)[index];
	      case opDnmGetAtReal4:
		return ((ArrayOfFloat)arr).getAt(index);
	      case opDnmGetAtReal8:
		return ((ArrayOfDouble)arr).getAt(index);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch(ArrayIndexOutOfBoundsException x) { 
	    if (right.tag == opIndexVar) { 
		throw new IndexOutOfRangeError(((IndexNode)right).loopId);
	    }
	    throw x;
	}
    }	    

    boolean evaluateBool(SearchThread t) { 
	Object arr = left.evaluateObj(t);
	int index = (int)right.evaluateInt(t);
	try { 
	    switch (tag) { 
	      case opGetAtBool:
		return ((boolean[])arr)[index];
	      case opDnmGetAtBool:
		return ((ArrayOfBoolean)arr).getAt(index);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch(ArrayIndexOutOfBoundsException x) { 
	    if (right.tag == opIndexVar) { 
		throw new IndexOutOfRangeError(((IndexNode)right).loopId);
	    }
	    throw x;
	}
    }

    String evaluateStr(SearchThread t) { 
	String[] arr = (String[])left.evaluateObj(t);
	long index = right.evaluateInt(t);

	if (right.tag == opIndexVar) { 
	    try { 
		if (index >= arr.length) {
		    throw new IndexOutOfRangeError(((IndexNode)right).loopId);
		}
	    } catch (IllegalArgumentException x) {
		throw new Error("Argument is not array");
	    }
	}
	return arr[(int)index];
    }

    Object evaluateObj(SearchThread t) { 
	Object arr = left.evaluateObj(t);
	int index = (int)right.evaluateInt(t);
	try { 
	    switch (tag) { 
	      case opGetAtObj:
		return ((Object[])arr)[index];
	      case opDnmGetAtObj:
		return ((ArrayOfObject)arr).getAt(index);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch(ArrayIndexOutOfBoundsException x) { 
	    if (right.tag == opIndexVar) { 
		throw new IndexOutOfRangeError(((IndexNode)right).loopId);
	    }
	    throw x;
	}
    }

    GetAtNode(int type, int tag, Node base, Node index) { 
	super(type, tag);
	left = base;
	right = index;
    }
}

class InvokeNode extends Node {
    Node target;
    Method mth;

    Class getType() { 
	return mth.getReturnType();
    }

    long evaluateInt(SearchThread t) {
	Object obj = target == null ? t.currObj : target.evaluateObj(t);
	try { 
	    return ((Number)mth.invoke(obj, null)).longValue();
	} catch (Exception x) { 
	    x.printStackTrace();
	    throw new Error("Method invocation error");
	}
    }

    double evaluateReal(SearchThread t) {
	Object obj = target == null ? t.currObj : target.evaluateObj(t);
	try { 
	    return ((Number)mth.invoke(obj, null)).doubleValue();
	} catch (Exception x) { 
	    x.printStackTrace();
	    throw new Error("Method invocation error");
	}
    }

    boolean evaluateBool(SearchThread t) {
	Object obj = target == null ? t.currObj : target.evaluateObj(t);
	try { 
	    return ((Boolean)mth.invoke(obj, null)).booleanValue();
	} catch (Exception x) { 
	    x.printStackTrace();
	    throw new Error("Method invocation error");
	}
    }

    String evaluateStr(SearchThread t) {
	Object obj = target == null ? t.currObj : target.evaluateObj(t);
	try { 
	    return (String)mth.invoke(obj, null);
	} catch (Exception x) { 
	    x.printStackTrace();
	    throw new Error("Method invocation error");
	}
    }

    Object evaluateObj(SearchThread t) {
	Object obj = target == null ? t.currObj : target.evaluateObj(t);
	try { 
	    return mth.invoke(obj, null);
	} catch (Exception x) { 
	    x.printStackTrace();
	    throw new Error("Method invocation error");
	}
    }    

    InvokeNode(int type, Method mth, Node target) { 
	super(type, opInvoke);
	this.target = target;
	this.mth = mth;
    }
}


class BinOpNode extends Node { 
    Node left;
    Node right;

    long evaluateInt(SearchThread t) {
	long lval = left.evaluateInt(t);
	long rval = right.evaluateInt(t);
	long res;
	switch (tag) { 
	  case opIntAdd:
	    return lval + rval;
	  case opIntSub:
	    return lval - rval;
	  case opIntMul:
	    return lval * rval;
	  case opIntDiv:
	    return lval / rval;
	  case opIntAnd:
	    return lval & rval;
	  case opIntOr:
	    return lval | rval;
	  case opIntPow:
	    res = 1;
	    if (rval < 0) {
		lval = 1/lval;
		rval = -rval;
	    }
	    while (rval != 0) {
		if ((rval & 1) != 0) { 
		    res *= lval;
		}
		lval *= lval;
		rval >>>= 1;
	    }
	    return res; 
	  default:
	    throw new Error("Invalid tag");
	}
    }

    double evaluateReal(SearchThread t) {
	double lval = left.evaluateReal(t);
	double rval = right.evaluateReal(t);
	switch (tag) { 
	  case opRealAdd:
	    return lval + rval;
	  case opRealSub:
	    return lval - rval;
	  case opRealMul:
	    return lval * rval;
	  case opRealDiv:
	    return lval / rval;
	  case opRealPow:
	    return Math.pow(lval, rval);
	  default:
	    throw new Error("Invalid tag");
	}
    }

    String evaluateStr(SearchThread t) {
	String lval = left.evaluateStr(t);
	String rval = right.evaluateStr(t);
	return lval + rval;
    }

    boolean evaluateBool(SearchThread t) {
	switch (tag) { 
	  case opBoolAnd:
	    return left.evaluateBool(t) && right.evaluateBool(t);
	  case opBoolOr:
	    return left.evaluateBool(t) || right.evaluateBool(t);

	  case opIntEq:
	    return left.evaluateInt(t) == right.evaluateInt(t);
	  case opIntNe:
	    return left.evaluateInt(t) != right.evaluateInt(t);
	  case opIntLt:
	    return left.evaluateInt(t) < right.evaluateInt(t);
	  case opIntLe:
	    return left.evaluateInt(t) <= right.evaluateInt(t);
	  case opIntGt:
	    return left.evaluateInt(t) > right.evaluateInt(t);
	  case opIntGe:
	    return left.evaluateInt(t) >= right.evaluateInt(t);

	  case opRealEq:
	    return left.evaluateReal(t) == right.evaluateReal(t);
	  case opRealNe:
	    return left.evaluateReal(t) != right.evaluateReal(t);
	  case opRealLt:
	    return left.evaluateReal(t) <  right.evaluateReal(t);
	  case opRealLe:
	    return left.evaluateReal(t) <= right.evaluateReal(t);
	  case opRealGt:
	    return left.evaluateReal(t) >  right.evaluateReal(t);
	  case opRealGe:
	    return left.evaluateReal(t) >= right.evaluateReal(t);

	  case opStrEq:
	    return left.evaluateStr(t).equals(right.evaluateStr(t));
	  case opStrNe:
	    return !left.evaluateStr(t).equals(right.evaluateStr(t));
	  case opStrLt:
	    return left.evaluateStr(t).compareTo(right.evaluateStr(t)) < 0;
	  case opStrLe:
	    return left.evaluateStr(t).compareTo(right.evaluateStr(t)) <= 0;
	  case opStrGt:
	    return left.evaluateStr(t).compareTo(right.evaluateStr(t)) > 0;
	  case opStrGe:
	    return left.evaluateStr(t).compareTo(right.evaluateStr(t)) >= 0;

	  case opBoolEq:
	    return left.evaluateBool(t) == right.evaluateBool(t);
	  case opBoolNe:
	    return left.evaluateBool(t) != right.evaluateBool(t);

	  case opObjEq:
	    return left.evaluateObj(t) == right.evaluateObj(t);
	  case opObjNe:
	    return left.evaluateObj(t) != right.evaluateObj(t);

	  case opScanArrayBool:
	  {
	    boolean val = left.evaluateBool(t);
	    boolean[] arr = (boolean[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayBool:
	  {
	    boolean val = left.evaluateBool(t);
	    return ((ArrayOfBoolean)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayInt1:
	  {
	    long val = left.evaluateInt(t);
	    byte[] arr = (byte[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayInt1:
	  {
	    byte val = (byte)left.evaluateInt(t);
	    return ((ArrayOfByte)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayChar:
	  {
	    long val = left.evaluateInt(t);
	    char[] arr = (char[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayChar:
	  {
	    char val = (char)left.evaluateInt(t);
	    return ((ArrayOfChar)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayInt2:
	  {
	    long val = left.evaluateInt(t);
	    short[] arr = (short[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayInt2:
	  {
	    short val = (short)left.evaluateInt(t);
	    return ((ArrayOfShort)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayInt4:
	  {
	    long val = left.evaluateInt(t);
	    int[] arr = (int[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayInt4:
	  {
	    int val = (int)left.evaluateInt(t);
	    return ((ArrayOfInt)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayInt8:
	  {
	    long val = left.evaluateInt(t);
	    long[] arr = (long[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayInt8:
	  {
	    long val = left.evaluateInt(t);
	    return ((ArrayOfLong)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayReal4:
	  {
	    double val = left.evaluateReal(t);
	    float[] arr = (float[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayReal4:
	  {
	    float val = (float)left.evaluateReal(t);
	    return ((ArrayOfFloat)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayReal8:
	  {
	    double val = left.evaluateReal(t);
	    double[] arr = (double[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (arr[i] == val) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayReal8:
	  {
	    double val = left.evaluateReal(t);
	    return ((ArrayOfDouble)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opScanArrayStr:
	  {
	    String val = left.evaluateStr(t);
	    String[] arr = (String[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (val.equals(arr[i])) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanArrayObj:
	  {
	    Object val = left.evaluateObj(t);
	    Object[] arr = (Object[])right.evaluateObj(t);  
	    for (int i = arr.length; --i >= 0;) { 
		if (val == arr[i]) { 
		    return true;
		}
	    }
	    return false;
	  }
	  case opScanDnmArrayObj:
	  {
	    Persistent val = (Persistent)left.evaluateObj(t);
	    return ((ArrayOfObject)right.evaluateObj(t)).indexOf(val) >= 0;  
	  }
	  case opInString:
	  {
	    String substr = left.evaluateStr(t);
	    String str = right.evaluateStr(t);
	    return str.indexOf(substr) >= 0;
	  }
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }

    BinOpNode(int type, int tag, Node left, Node right) {
	super(type, tag);
	this.left = left;
	this.right = right;
    }
}

class CompareNode extends Node {
    Node o1, o2, o3;

    boolean evaluateBool(SearchThread t) {
	switch(tag) { 
	  case opIntBetween:
	  {
	    long val = o1.evaluateInt(t);
	    return val >= o2.evaluateInt(t) && val <= o3.evaluateInt(t);
	  }
	  
	  case opRealBetween:
	  {
	    double val = o1.evaluateReal(t);
	    return val >= o2.evaluateReal(t) && val <= o3.evaluateReal(t);
	  }
	  
	  case opStrBetween:
	  {
	    String val = o1.evaluateStr(t);
	    return val.compareTo(o2.evaluateStr(t)) >= 0 
		&& val.compareTo(o3.evaluateStr(t)) <= 0;
	  }
	  
	  case opStrLike:
	  {
	    String str = o1.evaluateStr(t);
	    String pat = o2.evaluateStr(t);
	    int pi = 0, si = 0, pn = pat.length(), sn = str.length(); 
	    int wildcard = -1, strpos = -1;
	    while (true) { 
		if (pi < pn && pat.charAt(pi) == '%') { 
		    wildcard = ++pi;
		    strpos = si;
		} else if (si == sn) { 
		    return pi == pn;
		} else if (pi < pn && (str.charAt(si) == pat.charAt(pi) 
				       || pat.charAt(pi) == '_')) {
		    si += 1;
		    pi += 1;
		} else if (wildcard >= 0) { 
		    si = ++strpos;
		    pi = wildcard;
		} else { 
		    return false;
		}
	    }
	  }
	  case opStrLikeEsc:
	  {
	    String str = o1.evaluateStr(t);
	    String pat = o2.evaluateStr(t);
	    char escape = o3.evaluateStr(t).charAt(0);
	    int pi = 0, si = 0, pn = pat.length(), sn = str.length(); 
	    int wildcard = -1, strpos = -1;
	    while (true) { 
		if (pi < pn && pat.charAt(pi) == '%') { 
		    wildcard = ++pi;
		    strpos = si;
		} else if (si == sn) { 
		    return pi == pn;
		} else if (pi+1 < pn && pat.charAt(pi) == escape && 
			   pat.charAt(pi+1) == str.charAt(si)) {
		    si += 1;
		    pi += 2;
		} else if (pi < pn && ((pat.charAt(pi) != escape 
					&& (str.charAt(si) == pat.charAt(pi) 
					    || pat.charAt(pi) == '_')))) {
		    si += 1;
		    pi += 1;
		} else if (wildcard >= 0) { 
		    si = ++strpos;
		    pi = wildcard;
		} else { 
		    return false;
		}
	    }
	  }
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }

    CompareNode(int tag, Node a, Node b, Node c) { 
	super(tpBool, tag);
	o1 = a;
	o2 = b;
	o3 = c;
    }
}


class UnaryOpNode extends Node { 
    Node opd;

    long evaluateInt(SearchThread t) {
	long val;
	switch (tag) { 
	  case opIntNot:
	    return ~opd.evaluateInt(t);
	  case opIntNeg:
	    return -opd.evaluateInt(t);
	  case opIntAbs:
	    val = opd.evaluateInt(t);
	    return val < 0 ? -val : val;
	  case opRealToInt:
	    return (long)opd.evaluateReal(t);
	  case opLength:
	    try { 
	        return Array.getLength(opd.evaluateObj(t));
	    } catch (IllegalArgumentException x) {
		throw new Error("Argument is not array");
	    }
	  case opDnmLength:
	    return ((AnyArray)opd.evaluateObj(t)).length();
	  case opStrLength:
	    return opd.evaluateStr(t).length();
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }
    
    double evaluateReal(SearchThread t) { 
	double val;
	switch (tag) { 
	  case opRealNeg:
	    return -opd.evaluateReal(t);
	  case opRealAbs:
	    val = opd.evaluateReal(t);
	    return val < 0 ? -val : val;
	  case opRealSin:
	    return Math.sin(opd.evaluateReal(t));
	  case opRealCos:
	    return Math.cos(opd.evaluateReal(t));
	  case opRealTan:
	    return Math.tan(opd.evaluateReal(t));
	  case opRealAsin:
	    return Math.asin(opd.evaluateReal(t));
	  case opRealAcos:
	    return Math.acos(opd.evaluateReal(t));
	  case opRealAtan:
	    return Math.atan(opd.evaluateReal(t));
	  case opRealExp:
	    return Math.exp(opd.evaluateReal(t));
	  case opRealLog:
	    return Math.log(opd.evaluateReal(t));
	  case opRealSqrt:
	    return Math.sqrt(opd.evaluateReal(t));
	  case opRealCeil:
	    return Math.ceil(opd.evaluateReal(t));
	  case opRealFloor:
	    return Math.floor(opd.evaluateReal(t));
	  case opIntToReal:
	    return (double)opd.evaluateInt(t);
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }
    
    String evaluateStr(SearchThread t) { 
	switch (tag) { 
	  case opStrUpper:
	    return opd.evaluateStr(t).toUpperCase();
	  case opStrLower:
	    return opd.evaluateStr(t).toLowerCase();
	  case opIntToStr:
	    return Long.toString(opd.evaluateInt(t), 10);
	  case opRealToStr:
	    return Double.toString(opd.evaluateReal(t));
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }
    
    boolean evaluateBool(SearchThread t) { 
	switch (tag) { 
	  case opBoolNot:
	    return !opd.evaluateBool(t);
	  case opIsNull:
	    return opd.evaluateObj(t) == null;
	  default:
	    throw new Error("Invalid tag " + tag);
	}
    }

    UnaryOpNode(int type, int tag, Node node) { 
	super(type, tag);
	opd = node;
    }
}


class LoadNode extends Node {
    Field field;
    Node  base;

    Class getType() { 
	return field.getType();
    }

    long evaluateInt(SearchThread t) {
	Object obj;
	if (base != null) { 
	    obj = base.evaluateObj(t);
	    if (obj instanceof Persistent) { 
		Persistent pobj = (Persistent)obj;
		long val;
		pobj.metaobject.preDaemon(pobj, Metaobject.VARIABLE);
		try { 
		    switch (tag) { 
		      case opLoadChar:
			val = field.getChar(obj);
			break;
		      case opLoadInt1:
			val = field.getByte(obj);
			break;
		      case opLoadInt2:
			val = field.getShort(obj);
			break;
		      case opLoadInt4:
			val = field.getInt(obj);
			break;
		      case opLoadInt8:
			val = field.getLong(obj);
			break;
		      default:
			throw new Error("Invalid tag " + tag);
		    }
		} catch (IllegalAccessException x) { 
		    pobj.metaobject.postDaemon(pobj, 
		        Metaobject.VARIABLE|Metaobject.EXCEPTION, false);
		    throw new IllegalAccessError();
		}
		pobj.metaobject.postDaemon(pobj, Metaobject.VARIABLE, false);
		return val;
	    }
	} else { 
	    obj = t.currObj;
	}
	try { 
	    switch (tag) { 
	      case opLoadChar:
		return field.getChar(obj);
	      case opLoadInt1:
		return field.getByte(obj);
	      case opLoadInt2:
		return field.getShort(obj);
	      case opLoadInt4:
		return field.getInt(obj);
	      case opLoadInt8:
		return field.getLong(obj);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
    
    double evaluateReal(SearchThread t) { 
	Object obj;
	if (base != null) { 
	    obj = base.evaluateObj(t);
	    if (obj instanceof Persistent) { 
		Persistent pobj = (Persistent)obj;
		double val;
		pobj.metaobject.preDaemon(pobj, Metaobject.VARIABLE);
		try { 
		    switch (tag) { 
		      case opLoadReal4:
			val = field.getFloat(obj);
			break;
		      case opLoadReal8:
			val = field.getDouble(obj);
			break;
		      default:
			throw new Error("Invalid tag " + tag);
		    }
		} catch (IllegalAccessException x) { 
		    pobj.metaobject.postDaemon(pobj, 
		        Metaobject.VARIABLE|Metaobject.EXCEPTION, false);
		    throw new IllegalAccessError();
		}
		pobj.metaobject.postDaemon(pobj, Metaobject.VARIABLE, false);
		return val;
	    }
	} else { 
	    obj = t.currObj;
	}
	try { 
	    switch (tag) { 
	      case opLoadReal4:
		return field.getFloat(obj);
	      case opLoadReal8:
		return field.getDouble(obj);
	      default:
		throw new Error("Invalid tag " + tag);
	    }
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
    
    boolean evaluateBool(SearchThread t) { 
	Object obj;
	if (base != null) { 
	    obj = base.evaluateObj(t);
	    if (obj instanceof Persistent) { 
		Persistent pobj = (Persistent)obj;
		boolean val;
		pobj.metaobject.preDaemon(pobj, Metaobject.VARIABLE);
		try { 
		    val = field.getBoolean(obj);
		} catch (IllegalAccessException x) { 
		    pobj.metaobject.postDaemon(pobj, 
		        Metaobject.VARIABLE|Metaobject.EXCEPTION, false);
		    throw new IllegalAccessError();
		}
		pobj.metaobject.postDaemon(pobj, Metaobject.VARIABLE, false);
		return val;
	    }
	} else { 
	    obj = t.currObj;
	}
	try { 
	    return field.getBoolean(obj);
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
    
    String evaluateStr(SearchThread t) { 
	Object obj;
	if (base != null) { 
	    obj = base.evaluateObj(t);
	    if (obj instanceof Persistent) { 
		Persistent pobj = (Persistent)obj;
		String val;
		pobj.metaobject.preDaemon(pobj, Metaobject.VARIABLE);
		try {
		    val = (String)field.get(obj);
		} catch (IllegalAccessException x) { 
		    pobj.metaobject.postDaemon(pobj, 
		        Metaobject.VARIABLE|Metaobject.EXCEPTION, false);
		    throw new IllegalAccessError();
		}
		pobj.metaobject.postDaemon(pobj, Metaobject.VARIABLE, false);
		return val;		
	    }
	} else { 
	    obj = t.currObj;
	}
	try {
	    return (String)field.get(obj);
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
	
    Object evaluateObj(SearchThread t) { 
	Object obj;
	if (base != null) { 
	    obj = base.evaluateObj(t);
	    if (obj instanceof Persistent) { 
		Persistent pobj = (Persistent)obj;
		pobj.metaobject.preDaemon(pobj, Metaobject.VARIABLE);
		try {
		    obj = field.get(obj);
		} catch (IllegalAccessException x) { 
		    pobj.metaobject.postDaemon(pobj, 
		        Metaobject.VARIABLE|Metaobject.EXCEPTION, false);
		    throw new IllegalAccessError();
		}
		pobj.metaobject.postDaemon(pobj, Metaobject.VARIABLE, false);
		return obj;		
	    }
	} else { 
	    obj = t.currObj;
	}
	try {
	    return field.get(obj);
	} catch (IllegalAccessException x) { 
	    throw new IllegalAccessError();
	}
    }
    
    LoadNode(int type, int tag, Field f) { 
	super(type, tag);
	field = f;
	base = null;
    }
    LoadNode(int type, int tag, Field f, Node base) { 
	super(type, tag);
	field = f;
	this.base = base;
    }
}


class OrderNode {
    OrderNode next;
    boolean   ascent;
    Field     field;
    int       type;
    
    final static int fdBool  = 0;
    final static int fdChar  = 1;
    final static int fdInt1  = 2;
    final static int fdInt2  = 3;
    final static int fdInt4  = 4;
    final static int fdInt8  = 5;
    final static int fdReal4 = 6;
    final static int fdReal8 = 7;
    final static int fdStr   = 8;

    final int compare(Object a, Object b) { 
	int diff;
	Persistent pa = null;
	Persistent pb = null;
	if (a instanceof Persistent) { 
	    pa = (Persistent)a;
	    pa.metaobject.preDaemon(pa, Metaobject.VARIABLE);
	}
	if (b instanceof Persistent) { 
	    pb = (Persistent)b;
	    pb.metaobject.preDaemon(pb, Metaobject.VARIABLE);
	}
	try {
	    switch (type) { 
	      case fdBool:
		diff = field.getBoolean(a) ? field.getBoolean(b) ? 0 : 1 
		     : field.getBoolean(b) ? -1 : 0;
		break;
	      case fdChar:
		diff = field.getChar(a) - field.getChar(b);
		break;
	      case fdInt1:
		diff = field.getByte(a) - field.getByte(b);
		break;
	      case fdInt2:
		diff = field.getShort(a) - field.getShort(b);
		break;
	      case fdInt4:
		diff = field.getInt(a) - field.getInt(b);
		break;
	      case fdInt8:
	      {
		  long l = field.getLong(a);
		  long r = field.getLong(b);
		  diff = l < r ? -1 : l == r ? 0 : 1;
		  break;
	      }
	      case fdReal4:
	      {
		  float l = field.getFloat(a);
		  float r = field.getFloat(b);
		  diff = l < r ? -1 : l == r ? 0 : 1;
		  break;
	      }
	      case fdReal8:
	      {
		  double l = field.getDouble(a);
		  double r = field.getDouble(b);
		  diff = l < r ? -1 : l == r ? 0 : 1;
		  break;
	      }
	      case fdStr:
		diff = ((String)field.get(a)).compareTo((String)field.get(b));
		break;
	      default:
		throw new Error("Invalid order node type " + type);
	    }
	} catch(IllegalAccessException x) { 
	    if (pa != null) { 
		pa.metaobject.postDaemon(pa, 
		    Metaobject.EXCEPTION|Metaobject.VARIABLE, false);
	    }
	    if (pb != null) { 
		pb.metaobject.postDaemon(pb, 
		    Metaobject.EXCEPTION|Metaobject.VARIABLE, false);
	    }
	    throw new IllegalAccessError();
	}
	if (pa != null) { 
	    pa.metaobject.postDaemon(pa, Metaobject.VARIABLE, false);
	}
	if (pb != null) { 
	    pb.metaobject.postDaemon(pb, Metaobject.VARIABLE, false);
	}
	if (diff == 0 && next != null) { 
	    return next.compare(a, b);
	}
	if (!ascent) { 
	    diff = -diff;
	}
	return diff;
    }

    OrderNode(int type, Field field) { 
	this.type = type;
	this.field = field;
	ascent = true;
	next = null;
    }
}


class SearchThread extends Thread {
    int      id;
    Query    query;
    Object[] selection;
    int      used;
    int      pos;
    int      allocated;

    int[]    indexVar;
    Object   currObj;

    final static int initSize = 1024;
    final static int maxIndexVars = 32;

    SearchThread(int id, int nThreads, Query query) { 
	allocated = initSize;
	used = 0;
	pos = 0;
	selection = new Object[initSize];
	indexVar = new int[maxIndexVars];
	this.id = id;
	this.query = query;
    }

    public void run() {
	Node tree = query.tree;
	QueryIterator iterator = query.iterator;
	int n = query.nThreads;
	int i, j, k;
	int limit = query.limit;
	currObj = iterator.getFirst();
	for (i = 0, j = id; --j >= 0 && currObj != null; i++) {
	    currObj = iterator.getNext(currObj, i);
	}
	while (currObj != null) { 
	    boolean selected;
	    if (currObj instanceof Persistent) { 
		Persistent pobj = (Persistent)currObj;
		pobj.metaobject.preDaemon(pobj, 0);
		try { 
		    selected = tree.evaluateBool(this);
		} catch(RuntimeException x) { 
		    pobj.metaobject.postDaemon(pobj, Metaobject.EXCEPTION, 
					       false);
		    throw x;
		} catch(Error x) { 
		    pobj.metaobject.postDaemon(pobj, Metaobject.EXCEPTION, 
					       false);
		    throw x;
		}
		pobj.metaobject.postDaemon(pobj, 0, false);
	    } else { 
		selected = tree.evaluateBool(this);
	    }
	    if (selected) {
		if (used == allocated) { 
		    allocated *= 2;
		    Object[] newbuf = new Object[allocated];
		    System.arraycopy(selection, 0, newbuf, 0, used);
		    selection = newbuf;
		}
		selection[used++] = currObj;
		if (used == limit) { 
		    break;
		}
	    }
	    currObj = iterator.getNext(currObj, i);
	    for (j = n; currObj != null && --j != 0; i++) { 
		currObj = iterator.getNext(currObj, i);
	    }
	}
	if (query.order != null) {
	    OrderNode order = query.order;
	    Object top;
	    for (n = used, i = n/2, j = i; i >= 1; i--) { 
		k = i;
		top = selection[k-1];
		do { 
		    if (k*2 == n || 
			order.compare(selection[k*2-1], selection[k*2]) > 0) 
		    { 
			if (order.compare(top, selection[k*2-1]) >= 0) {
			    break;
			}
			selection[k-1] = selection[k*2-1];
			k = k*2;
		    } else { 
			if (order.compare(top, selection[k*2]) >= 0) {
			    break;
			}
			selection[k-1] = selection[k*2];
			k = k*2+1;
		    }
		} while (k <= j);
		selection[k-1] = top; 
	    }
	    for (i = n; i >= 2; i--) { 
		top = selection[i-1];
		selection[i-1] = selection[0];
		selection[0] = top;
		for (k = 1, j = (i-1)/2; k <= j;) { 
		    if (k*2 == i-1 || 
			order.compare(selection[k*2-1], selection[k*2]) > 0) 
		    { 
			if (order.compare(top, selection[k*2-1]) >= 0) {
			    break;
			}
			selection[k-1] = selection[k*2-1];
			k = k*2;
		    } else { 
			if (order.compare(top, selection[k*2]) >= 0) {
			    break;
			}
			selection[k-1] = selection[k*2];
			k = k*2+1;
		    }
		} 
		selection[k-1] = top;
	    } 
	}
    }
}


class Symbol { 
    int tkn;

    Symbol(int tkn) { 
	this.tkn = tkn;
    }
}


class Binding {
    Binding next;
    String  name;
    boolean used;
    int     loopId;

    Binding(String ident, int loop, Binding chain) { 
	name = ident;
	used = false;
	next = chain;
	loopId = loop;
    }
}

public class Query {
    public Object[] select(String className, QueryIterator iterator,
			   String query, int limit, int nThreads)
    throws CompileError
    {
	Object[] selection;
	int selected = 0;
	this.query = query;
	buf = query.toCharArray();
	str = new char[buf.length];
	this.iterator = iterator;
	this.nThreads = nThreads;
	this.limit = limit;

	try { 
	    cls = Class.forName(className); 
	} catch(Exception x) { 
	    throw new CompileError("Class name not found", 0);
	}
	compile();
	
	if ((tree.tag == Node.opBoolAnd 
	     && isIndexApplicable(((BinOpNode)tree).left))
	    || isIndexApplicable(tree)) 
	{ 
	    if (obj != null) { 
		if (tree.tag == Node.opBoolAnd) { 
		    SearchThread thr = new SearchThread(0, 1, this);
		    thr.currObj = obj;
		    selected = 
			((BinOpNode)tree).right.evaluateBool(thr) ? 1 : 0; 
		} else { 
		    selected = 1;
		}
	    }
	    try { 
		selection = (Object[])Array.newInstance(cls, selected);
	    } catch (NegativeArraySizeException x) { 
		throw new Error("Failed to create array instance");
	    }
	    if (selected > 0) { 
		selection[0] = obj;
	    }
	    return selection;
	}
	if (limit != 0) { 
	    nThreads = 1;
	}
	SearchThread[] thr = new SearchThread[nThreads];
	int i;
	int n = nThreads-1;
	for (i = 0; i < n; i++) { 
	    thr[i] = new SearchThread(i, nThreads, this);
	    thr[i].start();
	}
	thr[i] = new SearchThread(i, nThreads, this);
	thr[i].run();
	selected = thr[i].used;
	for (i = 0; i < n; i++) { 
	    try {
		thr[i].join();
		selected += thr[i].used;
	    } catch(InterruptedException x) { 
		throw new Error("Search thread was interrupted");
	    }
	}
	try { 
	    selection = (Object[])Array.newInstance(cls, selected);
	} catch (NegativeArraySizeException x) { 
	    throw new Error("Negative selection size");
	}
	selected = 0;
	if (order != null) { 
	    while (true) { 
		int min = -1;
		for (i = 0; i <= n; i++) { 
		    if (thr[i].pos < thr[i].used && (min < 0 || 
			order.compare(thr[i].selection[thr[i].pos], 
				      thr[min].selection[thr[min].pos]) < 0))
		    {
			min = i;
		    }
		}
		if (min < 0) { 
		    return selection;
		}
		selection[selected++] = thr[min].selection[thr[min].pos++];
	    }
	} else {
	    for (i = 0; i <= n; i++) { 
		System.arraycopy(thr[i].selection, 0,
				 selection, selected, thr[i].used);
		selected += thr[i].used;
	    }
	    return selection;
	}
    }


    public Object[] select(String className, QueryIterator iterator, 
			   String query)
    throws CompileError
    {
	return select(className, iterator, query, 0, 1);
    }

    public Object[] select(String className, QueryIterator iterator, 
			   String query, int limit)
    throws CompileError
    {
	return select(className, iterator, query, limit, 1);
    }

    

    int           pos;
    char[]        buf;
    char[]        str;
    String        query;
    long          ivalue;
    String        svalue;
    double        fvalue;
    Class         cls;
    Node          tree;
    String        ident;
    int           lex;
    int           vars;
    int           nThreads;
    int           limit;
    Object        obj;
    Binding       bindings;
    OrderNode     order;
    QueryIterator iterator;

    static Hashtable symtab;
    static Class[] profile = new Class[0];
    static Class[] getAtProfile = { Integer.TYPE };
    static final Object[] bypassFlag = { new Boolean(true) };
    static Method setBypass;
    static Class arrayOfObject;

    final static int tknIdent = 1;
    final static int tknLpar = 2;
    final static int tknRpar = 3;
    final static int tknLbr = 4;
    final static int tknRbr = 5;
    final static int tknDot = 6;
    final static int tknComma = 7;
    final static int tknPower = 8;
    final static int tknIconst = 9;
    final static int tknSconst = 10;
    final static int tknFconst = 11;
    final static int tknAdd = 12;
    final static int tknSub = 13;
    final static int tknMul = 14;
    final static int tknDiv = 15;
    final static int tknAnd = 16;
    final static int tknOr = 17;
    final static int tknNot = 18;
    final static int tknNull = 19;
    final static int tknNeg = 20;
    final static int tknEq = 21;
    final static int tknNe = 22;
    final static int tknGt = 23;
    final static int tknGe = 24;
    final static int tknLt = 25;
    final static int tknLe = 26;
    final static int tknBetween = 27;
    final static int tknEscape = 28;
    final static int tknExists = 29;
    final static int tknLike = 30;
    final static int tknIn = 31;
    final static int tknLength = 32;
    final static int tknLower = 33;
    final static int tknUpper = 34;
    final static int tknAbs = 35;
    final static int tknIs = 36;
    final static int tknInteger = 37;
    final static int tknReal = 38;
    final static int tknString = 39;
    final static int tknFirst = 40;
    final static int tknLast = 41;
    final static int tknCurrent = 42;
    final static int tknCol = 44;
    final static int tknTrue = 45;
    final static int tknFalse = 46;
    final static int tknWhere = 47;
    final static int tknOrder = 48;
    final static int tknAsc = 49;
    final static int tknDesc = 50;
    final static int tknEof = 51;
    final static int tknSin = 52;
    final static int tknCos = 53;
    final static int tknTan = 54;
    final static int tknAsin = 55;
    final static int tknAcos = 56;
    final static int tknAtan = 57;
    final static int tknSqrt = 58;
    final static int tknLog = 59;
    final static int tknExp = 60;
    final static int tknCeil = 61;
    final static int tknFloor = 62;
    final static int tknBy = 63;

    static { 
	symtab = new Hashtable();
	symtab.put("abs", new Symbol(tknAbs));
	symtab.put("acos", new Symbol(tknAcos));
	symtab.put("and", new Symbol(tknAnd));
	symtab.put("asc", new Symbol(tknAsc));
	symtab.put("asin", new Symbol(tknAsin));
	symtab.put("atan", new Symbol(tknAtan));
	symtab.put("between", new Symbol(tknBetween));
	symtab.put("by", new Symbol(tknBy));
	symtab.put("ceal", new Symbol(tknCeil));
	symtab.put("cos", new Symbol(tknCos));
	symtab.put("current", new Symbol(tknCurrent));
	symtab.put("desc", new Symbol(tknDesc));
	symtab.put("escape", new Symbol(tknEscape));
	symtab.put("exists", new Symbol(tknExists));
	symtab.put("exp", new Symbol(tknExp));
	symtab.put("false", new Symbol(tknFalse));
	symtab.put("floor", new Symbol(tknFloor));
	symtab.put("in", new Symbol(tknIn));
	symtab.put("is", new Symbol(tknIs));
	symtab.put("integer", new Symbol(tknInteger));
	symtab.put("last", new Symbol(tknLast));
	symtab.put("length", new Symbol(tknLength));
	symtab.put("like", new Symbol(tknLike));
	symtab.put("log", new Symbol(tknLog));
	symtab.put("lower", new Symbol(tknLower));
	symtab.put("not", new Symbol(tknNot));
	symtab.put("null", new Symbol(tknNull));
	symtab.put("or", new Symbol(tknOr));
	symtab.put("order", new Symbol(tknOrder));
	symtab.put("real", new Symbol(tknReal));
	symtab.put("sin", new Symbol(tknSin));
	symtab.put("sqrt", new Symbol(tknSqrt));
	symtab.put("string", new Symbol(tknString));
	symtab.put("true", new Symbol(tknTrue));
	symtab.put("upper", new Symbol(tknUpper));

	try { 
	    arrayOfObject = Class.forName("goodslib.ArrayOfObject");
	    Class c = Class.forName("java.lang.reflect.AccessibleObject");
	    Class[] param = { Boolean.TYPE };
	    // setAccessible() method is available only in JDK 1.2
	    setBypass = c.getMethod("setAccessible", param);
	} catch(Exception x) {}
    }


    final int scan() {
	int p = pos;
	int eol = buf.length;
	char ch = 0;
	int i;
	while (p < eol && Character.isWhitespace(ch = buf[p])) { 
	    p += 1;
	}
	if (p == eol) { 
	    return tknEof;
	}
	pos = ++p;
	switch (ch) { 
	  case '+':
	    return tknAdd;
	  case '-':
	    return tknSub;
	  case '*':
	    return tknMul;
	  case '/':
	    return tknDiv;
	  case '.':
	    return tknDot;
	  case ',':
	    return tknComma;
	  case '(':
	    return tknLpar;
	  case ')':
	    return tknRpar;
	  case '[':
	    return tknLbr;
	  case ']':
	    return tknRbr;
	  case ':':
	    return tknCol;
	  case '^':
	    return tknPower;
	  case '<':
	    if (p < eol) { 
		if (buf[p] == '=') { 
		    pos += 1;
		    return tknLe;
		} 
		if (buf[p] == '>') { 
		    pos += 1;
		    return tknNe;
		}
	    }
	    return tknLt;
	  case '>':
	    if (p < eol && buf[p] == '=') { 
		pos += 1;
		return tknGe;
	    } 
	    return tknGt;
	  case '=':
	    return tknEq;
	  case '!':
	    if (p == eol || buf[p] != '=') { 
		throw new CompileError("Invalid token '!'", p-1);
	    } 
	    pos += 1;
	    return tknNe;
	  case '|':
	    if (p == eol || buf[p] != '|') { 
		throw new CompileError("Invalid token '!'", p-1);
	    } 
	    pos += 1;
	    return tknAdd;
	  case '\'':
	    i = 0; 
	    while (true) { 
		if (p == eol) { 
		    throw new CompileError("Unexpected end of string constant",
					   p);
		}
		if (buf[p] == '\'') { 
		    if (++p == eol || buf[p] != '\'') { 
			svalue = new String(str, 0, i);
			pos = p;
			return tknSconst;
		    }
		}
		str[i++] = buf[p++];
	    }
	  case '0': case '1': case '2': case '3': case '4': 
	  case '5': case '6': case '7': case '8': case '9':
	    i = p - 1;
	    while (p < eol && Character.isDigit(ch = buf[p])) { 
		p += 1;
	    }
	    if (ch == '.' || ch == 'e' || ch == 'E') { 
		while (++p < eol && (Character.isDigit(buf[p]) 
		       || buf[p] == 'e' || buf[p] == 'E' || buf[p] == '.' ||
		       ((ch == 'e' || ch == 'E') 
			&& (buf[p] == '-' || buf[p] == '+'))));
		pos = p;
		try { 
		    fvalue = 
			Double.valueOf(query.substring(i, p)).doubleValue();
		} catch(NumberFormatException x) { 
		    throw new CompileError("Bad floating point constant", i);
		}
		return tknFconst;
	    } else { 
		pos = p;
		try { 
		    ivalue = Long.parseLong(query.substring(i, p), 10);
		} catch(NumberFormatException x) { 
		    throw new CompileError("Bad floating point constant", i);
		}
		return tknIconst;
	    }
	  default:
	    if (Character.isLetter(ch) || ch == '$' || ch == '_') { 
		i = p-1;
		while (p < eol && (Character.isLetterOrDigit(ch = buf[p]) 
				   || ch == '$' || ch == '_'))
		{
		    p += 1;
		}
		pos = p;
		ident = query.substring(i, p);
		Symbol s = (Symbol)symtab.get(ident);
		return (s == null) ? tknIdent : s.tkn;
	    } else { 
		throw new CompileError("Invalid symbol: " + ch, p-1);
	    }
	}
    }
	

    final Node disjunction() {
	Node left = conjunction();
	if (lex == tknOr) { 
	    int p = pos;
	    Node right = disjunction();
	    if (left.type == Node.tpInt && right.type == Node.tpInt) { 
		left = new BinOpNode(Node.tpInt, Node.opIntOr, left, right);
	    } else if (left.type == Node.tpBool && right.type == Node.tpBool) {
		left = new BinOpNode(Node.tpBool, Node.opBoolOr, left, right);
	    } else { 
		throw new CompileError("Bad operands for OR operator", p);
	    }
	}
	return left;
    }

    final Node conjunction() {
	Node left = comparison();
	if (lex == tknAnd) { 
	    int p = pos;
	    Node right = conjunction();
	    if (left.type == Node.tpInt && right.type == Node.tpInt) { 
		left = new BinOpNode(Node.tpInt, Node.opIntAnd, left, right);
	    } else if (left.type == Node.tpBool && right.type == Node.tpBool) {
		left = new BinOpNode(Node.tpBool, Node.opBoolAnd, left, right);
	    } else { 
		throw new CompileError("Bad operands for AND operator", p);
	    }
	}
	return left;
    }

    final static Node int2real(Node expr) {
	if (expr.tag == Node.opIntConst) { 
	    return new RealLiteralNode((double)((IntLiteralNode)expr).value);
	} 
	return new UnaryOpNode(Node.tpReal, Node.opIntToReal, expr);
    }

    final int compare(Node expr, BinOpNode list)
    {
	int n = 1;
	if (list.left != null) { 
	    n = compare(expr, (BinOpNode)list.left);
	}
	Node elem = list.right;
	int cop = Node.opNop;
	if (expr.type == Node.tpInt) { 
	    if (elem.type == Node.tpReal) { 
		expr = new UnaryOpNode(Node.tpReal, Node.opIntToReal, expr);
		cop = Node.opRealEq;
	    } else if (elem.type == Node.tpInt) { 
		cop = Node.opIntEq;
	    }
	} else if (expr.type == Node.tpReal) {
	    if (elem.type == Node.tpReal) { 
		cop = Node.opRealEq;
	    } else if (elem.type == Node.tpInt) { 
		cop = Node.opRealEq;
		elem = int2real(elem);
	    }
	} else if (expr.type == Node.tpStr && elem.type == Node.tpStr) {
	    cop = Node.opStrEq;
	} else if (expr.type == Node.tpObj && elem.type == Node.tpObj) {
	    cop = Node.opObjEq;
	} else if (expr.type == Node.tpBool && elem.type == Node.tpBool) {
	    cop = Node.opBoolEq;
	}
	if (cop == Node.opNop) { 
	    throw new CompileError("Expression "+n+" in right part of IN "+
				   "operator has incompatible type", pos);
	} 
	list.type = Node.tpBool;
	if (list.left != null) { 
	    list.right = new BinOpNode(Node.tpBool, cop, expr, elem);
	    list.tag = Node.opBoolOr;
	} else { 
	    list.left = expr;
	    list.tag = cop;
	}
	return ++n;
    }


    final Node comparison() {
	int  leftPos = pos;
	Node left, right;
	left = addition();
	int cop = lex;
	if (cop == tknEq || cop == tknNe || cop == tknGt || cop == tknGe
	    || cop == tknLe || cop == tknLt || cop == tknBetween 
	    || cop == tknLike || cop == tknNot || cop == tknIs || cop == tknIn)
	{
	    int rightPos = pos;
	    boolean not = false;
	    if (cop == tknNot) { 
		not = true;
		cop = scan();
		if (cop != tknLike && cop != tknBetween && cop != tknIn) { 
		    throw new CompileError("LIKE, BETWEEN or IN expected", 
					   rightPos);
		} 
		rightPos = pos;
	    } else if (cop == tknIs) {
		if (left.type < Node.tpObj) { 
		    throw new CompileError("IS [NOT] NULL predicate can be applied only to references,arrays or string", 
					   rightPos);
		} 
		rightPos = pos;
		if ((cop = scan()) == tknNull) { 
		    left = new UnaryOpNode(Node.tpBool, Node.opIsNull, left);
		} else if (cop == tknNot) { 
		    rightPos = pos;
		    if (scan() == tknNull) { 
			left = new UnaryOpNode(Node.tpBool, Node.opBoolNot, 
					       new UnaryOpNode(Node.tpBool, 
							       Node.opIsNull, 
							       left));
		    } else { 
			throw new CompileError("NULL expected", rightPos);
		    }
		} else { 
		    throw new CompileError("[NOT] NULL expected", rightPos);
		} 
		lex = scan();
		return left;
	    }	
	    right = addition();
	    if (cop == tknIn) { 
		int type;
		switch (right.type) {
		  case Node.tpArrayBool:
		    if (left.type != Node.tpBool) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayBool,
					 left, right);
		    break;
		  case Node.tpDnmArrayBool:
		    if (left.type != Node.tpBool) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayBool,
					 left, right);
		    break;
		  case Node.tpArrayInt1:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayInt1,
					 left, right);
		    break;
		  case Node.tpDnmArrayInt1:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayInt1,
					 left, right);
		    break;
		  case Node.tpArrayChar:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayChar,
					 left, right);
		    break;
		  case Node.tpDnmArrayChar:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayChar,
					 left, right);
		    break;
		  case Node.tpArrayInt2:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayInt2,
					 left, right);
		    break;
		  case Node.tpDnmArrayInt2:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayInt2,
					 left, right);
		    break;
		  case Node.tpArrayInt4:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayInt4,
					 left, right);
		    break;
		  case Node.tpDnmArrayInt4:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayInt4,
					 left, right);
		    break;
		  case Node.tpArrayInt8:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayInt8,
					 left, right);
		    break;
		  case Node.tpDnmArrayInt8:
		    if (left.type != Node.tpInt) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayInt8,
					 left, right);
		    break;
		  case Node.tpArrayReal4:
		    if (left.type == Node.tpInt) {
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayReal4,
					 left, right);
		    break;
		  case Node.tpDnmArrayReal4:
		    if (left.type == Node.tpInt) {
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayReal4,
					 left, right);
		    break;
		  case Node.tpArrayReal8:
		    if (left.type == Node.tpInt) {
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayReal8,
					 left, right);
		    break;
		  case Node.tpDnmArrayReal8:
		    if (left.type == Node.tpInt) {
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayReal8,
					 left, right);
		    break;
		  case Node.tpArrayObj:
		    if (left.type != Node.tpObj) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayObj,
					 left, right);
		    break;
		  case Node.tpDnmArrayObj:
		    if (left.type != Node.tpObj) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanDnmArrayObj,
					 left, right);
		    break;
		  case Node.tpArrayStr:
		    if (left.type != Node.tpStr) { 
			throw new CompileError("Incompatible types of IN operator operands", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opScanArrayStr,
					 left, right);
		    break;
		  case Node.tpStr:
		    if (left.type != Node.tpStr) { 
			throw new CompileError("Left operand of IN expression hasn't string type",
					       leftPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opInString,
					 left, right);
		    break;
		  case Node.tpList:
		    compare(left, (BinOpNode)right);
		    left = right;
		    break;
		  default:
		    throw new CompileError("List of expressions or array expected", 
					   rightPos);
		}
	    } else if (cop == tknBetween) { 
		int andPos = pos;
		if (lex != tknAnd) { 
		    throw new CompileError("AND expected", pos);
		}
		Node right2 = addition();
		if (left.type == Node.tpReal 
		    || right.type == Node.tpReal || right2.type == Node.tpReal)
		{
		    if (left.type == Node.tpInt) { 
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("operand of BETWEEN operator should be of integer, real or string type", 
					       leftPos);
		    }
		    if (right.type == Node.tpInt) {
			right = int2real(right);
		    } else if (right.type != Node.tpReal) { 
			throw new CompileError("operand of BETWEEN operator should be of integer, real or string type", 
					       rightPos);
		    }
		    if (right2.type == Node.tpInt) {
			right2 = int2real(right2);
		    } else if (right2.type != Node.tpReal) { 
			throw new CompileError("operand of BETWEEN operator should be of integer, real or string type", 
					       andPos);
		    }
		    left = new CompareNode(Node.opRealBetween, 
					   left, right, right2);
		} 
		else if (left.type == Node.tpInt 
			 && right.type == Node.tpInt 
			 && right2.type == Node.tpInt)
		{		    
		    left = new CompareNode(Node.opIntBetween, 
					   left, right, right2);
		}
		else if (left.type == Node.tpStr && right.type == Node.tpStr 
			 && right2.type == Node.tpStr)
		{
		    left = new CompareNode(Node.opStrBetween, 
					   left, right, right2);
		} else { 
		    throw new CompileError("operands of BETWEEN operator should be of integer, real or string type", 
					   rightPos);
		}
	    } else if (cop == tknLike) {  
		if (left.type != Node.tpStr || right.type != Node.tpStr) { 
		    throw new CompileError("operands of LIKE operator should be of string type", 
					   rightPos);
		}
		if (lex == tknEscape) { 
		    rightPos = pos;
		    if (scan() != tknSconst) { 
			throw new CompileError("String literal espected after ESCAPE", rightPos);
		    }
		    left = new CompareNode(Node.opStrLikeEsc, 
					   left, right,
					   new StrLiteralNode(svalue));
		    lex = scan();
		} else { 
		    left = new CompareNode(Node.opStrLike, left, right, null);
		}
	    } else { 
		if (left.type == Node.tpReal || right.type == Node.tpReal) { 
		    if (left.type == Node.tpInt) { 
			left = int2real(left);
		    } else if (left.type != Node.tpReal) { 
			throw new CompileError("operands of relation operator should be of intger, real or string type", 
					       leftPos);
		    }
		    if (right.type == Node.tpInt) { 
			right = int2real(right);
		    } else if (right.type != Node.tpReal) { 
			throw new CompileError("operands of relation operator should be of intger, real or string type", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opRealEq+cop-tknEq, 
					 left, right);
		} else if (left.type == Node.tpInt 
			   && right.type == Node.tpInt) 
		{ 
		    left = new BinOpNode(Node.tpBool, Node.opIntEq+cop-tknEq, 
					 left, right);
		} 
		else if (left.type == Node.tpStr && right.type == Node.tpStr) 
		{
		    left = new BinOpNode(Node.tpBool, Node.opStrEq+cop-tknEq, 
					 left, right);
		} 
		else if (left.type == Node.tpObj && right.type == Node.tpObj)
		{ 
		    if (cop != tknEq && cop != tknNe) { 
			throw new CompileError("References can be checked only for equality", 
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opObjEq+cop-tknEq, 
					 left, right);
		} else if (left.type == Node.tpBool 
			   && right.type == Node.tpBool) 
		{ 
		    if (cop != tknEq && cop != tknNe) { 
			throw new CompileError("Boolean variables can be checked only for equality",
					       rightPos);
		    }
		    left = new BinOpNode(Node.tpBool, Node.opBoolEq+cop-tknEq, 
					 left, right);
		} else { 
		    throw new CompileError("operands of relation operator should be of integer, real or string type", 
					   rightPos);
		}
	    }
	    if (not) { 
		left = new UnaryOpNode(Node.tpBool, Node.opBoolNot, left);
	    }
	}
	return left;
    }


    final Node addition() { 
	int leftPos = pos;
	Node left = multiplication();
	while (lex == tknAdd || lex == tknSub) { 
	    int cop = lex;
	    int rightPos = pos;
	    Node right = multiplication();
	    if (left.type == Node.tpReal || right.type == Node.tpReal) { 
		if (left.type == Node.tpInt) { 
		    left = int2real(left);
		} else if (left.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operators should be of integer or real type", 
					   leftPos);
		}
		if (right.type == Node.tpInt) { 
		    right = int2real(right);
		} else if (right.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operator should be of integer or real type", 
					   rightPos);
		}
		left = new BinOpNode(Node.tpReal, cop == tknAdd 
				     ? Node.opRealAdd : Node.opRealSub,
				     left, right);
	    } 
	    else if (left.type == Node.tpInt && right.type == Node.tpInt) 
	    { 
		left = new BinOpNode(Node.tpInt, cop == tknAdd 
				     ? Node.opIntAdd : Node.opIntSub,
				     left, right);
	    } else if (left.type == Node.tpStr && right.type == Node.tpStr) { 
		if (cop == tknAdd) { 
		    left = new BinOpNode(Node.tpStr, Node.opStrConcat, 
					 left, right);
		} else { 
		    throw new CompileError("Operation - is not defined for strings", 
					   rightPos);
		}
	    } else { 
		throw new CompileError("operands of arithmentic operator should be of integer or real type", 
				       rightPos);
	    }
	    leftPos = rightPos;
	}
	return left;
    }


    final Node multiplication() { 
	int leftPos = pos;
	Node left = power();
	while (lex == tknMul || lex == tknDiv) { 
	    int cop = lex;
	    int rightPos = pos;
	    Node right = power();
	    if (left.type == Node.tpReal || right.type == Node.tpReal) { 
		if (left.type == Node.tpInt) { 
		    left = int2real(left);
		} else if (left.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operators should be of integer or real type", 
					   leftPos);
		}
		if (right.type == Node.tpInt) { 
		    right = int2real(right);
		} else if (right.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operator should be of integer or real type", 
					   rightPos);
		}
		left = new BinOpNode(Node.tpReal, cop == tknMul 
				     ? Node.opRealMul : Node.opRealDiv,
				     left, right);
	    } else if (left.type == Node.tpInt && right.type == Node.tpInt) { 
		left = new BinOpNode(Node.tpInt, cop == tknMul 
				     ? Node.opIntMul : Node.opIntDiv,
				     left, right);
	    } else { 
		throw new CompileError("operands of arithmentic operator should be of integer or real type", 
				       rightPos);
	    }
	    leftPos = rightPos;
	}
	return left;
    }


    final Node power() { 
	int leftPos = pos;
	Node left = term();
	if (lex == tknPower) { 
	    int rightPos = pos;
	    Node right = power();
	    if (left.type == Node.tpReal || right.type == Node.tpReal) { 
		if (left.type == Node.tpInt) { 
		    left = int2real(left);
		} else if (left.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operators should be of integer or real type", 
					   leftPos);
		}
		if (right.type == Node.tpInt) { 
		    right = int2real(right);
		} else if (right.type != Node.tpReal) { 
		    throw new CompileError("operands of arithmetic operator should be of integer or real type", 
					   rightPos);
		}
		left = new BinOpNode(Node.tpReal, Node.opRealPow, left, right);
	    } else if (left.type == Node.tpInt && right.type == Node.tpInt) { 
		left = new BinOpNode(Node.tpInt, Node.opIntPow, left, right);
	    } else { 
		throw new CompileError("operands of arithmentic operator should be of integer or real type", 
				       rightPos);
	    }
	}
	return left;
    }

    final Node component(Node base, Class cls) { 
	String c;
	Field f = null;
	Class scope;
	for (scope = cls; scope != null; scope = scope.getSuperclass()) { 
	    try { 
		f = scope.getDeclaredField(ident);
		break;
	    } catch(Exception x) {} 
	}
	if (f == null) { 
	    Method m = null;
	    for (scope = cls; scope != null; scope = scope.getSuperclass()) { 
		try { 
		    m = scope.getDeclaredMethod(ident, profile); 
		    break;
		} catch(Exception x) {}
	    }
	    if (m == null) {
		throw new CompileError("No field '"+ident+"' in class "+
				       cls.getName(), pos);
	    }
	    if (setBypass != null) { 
		try { 
		    setBypass.invoke(m, bypassFlag);
		} catch(Exception x) {
		    throw new CompileError("Field '" + ident
					   + "' is not accessible", pos);
		}
	    }  
	    Class rt = m.getReturnType();
	    c = rt.getName();
	    if (c.equals("byte") || c.equals("char") || c.equals("int")
		|| c.equals("short") || c.equals("long")) 
	    { 
		return new InvokeNode(Node.tpInt, m, base);
	    } else if (c.equals("float") || c.equals("double")) {
		return new InvokeNode(Node.tpReal, m, base);
	    } else if (c.equals("java.lang.String")) { 
		return new InvokeNode(Node.tpStr, m, base);
	    } else if (c.equals("boolean")) { 
		return new InvokeNode(Node.tpBool, m, base);
	    } else if (c.equals("[B")) { 
		return new InvokeNode(Node.tpArrayInt1, m, base);
	    } else if (c.equals("[S")) { 
		return new InvokeNode(Node.tpArrayInt2, m, base);
	    } else if (c.equals("[I")) { 
		return new InvokeNode(Node.tpArrayInt4, m, base);
	    } else if (c.equals("[J")) { 
		return new InvokeNode(Node.tpArrayInt8, m, base);
	    } else if (c.equals("[F")) { 
		return new InvokeNode(Node.tpArrayReal4, m, base);
	    } else if (c.equals("[D")) { 
		return new InvokeNode(Node.tpArrayReal8, m, base);
	    } else if (c.equals("[Z")) { 
		return new InvokeNode(Node.tpArrayBool, m, base);
	    } else if (c.equals("[C")) {
		return new InvokeNode(Node.tpArrayChar, m, base);
	    } else if (c.equals("[Ljava.lang.String;")) { 
		return new InvokeNode(Node.tpArrayStr, m, base);
	    } else if (rt.isArray()) { 
		return new InvokeNode(Node.tpArrayObj, m, base);
	    } else {
		return new InvokeNode(Node.tpObj, m, base);
	    }
	}	  
	if (setBypass != null) { 
	    try { 
		setBypass.invoke(f, bypassFlag);
	    } catch(Exception x) {
		throw new CompileError("Field '"+ident+"' is not accessible", 
				       pos);
	    }
	}  
	Class type = f.getType();
	c = type.getName();
	if (c.equals("byte")) { 
	    return new LoadNode(Node.tpInt, Node.opLoadInt1, f, base);
	}
	if (c.equals("[B")) { 
	    return new LoadNode(Node.tpArrayInt1, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfByte")) { 
	    return new LoadNode(Node.tpDnmArrayInt1, Node.opLoadObj, f, base);
	}
	if (c.equals("short")) { 
	    return new LoadNode(Node.tpInt, Node.opLoadInt2, f, base);
	}
	if (c.equals("[S")) { 
	    return new LoadNode(Node.tpArrayInt2, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfShort")) { 
	    return new LoadNode(Node.tpDnmArrayInt2, Node.opLoadObj, f, base);
	}
	if (c.equals("int")) { 
	    return new LoadNode(Node.tpInt, Node.opLoadInt4, f, base);
	}
	if (c.equals("[I")) { 
	    return new LoadNode(Node.tpArrayInt4, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfInt")) { 
	    return new LoadNode(Node.tpDnmArrayInt4, Node.opLoadObj, f, base);
	}
	if (c.equals("long")) { 
	    return new LoadNode(Node.tpInt, Node.opLoadInt8, f, base);
	}
	if (c.equals("[J")) { 
	    return new LoadNode(Node.tpArrayInt8, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfLong")) { 
	    return new LoadNode(Node.tpDnmArrayInt8, Node.opLoadObj, f, base);
	}
	if (c.equals("float")) { 
	    return new LoadNode(Node.tpReal, Node.opLoadReal4, f, base);
	}
	if (c.equals("[F")) { 
	    return new LoadNode(Node.tpArrayReal4, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfFloat")) { 
	    return new LoadNode(Node.tpDnmArrayReal4, Node.opLoadObj, f, base);
	}
	if (c.equals("double")) { 
	    return new LoadNode(Node.tpReal, Node.opLoadReal8, f, base);
	} 
	if (c.equals("[D")) { 
	    return new LoadNode(Node.tpArrayReal8, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfDouble")) { 
	    return new LoadNode(Node.tpDnmArrayReal8, Node.opLoadObj, f, base);
	}
	if (c.equals("boolean")) { 
	    return new LoadNode(Node.tpBool, Node.opLoadBool, f, base);
	} 
	if (c.equals("[Z")) { 
	    return new LoadNode(Node.tpArrayBool, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfBoolean")) { 
	    return new LoadNode(Node.tpDnmArrayBool, Node.opLoadObj, f, base);
	}
	if (c.equals("char")) { 
	    return new LoadNode(Node.tpInt, Node.opLoadChar, f, base);
	} 
	if (c.equals("[C")) {
	    return new LoadNode(Node.tpArrayChar, Node.opLoadObj, f, base);
	}
	if (c.equals("goodslib.ArrayOfChar")) { 
	    return new LoadNode(Node.tpDnmArrayChar, Node.opLoadObj, f, base);
	}
	if (c.equals("java.lang.String")) { 
	    return new LoadNode(Node.tpStr, Node.opLoadObj, f, base);
	} 
	if (c.equals("[Ljava.lang.String;")) { 
	    return new LoadNode(Node.tpArrayStr, Node.opLoadObj, f, base);
	}
	if (type.isArray()) { 
	    return new LoadNode(Node.tpArrayObj, Node.opLoadObj, f, base);
	}	    
	if (arrayOfObject.isAssignableFrom(type)) { 
	    return new LoadNode(Node.tpDnmArrayObj, Node.opLoadObj, f, base);
	}	    
	return new LoadNode(Node.tpObj, Node.opLoadObj, f, base);
    }


    final Node field(Node expr) {
	int p = pos;
	int type;
	int tag;
	Class cls = expr.getType();
	while (true) { 
	    switch (lex) {
	      case tknDot:	
		if (scan() != tknIdent) { 
		    throw new CompileError("identifier expected", p);
		}
		if (expr.type != Node.tpObj) { 
		    throw new CompileError("Left operand of '.' should be structure or reference", 
					   p);
		}
		expr = component(expr, cls);
		cls = expr.getType();
		break;
	      case tknLbr:
		switch (expr.type) { 
		  case Node.tpArrayBool:
		    tag = Node.opGetAtBool;
		    type = Node.tpBool;
		    break;
		  case Node.tpArrayChar:
		    tag = Node.opGetAtChar;
		    type = Node.tpInt;
		    break;
		  case Node.tpStr:
		    tag = Node.opStrGetAt;
		    type = Node.tpInt;
		    break;
		  case Node.tpArrayInt1:
		    tag = Node.opGetAtInt1;
		    type = Node.tpInt;
		    break;
		  case Node.tpArrayInt2:
		    tag = Node.opGetAtInt2;
		    type = Node.tpInt;
		    break;
		  case Node.tpArrayInt4:
		    tag = Node.opGetAtInt4;
		    type = Node.tpInt;
		    break;
		  case Node.tpArrayInt8:
		    tag = Node.opGetAtInt8;
		    type = Node.tpInt;
		    break;
		  case Node.tpArrayReal4:
		    tag = Node.opGetAtReal4;
		    type = Node.tpReal;
		    break;
		  case Node.tpArrayReal8:
		    tag = Node.opGetAtReal8;
		    type = Node.tpReal;
		    break;
		  case Node.tpArrayStr:
		    tag = Node.opGetAtStr;
		    type = Node.tpStr;
		    break;
		  case Node.tpArrayObj:
		    tag = Node.opGetAtObj;
		    cls = cls.getComponentType();
		    type = cls.isArray() ? Node.tpArrayObj : Node.tpObj;
		    break;
		  case Node.tpDnmArrayBool:
		    tag = Node.opDnmGetAtBool;
		    type = Node.tpBool;
		    break;
		  case Node.tpDnmArrayChar:
		    tag = Node.opDnmGetAtChar;
		    type = Node.tpInt;
		    break;
		  case Node.tpDnmArrayInt1:
		    tag = Node.opDnmGetAtInt1;
		    type = Node.tpInt;
		    break;
		  case Node.tpDnmArrayInt2:
		    tag = Node.opDnmGetAtInt2;
		    type = Node.tpInt;
		    break;
		  case Node.tpDnmArrayInt4:
		    tag = Node.opDnmGetAtInt4;
		    type = Node.tpInt;
		    break;
		  case Node.tpDnmArrayInt8:
		    tag = Node.opDnmGetAtInt8;
		    type = Node.tpInt;
		    break;
		  case Node.tpDnmArrayReal4:
		    tag = Node.opDnmGetAtReal4;
		    type = Node.tpReal;
		    break;
		  case Node.tpDnmArrayReal8:
		    tag = Node.opDnmGetAtReal8;
		    type = Node.tpReal;
		    break;
		  case Node.tpDnmArrayObj:
		    tag = Node.opDnmGetAtObj;
		    try { 
			Method m = cls.getMethod("get", getAtProfile);
			cls = m.getReturnType();
		    } catch(Exception x) { 
			cls = null;
		    } 
		    type = Node.tpObj;
		    break;
		  default: 
		    throw new CompileError("Index can be applied only to arrays", 
					   p);
		}
		p = pos;
		Node index = disjunction();
		if (lex != tknRbr) { 
		    throw new CompileError("']' expected", pos);
		}
		if (index.type != Node.tpInt && 
		    index.type != Node.tpFreeVar) 
		{
		    throw new CompileError("Index should have integer type",p);
		}
		expr = new GetAtNode(type, tag, expr, index);
		break;
	      default:
		return expr;
	    }	
	    lex = scan();
	}
    }


    final Node term() {
	int cop = scan();
	int p = pos;
	Node expr;
	Binding bp;
	switch (cop) { 
	  case tknEof:
	  case tknOrder:
	    lex = cop;
	    return new EmptyNode();
	  case tknIdent:
	    for (bp = bindings; bp != null; bp = bp.next) { 
		if (bp.name.equals(ident)) { 
		    lex = scan();
		    bp.used = true;
		    return new IndexNode(bp.loopId);
		}
	    }
	    expr = component(null, cls);
	    lex = scan();
	    return field(expr);
	  case tknExists:
	    if (scan() != tknIdent) { 
		throw new CompileError("Free variable name expected", p);
	    }	    
	    bindings = bp = new Binding(ident, vars++, bindings);
	    if (vars >= SearchThread.maxIndexVars) { 
		throw new CompileError("Too many nested EXISTS clauses", p);
	    }
	    p = pos;
	    if (scan() != tknCol) { 
		throw new CompileError("':' expected", p);
	    }
	    p = pos;
	    expr = term();
	    if (expr.type != Node.tpBool) { 
		throw new CompileError("Expresion in EXISTS clause should be of boolean type", 
				       p);
	    }
	    if (bp.used) { 
		expr = new ExistsNode(expr, vars-1);
	    }
	    vars -= 1;	    
	    bindings = bp.next;
	    return expr;
	  case tknCurrent:
	    lex = scan();
	    return field(new ConstantNode(Node.tpObj, Node.opCurrent));
	  case tknFalse:
	    expr = new ConstantNode(Node.tpBool, Node.opFalse);
	    break;
	  case tknTrue:
	    expr = new ConstantNode(Node.tpBool, Node.opTrue);
	    break;
	  case tknNull:
	    expr = new ConstantNode(Node.tpObj, Node.opNull);
	    break;
	  case tknIconst:
	    expr = new IntLiteralNode(ivalue);
	    break;
	  case tknFconst:
	    expr = new RealLiteralNode(fvalue);
	    break;
	  case tknSconst:
	    expr = new StrLiteralNode(svalue);
	    lex = scan();
	    return field(expr); 
	  case tknSin:
	  case tknCos:
	  case tknTan:
	  case tknAsin:
	  case tknAcos:
	  case tknAtan:
	  case tknExp:
	  case tknLog:
	  case tknSqrt:
	  case tknCeil:
	  case tknFloor:
	    expr = term();
	    if (expr.type == Node.tpInt) { 
		expr = int2real(expr);
	    } else if (expr.type != Node.tpReal) { 
		throw new CompileError("Numeric argument expected", p);
	    }
	    return new UnaryOpNode(Node.tpReal, cop+Node.opRealSin-tknSin, 
				   expr);
	  case tknAbs:
	    expr = term();
	    if (expr.type == Node.tpInt) { 
		return new UnaryOpNode(Node.tpInt, Node.opIntAbs, expr);
	    } else if (expr.type == Node.tpReal) { 
		return new UnaryOpNode(Node.tpReal, Node.opRealAbs, expr);
	    } else { 
		throw new CompileError("ABS function can be applied only to integer or real expression", 
				       p);
	    }
	  case tknLength:
	    expr = term();
	    if (expr.type == Node.tpStr) { 
		return new UnaryOpNode(Node.tpInt, Node.opStrLength, expr);
	    } else if (expr.type >= Node.tpDnmArrayBool) { 
		return new UnaryOpNode(Node.tpInt, Node.opDnmLength, expr);
	    } else if (expr.type >= Node.tpArrayBool) { 
		return new UnaryOpNode(Node.tpInt, Node.opLength, expr);
	    } else { 
		throw new CompileError("LENGTH function is defined only for arrays and strings",
				       p);
	    } 
	  case tknLower:
	    expr = term();
	    if (expr.type == Node.tpStr) { 
		return field(new UnaryOpNode(Node.tpStr,Node.opStrLower,expr));
	    }
	    throw new CompileError("LOWER function can be applied only to string argument",
				       p);
	  case tknUpper:
	    expr = term();
	    if (expr.type == Node.tpStr) { 
		return field(new UnaryOpNode(Node.tpStr,Node.opStrUpper,expr));
	    }
	    throw new CompileError("UPPER function can be applied only to string argument",
				       p);
	  case tknInteger:
	    expr = term();
	    if (expr.type == Node.tpReal) { 
		return new UnaryOpNode(Node.tpInt, Node.opRealToInt, expr);
	    }  
	    throw new CompileError("INTEGER function can be applied only to expression of real type", 
				   p);
	  case tknReal:
	    expr = term();
	    if (expr.type == Node.tpInt) { 
		return new UnaryOpNode(Node.tpReal, Node.opIntToReal, expr);
	    }  
	    throw new CompileError("REAL function can be applied only to expression of integer type", 
				   p);
	  case tknString:
	    expr = term();
	    if (expr.type == Node.tpInt) { 
		return field(new UnaryOpNode(Node.tpStr, Node.opIntToStr, 
					     expr));
	    } else if (expr.type == Node.tpReal) { 
		return field(new UnaryOpNode(Node.tpStr, Node.opRealToStr, 
					     expr));
	    }
	    throw new CompileError("STRING function can be applied only to integer or real expression", 
				   p);
	  case tknLpar:
	  {
	    expr = disjunction();
	    Node list = null;
	    while (lex == tknComma) { 
		list = new BinOpNode(Node.tpList, Node.opNop, list, expr);
		expr = disjunction();
	    }
	    if (lex != tknRpar) { 
		throw new CompileError("')' expected", pos);
	    }
	    if (list != null) { 
		expr = new BinOpNode(Node.tpList, Node.opNop, list, expr);
	    }
	    break;
	  }
	  case tknNot:
	    expr = comparison();
	    if (expr.type == Node.tpInt) { 
		if (expr.tag == Node.opIntConst) { 
		    IntLiteralNode ic = (IntLiteralNode)expr;
		    ic.value = ~ic.value;
		} else {
		    expr = new UnaryOpNode(Node.tpInt, Node.opIntNot, expr);
		} 
		return expr;
	    } else if (expr.type == Node.tpBool) { 
		return new UnaryOpNode(Node.tpBool, Node.opBoolNot, expr);
	    } else { 
		throw new CompileError("NOT operator can be applied only to integer or boolean expressions", 
				       p);
	    }
	  case tknAdd:
	    throw new CompileError("Using of unary plus operator has no sense",
				   p);
	  case tknSub:
	    expr = term();
	    if (expr.type == Node.tpInt) { 
		if (expr.tag == Node.opIntConst) { 
		    IntLiteralNode ic = (IntLiteralNode)expr;
		    ic.value = -ic.value;
		} else {
		    expr = new UnaryOpNode(Node.tpInt, Node.opIntNeg, expr);
		} 
	    } else if (expr.type == Node.tpReal) { 
		if (expr.tag == Node.opRealConst) { 
		    RealLiteralNode fc = (RealLiteralNode)expr;
		    fc.value = -fc.value;
		} else {
		    expr = new UnaryOpNode(Node.tpReal, Node.opRealNeg, expr);
		} 
	    } else { 
		throw new CompileError("Unary minus can be applied only to numeric expressions", 
				       p);
	    }
	    return expr;
	  default:
	    throw new CompileError("operand expected", p);
	}
	lex = scan();
	return expr;
    }

    final boolean isIndexApplicable(Node expr) { 
	if (expr instanceof BinOpNode) { 
	    BinOpNode cmp = (BinOpNode)expr;
	    switch (expr.tag) { 
	      case Node.opStrEq:
		if (cmp.left.tag == Node.opStrConst &&
		    cmp.right instanceof LoadNode &&
		    ((LoadNode)cmp.right).base == null)
		{
		    try { 
			obj = iterator.getByKey
			    (((LoadNode)cmp.right).field.getName(), 
			     ((StrLiteralNode)cmp.left).value);
			return true;
		    } catch (NoIndexException x) { 
			return false;
		    }
		}
		if (cmp.right.tag == Node.opStrConst &&
		    cmp.left instanceof LoadNode &&
		    ((LoadNode)cmp.left).base == null)
		{
		    try { 
			obj = iterator.getByKey
			    (((LoadNode)cmp.left).field.getName(), 
			     ((StrLiteralNode)cmp.right).value);
			return true;
		    } catch (NoIndexException x) { 
			return false;
		    }
		}
		return false;
	      case Node.opIntEq:
		if (cmp.left.tag == Node.opIntConst &&
		    cmp.right instanceof LoadNode &&
		    ((LoadNode)cmp.right).base == null)
		{
		    try { 
			if (cmp.right.tag != Node.opLoadInt8) { 
			    ((IntLiteralNode)cmp.left).value &= 0xffffffff;
			}
			obj = iterator.getByKey
			    (((LoadNode)cmp.right).field.getName(), cmp.left);
			return true;
		    } catch (NoIndexException x) { 
			return false;
		    }
		}
		if (cmp.right.tag == Node.opIntConst &&
		    cmp.left instanceof LoadNode &&
		    ((LoadNode)cmp.left).base == null)
		{
		    try { 
			if (cmp.left.tag != Node.opLoadInt8) { 
			    ((IntLiteralNode)cmp.right).value &= 0xffffffff;
			}
			obj = iterator.getByKey
			    (((LoadNode)cmp.left).field.getName(), cmp.right);
			return true;
		    } catch (NoIndexException x) { 
			return false;
		    }
		}
		return false;
	    }
	}
	return false;
    }


    final void compile() {
	pos = 0;
	vars = 0;
	tree = disjunction();
	if (tree.type != Node.tpBool) { 
	    throw new CompileError("Boolean expression expected", pos);
	}
	OrderNode last = null;
	order = null;
	if (lex == tknEof) {	
	    return;
	}
	if (lex != tknOrder) { 
	    throw new CompileError("ORDER BY expected", pos);
	}
	int tkn;
	int p = pos;
	if (scan() != tknBy) { 
	    throw new CompileError("BY expected after ORDER", p);
	}
	do { 
	    p = pos;
	    if (scan() != tknIdent) { 
		throw new CompileError("field name expected", p);
	    }
	    Field f = null;
	    for (Class scope=cls; scope != null; scope=scope.getSuperclass()) {
		try { 
		    f = scope.getDeclaredField(ident);
		    break;
		} catch(Exception x) {} 
	    }

	    if (f == null) { 
		throw new CompileError("No field '"+ident+"' in class "+
				       cls.getName(), p);
	    }
	    if (setBypass != null) { 
		try { 
		    setBypass.invoke(f, bypassFlag);
		} catch(Exception x) {
		    throw new CompileError("Field '" + ident
					   + "' is not accessible", p);
		}
	    }  
	    int type;
	    String c = f.getType().getName();
	    if (c.equals("byte")) { 
		type = OrderNode.fdInt1;
	    } else if (c.equals("short")) { 
		type = OrderNode.fdInt2;
	    } else if (c.equals("int")) { 
		type = OrderNode.fdInt4;
	    } else if (c.equals("long")) { 
		type = OrderNode.fdInt8;
	    } else if (c.equals("float")) { 
		type = OrderNode.fdReal4;
	    } else if (c.equals("double")) { 
		type = OrderNode.fdReal8;
	    } else if (c.equals("boolean")) { 
		type = OrderNode.fdBool;
	    } else if (c.equals("char")) { 
		type = OrderNode.fdChar;
	    } else if (c.equals("java.lang.String")) { 
		type = OrderNode.fdStr;
	    } else { 
		throw new CompileError("Sort key should be of scalar or string type", 
				       p);
	    }
	    OrderNode node = new OrderNode(type, f);
	    if (last != null) { 
		last.next = node;
	    } else { 
		order = node;
	    }
	    last = node;
	    p = pos;
	    tkn = scan();
	    if (tkn == tknDesc) { 
		node.ascent = false;
		tkn = scan();
	    } else if (tkn == tknAsc) { 
		tkn = scan();
	    }
	} while (tkn == tknComma);
	if (tkn != tknEof) { 
	    throw new CompileError("',' expected", p);
	}
    }
}



